/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2009 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, 01 December, 2009
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.0, Nov 18, 2009
 * @author        Chittaranjan Tripathy (2007-2009)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */

/**
 * Description of the class
 */
public class myStericChecker {
    protected double __hydrogen_bond_overlap = 0.8; //TODO: ask Iv why it is 0.8
    protected double __soft_steric_overlap = 0.4;
    protected double __fatal_steric_overlap = 1.0;
    
    public myStericChecker() {        
    }
    
    public myStericChecker(double allowedStericThreshold) {
        __soft_steric_overlap = allowedStericThreshold;
    }
    
    public myStericChecker(double hBondThreshold, double allowedStericThreshold) {
        __hydrogen_bond_overlap = hBondThreshold;
        __soft_steric_overlap = allowedStericThreshold;
    }
    
    public myStericChecker(double fatalStericThresh, double hBondThreshold, double allowedStericThreshold) {
        __fatal_steric_overlap = fatalStericThresh;
        __hydrogen_bond_overlap = hBondThreshold;
        __soft_steric_overlap = allowedStericThreshold;
    }

    public double checkStericClash(myProtein p) {
        return checkStericClash(p, __soft_steric_overlap);
    }

    public double checkStericClash(myProtein p, double softStericOL) {
        int clashCount = 0;
        
//        myProtein.AtomIterator iter1;// = new myProtein.AtomIterator();//p.atomIterator();
//        Iterator<myAtom> iter2 = p.atomIterator();
//        while (iter.hasNext()) {
//            myAtom a = iter.next();
//            while (iter2.hasNext()) {
//                myAtom b = iter2.next();
        
        for (myResidue r1 : p) {
            for (myAtom a : r1) {
                for (myResidue r2 : p) { //if (r2.getResidueNumber() != r1.getResidueNumber())                
                    for (myAtom b : r2) {
                        if (a.getSerialNumber() < b.getSerialNumber()) {
                            if (hasStericClash(a, b, softStericOL)) {
                                clashCount++;
                                //System.out.print("Clash #" + clashCount + "\t: " + '\t' + a.getResidueName() + '\t' + a.getResidueNumber() + '\t' +
                                //        a.getAtomName() + '\t' + b.getResidueName() + '\t' + b.getResidueNumber() + '\t' +
                                //        b.getAtomName() + '\t' + (a.distance(b) - ((a.getVdwRadius() + b.getVdwRadius()) / 100)) + ' ' + a.getVdwRadius() + ' ' + b.getVdwRadius());
                                if (a.distance(b) < (a.getVdwRadius() + b.getVdwRadius()) / 100.0 - __fatal_steric_overlap) {
                                    //System.out.print("\tfatal steric clash!" + (a.distance(b) - ((a.getVdwRadius() + b.getVdwRadius()) / 100.0 - __fatal_steric_overlap)));
                                }
                                //System.out.println();
                            }

                        }
                    }
                }
            }
        }

        double clashScore = (clashCount + 0.0) / p.numberOfAtoms() * 1000; 
        //System.out.println("Clash Score: " + clashScore);
        return clashScore;
    }

    public boolean hasStericClash(myAtom a1, myAtom a2, double softStericOL) {
        
        if (a1.getAtomName().equals("N") && a1.getResidueNumber() == 24 && a2.getResidueNumber() == 25 && a2.getAtomName().equals("HD2")) {
                        System.out.print("found************************");
                                double dd = a1.distance(a2);
                                double vdw = (a1.getVdwRadius() + a2.getVdwRadius()) / 100.0 - softStericOL;
                                System.out.println("Actual Dist: " + dd + "Sum of vdw radii: " + ((a1.getVdwRadius() + a2.getVdwRadius()) / 100.0) + "    Allowed vdw cutoff Dist: " + vdw
                                       + nonBondedStericallyAccordingToProbe(a1, a2) );
        }
        
        double dist = a1.distance(a2);
        if (dist < (a1.getVdwRadius() + a2.getVdwRadius()) / 100.0 - softStericOL) { //There can be a potential clash
            if (nonBondedStericallyAccordingToProbe(a1, a2)) {
                if (HbondSuspected(a1, a2)) {
                    if (dist < (a1.getVdwRadius() + a2.getVdwRadius()) / 100.0 - __hydrogen_bond_overlap) {
                        return true;
                    }
                    else {
                        return false;
                    }
                }
                else {
                    return true;
                }
            }
            else {
                return false;
            }
        } else {
            return false;
        }
    }
    
    //In the Richardsons' Probe, for steric check purposes, two atoms are considered non-bonded if:
    //      1) They are more than 3 bonds apart if neither of the atoms is a hydrogen, or
    //      2) They are more than 4 bonds apart if at least one of the atoms is a hydrogen
    private boolean nonBondedStericallyAccordingToProbe(myAtom a1, myAtom a2) {
        if (a1.isEqualTo(a2) || a1.isBondedTo(a2))
            return false;
        // none of them H, so 4 bonds or more for them to be nonbonded
        // Do a depthFirstSearch with bounded depth
     
        // none of them H, so 4 bonds or more for them to be nonbonded
        if (!a1.getElementType().equalsIgnoreCase("H") && !a2.getElementType().equalsIgnoreCase("H")) {            
            for (myAtom atm1 : a1.getBondedAtoms()) {
                if (atm1.isEqualTo(a2))
                    return false;
                if (!atm1.isEqualTo(a1)) {
                    for (myAtom atm2 : atm1.getBondedAtoms()) {
                        if (atm2.isEqualTo(a2))
                            return false;
                        if (!atm2.isEqualTo(atm1) && !atm2.isEqualTo(a1)) {
                            for (myAtom atm3 : atm2.getBondedAtoms()) {
                                if (atm3.isEqualTo(a2))
                                    return false; // the two atoms are exactly 4 bonds apart
                            }
                        }
                    }
                }
            }            
        } 
        // at least one of them H, so 4 bonds or more for them to be nonbonded
        else if (a1.getElementType().equalsIgnoreCase("H") || a2.getElementType().equalsIgnoreCase("H")) {
            //System.out.println("here-1");
            for (myAtom atm1 : a1.getBondedAtoms()) {
                if (atm1.isEqualTo(a2))
                    return false;
                if (!atm1.isEqualTo(a1)) {
                    for (myAtom atm2 : atm1.getBondedAtoms()) {
                        if (atm2.isEqualTo(a2))
                            return false;
                        if (!atm2.isEqualTo(atm1) && !atm2.isEqualTo(a1)) {
                            for (myAtom atm3 : atm2.getBondedAtoms()) {
                                if (atm3.isEqualTo(a2))
                                    return false;
                                if (!atm3.isEqualTo(atm2) && !atm3.isEqualTo(atm1) && !atm3.isEqualTo(a1)) {
                                    for (myAtom atm4 : atm3.getBondedAtoms()) {
                                        if (atm4.isEqualTo(a2))
                                            return false;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        //System.out.println("here-2");
        return true;
    }

    private boolean HbondSuspected(myAtom a1, myAtom a2) {
        if (a1.isEqualTo(a2) || a1.isBondedTo(a2)) {
            return false;
        }
        if (a1.getAtomName().startsWith("H")) { //TODO: Consider sulphide bonded H too.
            if (a2.getAtomName().startsWith("N") || a2.getAtomName().startsWith("O") || a2.getAtomName().startsWith("S")) {
                return true;
            }
        }
        if (a2.getAtomName().startsWith("H")) { //TODO: Consider sulphide bonded H too.
            if (a1.getAtomName().startsWith("N") || a1.getAtomName().startsWith("O") || a1.getAtomName().startsWith("S")) {
                return true;
            }
        }
        return false;
    }
}
