/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2009 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, 01 December, 2009
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.0, Nov 18, 2009
 * @author        Chittaranjan Tripathy (2007-2009)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.util.*;
import java.util.regex.*;

/**
 * Description of the class
 */
public class myResidue implements Iterable<myAtom>, java.io.Serializable {

    protected Map<String, myAtom> __atoms = new TreeMap<String, myAtom>();
    protected Map<String, myAtom> __extremes = new TreeMap<String, myAtom>();
    protected String __residue_name = myMiscConstants.undefined; // Residue name in 3 letter IUPAC format.
    protected int __residue_number = 0; // Residue number in the current model in the PDB.
    protected boolean __is_terminal_residue = false; // TODO: this extension is to be done later
    //protected Vector<String, String> __bonds;
    // TODO: Other fields need to be added and then initialized, such as if a residue 
    // contains the noesy peaks or things like that. Perhaps a derived class would work better.
    protected static myResidue __ALA_residue = new myResidue("ALA");
    protected static myResidue __ARG_residue = new myResidue("ARG");
    protected static myResidue __ASN_residue = new myResidue("ASN");
    protected static myResidue __ASP_residue = new myResidue("ASP");
    protected static myResidue __CYS_residue = new myResidue("CYS");
    protected static myResidue __GLN_residue = new myResidue("GLN");
    protected static myResidue __GLU_residue = new myResidue("GLU");
    protected static myResidue __GLY_residue = new myResidue("GLY");
    protected static myResidue __HIS_residue = new myResidue("HIS");
    protected static myResidue __ILE_residue = new myResidue("ILE");
    protected static myResidue __LEU_residue = new myResidue("LEU");
    protected static myResidue __LYS_residue = new myResidue("LYS");
    protected static myResidue __MET_residue = new myResidue("MET");
    protected static myResidue __PHE_residue = new myResidue("PHE");
    protected static myResidue __PRO_residue = new myResidue("PRO");
    protected static myResidue __SER_residue = new myResidue("SER");
    protected static myResidue __THR_residue = new myResidue("THR");
    protected static myResidue __TRP_residue = new myResidue("TRP");
    protected static myResidue __TYR_residue = new myResidue("TYR");
    protected static myResidue __VAL_residue = new myResidue("VAL");
    //protected static myAminoAcidResidueMap AARMap = new myAminoAcidResidueMap();

    /**
     * Return a vector of PDB rows in the order found in PDB files.
     *
     * @return a vector of PDB rows in the order found in PDB files
     */
    public Vector<String> getResidueRowsInOrder() {
        Vector<String> res = new Vector<String>();
        for (String atomName : myAminoAcidResidueMap.__residue_atom_map.get(getResidueName())) {
            if (__atoms.containsKey(atomName)) {
                myAtom a = __atoms.get(atomName);
                myPdbResidueRow r = new myPdbResidueRow(a.getRecordName(), a.getSerialNumber(),
                        a.getAtomName(), a.getAlternateLocationIndicator(), getResidueName(),
                        a.getChainId(), getResidueNumber(), a.getInsertionCode(), a.getCoordinates(),
                        a.getOccupancy(), a.getTemperatureFactor(), a.getSegmentId(), a.getElementType(), a.getCharge());                
                res.add(r.getResidueRow());
            }
        }        
        // TODO: There has to be code for other non-default residue types (other than 20 amino acids) to go here
        return res;
    }
    
    /**
     * Print the residue in the order.
     */
    public void print() {        
        Vector<String> res = getResidueRowsInOrder();
        for (String row : res)
            System.out.println(row);
    }

    /**
     * Return an iterator that can iterate through the atoms in the residue.
     *
     * @return an iterator that can iterate through the atoms in the residue
     */
    public Iterator<myAtom> iterator() {
// Here we define an anonymous inner class for the Iterator<Atom>. 
// Here we actually don't need this as it suffices to use __atoms.values().iterator();
// If in future we need to implement an iterator that impose some ordering on the way
// it iterates then we can customize this by adding code to this template implementation.
//        return new Iterator<myAtom>() { 
//            private Iterator<myAtom> iter = __atoms.values().iterator();
//            
//            public boolean hasNext() {                
//                return iter.hasNext();
//            }
//            
//            public myAtom next() throws NoSuchElementException {
//                if (!iter.hasNext())
//                    throw new NoSuchElementException();       
//                return iter.next();
//            }
//            
//            public void remove() {
//                iter.remove();
//            }      
//        };
        return __atoms.values().iterator();
    }

    /**
     * Default constructor that constructs the default residue.
     */
    public myResidue() {
    }

    /**
     * Construct a residue with the residue name supplied as parameter.
     *
     * @param resName name of the residue
     */
    public myResidue(String resName) {
        __residue_name = resName;        
        initializeResidue();
    }

    /**
     * Construct a residue from a given residue type and residue number.
     *
     * @param resName residue name
     * @param resNum residue number
     */
    public myResidue(String resName, int resNum) {
        __residue_name = resName;
        __residue_number = resNum;
        initializeResidue();
    }

    /**
     * Construct a Residue from a block of rows representing a residue in PDB file.
     *
     * @param block a block of rows representing a residue in in PDB file
     */
    public myResidue(Vector<String> block) {
        __residue_name = block.elementAt(0).substring(17, 20);
        __residue_number = Integer.parseInt(block.elementAt(0).substring(22, 26).trim());
        //add atoms        
        myPdbResidueRow r;
        for (String thisRow : block) {
            r = new myPdbResidueRow(thisRow);
            myAtom a = r.toAtom();
            addAtom(a); // This automatically tests whether atom is not present and it is legal to add this atom, and then sets the bonds appropriately.
        }
    }

    /**
     * A copy constructor which takes an instance of a residue and constructs
     * another residue with the same parameters.
     *
     * @param r the residue to be cloned
     */
    public myResidue(final myResidue r) {
        __residue_name = r.getResidueName();
        __residue_number = r.getResidueNumber();
        for (myAtom a : r) {
            addAtom(new myAtom(a));
        }
    }

    /**
     * A static factory method which takes an instance of a residue and constructs
     * another residue with the same parameters.
     *
     * @param r the residue to be cloned
     */
    public static myResidue newInstance(final myResidue r) {
        return new myResidue(r);
    }

    /**
     * Return the residue name.
     * 
     * @return the residue name
     */
    public String getResidueName() {
        return __residue_name;
    }

    //public void setResidueName(String resName) { // TODO: if you want to implement this then update the atoms and residue properties correctly.
    //    __residue_name = resName;
    //}

    /**
     * Return the residue number.
     *
     * @return the residue number
     */
    public int getResidueNumber() {
        return __residue_number;
    }

    /**
     * Modify the residue number.
     *
     * @param resNum the new residue number
     */
    public void setResidueNumber(int resNum) {
        __residue_number = resNum;
        for (myAtom a : this)
            a.setResidueNumber(resNum);
    }

    /**
     * Mutate the residue to the new residue whose type is provided in argument.
     *
     * @param resName the new residue name
     */
    public void mutate(String resName) {
        if (this.getResidueName().equalsIgnoreCase(resName)) {
            return;
        }
        
        if (this.getResidueName().equalsIgnoreCase("PRO") && !resName.equalsIgnoreCase("PRO")) {
            System.out.println("Error: Mutation from PRO is yet to be implemented");
            System.exit(1);
        }

//        if (!this.getResidueName().equalsIgnoreCase("PRO") && resName.equalsIgnoreCase("PRO")) {
//            System.out.println("Error: Mutation to PRO is yet to be implemented");
//            System.exit(1);
//        }        
        
        __residue_name = resName;
        for (myAtom a: this) {
            a.setResidueName(resName);
        }

        // Mutate to GLY
        if (__residue_name.equalsIgnoreCase("GLY")) { // Note that the atom is removed and then added back, as what we are doing here is changing the key
            if (hasAtom(myAtomLabel.__HA)) {
                myAtom ac = getAtom(myAtomLabel.__HA);
                this.removeAtom(myAtomLabel.__HA);
                ac.setAtomName(myAtomLabel.__HA2);
                this.addAtom(ac);
            }
            if (hasAtom(myAtomLabel.__CB)) { // Note that the atom is removed and then added back, as what we are doing here is changing the key
                myAtom ac = getAtom(myAtomLabel.__CB);
                this.removeAtom(myAtomLabel.__CB);
                ac.setAtomName(myAtomLabel.__HA3);
                this.addAtom(ac);
            }
        }

        // Mutate from GLY
        if (!__residue_name.equalsIgnoreCase("GLY")) { // Note that the atom is removed and then added back, as what we are doing here is changing the key
            if (hasAtom(myAtomLabel.__HA2)) {
                myAtom ac = getAtom(myAtomLabel.__HA2);
                this.removeAtom(myAtomLabel.__HA2);
                ac.setAtomName(myAtomLabel.__HA);
                this.addAtom(ac);
            }
            if (hasAtom(myAtomLabel.__HA3)) {
                myAtom ac = getAtom(myAtomLabel.__HA3);
                this.removeAtom(myAtomLabel.__HA3);
                ac.setAtomName(myAtomLabel.__CB);
                this.addAtom(ac);
            }
        }

        // Remove illegal atoms for this residue
        Vector<myAtom> listOfIllegalAtoms = new Vector<myAtom>();
        for (myAtom a : this) {
            if (!this.canHaveAtom(a.getAtomName())) {
                listOfIllegalAtoms.add(a);
            }
        }
        for (myAtom a : listOfIllegalAtoms) {
            this.removeAtom(a.getAtomName());
        }
    }

    /**
     * Return the number of atoms in the residue.
     *
     * @return the number of atoms in the residue
     */
    public int numberOfAtoms() {
        return __atoms.size();
    }

    /**
     * Return the number of bonds in the residue.
     *
     * @return the number of bonds in the residue
     */
    public int numberOfBonds() {
        int noOfBonds = 0;
        for (String atomName : __atoms.keySet()) {
            myAtom a = __atoms.get(atomName);
            noOfBonds += a.numberOfBonds();
        }
        return noOfBonds / 2 + 1; // +1 is valid when the residue is connected at both the ends    
    }

    /**
     * Return true if the residue has the atom whose name is supplied as argument.
     *
     * @param IUPAC_atom_name the atom name
     * @return true if the residue contains the atom with the name specified
     */
    public boolean hasAtom(String IUPAC_atom_name) {
        return __atoms.containsKey(IUPAC_atom_name);
    }

    /**
     * Returns true if the residue type is one of the 20 naturally occuring
     * amino acid residues and can contain this atom. If the residue type is
     * different then return true by default.
     *
     * @param IUPAC_atom_name
     * @return true if the residue type is one of the 20 naturally occuring amino
     * acid residues and can contain this atom. If the residue type is different
     * then return true by default.
     */
    public boolean canHaveAtom(String IUPAC_atom_name) { 
        String resName = __residue_name;
        if (/*AARMap*/myAminoAcidResidueMap.validResidueName(resName)) {
            if (resName.equalsIgnoreCase("ALA")) { //10 atoms              
                return __ALA_residue.hasAtom(IUPAC_atom_name);
            } else if (resName.equalsIgnoreCase("ARG")) { //24 atoms
                return __ARG_residue.hasAtom(IUPAC_atom_name);
            } else if (resName.equalsIgnoreCase("ASN")) { //14 atoms 
                return __ASN_residue.hasAtom(IUPAC_atom_name);
            } else if (resName.equalsIgnoreCase("ASP")) { //12 atoms
                return __ASP_residue.hasAtom(IUPAC_atom_name);
            } else if (resName.equalsIgnoreCase("CYS")) { //11 atoms
                return __CYS_residue.hasAtom(IUPAC_atom_name);
            } else if (resName.equalsIgnoreCase("GLN")) { //17 atoms
                return __GLN_residue.hasAtom(IUPAC_atom_name);
            } else if (resName.equalsIgnoreCase("GLU")) { //15 atoms
                return __GLU_residue.hasAtom(IUPAC_atom_name);
            } else if (resName.equalsIgnoreCase("GLY")) { //7 atoms
                return __GLY_residue.hasAtom(IUPAC_atom_name);
            } else if (resName.equalsIgnoreCase("HIS")) {//18 atoms
                return __HIS_residue.hasAtom(IUPAC_atom_name);
            } else if (resName.equalsIgnoreCase("ILE")) {//19 atoms
                return __ILE_residue.hasAtom(IUPAC_atom_name);
            } else if (resName.equalsIgnoreCase("LEU")) {//19 atoms
                return __LEU_residue.hasAtom(IUPAC_atom_name);
            } else if (resName.equalsIgnoreCase("LYS")) {//22 atoms
                return __LYS_residue.hasAtom(IUPAC_atom_name);
            } else if (resName.equalsIgnoreCase("MET")) {//17 atoms
                return __MET_residue.hasAtom(IUPAC_atom_name);
            } else if (resName.equalsIgnoreCase("PHE")) {//20 atoms
                return __PHE_residue.hasAtom(IUPAC_atom_name);
            } else if (resName.equalsIgnoreCase("PRO")) {//14 atoms
                return __PRO_residue.hasAtom(IUPAC_atom_name);
            } else if (resName.equalsIgnoreCase("SER")) {//11 atom
                return __SER_residue.hasAtom(IUPAC_atom_name);
            } else if (resName.equalsIgnoreCase("THR")) {//14 atoms
                return __THR_residue.hasAtom(IUPAC_atom_name);
            } else if (resName.equalsIgnoreCase("TRP")) {//24 atoms
                return __TRP_residue.hasAtom(IUPAC_atom_name);
            } else if (resName.equalsIgnoreCase("TYR")) {//21 atoms
                return __TYR_residue.hasAtom(IUPAC_atom_name);
            } else if (resName.equalsIgnoreCase("VAL")) {//16 atoms
                return __VAL_residue.hasAtom(IUPAC_atom_name);
            }
        }
        
        // if the residue is not one of the 20 amino acid residues, then you are
        // creating a new type of residue. Under this assumption, this method
        // returns true.
        return true;
    }

    /**
     * Add an atom to the residue.
     *
     * @param a the atom to be added to the residue
     */
    public void addAtom(myAtom a) {
        if (canHaveAtom(a.getAtomName()) && !hasAtom(a.getAtomName())) {
            __atoms.put(a.getAtomName(), a);
            addAllBonds();
            updateVdwRadius();
            addAllTerminalInfo();
        }
    }
    
    /**
     * Remove an atom with the specified name from the residue.
     *
     * @param IUPAC_atom_name the atom to be removed
     */
    public void removeAtom(String IUPAC_atom_name) {
        __atoms.remove(IUPAC_atom_name);
    }

    /**
     * Return a reference to the atom whose name is specified in the argument.
     *
     * @param IUPAC_atom_name the atom being sought in the residue
     * @return a reference to the atom
     */
    public myAtom getAtom(String IUPAC_atom_name) {
        return __atoms.get(IUPAC_atom_name);
    }

    /**
     * Add a list of bonds to the residue.
     *
     * @param listOfBonds a String representing a list of bonds,
     * e.g., "LINK-N, N-H, N-CA, CA-HA, CA-CB, CA-C, C-O, C-LINK, CB-HB1, CB-HB2, CB-HB3"
     */
    //TODO: adding bonds to a terminal residue is yet to be implemented
    private void addBonds(String listOfBonds) {                
        // Take the back up of the links to the previous and next residues
        myAtom N = this.getAtom(myAtomLabel.__N);
        myAtom C_i_1 = null;
        if (N != null) {
            myAtom.BondIterator iter = N.bondIterator();
            while (iter.hasNext()) {
                myAtom a = iter.next();
                if (a.getAtomName().equals(myAtomLabel.__C)) {
                    C_i_1 = a;
                    break;
                }
            }
        }

        myAtom C = this.getAtom(myAtomLabel.__C);
        myAtom N_i_p_1 = null;
        if (C != null) {
            myAtom.BondIterator iter = C.bondIterator();
            while (iter.hasNext()) {
                myAtom a = iter.next();
                if (a.getAtomName().equals(myAtomLabel.__N)) {
                    N_i_p_1 = a;
                    break;
                }
            }
        }

        // Clear any existing bonds for this residue
        for (myAtom a : this) {
            a.clearAllBonds();
        }

        Pattern p = Pattern.compile(",(\\s*)");
        String[] bonds = p.split(listOfBonds);
        for (String bond : bonds) {
            int index = bond.indexOf('-');
            assert index != -1 : "Error: Invalid bond information";
            String a1Name = bond.substring(0, index);
            String a2Name = bond.substring(index + 1, bond.length());
            //System.out.println("atom 1: " + a1Name + "\t\t\tatom 2: " + a2Name);
            myAtom atom1 = __atoms.get(a1Name);
            myAtom atom2 = __atoms.get(a2Name);
            if (atom1 != null && atom2 != null) { //to take care of inter-residue link and other unavailable atoms if not all atoms are present
                atom1.addBond(atom2);
                atom2.addBond(atom1);
            }
        }

        // Graft back the links to the previous and next residue
        if (C_i_1 != null) {
            N.addBond(C_i_1);
        }
        if (N_i_p_1 != null) {
            C.addBond(N_i_p_1);
        }
    }

    /**
     * Add all atoms in the residue, add all bonds, update van der Waal radii
     * of atoms, and add all terminal information.
     */
    private void initializeResidue() {
        addAllAtoms();
        addAllBonds();        
        updateVdwRadius();
        addAllTerminalInfo();
    }
    
    /**
     * Update the van der Waal radii of the atoms in the residue.
     */
    private void updateVdwRadius() {
        for (myAtom a : this)
            a.setAtomProperties();
    }

    /**
     * Add the terminal information.
     */
    private void addAllTerminalInfo() {
//        String resName = __residue_name;
//        System.out.println("TODO: The terminal info for the residue " + resName + " is yet to be initialized");    //TODO: add term info
    }

    /**
     * Add all bonds to the residue. Note that this method only add bonds to the
     * available atoms in the residue and it does not add the bond information for
     * the missing atoms.
     *
     * @return true if the residue type is one of the 20 naturally occuring 
     * amino acids. Otherwise, return false.
     */
    private boolean addAllBonds() {
        String resName = __residue_name;
        if (/*AARMap*/myAminoAcidResidueMap.validResidueName(resName)) {
            if (resName.equalsIgnoreCase("ALA")) { //10 atoms
                addBonds("LINK-N, N-H, N-CA, CA-HA, CA-CB, CA-C, C-O, C-LINK, CB-HB1, CB-HB2, CB-HB3");
            } else if (resName.equalsIgnoreCase("ARG")) { //24 atoms
                addBonds("LINK-N, N-H, N-CA, CA-HA, CA-CB, CA-C, C-O, C-LINK, CB-HB2, CB-HB3, CB-CG, CG-HG2, CG-HG3, CG-CD, CD-HD2, CD-HD3, CD-NE, NE-HE, NE-CZ, CZ-NH1, CZ-NH2, NH1-HH11, NH1-HH12, NH2-HH21, NH2-HH22");
            } else if (resName.equalsIgnoreCase("ASN")) { //14 atoms
                addBonds("LINK-N, N-H, N-CA, CA-HA, CA-CB, CA-C, C-O, C-LINK, CB-HB2, CB-HB3, CB-CG, CG-OD1, CG-ND2, ND2-HD21, ND2-HD22");
            } else if (resName.equalsIgnoreCase("ASP")) { //12 atoms
                addBonds("LINK-N, N-H, N-CA, CA-HA, CA-CB, CA-C, C-O, C-LINK, CB-HB2, CB-HB3, CB-CG, CG-OD1, CG-OD2");
            } else if (resName.equalsIgnoreCase("CYS")) { //11 atoms
                addBonds("LINK-N, N-H, N-CA, CA-HA, CA-CB, CA-C, C-O, C-LINK, CB-HB2, CB-HB3, CB-SG, SG-HG");
            } else if (resName.equalsIgnoreCase("GLN")) { //17 atoms
                addBonds("LINK-N, N-H, N-CA, CA-HA, CA-CB, CA-C, C-O, C-LINK, CB-HB2, CB-HB3, CB-CG, CG-HG2, CG-HG3, CG-CD, CD-OE1, CD-NE2, NE2-HE21, NE2-HE22");
            } else if (resName.equalsIgnoreCase("GLU")) { //15 atoms
                addBonds("LINK-N, N-H, N-CA, CA-HA, CA-CB, CA-C, C-O, C-LINK, CB-HB2, CB-HB3, CB-CG, CG-HG2, CG-HG3, CG-CD, CD-OE1, CD-OE2");
            } else if (resName.equalsIgnoreCase("GLY")) { //7 atoms
                addBonds("LINK-N, N-H, N-CA, CA-HA2, CA-HA3, CA-C, C-O, C-LINK");
            } else if (resName.equalsIgnoreCase("HIS")) {//18 atoms
                addBonds("LINK-N, N-H, N-CA, CA-HA, CA-CB, CA-C, C-O, C-LINK, CB-HB2, CB-HB3, CB-CG, CG-ND1, CG-CD2, ND1-HD1, ND1-CE1, CD2-HD2, CD2-NE2, CE1-HE1, CE1-NE2");/*, NE2-HE2*/
            } else if (resName.equalsIgnoreCase("ILE")) {//19 atoms
                addBonds("LINK-N, N-H, N-CA, CA-HA, CA-CB, CA-C, C-O, C-LINK, CB-HB, CB-CG1, CB-CG2, CG1-HG12, CG1-HG13, CG1-CD1, CG2-HG21, CG2-HG22, CG2-HG23, CD1-HD11, CD1-HD12, CD1-HD13");
            } else if (resName.equalsIgnoreCase("LEU")) {//19 atoms
                addBonds("LINK-N, N-H, N-CA, CA-HA, CA-CB, CA-C, C-O, C-LINK, CB-HB2, CB-HB3, CB-CG, CG-HG, CG-CD1, CG-CD2, CD1-HD11, CD1-HD12, CD1-HD13, CD2-HD21, CD2-HD22, CD2-HD23");
            } else if (resName.equalsIgnoreCase("LYS")) {//22 atoms
                addBonds("LINK-N, N-H, N-CA, CA-HA, CA-CB, CA-C, C-O, C-LINK, CB-HB2, CB-HB3, CB-CG, CG-HG2, CG-HG3, CG-CD, CD-HD2, CD-HD3, CD-CE, CE-HE2, CE-HE3, CE-NZ, NZ-HZ1, NZ-HZ2, NZ-HZ3");
            } else if (resName.equalsIgnoreCase("MET")) {//17 atoms
                addBonds("LINK-N, N-H, N-CA, CA-HA, CA-CB, CA-C, C-O, C-LINK, CB-HB2, CB-HB3, CB-CG, CG-HG2, CG-HG3, CG-SD, SD-CE, CE-HE1, CE-HE2, CE-HE3");
            } else if (resName.equalsIgnoreCase("PHE")) {//20 atoms
                addBonds("LINK-N, N-H, N-CA, CA-HA, CA-CB, CA-C, C-O, C-LINK, CB-HB2, CB-HB3, CB-CG, CG-CD1, CG-CD2, CD1-HD1, CD1-CE1, CD2-HD2, CD2-CE2, CE1-HE1, CE1-CZ, CE2-HE2, CE2-CZ, CZ-HZ");
            } else if (resName.equalsIgnoreCase("PRO")) {//14 atoms
                addBonds("LINK-N, N-CA, CA-HA, CA-CB, CA-C, C-O, C-LINK, CB-HB2, CB-HB3, CB-CG, CG-HG2, CG-HG3, CG-CD, CD-HD2, CD-HD3, CD-N");
            } else if (resName.equalsIgnoreCase("SER")) {//11 atom
                addBonds("LINK-N, N-H, N-CA, CA-HA, CA-CB, CA-C, C-O, C-LINK, CB-HB2, CB-HB3, CB-OG, OG-HG");
            } else if (resName.equalsIgnoreCase("THR")) {//14 atoms
                addBonds("LINK-N, N-H, N-CA, CA-HA, CA-CB, CA-C, C-O, C-LINK, CB-HB, CB-OG1, CB-CG2, OG1-HG1, CG2-HG21, CG2-HG22, CG2-HG23");
            } else if (resName.equalsIgnoreCase("TRP")) {//24 atoms
                addBonds("LINK-N, N-H, N-CA, CA-HA, CA-CB, CA-C, C-O, C-LINK, CB-HB2, CB-HB3, CB-CG, CG-CD1, CG-CD2, CD1-HD1, CD1-NE1, NE1-HE1, NE1-CE2, CE2-CD2, CE2-CZ2, CZ2-HZ2, CZ2-CH2, CH2-HH2, CH2-CZ3, CZ3-HZ3, CZ3-CE3, CE3-HE3, CE3-CD2");
            } else if (resName.equalsIgnoreCase("TYR")) {//21 atoms
                addBonds("LINK-N, N-H, N-CA, CA-HA, CA-CB, CA-C, C-O, C-LINK, CB-HB2, CB-HB3, CB-CG, CG-CD1, CG-CD2, CD1-HD1, CD1-CE1, CD2-HD2, CD2-CE2, CE1-HE1, CE1-CZ, CE2-HE2, CE2-CZ, CZ-OH, OH-HH");
            } else if (resName.equalsIgnoreCase("VAL")) {//16 atoms
                addBonds("LINK-N, N-H, N-CA, CA-HA, CA-CB, CA-C, C-O, C-LINK, CB-HB, CB-CG1, CB-CG2, CG1-HG11, CG1-HG12, CG1-HG13, CG2-HG21, CG2-HG22, CG2-HG23");
            }
            return true;
        } else {
            System.out.println("Error: Invalid residue name " + resName + ". Therefore bonds could not be added");
            return false;
        }
    }

    /**
     * Add all atoms to the residue. This method adds any missing atoms, and 
     * overwrites the existing atoms. Therefore, one has to be cautious while
     * using this method. This method is basically meant to be used during
     * initialization of a residue.
     *
     * @return true if the residue type is one of the 20 naturally occuring
     * amino acids. Otherwise, return false.
     */
    private boolean addAllAtoms() {
        String resName = __residue_name;
        if (/*AARMap*/myAminoAcidResidueMap.validResidueName(resName)) {
            //if (resName.equalsIgnoreCase("ACE")) {}
            if (resName.equalsIgnoreCase("ALA")) { //10 atoms
                __atoms.put(myAtomLabel.__N, new myAtom(myAtomLabel.__N));
                __atoms.put(myAtomLabel.__CA, new myAtom(myAtomLabel.__CA));
                __atoms.put(myAtomLabel.__C, new myAtom(myAtomLabel.__C));
                __atoms.put(myAtomLabel.__O, new myAtom(myAtomLabel.__O));
                __atoms.put(myAtomLabel.__CB, new myAtom(myAtomLabel.__CB));
                __atoms.put(myAtomLabel.__H, new myAtom(myAtomLabel.__H));
                __atoms.put(myAtomLabel.__HA, new myAtom(myAtomLabel.__HA));
                __atoms.put(myAtomLabel.__HB1, new myAtom(myAtomLabel.__HB1));
                __atoms.put(myAtomLabel.__HB2, new myAtom(myAtomLabel.__HB2));
                __atoms.put(myAtomLabel.__HB3, new myAtom(myAtomLabel.__HB3));

            } else if (resName.equalsIgnoreCase("ARG")) { //24 atoms
                __atoms.put(myAtomLabel.__N, new myAtom(myAtomLabel.__N));
                __atoms.put(myAtomLabel.__CA, new myAtom(myAtomLabel.__CA));
                __atoms.put(myAtomLabel.__C, new myAtom(myAtomLabel.__C));
                __atoms.put(myAtomLabel.__O, new myAtom(myAtomLabel.__O));
                __atoms.put(myAtomLabel.__CB, new myAtom(myAtomLabel.__CB));
                __atoms.put(myAtomLabel.__CG, new myAtom(myAtomLabel.__CG));
                __atoms.put(myAtomLabel.__CD, new myAtom(myAtomLabel.__CD));
                __atoms.put(myAtomLabel.__NE, new myAtom(myAtomLabel.__NE));
                __atoms.put(myAtomLabel.__CZ, new myAtom(myAtomLabel.__CZ));
                __atoms.put(myAtomLabel.__NH1, new myAtom(myAtomLabel.__NH1));
                __atoms.put(myAtomLabel.__NH2, new myAtom(myAtomLabel.__NH2));
                __atoms.put(myAtomLabel.__H, new myAtom(myAtomLabel.__H));
                __atoms.put(myAtomLabel.__HA, new myAtom(myAtomLabel.__HA));
                __atoms.put(myAtomLabel.__HB2, new myAtom(myAtomLabel.__HB2));
                __atoms.put(myAtomLabel.__HB3, new myAtom(myAtomLabel.__HB3));
                __atoms.put(myAtomLabel.__HG2, new myAtom(myAtomLabel.__HG2));
                __atoms.put(myAtomLabel.__HG3, new myAtom(myAtomLabel.__HG3));
                __atoms.put(myAtomLabel.__HD2, new myAtom(myAtomLabel.__HD2));
                __atoms.put(myAtomLabel.__HD3, new myAtom(myAtomLabel.__HD3));
                __atoms.put(myAtomLabel.__HE, new myAtom(myAtomLabel.__HE));
                __atoms.put(myAtomLabel.__HH11, new myAtom(myAtomLabel.__HH11));
                __atoms.put(myAtomLabel.__HH12, new myAtom(myAtomLabel.__HH12));
                __atoms.put(myAtomLabel.__HH21, new myAtom(myAtomLabel.__HH21));
                __atoms.put(myAtomLabel.__HH22, new myAtom(myAtomLabel.__HH22));

            } else if (resName.equalsIgnoreCase("ASN")) { //14 atoms
                __atoms.put(myAtomLabel.__N, new myAtom(myAtomLabel.__N));
                __atoms.put(myAtomLabel.__CA, new myAtom(myAtomLabel.__CA));
                __atoms.put(myAtomLabel.__C, new myAtom(myAtomLabel.__C));
                __atoms.put(myAtomLabel.__O, new myAtom(myAtomLabel.__O));
                __atoms.put(myAtomLabel.__CB, new myAtom(myAtomLabel.__CB));
                __atoms.put(myAtomLabel.__CG, new myAtom(myAtomLabel.__CG));
                __atoms.put(myAtomLabel.__OD1, new myAtom(myAtomLabel.__OD1));
                __atoms.put(myAtomLabel.__ND2, new myAtom(myAtomLabel.__ND2));
                __atoms.put(myAtomLabel.__H, new myAtom(myAtomLabel.__H));
                __atoms.put(myAtomLabel.__HA, new myAtom(myAtomLabel.__HA));
                __atoms.put(myAtomLabel.__HB2, new myAtom(myAtomLabel.__HB2));
                __atoms.put(myAtomLabel.__HB3, new myAtom(myAtomLabel.__HB3));
                __atoms.put(myAtomLabel.__HD21, new myAtom(myAtomLabel.__HD21));
                __atoms.put(myAtomLabel.__HD22, new myAtom(myAtomLabel.__HD22));

            } else if (resName.equalsIgnoreCase("ASP")) { //12 atoms
                __atoms.put(myAtomLabel.__N, new myAtom(myAtomLabel.__N));
                __atoms.put(myAtomLabel.__CA, new myAtom(myAtomLabel.__CA));
                __atoms.put(myAtomLabel.__C, new myAtom(myAtomLabel.__C));
                __atoms.put(myAtomLabel.__O, new myAtom(myAtomLabel.__O));
                __atoms.put(myAtomLabel.__CB, new myAtom(myAtomLabel.__CB));
                __atoms.put(myAtomLabel.__CG, new myAtom(myAtomLabel.__CG));
                __atoms.put(myAtomLabel.__OD1, new myAtom(myAtomLabel.__OD1));
                __atoms.put(myAtomLabel.__OD2, new myAtom(myAtomLabel.__OD2));
                __atoms.put(myAtomLabel.__H, new myAtom(myAtomLabel.__H));
                __atoms.put(myAtomLabel.__HA, new myAtom(myAtomLabel.__HA));
                __atoms.put(myAtomLabel.__HB2, new myAtom(myAtomLabel.__HB2));
                __atoms.put(myAtomLabel.__HB3, new myAtom(myAtomLabel.__HB3));

            } else if (resName.equalsIgnoreCase("CYS")) { //11 atoms
                __atoms.put(myAtomLabel.__N, new myAtom(myAtomLabel.__N));
                __atoms.put(myAtomLabel.__CA, new myAtom(myAtomLabel.__CA));
                __atoms.put(myAtomLabel.__C, new myAtom(myAtomLabel.__C));
                __atoms.put(myAtomLabel.__O, new myAtom(myAtomLabel.__O));
                __atoms.put(myAtomLabel.__CB, new myAtom(myAtomLabel.__CB));
                __atoms.put(myAtomLabel.__SG, new myAtom(myAtomLabel.__SG));
                __atoms.put(myAtomLabel.__H, new myAtom(myAtomLabel.__H));
                __atoms.put(myAtomLabel.__HA, new myAtom(myAtomLabel.__HA));
                __atoms.put(myAtomLabel.__HB2, new myAtom(myAtomLabel.__HB2));
                __atoms.put(myAtomLabel.__HB3, new myAtom(myAtomLabel.__HB3));
                __atoms.put(myAtomLabel.__HG, new myAtom(myAtomLabel.__HG));

            } else if (resName.equalsIgnoreCase("GLN")) { //17 atoms
                __atoms.put(myAtomLabel.__N, new myAtom(myAtomLabel.__N));
                __atoms.put(myAtomLabel.__CA, new myAtom(myAtomLabel.__CA));
                __atoms.put(myAtomLabel.__C, new myAtom(myAtomLabel.__C));
                __atoms.put(myAtomLabel.__O, new myAtom(myAtomLabel.__O));
                __atoms.put(myAtomLabel.__CB, new myAtom(myAtomLabel.__CB));
                __atoms.put(myAtomLabel.__CG, new myAtom(myAtomLabel.__CG));
                __atoms.put(myAtomLabel.__CD, new myAtom(myAtomLabel.__CD));
                __atoms.put(myAtomLabel.__OE1, new myAtom(myAtomLabel.__OE1));
                __atoms.put(myAtomLabel.__NE2, new myAtom(myAtomLabel.__NE2));
                __atoms.put(myAtomLabel.__H, new myAtom(myAtomLabel.__H));
                __atoms.put(myAtomLabel.__HA, new myAtom(myAtomLabel.__HA));
                __atoms.put(myAtomLabel.__HB2, new myAtom(myAtomLabel.__HB2));
                __atoms.put(myAtomLabel.__HB3, new myAtom(myAtomLabel.__HB3));
                __atoms.put(myAtomLabel.__HG2, new myAtom(myAtomLabel.__HG2));
                __atoms.put(myAtomLabel.__HG3, new myAtom(myAtomLabel.__HG3));
                __atoms.put(myAtomLabel.__HE21, new myAtom(myAtomLabel.__HE21));
                __atoms.put(myAtomLabel.__HE22, new myAtom(myAtomLabel.__HE22));

            } else if (resName.equalsIgnoreCase("GLU")) { //15 atoms
                __atoms.put(myAtomLabel.__N, new myAtom(myAtomLabel.__N));
                __atoms.put(myAtomLabel.__CA, new myAtom(myAtomLabel.__CA));
                __atoms.put(myAtomLabel.__C, new myAtom(myAtomLabel.__C));
                __atoms.put(myAtomLabel.__O, new myAtom(myAtomLabel.__O));
                __atoms.put(myAtomLabel.__CB, new myAtom(myAtomLabel.__CB));
                __atoms.put(myAtomLabel.__CG, new myAtom(myAtomLabel.__CG));
                __atoms.put(myAtomLabel.__CD, new myAtom(myAtomLabel.__CD));
                __atoms.put(myAtomLabel.__OE1, new myAtom(myAtomLabel.__OE1));
                __atoms.put(myAtomLabel.__OE2, new myAtom(myAtomLabel.__OE2));
                __atoms.put(myAtomLabel.__H, new myAtom(myAtomLabel.__H));
                __atoms.put(myAtomLabel.__HA, new myAtom(myAtomLabel.__HA));
                __atoms.put(myAtomLabel.__HB2, new myAtom(myAtomLabel.__HB2));
                __atoms.put(myAtomLabel.__HB3, new myAtom(myAtomLabel.__HB3));
                __atoms.put(myAtomLabel.__HG2, new myAtom(myAtomLabel.__HG2));
                __atoms.put(myAtomLabel.__HG3, new myAtom(myAtomLabel.__HG3));

            } else if (resName.equalsIgnoreCase("GLY")) { //7 atoms
                __atoms.put(myAtomLabel.__N, new myAtom(myAtomLabel.__N));
                __atoms.put(myAtomLabel.__CA, new myAtom(myAtomLabel.__CA));
                __atoms.put(myAtomLabel.__C, new myAtom(myAtomLabel.__C));
                __atoms.put(myAtomLabel.__O, new myAtom(myAtomLabel.__O));
                __atoms.put(myAtomLabel.__H, new myAtom(myAtomLabel.__H));
                __atoms.put(myAtomLabel.__HA2, new myAtom(myAtomLabel.__HA2));
                __atoms.put(myAtomLabel.__HA3, new myAtom(myAtomLabel.__HA3));

            } else if (resName.equalsIgnoreCase("HIS")) {//18 atoms
                __atoms.put(myAtomLabel.__N, new myAtom(myAtomLabel.__N));
                __atoms.put(myAtomLabel.__CA, new myAtom(myAtomLabel.__CA));
                __atoms.put(myAtomLabel.__C, new myAtom(myAtomLabel.__C));
                __atoms.put(myAtomLabel.__O, new myAtom(myAtomLabel.__O));
                __atoms.put(myAtomLabel.__CB, new myAtom(myAtomLabel.__CB));
                __atoms.put(myAtomLabel.__CG, new myAtom(myAtomLabel.__CG));
                __atoms.put(myAtomLabel.__ND1, new myAtom(myAtomLabel.__ND1));
                __atoms.put(myAtomLabel.__CD2, new myAtom(myAtomLabel.__CD2));
                __atoms.put(myAtomLabel.__CE1, new myAtom(myAtomLabel.__CE1));
                __atoms.put(myAtomLabel.__NE2, new myAtom(myAtomLabel.__NE2));
                __atoms.put(myAtomLabel.__H, new myAtom(myAtomLabel.__H));
                __atoms.put(myAtomLabel.__HA, new myAtom(myAtomLabel.__HA));
                __atoms.put(myAtomLabel.__HB2, new myAtom(myAtomLabel.__HB2));
                __atoms.put(myAtomLabel.__HB3, new myAtom(myAtomLabel.__HB3));
                __atoms.put(myAtomLabel.__HD1, new myAtom(myAtomLabel.__HD1));
                __atoms.put(myAtomLabel.__HD2, new myAtom(myAtomLabel.__HD2));
                __atoms.put(myAtomLabel.__HE1, new myAtom(myAtomLabel.__HE1));
            //__atoms.put(myAtomLabel.__HE2, new myAtom(myAtomLabel.__HE2));

            } else if (resName.equalsIgnoreCase("ILE")) {//19 atoms
                __atoms.put(myAtomLabel.__N, new myAtom(myAtomLabel.__N));
                __atoms.put(myAtomLabel.__CA, new myAtom(myAtomLabel.__CA));
                __atoms.put(myAtomLabel.__C, new myAtom(myAtomLabel.__C));
                __atoms.put(myAtomLabel.__O, new myAtom(myAtomLabel.__O));
                __atoms.put(myAtomLabel.__CB, new myAtom(myAtomLabel.__CB));
                __atoms.put(myAtomLabel.__CG1, new myAtom(myAtomLabel.__CG1));
                __atoms.put(myAtomLabel.__CG2, new myAtom(myAtomLabel.__CG2));
                __atoms.put(myAtomLabel.__CD1, new myAtom(myAtomLabel.__CD1));
                __atoms.put(myAtomLabel.__H, new myAtom(myAtomLabel.__H));
                __atoms.put(myAtomLabel.__HA, new myAtom(myAtomLabel.__HA));
                __atoms.put(myAtomLabel.__HB, new myAtom(myAtomLabel.__HB));
                __atoms.put(myAtomLabel.__HG12, new myAtom(myAtomLabel.__HG12));
                __atoms.put(myAtomLabel.__HG13, new myAtom(myAtomLabel.__HG13));
                __atoms.put(myAtomLabel.__HG21, new myAtom(myAtomLabel.__HG21));
                __atoms.put(myAtomLabel.__HG22, new myAtom(myAtomLabel.__HG22));
                __atoms.put(myAtomLabel.__HG23, new myAtom(myAtomLabel.__HG23));
                __atoms.put(myAtomLabel.__HD11, new myAtom(myAtomLabel.__HD11));
                __atoms.put(myAtomLabel.__HD12, new myAtom(myAtomLabel.__HD12));
                __atoms.put(myAtomLabel.__HD13, new myAtom(myAtomLabel.__HD13));

            } else if (resName.equalsIgnoreCase("LEU")) {//19 atoms
                __atoms.put(myAtomLabel.__N, new myAtom(myAtomLabel.__N));
                __atoms.put(myAtomLabel.__CA, new myAtom(myAtomLabel.__CA));
                __atoms.put(myAtomLabel.__C, new myAtom(myAtomLabel.__C));
                __atoms.put(myAtomLabel.__O, new myAtom(myAtomLabel.__O));
                __atoms.put(myAtomLabel.__CB, new myAtom(myAtomLabel.__CB));
                __atoms.put(myAtomLabel.__CG, new myAtom(myAtomLabel.__CG));
                __atoms.put(myAtomLabel.__CD1, new myAtom(myAtomLabel.__CD1));
                __atoms.put(myAtomLabel.__CD2, new myAtom(myAtomLabel.__CD2));
                __atoms.put(myAtomLabel.__H, new myAtom(myAtomLabel.__H));
                __atoms.put(myAtomLabel.__HA, new myAtom(myAtomLabel.__HA));
                __atoms.put(myAtomLabel.__HB2, new myAtom(myAtomLabel.__HB2));
                __atoms.put(myAtomLabel.__HB3, new myAtom(myAtomLabel.__HB3));
                __atoms.put(myAtomLabel.__HG, new myAtom(myAtomLabel.__HG));
                __atoms.put(myAtomLabel.__HD11, new myAtom(myAtomLabel.__HD11));
                __atoms.put(myAtomLabel.__HD12, new myAtom(myAtomLabel.__HD12));
                __atoms.put(myAtomLabel.__HD13, new myAtom(myAtomLabel.__HD13));
                __atoms.put(myAtomLabel.__HD21, new myAtom(myAtomLabel.__HD21));
                __atoms.put(myAtomLabel.__HD22, new myAtom(myAtomLabel.__HD22));
                __atoms.put(myAtomLabel.__HD23, new myAtom(myAtomLabel.__HD23));

            } else if (resName.equalsIgnoreCase("LYS")) {//22 atoms
                __atoms.put(myAtomLabel.__N, new myAtom(myAtomLabel.__N));
                __atoms.put(myAtomLabel.__CA, new myAtom(myAtomLabel.__CA));
                __atoms.put(myAtomLabel.__C, new myAtom(myAtomLabel.__C));
                __atoms.put(myAtomLabel.__O, new myAtom(myAtomLabel.__O));
                __atoms.put(myAtomLabel.__CB, new myAtom(myAtomLabel.__CB));
                __atoms.put(myAtomLabel.__CG, new myAtom(myAtomLabel.__CG));
                __atoms.put(myAtomLabel.__CD, new myAtom(myAtomLabel.__CD));
                __atoms.put(myAtomLabel.__CE, new myAtom(myAtomLabel.__CE));
                __atoms.put(myAtomLabel.__NZ, new myAtom(myAtomLabel.__NZ));
                __atoms.put(myAtomLabel.__H, new myAtom(myAtomLabel.__H));
                __atoms.put(myAtomLabel.__HA, new myAtom(myAtomLabel.__HA));
                __atoms.put(myAtomLabel.__HB2, new myAtom(myAtomLabel.__HB2));
                __atoms.put(myAtomLabel.__HB3, new myAtom(myAtomLabel.__HB3));
                __atoms.put(myAtomLabel.__HG2, new myAtom(myAtomLabel.__HG2));
                __atoms.put(myAtomLabel.__HG3, new myAtom(myAtomLabel.__HG3));
                __atoms.put(myAtomLabel.__HD2, new myAtom(myAtomLabel.__HD2));
                __atoms.put(myAtomLabel.__HD3, new myAtom(myAtomLabel.__HD3));
                __atoms.put(myAtomLabel.__HE2, new myAtom(myAtomLabel.__HE2));
                __atoms.put(myAtomLabel.__HE3, new myAtom(myAtomLabel.__HE3));
                __atoms.put(myAtomLabel.__HZ1, new myAtom(myAtomLabel.__HZ1));
                __atoms.put(myAtomLabel.__HZ2, new myAtom(myAtomLabel.__HZ2));
                __atoms.put(myAtomLabel.__HZ3, new myAtom(myAtomLabel.__HZ3));

            } else if (resName.equalsIgnoreCase("MET")) {//17 atoms
                __atoms.put(myAtomLabel.__N, new myAtom(myAtomLabel.__N));
                __atoms.put(myAtomLabel.__CA, new myAtom(myAtomLabel.__CA));
                __atoms.put(myAtomLabel.__C, new myAtom(myAtomLabel.__C));
                __atoms.put(myAtomLabel.__O, new myAtom(myAtomLabel.__O));
                __atoms.put(myAtomLabel.__CB, new myAtom(myAtomLabel.__CB));
                __atoms.put(myAtomLabel.__CG, new myAtom(myAtomLabel.__CG));
                __atoms.put(myAtomLabel.__SD, new myAtom(myAtomLabel.__SD));
                __atoms.put(myAtomLabel.__CE, new myAtom(myAtomLabel.__CE));
                __atoms.put(myAtomLabel.__H, new myAtom(myAtomLabel.__H));
                __atoms.put(myAtomLabel.__HA, new myAtom(myAtomLabel.__HA));
                __atoms.put(myAtomLabel.__HB2, new myAtom(myAtomLabel.__HB2));
                __atoms.put(myAtomLabel.__HB3, new myAtom(myAtomLabel.__HB3));
                __atoms.put(myAtomLabel.__HG2, new myAtom(myAtomLabel.__HG2));
                __atoms.put(myAtomLabel.__HG3, new myAtom(myAtomLabel.__HG3));
                __atoms.put(myAtomLabel.__HE1, new myAtom(myAtomLabel.__HE1));
                __atoms.put(myAtomLabel.__HE2, new myAtom(myAtomLabel.__HE2));
                __atoms.put(myAtomLabel.__HE3, new myAtom(myAtomLabel.__HE3));

            } else if (resName.equalsIgnoreCase("PHE")) {//20 atoms
                __atoms.put(myAtomLabel.__N, new myAtom(myAtomLabel.__N));
                __atoms.put(myAtomLabel.__CA, new myAtom(myAtomLabel.__CA));
                __atoms.put(myAtomLabel.__C, new myAtom(myAtomLabel.__C));
                __atoms.put(myAtomLabel.__O, new myAtom(myAtomLabel.__O));
                __atoms.put(myAtomLabel.__CB, new myAtom(myAtomLabel.__CB));
                __atoms.put(myAtomLabel.__CG, new myAtom(myAtomLabel.__CG));
                __atoms.put(myAtomLabel.__CD1, new myAtom(myAtomLabel.__CD1));
                __atoms.put(myAtomLabel.__CD2, new myAtom(myAtomLabel.__CD2));
                __atoms.put(myAtomLabel.__CE1, new myAtom(myAtomLabel.__CE1));
                __atoms.put(myAtomLabel.__CE2, new myAtom(myAtomLabel.__CE2));
                __atoms.put(myAtomLabel.__CZ, new myAtom(myAtomLabel.__CZ));
                __atoms.put(myAtomLabel.__H, new myAtom(myAtomLabel.__H));
                __atoms.put(myAtomLabel.__HA, new myAtom(myAtomLabel.__HA));
                __atoms.put(myAtomLabel.__HB2, new myAtom(myAtomLabel.__HB2));
                __atoms.put(myAtomLabel.__HB3, new myAtom(myAtomLabel.__HB3));
                __atoms.put(myAtomLabel.__HD1, new myAtom(myAtomLabel.__HD1));
                __atoms.put(myAtomLabel.__HD2, new myAtom(myAtomLabel.__HD2));
                __atoms.put(myAtomLabel.__HE1, new myAtom(myAtomLabel.__HE1));
                __atoms.put(myAtomLabel.__HE2, new myAtom(myAtomLabel.__HE2));
                __atoms.put(myAtomLabel.__HZ, new myAtom(myAtomLabel.__HZ));

            } else if (resName.equalsIgnoreCase("PRO")) {//14 atoms
                __atoms.put(myAtomLabel.__N, new myAtom(myAtomLabel.__N));
                __atoms.put(myAtomLabel.__CA, new myAtom(myAtomLabel.__CA));
                __atoms.put(myAtomLabel.__C, new myAtom(myAtomLabel.__C));
                __atoms.put(myAtomLabel.__O, new myAtom(myAtomLabel.__O));
                __atoms.put(myAtomLabel.__CB, new myAtom(myAtomLabel.__CB));
                __atoms.put(myAtomLabel.__CG, new myAtom(myAtomLabel.__CG));
                __atoms.put(myAtomLabel.__CD, new myAtom(myAtomLabel.__CD));
                __atoms.put(myAtomLabel.__HA, new myAtom(myAtomLabel.__HA));
                __atoms.put(myAtomLabel.__HB2, new myAtom(myAtomLabel.__HB2));
                __atoms.put(myAtomLabel.__HB3, new myAtom(myAtomLabel.__HB3));
                __atoms.put(myAtomLabel.__HG2, new myAtom(myAtomLabel.__HG2));
                __atoms.put(myAtomLabel.__HG3, new myAtom(myAtomLabel.__HG3));
                __atoms.put(myAtomLabel.__HD2, new myAtom(myAtomLabel.__HD2));
                __atoms.put(myAtomLabel.__HD3, new myAtom(myAtomLabel.__HD3));

            } else if (resName.equalsIgnoreCase("SER")) {//11 atom
                __atoms.put(myAtomLabel.__N, new myAtom(myAtomLabel.__N));
                __atoms.put(myAtomLabel.__CA, new myAtom(myAtomLabel.__CA));
                __atoms.put(myAtomLabel.__C, new myAtom(myAtomLabel.__C));
                __atoms.put(myAtomLabel.__O, new myAtom(myAtomLabel.__O));
                __atoms.put(myAtomLabel.__CB, new myAtom(myAtomLabel.__CB));
                __atoms.put(myAtomLabel.__OG, new myAtom(myAtomLabel.__OG));
                __atoms.put(myAtomLabel.__H, new myAtom(myAtomLabel.__H));
                __atoms.put(myAtomLabel.__HA, new myAtom(myAtomLabel.__HA));
                __atoms.put(myAtomLabel.__HB2, new myAtom(myAtomLabel.__HB2));
                __atoms.put(myAtomLabel.__HB3, new myAtom(myAtomLabel.__HB3));
                __atoms.put(myAtomLabel.__HG, new myAtom(myAtomLabel.__HG));

            } else if (resName.equalsIgnoreCase("THR")) {//14 atoms
                __atoms.put(myAtomLabel.__N, new myAtom(myAtomLabel.__N));
                __atoms.put(myAtomLabel.__CA, new myAtom(myAtomLabel.__CA));
                __atoms.put(myAtomLabel.__C, new myAtom(myAtomLabel.__C));
                __atoms.put(myAtomLabel.__O, new myAtom(myAtomLabel.__O));
                __atoms.put(myAtomLabel.__CB, new myAtom(myAtomLabel.__CB));
                __atoms.put(myAtomLabel.__OG1, new myAtom(myAtomLabel.__OG1));
                __atoms.put(myAtomLabel.__CG2, new myAtom(myAtomLabel.__CG2));
                __atoms.put(myAtomLabel.__H, new myAtom(myAtomLabel.__H));
                __atoms.put(myAtomLabel.__HA, new myAtom(myAtomLabel.__HA));
                __atoms.put(myAtomLabel.__HB, new myAtom(myAtomLabel.__HB));
                __atoms.put(myAtomLabel.__HG1, new myAtom(myAtomLabel.__HG1));
                __atoms.put(myAtomLabel.__HG21, new myAtom(myAtomLabel.__HG21));
                __atoms.put(myAtomLabel.__HG22, new myAtom(myAtomLabel.__HG22));
                __atoms.put(myAtomLabel.__HG23, new myAtom(myAtomLabel.__HG23));

            } else if (resName.equalsIgnoreCase("TRP")) {//24 atoms
                __atoms.put(myAtomLabel.__N, new myAtom(myAtomLabel.__N));
                __atoms.put(myAtomLabel.__CA, new myAtom(myAtomLabel.__CA));
                __atoms.put(myAtomLabel.__C, new myAtom(myAtomLabel.__C));
                __atoms.put(myAtomLabel.__O, new myAtom(myAtomLabel.__O));
                __atoms.put(myAtomLabel.__CB, new myAtom(myAtomLabel.__CB));
                __atoms.put(myAtomLabel.__CG, new myAtom(myAtomLabel.__CG));
                __atoms.put(myAtomLabel.__CD1, new myAtom(myAtomLabel.__CD1));
                __atoms.put(myAtomLabel.__CD2, new myAtom(myAtomLabel.__CD2));
                __atoms.put(myAtomLabel.__NE1, new myAtom(myAtomLabel.__NE2));
                __atoms.put(myAtomLabel.__CE2, new myAtom(myAtomLabel.__CE2));
                __atoms.put(myAtomLabel.__CE3, new myAtom(myAtomLabel.__CE3));
                __atoms.put(myAtomLabel.__CZ2, new myAtom(myAtomLabel.__CZ2));
                __atoms.put(myAtomLabel.__CZ3, new myAtom(myAtomLabel.__CZ3));
                __atoms.put(myAtomLabel.__CH2, new myAtom(myAtomLabel.__CH2));
                __atoms.put(myAtomLabel.__H, new myAtom(myAtomLabel.__H));
                __atoms.put(myAtomLabel.__HA, new myAtom(myAtomLabel.__HA));
                __atoms.put(myAtomLabel.__HB2, new myAtom(myAtomLabel.__HB2));
                __atoms.put(myAtomLabel.__HB3, new myAtom(myAtomLabel.__HB3));
                __atoms.put(myAtomLabel.__HD1, new myAtom(myAtomLabel.__HD1));
                __atoms.put(myAtomLabel.__HE1, new myAtom(myAtomLabel.__HE1));
                __atoms.put(myAtomLabel.__HE3, new myAtom(myAtomLabel.__HE3));
                __atoms.put(myAtomLabel.__HZ2, new myAtom(myAtomLabel.__HZ2));
                __atoms.put(myAtomLabel.__HZ3, new myAtom(myAtomLabel.__HZ3));
                __atoms.put(myAtomLabel.__HH2, new myAtom(myAtomLabel.__HH2));

            } else if (resName.equalsIgnoreCase("TYR")) {//21 atoms
                __atoms.put(myAtomLabel.__N, new myAtom(myAtomLabel.__N));
                __atoms.put(myAtomLabel.__CA, new myAtom(myAtomLabel.__CA));
                __atoms.put(myAtomLabel.__C, new myAtom(myAtomLabel.__C));
                __atoms.put(myAtomLabel.__O, new myAtom(myAtomLabel.__O));
                __atoms.put(myAtomLabel.__CB, new myAtom(myAtomLabel.__CB));
                __atoms.put(myAtomLabel.__CG, new myAtom(myAtomLabel.__CG));
                __atoms.put(myAtomLabel.__CD1, new myAtom(myAtomLabel.__CD1));
                __atoms.put(myAtomLabel.__CD2, new myAtom(myAtomLabel.__CD2));
                __atoms.put(myAtomLabel.__CE1, new myAtom(myAtomLabel.__CE2));
                __atoms.put(myAtomLabel.__CE2, new myAtom(myAtomLabel.__CE2));
                __atoms.put(myAtomLabel.__CZ, new myAtom(myAtomLabel.__CZ));
                __atoms.put(myAtomLabel.__OH, new myAtom(myAtomLabel.__OH));
                __atoms.put(myAtomLabel.__H, new myAtom(myAtomLabel.__H));
                __atoms.put(myAtomLabel.__HA, new myAtom(myAtomLabel.__HA));
                __atoms.put(myAtomLabel.__HB2, new myAtom(myAtomLabel.__HB2));
                __atoms.put(myAtomLabel.__HB3, new myAtom(myAtomLabel.__HB3));
                __atoms.put(myAtomLabel.__HD1, new myAtom(myAtomLabel.__HD1));
                __atoms.put(myAtomLabel.__HD2, new myAtom(myAtomLabel.__HD2));
                __atoms.put(myAtomLabel.__HE1, new myAtom(myAtomLabel.__HE1));
                __atoms.put(myAtomLabel.__HE2, new myAtom(myAtomLabel.__HE2));
                __atoms.put(myAtomLabel.__HH, new myAtom(myAtomLabel.__HH));

            } else if (resName.equalsIgnoreCase("VAL")) {//16 atoms
                __atoms.put(myAtomLabel.__N, new myAtom(myAtomLabel.__N));
                __atoms.put(myAtomLabel.__CA, new myAtom(myAtomLabel.__CA));
                __atoms.put(myAtomLabel.__C, new myAtom(myAtomLabel.__C));
                __atoms.put(myAtomLabel.__O, new myAtom(myAtomLabel.__O));
                __atoms.put(myAtomLabel.__CB, new myAtom(myAtomLabel.__CB));
                __atoms.put(myAtomLabel.__CG1, new myAtom(myAtomLabel.__CG1));
                __atoms.put(myAtomLabel.__CG2, new myAtom(myAtomLabel.__CG2));
                __atoms.put(myAtomLabel.__H, new myAtom(myAtomLabel.__H));
                __atoms.put(myAtomLabel.__HA, new myAtom(myAtomLabel.__HA));
                __atoms.put(myAtomLabel.__HB, new myAtom(myAtomLabel.__HB));
                __atoms.put(myAtomLabel.__HG11, new myAtom(myAtomLabel.__HG11));
                __atoms.put(myAtomLabel.__HG12, new myAtom(myAtomLabel.__HG12));
                __atoms.put(myAtomLabel.__HG13, new myAtom(myAtomLabel.__HG13));
                __atoms.put(myAtomLabel.__HG21, new myAtom(myAtomLabel.__HG21));
                __atoms.put(myAtomLabel.__HG22, new myAtom(myAtomLabel.__HG22));
                __atoms.put(myAtomLabel.__HG23, new myAtom(myAtomLabel.__HG23));

            }
            return true;
        } else {
            System.out.println("Error: Invalid residue name " + resName + ". Therefore atoms could not be added");
            return false;
        }
    }

    /**
     * Remove all atoms from the residue. The residue will be empty after this
     * call returns.
     */
    public void clear() {
        __atoms.clear();
    }

    /**
     * This method tests the various methods in this class.
     * 
     * @param args
     */
    public static void main(String[] args) {
        String[][] residueBlock = {
            {"ATOM    738  N   ALA A  46      47.073 -72.636   2.682  1.00  6.53           N  ",
                "ATOM    739  CA  ALA A  46      46.642 -71.407   3.330  1.00  7.15           C  ",
                "ATOM    740  C   ALA A  46      46.824 -70.193   2.418  1.00  9.00           C  ",
                "ATOM    741  O   ALA A  46      47.168 -69.104   2.877  1.00 11.15           O  ",
                "ATOM    742  CB  ALA A  46      47.402 -71.223   4.646  1.00  8.99           C  ",
                "ATOM    743  H   ALA A  46      46.482 -73.417   2.700  1.00  0.00           H  ",
                "ATOM    744  HA  ALA A  46      45.590 -71.500   3.561  1.00  0.00           H  ",
                "ATOM    745  HB1 ALA A  46      46.888 -70.498   5.259  1.00  0.00           H  ",
                "ATOM    746  HB2 ALA A  46      48.402 -70.874   4.437  1.00  0.00           H  ",
                "ATOM    747  HB3 ALA A  46      47.451 -72.167   5.169  1.00  0.00           H  "
            },
            {"ATOM     81  N   TRP A   6       7.338 -10.067  -7.993  1.00  0.00           N  ",
                "ATOM     82  CA  TRP A   6       6.886 -11.441  -7.811  1.00  0.00           C  ",
                "ATOM     83  C   TRP A   6       8.059 -12.362  -7.491  1.00  0.00           C  ",
                "ATOM     84  O   TRP A   6       8.902 -12.628  -8.347  1.00  0.00           O  ",
                "ATOM     85  CB  TRP A   6       6.163 -11.933  -9.065  1.00  0.00           C  ",
                "ATOM     86  CG  TRP A   6       4.698 -11.616  -9.070  1.00  0.00           C  ",
                "ATOM     87  CD1 TRP A   6       3.667 -12.504  -8.948  1.00  0.00           C  ",
                "ATOM     88  CD2 TRP A   6       4.102 -10.321  -9.204  1.00  0.00           C  ",
                "ATOM     89  NE1 TRP A   6       2.466 -11.838  -8.999  1.00  0.00           N  ",
                "ATOM     90  CE2 TRP A   6       2.705 -10.499  -9.154  1.00  0.00           C  ",
                "ATOM     91  CE3 TRP A   6       4.612  -9.030  -9.359  1.00  0.00           C  ",
                "ATOM     92  CZ2 TRP A   6       1.816  -9.432  -9.256  1.00  0.00           C  ",
                "ATOM     93  CZ3 TRP A   6       3.728  -7.972  -9.460  1.00  0.00           C  ",
                "ATOM     94  CH2 TRP A   6       2.343  -8.178  -9.407  1.00  0.00           C  ",
                "ATOM     95  H   TRP A   6       7.424  -9.708  -8.902  1.00  0.00           H  ",
                "ATOM     96  HA  TRP A   6       6.196 -11.455  -6.980  1.00  0.00           H  ",
                "ATOM     97  HB2 TRP A   6       6.607 -11.470  -9.934  1.00  0.00           H  ",
                "ATOM     98  HB3 TRP A   6       6.273 -13.006  -9.138  1.00  0.00           H  ",
                "ATOM     99  HD1 TRP A   6       3.793 -13.569  -8.832  1.00  0.00           H  ",
                "ATOM    100  HE1 TRP A   6       1.581 -12.256  -8.933  1.00  0.00           H  ",
                "ATOM    101  HE3 TRP A   6       5.676  -8.851  -9.402  1.00  0.00           H  ",
                "ATOM    102  HZ2 TRP A   6       0.746  -9.575  -9.216  1.00  0.00           H  ",
                "ATOM    103  HZ3 TRP A   6       4.103  -6.967  -9.581  1.00  0.00           H  ",
                "ATOM    104  HH2 TRP A   6       1.690  -7.323  -9.490  1.00  0.00           H  "
            }
        };
        Vector<String> block = new Vector<String>(Arrays.asList(residueBlock[1]));

        myResidue myRes = new myResidue(block);
        myRes.print();
        System.out.println("Can have HB3? " + myRes.canHaveAtom("HB3"));
        System.out.println("Has atom CZ? " + myRes.canHaveAtom("CZ"));
        System.out.println("Number of atoms: " + myRes.numberOfAtoms());
        System.out.println("Number of bonds: " + myRes.numberOfBonds());
        myRes.addAtom(new myAtom("CZ"));

        for (myAtom a : myRes) {
            a.print();
        }
        Iterator<myAtom> it = myRes.iterator();
        while (it.hasNext()) {
            myAtom ass = it.next();
        //ass.setOccupancy(4.00);        
        //it.remove();
        }
        System.out.println("Good bye...");
        myRes.print();
    }
}
