/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2009 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, 01 December, 2009
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.0, Nov 18, 2009
 * @author        Chittaranjan Tripathy (2007-2009)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.util.*;

/**
 * Description of the class
 */
public class myForKin {

    static final double degToRad = Math.PI / 180.0; // convert degree to radian
    static final double radToDeg = 180.0 / Math.PI; // convert radian to degree
    static final double bigOmega = 40 * degToRad;

    static final Matrix RLCaCForPhi = R(myConstantsForProteinGeometry.angHN_N_CA, "-x").
            times(R(Math.PI - myConstantsForProteinGeometry.dihC_N_CA_HN, "+z"));
    static final Matrix RRCaCForPhi = R(Math.PI - myConstantsForProteinGeometry.angN_CA_C, "-x");

    static final Matrix RLCaHaForPhi = RLCaCForPhi;
    static final Matrix RRCaHaForPhi = R(myConstantsForProteinGeometry.dihC_N_CA_HA, "+z"). //
            times(R(Math.PI - myConstantsForProteinGeometry.angN_CA_HA, "-x"));

    static final Matrix RLCaCbForPhi = RLCaCForPhi;
    static final Matrix RRCaCbForPhi = R(myConstantsForProteinGeometry.dihC_N_CA_CB, "+z"). //
            times(R(Math.PI - myConstantsForProteinGeometry.angN_CA_CB, "-x"));

    static final Matrix RLCNForPsi = RLCaCForPhi;
    static final Matrix RMCNForPsi = RRCaCForPhi;
    static final Matrix RRCNForPsi = R(Math.PI - myConstantsForProteinGeometry.angCA_C_N, "-x");

    static final Matrix RLNCaForPsi = RLCaCForPhi;
    static final Matrix RMNCaForPsi = RMCNForPsi;
    static final Matrix RRNCaForPsi = RRCNForPsi.times(R(myConstantsForProteinGeometry.dihCA_C_N_CA, "+z")).
            times(R(Math.PI - myConstantsForProteinGeometry.angC_N_CA, "-x"));

    static final Matrix RLNHnForPsi = RLCaCForPhi;
    static final Matrix RMNHnForPsi = RMCNForPsi;
    static final Matrix RRNHnForPsi = RRCNForPsi.times(R(myConstantsForProteinGeometry.dihCA_C_N_CA, "+z")).
            times(R(Math.PI - myConstantsForProteinGeometry.angC_N_CA, "-x")).
            times(R(myConstantsForProteinGeometry.dihC_N_CA_HN, "+z")).
            times(R(myConstantsForProteinGeometry.angHN_N_CA, "-x"));

    static final Matrix RLCOForPsi = RLCaCForPhi;
    static final Matrix RMCOForPsi = RMCNForPsi;
    static final Matrix RRCOForPsi = RRCNForPsi.times(R(myConstantsForProteinGeometry.dihCA_C_N_O, "+z")).
            times(R(myConstantsForProteinGeometry.angN_C_O, "-x"));

    static final Matrix RLv1CsaForPsi = RLCaCForPhi;
    static final Matrix RMv1CsaForPsi = RMCNForPsi;
    static final Matrix RRv1CsaForPsi = RRCNForPsi.
            //times(R(myConstantsForProteinGeometry.dihCA_C_N_O, "+z")).
            times(R(bigOmega, "+x")); // use "-x" if the line above is uncommented

    static final Matrix RLv2CsaForPsi = RLCaCForPhi;
    static final Matrix RMv2CsaForPsi = RMCNForPsi;
    static final Matrix RRv2CsaForPsi = RRCNForPsi.
            //times(R(myConstantsForProteinGeometry.dihCA_C_N_O, "+z")).
            times(R(bigOmega + Math.PI / 2, "+x")); // use "-x" if the line above is uncommented

    /**
     * Return the rotation matrix M that rotates a vector v1 to v2 about +x-axis
     * in the frame F. That is, v2 = M.v1, where v1 and v2 are in the same frame F.
     *
     * @param theta the angle of rotation
     * @return the rotation matrix
     */
    public static Matrix Rx(double angle) {
        return R(angle, "+x");
    }

    /**
     * Return the rotation matrix M that rotates a vector v1 to v2 about +y-axis
     * in the frame F. That is, v2 = M.v1, where v1 and v2 are in the same frame F.
     *
     * @param theta the angle of rotation
     * @return the rotation matrix
     */
    public static Matrix Ry(double angle) {
        return R(angle, "+y");
    }

    /**
     * Return the rotation matrix M that rotates a vector v1 to v2 about +y-axis
     * in the frame F. That is, v2 = M.v1, where v1 and v2 are in the same frame F.
     *
     * @param theta the angle of rotation
     * @return the rotation matrix
     */
    public static Matrix Rz(double angle) {
        return R(angle, "+z");
    }

    /**
     * Return the rotation matrix M that rotates a vector v1 to v2 in the same
     * frame F. That is, v2 = M.v1, where v1 and v2 are in the same frame F.
     *
     * @param theta the angle of rotation
     * @param axis the axis of rotation which can be [+-]?[xyzXYZ]
     * @return the rotation matrix
     */
    public static Matrix R(double angle, String axis) {
        return Rvec(angle, axis);
    }

    /**
     * Return the rotation matrix M that rotates a vector v1 to v2 in the same
     * frame F. That is, v2 = M.v1, where v1 and v2 are in the same frame F.
     *
     * @param theta the angle of rotation
     * @param axis the axis of rotation which can be [+-]?[xyzXYZ]
     * @return the rotation matrix
     */
    public static Matrix Rvec(double theta, String axis) {
        return Raxis(theta, axis).transpose(); // since inverse = transpose
    }

    /**
     * Return the rotation matrix M that takes a vector v from frame F0 to frame
     * F1. In other words, If v is the representation of a vector in frame F0,
     * then M.v will be the representation of the vector in frame F1.
     *
     * @param theta the angle of rotation
     * @param axis the axis of rotation which can be [+-]?[xyzXYZ]
     * @return the rotation matrix
     */
    public static Matrix Raxis(double theta, String axis){
        axis = axis.trim();
        if (axis.length() == 1) {
            axis = "+" + axis;
        }
        if (!axis.matches("[+-]?[xyzXYZ]")) {
            System.out.println("Error: the axis specification " + axis + " for the rotation is not correct");
            System.exit(1);
        }
        if (axis.charAt(0) == '-') {
            theta = -theta;
        }

        double [][] M = new double[3][3];

        double c = Math.cos(theta);
        double s = Math.sin(theta);

        if (Character.toLowerCase(axis.charAt(1)) == 'x') {
	    M[0][0] = 1.0;
	    M[0][1] = 0.0;
	    M[0][2] = 0.0;
	    M[1][0] = 0.0;
	    M[1][1] = c;
	    M[1][2] = s;
	    M[2][0] = 0.0;
	    M[2][1] = -s;
	    M[2][2] = c;
        } else if (Character.toLowerCase(axis.charAt(1)) == 'y') {
	    M[0][0] = c;
	    M[0][1] = 0.0;
	    M[0][2] = -s;
	    M[1][0] = 0.0;
	    M[1][1] = 1.0;
	    M[1][2] = 0.0;
	    M[2][0] = s;
	    M[2][1] = 0.0;
	    M[2][2] = c;
        } else if (Character.toLowerCase(axis.charAt(1)) == 'z') {
	    M[0][0] = c;
	    M[0][1] = s;
	    M[0][2] = 0.0;
	    M[1][0] = -s;
	    M[1][1] = c;
	    M[1][2] = 0.0;
	    M[2][0] = 0.0;
	    M[2][1] = 0.0;
	    M[2][2] = 1.0;
        }

        return new Matrix(M, 3, 3);
    }

    /**
     * Let F0 denote the global reference frame, and F1 be the reference frame
     * attached to the peptide plane with +z-axis is along N-HN direction, +y-axis
     * is along the direction such that the N-CA vector makes an acute angle, and
     * +z-axis is defined by the right-hand rule. The rotation matrix returned is
     * the rotation of F1 with respect to F0.
     *
     * @param pPlane the initial peptide plane
     * @return the rotation matrix of F1 wrt. F0
     */
    public static Matrix getRotationWrtGlobalFrame(final myPeptidePlane pPlane) {
        myVector3D vN2HN = new myVector3D(pPlane.getN(), pPlane.getH());
        myVector3D vN2Ca = new myVector3D(pPlane.getN(), pPlane.getCa());

        myVector3D vZ = myVector3D.normalize(vN2HN);
        myVector3D vX = myVector3D.normalize(myVector3D.crossProduct(vN2Ca, vZ));
        myVector3D vY = myVector3D.normalize(myVector3D.crossProduct(vZ, vX));
        //System.out.println("vX: " + vX.toString() + "\nvY: " + vY.toString() + "\nvZ: " + vZ.toString());
        double[][] rot = {{vX.getX(), vY.getX(), vZ.getX()}, {vX.getY(), vY.getY(), vZ.getY()}, {vX.getZ(), vY.getZ(), vZ.getZ()}};
        Matrix rotMat = new Matrix(rot);

        //System.out.println("The rotation matrix is: ");
        //rotMat.print(8, 8);
        return rotMat;
    }

    public static Matrix getRotationWrtGlobalFrame(myPoint p1, myPoint p2, myPoint p3) {
        myVector3D vN2HN = new myVector3D(p2, p1);
        myVector3D vN2Ca = new myVector3D(p2, p3);

        myVector3D vZ = myVector3D.normalize(vN2HN);
        myVector3D vX = myVector3D.normalize(myVector3D.crossProduct(vN2Ca, vZ));
        myVector3D vY = myVector3D.normalize(myVector3D.crossProduct(vZ, vX));
        //System.out.println("vX: " + vX.toString() + "\nvY: " + vY.toString() + "\nvZ: " + vZ.toString());
        double[][] rot = {{vX.getX(), vY.getX(), vZ.getX()}, {vX.getY(), vY.getY(), vZ.getY()}, {vX.getZ(), vY.getZ(), vZ.getZ()}};
        Matrix rotMat = new Matrix(rot);

        //System.out.println("The rotation matrix is: ");
        //rotMat.print(8, 8);
        return rotMat;
    }
 
    /**
     * This method computes the rotation matrix for the peptide plane (i+1) given
     * the rotation matrix for peptide plane i, the dihedrals phi_i and psi_i,
     * and the chirality information (true denotes the right handedness).
     * 
     * @param thisPhi the phi angle of residue i
     * @param thisPsi the psi angle of residue i
     * @param Rg the rotation matrix of the coordinate frame defined in the peptide plane i wrt. POF
     * @param rightHand rightHand chirality information (true denotes right handedness)
     * @return a new rotation matrix which is the rotation matrix for the residue (i+1)
     */
    public static Matrix newRG__(double thisPhi, double thisPsi, Matrix Rg, boolean rightHand) {
/*          Matrix CaCMatrix = Rg.times(RLCaCForPhi).times(R(thisPhi, "+z")).times(RRCaCForPhi);
            Matrix CaHaMatrix = Rg.times(RLCaHaForPhi).times(R(thisPhi, "+z")).times(RRCaHaForPhi);
            Matrix CaCbMatrix = Rg.times(RLCaCbForPhi).times(R(thisPhi, "+z")).times(RRCaCbForPhi);
            Matrix Nip1Matrix = Rg.times(RLCNForPsi).times(R(thisPhi, "+z")).times(RMCNForPsi).times(R(thisPsi, "+z")).times(RRCNForPsi);
            Matrix CAip1Matrix = Rg.times(RLNCaForPsi).times(R(thisPhi, "+z")).times(RMNCaForPsi).times(R(thisPsi, "+z")).times(RRNCaForPsi);
            Matrix HNip1Matrix = Rg.times(RLNHnForPsi).times(R(thisPhi, "+z")).times(RMNHnForPsi).times(R(thisPsi, "+z")).times(RRNHnForPsi);
            Matrix COMatrix = Rg.times(RLCOForPsi).times(R(thisPhi, "+z")).times(RMCOForPsi).times(R(thisPsi, "+z")).times(RRCOForPsi);
            Matrix v1CsaMatrix = Rg.times(RLv1CsaForPsi).times(R(thisPhi, "+z")).times(RMv1CsaForPsi).times(R(thisPsi, "+z")).times(RRv1CsaForPsi);
            Matrix v2CsaMatrix = Rg.times(RLv2CsaForPsi).times(R(thisPhi, "+z")).times(RMv2CsaForPsi).times(R(thisPsi, "+z")).times(RRv2CsaForPsi);
*/
//        I need N here
        //Matrix Nip1Matrix = Rg.times(RLCNForPsi).times(R(thisPhi, "+z")).times(RMCNForPsi).times(R(thisPsi, "+z")).times(RRCNForPsi);
        //Matrix CAip1Matrix = Rg.times(RLNCaForPsi).times(R(thisPhi, "+z")).times(RMNCaForPsi).times(R(thisPsi, "+z")).times(RRNCaForPsi);
        //Matrix HNip1Matrix = Rg.times(RLNHnForPsi).times(R(thisPhi, "+z")).times(RMNHnForPsi).times(R(thisPsi, "+z")).times(RRNHnForPsi);

        Matrix CAip1Matrix = Rg.times(RLNCaForPsi).times(R(thisPhi, "+z")).times(RMNCaForPsi).times(R(thisPsi, "+z")).times(RRNCaForPsi);
        Matrix HNip1Matrix = Rg.times(RLNHnForPsi).times(R(thisPhi, "+z")).times(RMNHnForPsi).times(R(thisPsi, "+z")).times(RRNHnForPsi);

        myPoint CAi1 = new myPoint(0.0, 0.0, myConstantsForProteinGeometry.dN_CA);
        myVector3D N2CaVec1 = new myVector3D(CAip1Matrix.times(CAi1.getXYZ()));

        myPoint Hi1 = new myPoint(0.0, 0.0, myConstantsForProteinGeometry.dN_HN);
        myVector3D N2HVec1 = new myVector3D(HNip1Matrix.times(Hi1.getXYZ()));

        myPeptidePlane pPlane = new myPeptidePlane(new myPoint(0.0, 0.0, 0.0), N2HVec1, N2CaVec1);
/*
        myVector3D vN2HN = N2HVec1;
        myVector3D vN2Ca = N2CaVec1;

        myVector3D vZ = myVector3D.normalize(vN2HN);
        myVector3D vX = myVector3D.normalize(myVector3D.crossProduct(vN2Ca, vZ));
        myVector3D vY = myVector3D.normalize(myVector3D.crossProduct(vZ, vX));
        //System.out.println("vX: " + vX.toString() + "\nvY: " + vY.toString() + "\nvZ: " + vZ.toString());
        double[][] rot = {{vX.getX(), vY.getX(), vZ.getX()}, {vX.getY(), vY.getY(), vZ.getY()}, {vX.getZ(), vY.getZ(), vZ.getZ()}};
        Matrix rotMat = new Matrix(rot);

        //System.out.println("The rotation matrix is: ");
        //rotMat.print(8, 8);
        return rotMat;
 */
        return getRotationWrtGlobalFrame(pPlane);
    }

    /**
     * Compute the coordinates of the backbone atoms (N, H, CA, HA, CB, C and O)
     * of a protein fragment given the first peptide plane and a sequence of PHI
     * and PSI angles using ideal peptide geometry assumptions.
     *
     * @param phiPsiVec phiPsiVec a vector of (PHI, PSI) pairs
     * @param initialPeptidePlane initialPeptidePlane the initial peptide plane
     * @return the constructed protein fragment
     */
    public static myProtein buildBackbone(final Vector<myPhiPsi> phiPsiVec, final myPeptidePlane initialPeptidePlane) {
        myPeptidePlane pPlane = new myPeptidePlane(initialPeptidePlane); //get a local copy of the initial pp and use that.
        myProtein thisFragment = new myProtein();
        int serialNum = 1;
        Vector<myAtom> atomsOfThisResidue = new Vector<myAtom>(7);

        for (myPhiPsi thisPhiPsiPair : phiPsiVec) {
            atomsOfThisResidue.clear(); // clear this vector so that it can be reused

            int thisResidueNumber = thisPhiPsiPair.getResidueNumber();
            String thisResidueName = thisPhiPsiPair.getResidueName();
            double thisPhi = thisPhiPsiPair.getPhi();
            double thisPsi = thisPhiPsiPair.getPsi();
            //System.out.println("thisPhi: " + thisPhi + "    thisPsi: " + thisPsi);

            myPoint Ni = pPlane.getN();
            myPoint Hi = pPlane.getH();
            myPoint CAi = pPlane.getCa();
            //System.out.println("Ni: " + Ni.toString() + "\nHi: " + Hi.toString() + "\nCAi: " + CAi.toString());

            atomsOfThisResidue.add(new myAtom(myAtomLabel.__N, Ni));
            atomsOfThisResidue.add(new myAtom(myAtomLabel.__H, Hi));
            atomsOfThisResidue.add(new myAtom(myAtomLabel.__CA, CAi));

            Matrix Rg = getRotationWrtGlobalFrame(pPlane);

/*
            
            Matrix CaCMatrix = Rg.times(R(myConstantsForProteinGeometry.angHN_N_CA, "-x")). 
                    times(R(Math.PI - myConstantsForProteinGeometry.dihC_N_CA_HN + thisPhi, "+z")).  //
                    times(R(Math.PI - myConstantsForProteinGeometry.angN_CA_C, "-x"));

            Matrix CaHaMatrix = Rg.times(R(myConstantsForProteinGeometry.angHN_N_CA, "-x")).
                    times(R(Math.PI  - myConstantsForProteinGeometry.dihC_N_CA_HN + thisPhi + myConstantsForProteinGeometry.dihHA_N_CA_C, "+z")).
                    times(R(Math.PI - myConstantsForProteinGeometry.angN_CA_HA, "-x"));

            Matrix CaCbMatrix = Rg.times(R(myConstantsForProteinGeometry.angHN_N_CA, "-x")).
                    times(R(Math.PI + -myConstantsForProteinGeometry.dihC_N_CA_HN + thisPhi + myConstantsForProteinGeometry.dihCB_N_CA_C, "+z")).
                    times(R(Math.PI - myConstantsForProteinGeometry.angN_CA_CB, "-x"));

            Matrix Nip1Matrix = CaCMatrix.times(R(thisPsi, "+z")).
                    times(R(Math.PI - myConstantsForProteinGeometry.angCA_C_N, "-x"));
            
//            Matrix Nip1Matrix = CaCMatrix.times(R(Math.PI / 2, "+x")).times(R(thisPsi, "+y")).
//                    times(R(myConstantsForProteinGeometry.angCA_C_N - Math.PI / 2, "-x"));

            Matrix CAip1Matrix = Nip1Matrix.times(R(myConstantsForProteinGeometry.dihCA_C_N_CA, "+z")).
                    times(R(Math.PI - myConstantsForProteinGeometry.angC_N_CA, "-x"));

            Matrix HNip1Matrix = Nip1Matrix.times(R(myConstantsForProteinGeometry.dihCA_C_N_CA, "+z")).
                    times(R(myConstantsForProteinGeometry.angC_N_CA - Math.PI / 2, "+x")).
                    times(R(Math.PI + myConstantsForProteinGeometry.dihC_N_CA_HN, "+y")).
                    times(R(myConstantsForProteinGeometry.angHN_N_CA - Math.PI / 2, "+x"));

            Matrix COMatrix = Nip1Matrix.times(R(Math.PI / 2, "+x")).
                    times(R(Math.PI + myConstantsForProteinGeometry.dihCA_C_N_O, "+y")).
                    times(R(myConstantsForProteinGeometry.angN_C_O - Math.PI / 2, "+x"));

            Matrix v1CsaMatrix = Nip1Matrix.times(R(Math.PI / 2, "+x")).
                    //times(R(Math.PI + myConstantsForProteinGeometry.dihCA_C_N_O, "+y")).
                    times(R(Math.PI / 2 - bigOmega, "-x"));

            Matrix v2CsaMatrix = Nip1Matrix.times(R(Math.PI / 2, "+x")).
                    //times(R(Math.PI + myConstantsForProteinGeometry.dihCA_C_N_O, "+y")).
                    times(R(bigOmega, "+x"));
*/

            Matrix CaCMatrix = Rg.times(RLCaCForPhi).times(R(thisPhi, "+z")).times(RRCaCForPhi);
            Matrix CaHaMatrix = Rg.times(RLCaHaForPhi).times(R(thisPhi, "+z")).times(RRCaHaForPhi);
            Matrix CaCbMatrix = Rg.times(RLCaCbForPhi).times(R(thisPhi, "+z")).times(RRCaCbForPhi);
            Matrix Nip1Matrix = Rg.times(RLCNForPsi).times(R(thisPhi, "+z")).times(RMCNForPsi).times(R(thisPsi, "+z")).times(RRCNForPsi);
            Matrix CAip1Matrix = Rg.times(RLNCaForPsi).times(R(thisPhi, "+z")).times(RMNCaForPsi).times(R(thisPsi, "+z")).times(RRNCaForPsi);
            Matrix HNip1Matrix = Rg.times(RLNHnForPsi).times(R(thisPhi, "+z")).times(RMNHnForPsi).times(R(thisPsi, "+z")).times(RRNHnForPsi);
            Matrix COMatrix = Rg.times(RLCOForPsi).times(R(thisPhi, "+z")).times(RMCOForPsi).times(R(thisPsi, "+z")).times(RRCOForPsi);
            Matrix v1CsaMatrix = Rg.times(RLv1CsaForPsi).times(R(thisPhi, "+z")).times(RMv1CsaForPsi).times(R(thisPsi, "+z")).times(RRv1CsaForPsi);
            Matrix v2CsaMatrix = Rg.times(RLv2CsaForPsi).times(R(thisPhi, "+z")).times(RMv2CsaForPsi).times(R(thisPsi, "+z")).times(RRv2CsaForPsi);


            // Compute the C coordinate residue (i)
            myPoint Ci = new myPoint(0.0, 0.0, myConstantsForProteinGeometry.dCA_C);
            myVector3D Ca2CoVec = new myVector3D(CaCMatrix.times(Ci.getXYZ()));
            Ci = myPoint.translate(CAi, Ca2CoVec);
            atomsOfThisResidue.add(new myAtom(myAtomLabel.__C, Ci));

            // Compute the HA coordinate residue (i)
            myPoint HAi = new myPoint(0.0, 0.0, myConstantsForProteinGeometry.dCA_HA);
            myVector3D Ca2HaVec = new myVector3D(CaHaMatrix.times(HAi.getXYZ()));
            HAi = myPoint.translate(CAi, Ca2HaVec);
            atomsOfThisResidue.add(new myAtom(myAtomLabel.__HA, HAi));

            // Compute the CB coordinate of residue (i)
            myPoint CBi = new myPoint(0.0, 0.0, myConstantsForProteinGeometry.dCA_CB);
            myVector3D Ca2CbVec = new myVector3D(CaCbMatrix.times(CBi.getXYZ()));
            CBi = myPoint.translate(CAi, Ca2CbVec);
            atomsOfThisResidue.add(new myAtom(myAtomLabel.__CB, CBi));

            // Compute the O coordinate of residue (i)
            myPoint Oi = new myPoint(0.0, 0.0, myConstantsForProteinGeometry.dC_O);
            myVector3D C2OVec = new myVector3D(COMatrix.times(Oi.getXYZ()));
            Oi = myPoint.translate(Ci, C2OVec);
            atomsOfThisResidue.add(new myAtom(myAtomLabel.__O, Oi));

            // Compute the N coordinates for residue (i + 1)
            myPoint Ni1 = new myPoint(0.0, 0.0, myConstantsForProteinGeometry.dC_N);
            //matT = matT.times(r4zInv.times(Const.r5xInv));
            myVector3D C2NVec = new myVector3D(Nip1Matrix.times(Ni1.getXYZ()));
            Ni1 = myPoint.translate(Ci, C2NVec);
            pPlane.setN(Ni1);

            // Compute the CA coordinate residue (i + 1)
            myPoint CAi1 = new myPoint(0.0, 0.0, myConstantsForProteinGeometry.dN_CA);
            myVector3D N2CaVec1 = new myVector3D(CAip1Matrix.times(CAi1.getXYZ()));
            CAi1 = myPoint.translate(Ni1, N2CaVec1);
            pPlane.setCa(CAi1);

            myPoint Hi1 = new myPoint(0.0, 0.0, myConstantsForProteinGeometry.dN_HN);
            myVector3D N2HVec1 = new myVector3D(HNip1Matrix.times(Hi1.getXYZ()));
            Hi1 = myPoint.translate(Ni1, N2HVec1);
            pPlane.setH(Hi1);

            // Given Ni, Hi, CAi, we computed HAi, CBi COi, Oi, Ni+1, Hi+1, CAi+1
            myResidue r = new myResidue(thisResidueName, thisResidueNumber);
            r.clear();
            for (myAtom a : atomsOfThisResidue) {
                myAtom thisAtom = new myAtom("ATOM", serialNum++, a.getAtomName(), thisResidueName, 'A',
                        thisResidueNumber, new myPoint(a.getCoordinates()), 1.00, 0.00, "    ", "  ", "  ");
                r.addAtom(thisAtom);
            }
            thisFragment.addResidue(r);
        }

        // Add the last half residue computed since PSI for this is given.
        myResidue r = new myResidue("ALA", phiPsiVec.lastElement().getResidueNumber() + 1);
        r.clear();
        r.addAtom(new myAtom("ATOM", serialNum++, myAtomLabel.__N, "ALA", 'A',
                phiPsiVec.lastElement().getResidueNumber() + 1, new myPoint(pPlane.getN()), 1.00, 0.00, "    ", "  ", "  "));
        r.addAtom(new myAtom("ATOM", serialNum++, myAtomLabel.__H, "ALA", 'A',
                phiPsiVec.lastElement().getResidueNumber() + 1, new myPoint(pPlane.getH()), 1.00, 0.00, "    ", "  ", "  "));
        r.addAtom(new myAtom("ATOM", serialNum++, myAtomLabel.__CA, "ALA", 'A',
                phiPsiVec.lastElement().getResidueNumber() + 1, new myPoint(pPlane.getCa()), 1.00, 0.00, "    ", "  ", "  "));
        thisFragment.addResidue(r);

        thisFragment.removeResidue(thisFragment.getEndResidueNumber()); // Note: we chop off the last half residue as it cannot be computed reliably without the phi value

        thisFragment.synchronizeProtein();
        return thisFragment;
    }

    /**
     * Build a poly-alanine ideal SSE (helix or strand) model using the standard
     * (ideal) phi and psi values for an ideal SSE.
     *
     * @param sseId contains SSE boundary and type
     * @return the ideal SSE coordinates
     */
    public static myProtein buildIdealSSE(mySseInfo sseId) {
        myPeptidePlane pPlane = new myPeptidePlane(new myPoint(0.0, 0.0, 0.0),
                new myPoint(0.0, 0.0, myConstantsForProteinGeometry.dN_HN),
                new myPoint(0.0, myConstantsForProteinGeometry.dN_CA * Math.cos(myConstantsForProteinGeometry.angHN_N_CA - Math.PI / 2),
                -myConstantsForProteinGeometry.dN_CA * Math.sin(myConstantsForProteinGeometry.angHN_N_CA - Math.PI / 2)));

//        pPlane = new myPeptidePlane(new myPoint(2.404230868423198, 0.9976986297436247, -0.8684695439771458),
//                new myPoint(2.329036060911908, 0.613398768504605, 0.029895280342579156),
//                new myPoint(3.7169224372116045, 1.085457230817374, -1.4968828420297964));

        System.out.println("Printing the initial peptide plane:");
        pPlane.print();

        Vector<myPhiPsi> phiPsiVec = new Vector<myPhiPsi>();

        for (int i = sseId.getBeginResidueNumber(); i <= sseId.getEndResidueNumber(); i++) {
            if (sseId.isHelix()) {
                phiPsiVec.add(new myPhiPsi(i, "ALA", Const.phiAveHelix, Const.psiAveHelix));
            } else if (sseId.isStrand()) {
                phiPsiVec.add(new myPhiPsi(i, "ALA", Const.phiAveBeta, Const.psiAveBeta));
            } else {
                System.out.println("Error: " + sseId.toString() + " is not an SSE");
                System.exit(1);
            }
        }
        return buildBackbone(phiPsiVec, pPlane);
    }

    public static myProtein buildIdealSSE(mySseInfo sseId, myPeptidePlane pPlane) {
//        myPeptidePlane pPlane = new myPeptidePlane(new myPoint(0.0, 0.0, 0.0),
//                new myPoint(0.0, 0.0, myConstantsForProteinGeometry.dN_HN),
//                new myPoint(0.0, myConstantsForProteinGeometry.dN_CA * Math.cos(myConstantsForProteinGeometry.angHN_N_CA - Math.PI / 2),
//                -myConstantsForProteinGeometry.dN_CA * Math.sin(myConstantsForProteinGeometry.angHN_N_CA - Math.PI / 2)));

        System.out.println("Printing the initial peptide plane:");
        pPlane.print();

        Vector<myPhiPsi> phiPsiVec = new Vector<myPhiPsi>();

        for (int i = sseId.getBeginResidueNumber(); i <= sseId.getEndResidueNumber(); i++) {
            if (sseId.isHelix()) {
                phiPsiVec.add(new myPhiPsi(i, "ALA", Const.phiAveHelix, Const.psiAveHelix));
            } else if (sseId.isStrand()) {
                phiPsiVec.add(new myPhiPsi(i, "ALA", Const.phiAveBeta, Const.psiAveBeta));
            } else {
                System.out.println("Error: " + sseId.toString() + " is not an SSE");
                System.exit(1);
            }
        }
        return buildBackbone(phiPsiVec, pPlane);
    }

    public static void main(String... args) {

        myVector3D v1 = new myVector3D(0, 3, 2);
        myVector3D v2 = new myVector3D(Rx(Math.PI/2).times(v1.getXYZ()));

        System.out.println("v1: " + v1.toString() + "\nv2: " + v2.toString());
        //System.exit(1);

        mySseInfo thisSseId = new mySseInfo(60, 70, "H");
        myProtein p = buildIdealSSE(thisSseId);
        System.out.println("Printing the ideal SSE");
        p.print();

        double[][] rot = {{2., 3., 4.}, {5., 6., 7.}, {8., 9., 10.}};
        Matrix rotMat = new Matrix(rot);

        Matrix rm = new Matrix(new double[][] {{2., 3., 4.}, {5., 6., 7.}, {8., 9., 10.}});

        double i = rot[1][1] ;
        i = 100;
        System.out.println(rot[1][1]);

        R(30 * degToRad, "+z").print(8, 8);
    }


    /**
     * This method takes the coefficients of a quartic equation and returns the
     * real roots of the quartic equation.
     *
     * @param coeffs the coefficients of the quartic equation
     * @return the real roots
     */
    static Vector<Double> myQuarticSolver(double[] coeffs) {
//        double L4 = coeffs[0];
//        double L3 = coeffs[1];
//        double L2 = coeffs[2];
//        double L1 = coeffs[3];
//        double L0 = coeffs[4];

        double L4 = coeffs[4];
        double L3 = coeffs[3];
        double L2 = coeffs[2];
        double L1 = coeffs[1];
        double L0 = coeffs[0];

        Vector<Double> roots = new Vector<Double>();

        Matrix mm = null;

        //if (L4 != 0.0) {
        if (Math.abs(L4) > myMiscConstants.eps) {
            double[][] mArr = {{-L3 / L4, -L2 / L4, -L1 / L4, -L0 / L4},
                                {1.0,     0.0,      0.0,      0.0},
                                {0.0,     1.0,      0.0,      0.0},
                                {0.0,     0.0,      1.0,      0.0}
            };
            mm = new Matrix(mArr);
        }

        if (mm == null) {
            System.out.println("Error: The companion matrix is null");
            System.exit(1);
        }

        EigenvalueDecomposition eigs = new EigenvalueDecomposition(mm, false);

        if (eigs == null) {
            System.out.println("Warning: The vector of eigenvalues is null");
            return roots;
        }

        double[] realValues = eigs.getRealEigenvalues();
        double[] imaginaryValues = eigs.getImagEigenvalues();

//        System.out.println("number of real roots: " + realValues.length + "  number of imaginary roots: " + imaginaryValues.length);
//
//        for (int i = 0; i < realValues.length; i++) {
//            System.out.println("Re: " + realValues[i] + "    Im: " + imaginaryValues[i]);
//        }

        // Return the real solutions only.
        for (int k = 0; k < imaginaryValues.length; k++) {
            if (Math.abs(imaginaryValues[k] - 0.0) < myMiscConstants.eps) {
                roots.add(new Double(realValues[k]));
                //System.out.println(realValues[k] + "  " + imageValues[k]);
            }
        }
        
        return roots;
    }

    static double halfTangentToAngle(double u) {
        double denom = (1 + u * u);
        double sinTheta = 2 * u / denom;
        double cosTheta = (1 - u * u) / denom;
        return Math.atan2(sinTheta, cosTheta);
    }
    
    static Vector<Double> halfTangentToAngle(Vector<Double> v) {
        Vector<Double> angles = new Vector<Double>();
        for (double u : v) {
            angles.add(halfTangentToAngle(u));
        }
        return angles;
    }

    /**
     * Returns the set of phi angles that are within the Ramachandran region of
     * the secondary structure element.
     *
     * @param Rg the rotation matrix that rotates a POF to the frame defined on the current peptide plane (for residue i)
     * @param Syy the diagonilized alignment tensor component
     * @param Szz the diagonilized alignment tensor component
     * @param dc the dipolar coupling that constrains the dihedral phi
     * @param isHelix true if the secondary structure element is helix, false if it is a strand
     * @return a vector of solution phi angles; if no solution exists, the vector is an empty vector
     */
    public static Vector<Double> computePhi(Matrix Rg, double Syy, double Szz, myDipolarCoupling dc, boolean isHelix) {
        //System.out.println(" I am here in computePhi");
        double[] coeffs = computeQuarticCoefficientsForPhi(Rg, Syy, Szz, dc);
        Vector<Double> roots = myQuarticSolver(coeffs);
        Vector<Double> phiVec = halfTangentToAngle(roots);

        double phiHigh = 0.0;
        double phiLow = 0.0;
        double phiRamHigh = 0.0;
        double phiRamLow = 0.0;
        if (isHelix) {
            phiRamHigh = Const.phiHighHelix;
            phiRamLow = Const.phiLowHelix;
        } else {
            phiRamHigh = Const.phiHighBeta;
            phiRamLow = Const.phiLowBeta;
        }
        phiHigh = phiRamHigh; 
        phiLow = phiRamLow; 

        //phiHigh = Math.PI;
        //phiLow = -Math.PI;

        Vector<Double> phisInRamaRegion = new Vector<Double>();
        for (double phi : phiVec) {
            if (phiLow < phi && phi < phiHigh) {
                phisInRamaRegion.add(phi);
            }
        }
        //System.out.println("Printing the Phi vector from computePhi: " + phiVec + phisInRamaRegion);
        return phisInRamaRegion;
    }

    /**
     * Returns the set of psi angles that are within the Ramachandran region of
     * the secondary structure element.
     *
     * @param Rg the rotation matrix that rotates a POF to the frame defined on the current peptide plane (for residue i)
     * @param Syy the diagonilized alignment tensor component
     * @param Szz the diagonilized alignment tensor component
     * @param dc the dipolar coupling that constrains the dihedral phi
     * @param phi the computed phi angle for the same residue
     * @param isHelix true if the secondary structure element is helix, false if it is a strand
     * @return a vector of solution phi angles; if no solution exists, the vector is an empty vector
     */
    public static Vector<Double> computePsi(Matrix Rg, double Syy, double Szz, myDipolarCoupling dc, double phi, boolean isHelix) {
        //System.out.println(" I am here in computePsi");
        double[] coeffs = computeQuarticCoefficientsForPsi(Rg, Syy, Szz, phi, dc);
        Vector<Double> roots = myQuarticSolver(coeffs);
        Vector<Double> phiVec = halfTangentToAngle(roots);

        double psiHigh = 0.0;
        double psiLow = 0.0;
        double psiRamHigh = 0.0;
        double psiRamLow = 0.0;
        if (isHelix) {
            psiRamHigh = Const.psiHighHelix;
            psiRamLow = Const.psiLowHelix;
        } else {
            psiRamHigh = Const.psiHighBeta;
            psiRamLow = Const.psiLowBeta;
        }
        psiHigh = psiRamHigh; 
        psiLow = psiRamLow;

        //psiHigh = Math.PI;
        //psiLow = -Math.PI;

        Vector<Double> psisInRamaRegion = new Vector<Double>();
        for (double psi : phiVec) {
            if (psiLow < psi && psi < psiHigh) {
                psisInRamaRegion.add(psi);
            }
        }
        //System.out.println("Printing the Psi vector from computePsi: " + psiVec + psisInRamaRegion);
        return psisInRamaRegion;
    }

    public static double[] computeQuarticCoefficientsForPhi(Matrix Rg, double Syy, double Szz, myDipolarCoupling dc) {
        double[] F = null;
        Matrix L = null, R = null;

        switch (dc.getType()) {
            case CA_HA:
                //System.out.println("Inside computeQuarticCoefficientsForPhi CA-HA");
                //System.out.println("CA-HA Rdc: " + dc.toString());
                L = Rg.times(RLCaHaForPhi);
                R = RRCaHaForPhi;
                F = computeQuarticCoefficientsForRdcOnSingleVector(L, R, -(Syy + Szz), Syy, Szz, dc.getRdc() / myConstantsForRdcs.DmaxScaled.get(dc.getType().toString()));
                //System.out.println(Arrays.toString(F));
                break;
            default:
                System.out.println("Error: wrong matrix while computing phi");
                System.exit(1);

        }
        
        return F;
    }

    public static double[] computeQuarticCoefficientsForPsi(Matrix Rg, double Syy, double Szz, double phi, myDipolarCoupling dc) {
        double proxy1 = 0.0, proxy2 = 0.0; // they are never used
        return computeQuarticCoefficientsForPsi(Rg, Syy, Szz, phi, dc, proxy1, proxy2);
    }

    public static double[] computeQuarticCoefficientsForPsi(Matrix Rg, double Syy, double Szz, double phi, myDipolarCoupling dc, double e1, double e2) {        
        double[] F = null;
        Matrix L = null, R = null;

        switch(dc.getType()) {
            case N_HN:
                //System.out.println("Inside computeQuarticCoefficientsForPhi N_HN");
                //System.out.println("N-HN Rdc :" + dc.toString());
                L = Rg.times(RLNHnForPsi).times(Rz(phi)).times(RMNHnForPsi);
                R = RRNHnForPsi;
                F = computeQuarticCoefficientsForRdcOnSingleVector(L, R, -(Syy + Szz), Syy, Szz, dc.getRdc() / myConstantsForRdcs.DmaxScaled.get(dc.getType().toString()));
                break;
            default:
                System.out.println("Error: wrong matrix while computing psi");
                System.exit(1);
        }
        
        return F;
    }

    // The representation is v = L * Rz(unknownAngle) * R
    private static double[] computeQuarticCoefficientsForRdcOnSingleVector(Matrix L, Matrix R, double Sxx, double Syy, double Szz, double rdc) {
        double a = Sxx - Szz;
        double b = Syy - Szz;
        double d = Szz - rdc;
        
        //double x = L.get(0, 2) * R.get(2, 2) + (L.get(0, 0) * R.get(0, 2) + L.get(0, 1) * R.get(1, 2)) * c + (L.get(0, 1) * R.get(0, 2) - L.get(0, 0) * R.get(1, 2)) * s;
        //double y = L.get(1, 2) * R.get(2, 2) + (L.get(1, 0) * R.get(0, 2) + L.get(1, 1) * R.get(1, 2)) * c + (L.get(1, 1) * R.get(0, 2) - L.get(1, 0) * R.get(1, 2)) * s;
        //double z = L.get(2, 2) * R.get(2, 2) + (L.get(2, 0) * R.get(0, 2) + L.get(2, 1) * R.get(1, 2)) * c + (L.get(2, 1) * R.get(0, 2) - L.get(2, 0) * R.get(1, 2)) * s;
    
        double A0 = L.get(0, 2) * R.get(2, 2);
        double A1 = L.get(0, 0) * R.get(0, 2) + L.get(0, 1) * R.get(1, 2);
        double A2 = L.get(0, 1) * R.get(0, 2) - L.get(0, 0) * R.get(1, 2);

        double B0 = L.get(1, 2) * R.get(2, 2);
        double B1 = L.get(1, 0) * R.get(0, 2) + L.get(1, 1) * R.get(1, 2);
        double B2 = L.get(1, 1) * R.get(0, 2) - L.get(1, 0) * R.get(1, 2);
/*
        // coeffs of quartics
        double F4 = a * (A0 - A1) * (A0 - A1) + b * (B0 - B1) * (B0 - B1) + d; // a*A0^2 - 2*a*A0*A1 + a*A1^2 + b*B0^2 - 2*b*B0*B1 + b*B1^2 + d
        double F3 = 4 * a * A2 * (A0 - A1) + 4 * b * B2 * (B0 -  B1); // 4*a*A0*A2 - 4*a*A1*A2 + 4*b*B0*B2 - 4*b*B1*B2
        double F2 = 2 * a * (A0 * A0 - A1 * A1 + 2 * A2 * A2) + 2 * b * (B0 * B0 - B1 * B1 + 2 * B2 * B2) + 2 * d; // 2*a*A0^2 - 2*a*A1^2 + 4*a*A2^2 + 2*b*B0^2 - 2*b*B1^2 + 4*b*B2^2 + 2*d
        double F1 = 4 * a * A2 * (A0 + A1) + 4 * b * B2 * (B0 + B1); // 4*a*A0*A2 + 4*a*A1*A2 + 4*b*B0*B2 + 4*b*B1*B2
        double F0 = a * (A0 + A1) * (A0 + A1) + b * (B0 + B1) * (B0 + B1) + d; // a*A0^2 + 2*a*A0*A1 + a*A1^2 + b*B0^2 + 2*b*B0*B1 + b*B1^2 + d
*/

        // coeffs of quartics
        double F4 = a*A0*A0 - 2*a*A0*A1 + a*A1*A1 + b*B0*B0 - 2*b*B0*B1 + b*B1*B1 + d;
        double F3 = 4*a*A0*A2 - 4*a*A1*A2 + 4*b*B0*B2 - 4*b*B1*B2;
        double F2 = 2*a*A0*A0 - 2*a*A1*A1 + 4*a*A2*A2 + 2*b*B0*B0 - 2*b*B1*B1 + 4*b*B2*B2 + 2*d;
        double F1 = 4*a*A0*A2 + 4*a*A1*A2 + 4*b*B0*B2 + 4*b*B1*B2;
        double F0 = a*A0*A0 + 2*a*A0*A1 + a*A1*A1 + b*B0*B0 + 2*b*B0*B1 + b*B1*B1 + d;    

        return new double[] {F0, F1, F2, F3, F4};
    }
}

