/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2009 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, 01 December, 2009
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.0, Nov 18, 2009
 * @author        Chittaranjan Tripathy (2007-2009)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.util.*;
import java.io.*;

/**
 * Description of the class
 */
public class myConstantsForRdcs {

    /**
     * Default constructor which creates an instance of this class. This
     * constructor is private to enforce the singleton property of the class in
     * future.
     */
    private myConstantsForRdcs() {
    }

    /**
     * This class parses the canonical residual dipolar coupling and chemical
     * shift anistropy parameters from a file and stores them as constants.
     */
    private static class rdcParameterParser {

        /**
         * The parameters read from the parameter file is stored in this map.
         */
        private Map<String, Double> __parameter_map = new TreeMap<String, Double>();

        /**
         * This is the default constructor which reads from a file the RDC and
         * CSA parameters from a file and stores them as constants.
         */
        public rdcParameterParser() {
            String rdcCsaParameterFile = System.getProperty("user.dir") + System.getProperty("file.separator") +
                    "mytoppar" + System.getProperty("file.separator") + "rdc_constants.txt";
            try {
                Scanner scanner = new Scanner(new File(rdcCsaParameterFile));
                scanner.useDelimiter(System.getProperty("line.separator"));
                while (scanner.hasNextLine()) {
                    String line = myMiscUtilities.stripLineComment(scanner.nextLine()).trim();
                    if (line.length() != 0) {
                        String[] words = line.split("\\s+");
                        //System.out.println("words: " + Arrays.toString(words));
                        __parameter_map.put(words[0], Double.valueOf(words[1]));
                    }
                }
                scanner.close();
            } catch (FileNotFoundException e) {
                System.out.println("Error: Input file " + rdcCsaParameterFile + " not found");
                e.printStackTrace();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * This object when constructed contains the set of constants that are
     * essential for the interpretation of RDCs and CSAs.
     */
    private static rdcParameterParser __parser = new rdcParameterParser();
    
    // Gyromagnetic ratios for different atoms.
    static final double gamma_H = __parser.__parameter_map.get("gamma_H");
    static final double gamma_C = __parser.__parameter_map.get("gamma_C");
    static final double gamma_N = __parser.__parameter_map.get("gamma_N");
    static final double gamma_P = __parser.__parameter_map.get("gamma_P");

    // Canonical bond distances used for RDC scaling.
    static final double rdcNi_HNi = __parser.__parameter_map.get("N(i)-HN(i)");
    static final double rdcCAi_HAi = __parser.__parameter_map.get("CA(i)-HA(i)");
    static final double rdcCi_Nip1 = __parser.__parameter_map.get("C(i)-N(i+1)");
    static final double rdcCi_HNip1 = __parser.__parameter_map.get("C(i)-HN(i+1)");
    static final double rdcCAi_Ci = __parser.__parameter_map.get("CA(i)-C(i)");
    static final double rdcCAi_CBi = __parser.__parameter_map.get("CA(i)-CB(i)");
    static final double rdcCAi_HNip1 = __parser.__parameter_map.get("CA(i)-HN(i+1)");
    static final double rdcCBi_HBi = __parser.__parameter_map.get("CB(i)-HB(i)");
    static final double rdcNi_Ci = __parser.__parameter_map.get("N(i)-C(i)");
    static final double rdcNi_CAi = __parser.__parameter_map.get("N(i)-CA(i)");
    static final double rdcCAi_Nip1 = __parser.__parameter_map.get("CA(i)-N(i+1)");
    static final double rdcCAi_HNi = __parser.__parameter_map.get("CA(i)-HN(i)");
    static final double rdcCi_HAi = __parser.__parameter_map.get("C(i)-HA(i)");

    // for convenience
    static final double rdcN_HN = rdcNi_HNi;
    static final double rdcCA_HA = rdcCAi_HAi;
    static final double rdcCA_C = rdcCAi_Ci;
    static final double rdcC_N = rdcCi_Nip1;
    static final double rdcCA_CB = rdcCAi_CBi;
    static final double rdcCB_HB = rdcCBi_HBi;
    static final double rdcHN_C = rdcCi_HNip1;
    static final double rdcCA_HN = rdcCAi_HNip1;

    /**
     * It contains of the prefactors (i.e., gamma_A * gamma_B / r_{AB}^3)
     * for an internuclear vector A-B. The prefactors are used to (re)scale
     * the RDCs.
     */
    static final Map<String, Double> Dmax = new TreeMap<String, Double>() {

        {            
            put("N_HN", gamma_N * gamma_H / Math.pow(rdcN_HN, 3));
            put("C_N", gamma_N * gamma_C / Math.pow(rdcC_N, 3));
            put("C_HN", gamma_H * gamma_C / Math.pow(rdcHN_C, 3));
            put("CA_C", gamma_C * gamma_C / Math.pow(rdcCA_C, 3));
            put("CA_CB", gamma_C * gamma_C / Math.pow(rdcCA_CB, 3));
            put("CA_HA", gamma_C * gamma_H / Math.pow(rdcCA_HA, 3));
            put("CA_HN", gamma_C * gamma_H / Math.pow(rdcCA_HN, 3));
            put("H_H", gamma_H * gamma_H);
            put("H_P", gamma_H * gamma_P);
        }
    };

    /**
     * It contains of the scaled prefactors (e.g., Dmax("A_B") / Dmax("N_HN"))
     * for an internuclear vector A-B with respect to the scale used.
     */
    static final Map<String, Double> DmaxScaled = new TreeMap<String, Double>();

    // This static initialization block reads the parameter file to get the
    // scaling information and fills in the map for scaled Dmax values appropriately.
    static {
        String programParameterFile = System.getProperty("user.dir") + System.getProperty("file.separator") +
                myInputDataAndParameterManager.getInputDirectory("dirArch.txt") + System.getProperty("file.separator") +
                myInputDataAndParameterManager.getInputParameterDirectory("dirArch.txt") + System.getProperty("file.separator") +
                myInputDataAndParameterManager.getProgramParameterFile("dirArch.txt");
        try {
            Scanner scanner = new Scanner(new File(programParameterFile));
            scanner.useDelimiter(System.getProperty("line.separator"));
            while (scanner.hasNextLine()) {
                String line = scanner.nextLine().trim();
                if (line.length() != 0) {
                    String[] words = line.split("\\s+");

                    // Set the RDC scale properly based on user input
                    if (words[0].equalsIgnoreCase("scaleRdcTo") && words.length == 2) {
                        if (words[1].equalsIgnoreCase("N_HN")) {
                            scaleToNH();
                        } else if (words[1].equalsIgnoreCase("CA_HA")) {
                            scaleToCH();
                        } else if (words[1].equalsIgnoreCase("scaled")) {
                            scaledAlready();
                        } else {
                            System.out.println("Error: In parameterfile " + programParameterFile + " the RDC scaling is not specified");
                            System.exit(1);
                        }
                    }
                }
            }
            scanner.close();
        } catch (FileNotFoundException e) {
            System.out.println("Error: Input file " + programParameterFile + " not found");
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * Set the scaling factor for RDCs such that Dmax(A_B) = scale * Dmax(N_HN).
     */
    public static void scaleToNH() {
        DmaxScaled.clear();
        for (Map.Entry<String, Double> entry : Dmax.entrySet()) {
            DmaxScaled.put(entry.getKey(), entry.getValue() / Dmax.get("N_HN"));
        }
// for example:
//        DmaxScaled.put("N_HN", Dmax.get("N_HN") / Dmax.get("N_HN"));
//        DmaxScaled.put("N_CO", Dmax.get("N_CO") / Dmax.get("N_HN"));
//        DmaxScaled.put("HN_C", Dmax.get("HN_C") / Dmax.get("N_HN"));
//        DmaxScaled.put("CA_CO", Dmax.get("CA_CO") / Dmax.get("N_HN"));
//        DmaxScaled.put("CA_CB", Dmax.get("CA_CB") / Dmax.get("N_HN"));
//        DmaxScaled.put("CA_HA", Dmax.get("CA_HA") / Dmax.get("N_HN"));
//        DmaxScaled.put("CA_HN", Dmax.get("CA_HN") / Dmax.get("N_HN"));
//        DmaxScaled.put("H_H", Dmax.get("H_H") / Dmax.get("N_HN"));
//        DmaxScaled.put("H_P", Dmax.get("H_P") / Dmax.get("N_HN"));
    }

    /**
     * Set the scaling factor for RDCs such that Dmax(A_B) = scale * Dmax(CA_HA).
     */
    public static void scaleToCH() {
        DmaxScaled.clear();
        for (Map.Entry<String, Double> entry : Dmax.entrySet()) {
            DmaxScaled.put(entry.getKey(), entry.getValue() / Dmax.get("CA_HA"));
        }
    }

    /**
     * Set the scaling factors for RDCs to 1.0 to mean that they are scaled already.
     */
    public static void scaledAlready() {
        DmaxScaled.clear();
        for (Map.Entry<String, Double> entry : Dmax.entrySet()) {
            DmaxScaled.put(entry.getKey(), 1.0);
        }
    }

    // The followings are used for the sake of convenience
    static final double nhRatio = DmaxScaled.get("N_HN");
    static final double cahaRatio = DmaxScaled.get("CA_HA");
    // Similarly define other RDC ratios here or just get it from DmaxScaled too.

    /**
     * The main method tests the various parts of this class.
     * 
     * @param args the argument to be passed to the main method
     */
    public static void main(String... args) {
        System.out.println("gamma_H: " + gamma_H);
        System.out.println("gamma_C: " + gamma_C);
        System.out.println("gamma_N: " + gamma_N);
        System.out.println("gamma_P: " + gamma_P);

        System.out.println("N(i)-HN(i): " + rdcNi_HNi);

        System.out.println(__parser.__parameter_map.size());

        System.out.println("Printing Dmax = gamma_A * gamma_B / r_{AB}^3");
        for (Map.Entry<String, Double> entry : Dmax.entrySet()) {
            System.out.println("d: " + entry.getKey() + "  " + entry.getValue());
        }

        System.out.println("Printing DmaxScaled");
        for (Map.Entry<String, Double> entry : DmaxScaled.entrySet()) {
            System.out.println("d: " + entry.getKey() + "  " + entry.getValue());
        }

        System.out.println("New nhRatio: " + nhRatio + "  New cahaRatio: " + cahaRatio);

    }
}
