/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2012 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, August 04, 2012
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */
/**
 * @version 1.0.1, August 04, 2012
 * @author Chittaranjan Tripathy (2007-2012)
 * @email chittu@cs.duke.edu
 * @organization Duke University
 */
/**
 * Package specification
 */
package utilities;

/**
 * Import statement(s)
 */
import java.util.List;
import java.util.ArrayList;
import java.util.Map;
import java.util.TreeMap;
import java.util.Vector;


import analytic.myDipolarCoupling;
import analytic.myProtein;
import analytic.Matrix;
import analytic.myPdbParser;
import analytic.myMatrix;
import analytic.myInputDataAndParameterManager;
import analytic.myAlignmentTensorEstimator;
import analytic.myDipolarCouplingTable;
import analytic.myMiscUtilities;
import analytic.myPair;

/**
 * Description of the class
 */
public class OrientationalRestraintAndStructureFitter {

    List<myProtein> __structures = null;
    String __alignment_medium_name = "medium1";
    String __master_dir = null;
    String __directory_architecture_file = null;
    List<myDipolarCoupling.Type> __rdc_types_for_fit = new ArrayList<>();
    List<myPair<Integer, Integer>> __ranges = new ArrayList<>();
    boolean __print_backbone_only_in_POF = false;
    boolean __print_back_computed_rdc_table_in_xplor_format = false;
    boolean __print_back_computed_rdc_table_in_csrosetta_format = false;
    boolean __print_backbone_dihedrals = false;

    private myProtein trimProteinToBeInTheResidueRanges(myProtein p, List<myPair<Integer, Integer>> ranges) {
        myProtein pRet = new myProtein(p);
        int beginResNum = pRet.getBeginResidueNumber();
        int endResNum = pRet.getEndResidueNumber();

        for (int i = beginResNum; i <= endResNum; i++) {
            boolean iInRange = false;
            for (myPair<Integer, Integer> thisRange : ranges) {
                int lb = thisRange.first();
                int ub = thisRange.second();
                if (lb <= i && i <= ub) {
                    iInRange = true;
                    break;
                }
            }
            if (!iInRange) { // Delete the residue that is not in the ranges
                pRet.removeResidue(i);               
            }
        }
        
        pRet.synchronizeProtein();
        return pRet;
    }

    /**
     * Check for a valid option.
     *
     * @param s string that specifies an option
     * @return return true if the option is valid
     */
    private static boolean validOption(String s) {
        return s.startsWith("-");
    }

    // java utilities/StrictureAligner -pdbfile file1.pdb -pdbfile file2.pdb -atomtypes N CA C -ranges 2 7 12 17
    public void parseCommandLineArgumentsAndInvokeOrientationalRestraintAndStructureFitter(String[] args) {
        String masterDir = null;
        String directoryArchitectureFile = null;
        String pdbFile = null;

        List<String> rdcTypes = new ArrayList<>();
        List<Integer> ranges = new ArrayList<>();

        if (args.length == 0) {
            System.out.println("To learn more about this command, please type: " + new Exception().getStackTrace()[0].getClassName() + " -help");
        }

        // Parse the command line arguments 
        for (int i = 0; i < args.length && validOption(args[i]);) {
            String thisArgument = args[i++];

            if (thisArgument.equals("-help") || thisArgument.equals("-verbose")) {
                String helpString = "Usage: " + new Exception().getStackTrace()[0].getClassName() + " <options> \nwhere possible options include: \n"
                        + "-masterdir <file name>                  Specify the master directory name which contains all the input files and the output will also be redirected to this directory  " + '\n'
                        + "-directoryarchitecturefile <file name>  Specify the directory architecture file name which contains the information about the input and output directories and files under the master directory" + '\n'
                        + "-pdbfile <file name>                    Specify the name of the input PDB file (one model and must end with TER \\n END" + '\n'
                        + "-rdctypes                               Specify the RDC types to be used for superposition" + '\n'
                        + "-ranges                                 Specify the range of residues to be considered" + '\n'
                        + "-printbackboneinpof                     Specify that the protein backbone to be printed in POF (if -noalign is flag is not supplied)" + '\n'
                        + "-printbackcomputedrdcsincsrosettaformat Specify that the back-computed RDCs to be printed in XPLOR format" + '\n'
                        + "-printbackcomputedrdcsincsrosettaformat Specify that the back-computed RDCs to be printed in csRosetta format" + '\n'
                        + "-printbackbonedihedrals                 Specify that the backbone dihedrals to be printed" + '\n'
                        + "-verbose                                Print a synopsis of standard options and return" + '\n'
                        + "-help                                   Print a synopsis of standard options and return" + '\n';
                System.out.println(helpString);
                return;
            } else if (thisArgument.equals("-masterdir")) {
                if (i < args.length) {
                    masterDir = args[i++];
                    if (validOption(masterDir)) {
                        System.out.println("Error: incorrect master directory name or missing argument(s)");
                        System.exit(1);
                    }
                } else {
                    System.out.println("Error: master directory name is not supplied");
                    System.exit(1);
                }
            } else if (thisArgument.equals("-directoryarchitecturefile")) {
                if (i < args.length) {
                    directoryArchitectureFile = args[i++];
                    if (validOption(directoryArchitectureFile)) {
                        System.out.println("Error: incorrect directory architecture file name or missing argument(s)");
                        System.exit(1);
                    }
                } else {
                    System.out.println("Error: directory architecture file name is not supplied");
                    System.exit(1);
                }
            } else if (thisArgument.equals("-pdbfile")) {
                if (i < args.length) {
                    pdbFile = args[i++];
                    if (validOption(pdbFile)) {
                        System.out.println("Error: incorrect pdb file name or missing argument(s)");
                        System.exit(1);
                    }
                } else {
                    System.out.println("Error: pdb file name is not supplied");
                    System.exit(1);
                }
            } else if (thisArgument.equals("-printbackboneinpof")) {
                __print_backbone_only_in_POF = true;
            } else if (thisArgument.equals("-printbackcomputedrdcsinxplorformat")) {
                __print_back_computed_rdc_table_in_xplor_format = true;
            } else if (thisArgument.equals("-printbackcomputedrdcsincsrosettaformat")) {
                __print_back_computed_rdc_table_in_csrosetta_format = true;
            } else if (thisArgument.equals("-printbackbonedihedrals")) {
                __print_backbone_dihedrals = true;
            } else if (thisArgument.equals("-rdctypes")) {
                while (true) {
                    //System.out.println("asdf");
                    if (i < args.length) {
                        String thisType = args[i++];
                        if (validOption(thisType)) {
                            if (rdcTypes.isEmpty()) {
                                System.out.println("Error: incorrect orientational restraint types or missing argument(s)");
                                System.exit(1);
                            } else {
                                i--;
                                break;
                            }
                        } else {
                            rdcTypes.add(thisType);
                        }
                    } else {
                        if (!rdcTypes.isEmpty()) {
                            break;
                        }
                        System.out.println("Error: orientational restraint types are not supplied");
                        System.exit(1);
                    }
                }
            } else if (thisArgument.equals("-ranges")) {
                while (true) {
                    //System.out.println("asdf");
                    if (i < args.length) {
                        String thisLbOrUb = args[i++];
                        if (validOption(thisLbOrUb)) {
                            if (ranges.isEmpty()) {
                                System.out.println("Error: incorrect ranges or missing argument(s)");
                                System.exit(1);
                            } else {
                                i--;
                                break;
                            }
                        } else {
                            ranges.add(Integer.parseInt(thisLbOrUb));
                        }
                    } else {
                        if (!ranges.isEmpty()) {
                            break;
                        }
                        System.out.println("Error: ranges are not supplied");
                        System.exit(1);
                    }
                }
            } else {
                System.out.println("Error: incorrect argument specification: " + thisArgument);
                System.exit(1);
            }
        }

        if (masterDir == null) {
            System.out.println("Error: master directory name is not supplied");
            System.exit(1);
        }
        __master_dir = masterDir;

        if (directoryArchitectureFile == null) {
            System.out.println("Warning: directory architecture file name is not supplied. It is defaulted to dirArch.txt");
            directoryArchitectureFile = "dirArch.txt";
        }
        __directory_architecture_file = directoryArchitectureFile;

        if (pdbFile == null) {
            System.out.println("Error: input pdb file must be supplied");
            System.exit(1);
        }

        if (rdcTypes.isEmpty()) {
            System.out.println("Error: RDC types not supplied");
            System.exit(1);
        } else {
            for (String t : rdcTypes) {
                t = t.trim();                
                switch (t) {
                    case "N_HN":
                        __rdc_types_for_fit.add(myDipolarCoupling.Type.N_HN);
                        break;
                    case "CA_C":
                        __rdc_types_for_fit.add(myDipolarCoupling.Type.CA_C);
                        break;
                    case "CA_HA":
                        __rdc_types_for_fit.add(myDipolarCoupling.Type.CA_HA);
                        break;
                    case "C_N":
                        __rdc_types_for_fit.add(myDipolarCoupling.Type.C_N);
                        break;
                    case "C_HN":
                        __rdc_types_for_fit.add(myDipolarCoupling.Type.C_HN);
                        break;
                    default:
                        System.out.println("Error: incorrect RDC type supplied");
                        System.exit(1);
                }
            }
        }
        
        if (ranges.isEmpty()) {
            System.out.println("Warning: ranges are not supplied. Default range is the entire set of residues in the structure(s)");
        } else {
            if (ranges.size() % 2 != 0) {
                System.out.println("Error: incorrect ranges or missing argument(s)");
            }
        }

        for (int i = 0; i < ranges.size(); i += 2) {
            int lb = ranges.get(i);
            int ub = ranges.get(i + 1);
            __ranges.add(new myPair<>(lb, ub));
        }

        // Check if the intervals overlap. If so then report to the user.
        if (myMiscUtilities.existsOverlap(__ranges)) {
            System.out.println("Error: there exists overlap in the intervals");
            System.exit(1);
        }        
        
        setUpMasterDirectory:
        {            
            myInputDataAndParameterManager mdpm = myInputDataAndParameterManager.getInstance();
            mdpm.setMasterDirectory(__master_dir);
            mdpm.setDirectoryArchitectureFile(__directory_architecture_file);            
            
        }
        
        ReadProtein:
        {
            List<myProtein> lp = new ArrayList<>();
            myPdbParser pParser = new myPdbParser(pdbFile);
            while (pParser.hasNextProtein()) {
                myProtein p = new myProtein(pParser.nextProtein());
                if (!__ranges.isEmpty()) {
                    p = trimProteinToBeInTheResidueRanges(p, __ranges);
                }
                lp.add(p);
            }
            __structures = lp;            
        }
        
        // The command line arguments are parsed, and the protein models are read. Now do the tasks.
        doTasks();
    }

    private void doTasks() {     
        myInputDataAndParameterManager thisParameterManager = myInputDataAndParameterManager.getInstance();

        myDipolarCouplingTable thisRdcCsaTable = thisParameterManager.getDipolarCouplingTable();
        thisRdcCsaTable.print();

        String[] alignmentMediaNames = thisRdcCsaTable.getAllAlignmentMediaNames();
        
        for (String s : alignmentMediaNames) {
            System.out.println("-------" + s);
        }
        //System.exit(1);
        
        for (myProtein thisStructure : __structures) {
            System.out.println("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");
            System.out.println("**************Analyzing one structure****************************");
            System.out.println("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");
            for (String thisAlignmentMedium : alignmentMediaNames) {
                Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcTableForThisMedium = thisRdcCsaTable.getRdcMapForThisMedium(thisAlignmentMedium);
                Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMediumBackCal = new TreeMap<>();

                double[] saupe = new double[4];
                Matrix rotToPOF = myAlignmentTensorEstimator.bestFitUsingSVD(new myProtein(thisStructure), rdcTableForThisMedium,
                        new Vector<>(__rdc_types_for_fit), rdcCsaTableForThisMediumBackCal, saupe);
                System.out.println("The rotation matrix for rotating the structure to a POF is");
                rotToPOF.print(5, 12);

                myProtein thisFragmentInPOF = new myProtein(thisStructure);
                thisFragmentInPOF.rotate(new myMatrix(rotToPOF.getArrayCopy()));
                if (__print_backbone_only_in_POF) {
                    myProtein backboneInPOF = myProtein.extractBackbone(thisFragmentInPOF);
                    System.out.println("Printing the protein backbone in POF");
                    backboneInPOF.print();
                } else {
                    System.out.println("Printing the protein in POF");
                    thisFragmentInPOF.print();
                }

                if (__print_back_computed_rdc_table_in_xplor_format) {
                    System.out.println("Printing backcomputed RDCs in XPLOR format");
                    myDipolarCouplingTable.printInXplorFormat(rdcCsaTableForThisMediumBackCal);
                }

                if (__print_back_computed_rdc_table_in_csrosetta_format) {
                    System.out.println("Printing backcomputed RDCs in CSRosetta format (warn: without RDC scaling)");
                    myDipolarCouplingTable.printInCSRosettaFormat(rdcCsaTableForThisMediumBackCal, false);
                }
            }

            if (__print_backbone_dihedrals) {
                System.out.println("Printing the backbone dihedrals for this structure");
                thisStructure.printBackboneDihedrals();
            }
        }
    }
        
    public static void main(String... args) {
        long startTime = System.currentTimeMillis(); // Log the start time
        
//        String arguments = "-masterdir EXPERIMENTS_TO_TEST_UTILITIES/1ghh -pdbfile EXPERIMENTS_TO_TEST_UTILITIES/1ghh/1GHHModel1And2.pdb "
//                + "-rdctypes N_HN CA_C CA_HA -printbackcomputedrdcsincsrosettaformat -printbackcomputedrdcsinxplorformat -ranges 20 30 "
//                + "-printbackboneinpof -printbackbonedihedrals";//
//        args = arguments.split("\\s+"); // Give a set of arguments for testing
//        
//        for (String s : args) {
//            System.out.println(s);
//        }

        OrientationalRestraintAndStructureFitter ftr = new OrientationalRestraintAndStructureFitter();
        ftr.parseCommandLineArgumentsAndInvokeOrientationalRestraintAndStructureFitter(args);
        
        long endTime = System.currentTimeMillis(); // Log the end time
        double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
        System.out.println("Time elapsed: " + totalTime + " minutes");                  
    }
    
}
