/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2012 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, August 04, 2012
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.1, August 04, 2012
 * @author        Chittaranjan Tripathy (2007-2012)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.io.*;
import java.util.*;

/**
 * Description of the class
 */
public class myRama {
/*
    private static final myRama __instance = new myRama();

    public static synchronized myRama getInstance() {
        return __instance;
    }

    @Override
    public Object clone() throws CloneNotSupportedException {
        throw new CloneNotSupportedException();
    }

    private myRama() {
        //throw new AssertionError(); // to indicate that the object construction failed
    }
*/

    private Vector<myTriple<Double, Double, Double>> __rama_data_raw = null;
    private byte[][] __rama_map = null;
    private Type __type = null;
    
    private myRedPencil thisRedPencil = new myRedPencil("off"); // Let's default each local red pencil off    

    /**
     * Ramachandran Map Type.
     */
    public static enum Type {
        GENERAL, GLY, PRO, PREPRO
    }

    private static double __favored_threshold; // = 0.02 covers 98.00%
    private static double __allowed_threshold; // = 0.0005 covers 99.95% in general case, and 0.0020 covers 99.80% for gly, pro and prepro
    //private static double __gly_pro_prepro_allowed_threshold = 0.0020; // 99.80%

    private static final byte __outlier = 0;
    private static final byte __favored = 1;
    private static final byte __allowed = 2;
    //private static byte __gly_pro_prepro_allowed = 2;

    Vector<Vector<myPair<Integer, Integer>>> __all_psi_intervals_favored = null;
    Vector<Vector<myPair<Integer, Integer>>> __all_psi_intervals_allowed = null;
    
    Vector<myPair<Integer, Integer>> __all_phi_intervals_allowed_projected = null;
    Vector<myPair<Integer, Integer>> __all_phi_intervals_favored_projected = null;

    /**
     * Return the type of Ramachandran region.
     */
    public Type getType() {
        return __type;
    }

    /**
     * This constructor constructs a Ramachandran map from the data file for a
     * specified type of residue.
     *
     * @param ramaDataFileName
     * @param type
     */
    public myRama(String ramaDataFileName, String type) {
        __rama_data_raw = parseRamaFile(ramaDataFileName);

        // These parameters are from Richardsons'
        __favored_threshold = 0.0200;

        type = type.trim().toUpperCase();
        if (type.equalsIgnoreCase("general")) {
            __type = Type.GENERAL;
            __allowed_threshold = 0.0005;
        } else if (type.equalsIgnoreCase("gly")) {
            __type = Type.GLY;
            __allowed_threshold = 0.0020;
        } else if (type.equalsIgnoreCase("pro")) {
            __type = Type.PRO;
            __allowed_threshold = 0.0020;
        } else if (type.equalsIgnoreCase("prepro")) {
            __type = Type.PREPRO;
            __allowed_threshold = 0.0020;
        } else {
            System.out.println("Error: Invalid type specification");
            System.exit(1);
        }

        // Represent the Ramachandran as a 2D map
        __rama_map = new byte[361][361]; // 0 to +/-180
        for (int i = 0; i < 361; i++) {
            for (int j = 0; j < 361; j++) {
                __rama_map[i][j] = __outlier;
            }
        }

        preprocessRamaMap();

        __all_psi_intervals_allowed = extractAllPsiIntervals("allowed");
        __all_psi_intervals_favored = extractAllPsiIntervals("favored");

        __all_phi_intervals_allowed_projected = extractAllProjectedPhiIntervals("allowed");
        __all_phi_intervals_favored_projected = extractAllProjectedPhiIntervals("favored");
    }

    private void preprocessRamaMap() {
        for (myTriple<Double, Double, Double> ppt : __rama_data_raw) {
            int thisPhi = (int) (ppt.first().doubleValue());
            int thisPsi = (int) (ppt.second().doubleValue());

            // transform to origin so that all values are >= 0
            thisPhi += 180;
            thisPsi += 180;

            if (ppt.third() > __favored_threshold) {
                __rama_map[thisPhi][thisPsi] = __favored;
            } else if (ppt.third() > __allowed_threshold) {
                __rama_map[thisPhi][thisPsi] = __allowed;
            } else {
                __rama_map[thisPhi][thisPsi] = __outlier;
            }
        }

        // Fill missing column
        for (int i = 1; i < 360; i++) {
            for (int j = 1; j < 360; j++) {
                if (__rama_map[i][j] == __outlier
                        && __rama_map[i][j - 1] != __outlier
                        && __rama_map[i][j + 1] != __outlier) {
                    if (__rama_map[i][j - 1] == __favored || __rama_map[i][j + 1] == __favored) {
                        __rama_map[i][j] = __favored;
                    } else {
                        __rama_map[i][j] = __allowed;
                    }
                }
            }
        }

        // Fill missing row
        for (int i = 1; i < 360; i++) {
            for (int j = 1; j < 360; j++) {
                if (__rama_map[i][j] == __outlier
                        && __rama_map[i - 1][j] != __outlier
                        && __rama_map[i + 1][j] != __outlier) {
                    if (__rama_map[i - 1][j] == __favored || __rama_map[i + 1][j] == __favored) {
                        __rama_map[i][j] = __favored;
                    } else {
                        __rama_map[i][j] = __allowed;
                    }
                }
            }
        }

        // Set the first column and the last column
        for (int j = 0; j <= 360; j++) {
            if (__rama_map[1][j] == __outlier) {
                __rama_map[0][j] = __outlier;
            } else {
                __rama_map[0][j] = __rama_map[1][j];
            }

            if (__rama_map[359][j] == __outlier) {
                __rama_map[360][j] = __outlier;
            } else {
                __rama_map[360][j] = __rama_map[359][j];
            }
        }

        // Set the first row and the last row
        for (int i = 0; i <= 360; i++) {
            if (__rama_map[i][1] == __outlier) {
                __rama_map[i][0] = __outlier;
            } else {
                __rama_map[i][0] = __rama_map[i][1];
            }

            if (__rama_map[i][359] == __outlier) {
                __rama_map[i][360] = __outlier;
            } else {
                __rama_map[i][360] = __rama_map[i][359];
            }
        }
    }


    private static boolean isValidNumber(String s) {
        try {
            Double.parseDouble(s);
        } catch (NumberFormatException e) {
            //e.printStackTrace();
            return false;
        }
        return true;
    }

    public static Vector<myTriple<Double, Double, Double>> parseRamaFile(String ramaDataFileName) {
        Vector<myTriple<Double, Double, Double>> proRamaData = new Vector<myTriple<Double, Double, Double>>();
        try {
            Scanner scanner = new Scanner(new File(ramaDataFileName));
            scanner.useDelimiter(System.getProperty("line.separator"));
            while (scanner.hasNextLine()) {
                String line = myMiscUtilities.stripLineComment(scanner.nextLine()).trim();
                if (line.length() != 0) {
                    String[] words = line.split("\\s+");
                    //System.out.println("words: " + Arrays.toString(words));

                    if (!(words.length == 3 && isValidNumber(words[0]) && isValidNumber(words[1]) && isValidNumber(words[2]))) {
                        continue;
                    }
                    myTriple<Double, Double, Double> thisPhiPsiDensityTriple = new myTriple<Double, Double, Double>(Double.parseDouble(words[0]), Double.parseDouble(words[1]), Double.parseDouble(words[2]));
                    //__parameter_map.put(words[0], words[1]);
                    proRamaData.add(thisPhiPsiDensityTriple);
                }
            }
            scanner.close();
        } catch (FileNotFoundException e) {
            System.out.println("Error: Input file " + ramaDataFileName + " not found");
            e.printStackTrace();
        } 

        return proRamaData.isEmpty() ? null : proRamaData;
    }

    private boolean isValidDihedralValue(double d) {
        return (-180 <= d && d <= 180) ? true : false;
    }

    private void trapInvalidDihedralValue(double d) {
        if (!(isValidDihedralValue(d))) {
            System.out.println("Error: Invalid value of " + d + " for a dihedral");
            System.exit(1);
        }
    }

    private Vector<myPair<Integer, Integer>> extractAllPsiIntervalsGivenPhi(double phi, String favoredOrAllowed) {
        trapInvalidDihedralValue(phi);

        favoredOrAllowed = favoredOrAllowed.trim().toUpperCase();
        int thisPhi = (int) Math.round(phi + 180); // Transform the angle first

        if (thisPhi > 360) {
            System.out.println("Error: invalid value for the argument phi");
            System.exit(1);
        }

        Vector<myPair<Integer, Integer>> vectorOfIntervals = new Vector<myPair<Integer, Integer>>();
        Vector<Integer> vd = new Vector<Integer>();

        if (favoredOrAllowed.equalsIgnoreCase("favored")) {
            for (int j = 0; j <= 360; j++) {
                if (j == 0) {
                    if (__rama_map[thisPhi][j] == __favored) {
                        vd.add(j);
                    }
                } else if (j == 360) {
                    if (__rama_map[thisPhi][j] == __favored) {
                        vd.add(j);
                    }
                } else { // j \in (0, 360)
                    if (__rama_map[thisPhi][j] != __favored && __rama_map[thisPhi][j - 1] == __favored) {
                        vd.add(j - 1);
                    } else if (__rama_map[thisPhi][j] == __favored && __rama_map[thisPhi][j - 1] != __favored) {
                        vd.add(j);
                    }
                }
            }
        } else if (favoredOrAllowed.equalsIgnoreCase("allowed")) {
            for (int j = 0; j <= 360; j++) {
                if (j == 0) {
                    if ((__rama_map[thisPhi][j] == __favored || __rama_map[thisPhi][j] == __allowed/*--- allowed for gly pro*/)) {
                        vd.add(j);
                    }
                } else if (j == 360) {
                    if (__rama_map[thisPhi][j] == __favored || __rama_map[thisPhi][j] == __allowed) {
                        vd.add(j);
                    }
                } else { // j \in (0, 360)
                    if (!(__rama_map[thisPhi][j] == __favored || __rama_map[thisPhi][j] == __allowed) &&
                            (__rama_map[thisPhi][j - 1] == __favored || __rama_map[thisPhi][j - 1] == __allowed)) {
                        vd.add(j - 1);
                    } else if ((__rama_map[thisPhi][j] == __favored  || __rama_map[thisPhi][j] == __allowed)  &&
                            (!(__rama_map[thisPhi][j - 1] == __favored || __rama_map[thisPhi][j - 1] == __allowed))) {
                        vd.add(j);
                    }
                }
            }
        }

        //for (Integer i : vd) System.out.println("-> " + (i - 180));
        if (vd.size() % 2 != 0) { // vd's size must be even since it stores a number of intervals
            System.out.println("Error: invalid interval specification");
            System.exit(1);
        }

        for (int i = 0; i < vd.size(); i += 2) {
            vectorOfIntervals.add(new myPair<Integer, Integer>(vd.elementAt(i) - 180, vd.elementAt(i + 1) - 180));
        }

        return vectorOfIntervals;
    }

    private Vector<Vector<myPair<Integer, Integer>>> extractAllPsiIntervals(String favoredOrAllowed) {
        Vector<Vector<myPair<Integer, Integer>>> allIntervals = new Vector<Vector<myPair<Integer, Integer>>>();
        for (int phi = -180; phi <= 180; phi++) {
            Vector<myPair<Integer, Integer>> vp = extractAllPsiIntervalsGivenPhi(phi, favoredOrAllowed);
            allIntervals.add(vp);
        }
        return allIntervals;
    }

    private Vector<myPair<Integer, Integer>> extractAllProjectedPhiIntervals(String favoredOrAllowed) {
        Vector<myPair<Integer, Integer>> phiIntervals = new Vector<myPair<Integer, Integer>>();
        Vector<Vector<myPair<Integer, Integer>>> allIntervals = null;

        if (favoredOrAllowed.trim().equalsIgnoreCase("favored")) {
            allIntervals = __all_psi_intervals_favored;
        } else if (favoredOrAllowed.trim().equalsIgnoreCase("allowed")) {
            allIntervals = __all_psi_intervals_allowed;
        } else {
            System.out.println("Error: invalid Ramachandran region (favored/allowed) specification");
            System.exit(1);
        }

        int lb = 999, ub = 999;
        boolean lbFound = false, ubFound = false;
        int currPhi = -180;

        for (Vector<myPair<Integer, Integer>> vi : allIntervals) {
            if (lbFound == false && ubFound == false) {
                if (vi.size() != 0) {
                    lbFound = true;
                    lb = currPhi;
                }
            }
            if (lbFound == true && ubFound == false) {
                if (vi.size() == 0) {
                    ubFound = true;
                    ub = currPhi - 1;

                    phiIntervals.add(new myPair<Integer, Integer>(lb, ub));
                    lbFound = ubFound = false;
                    lb = ub = 999;
                } else if (currPhi == 180 && lbFound == true) {
                    ub = currPhi;
                    phiIntervals.add(new myPair<Integer, Integer>(lb, ub));
                }
            }
            currPhi++;
        }

        return phiIntervals;
    }

    public Vector<myPair<Integer, Integer>> queryAllPsiIntervalsGivenPhi(double phi, String favoredOrAllowed) {
        trapInvalidDihedralValue(phi);
        Vector<Vector<myPair<Integer, Integer>>> allIntervals = null;

        if (favoredOrAllowed.trim().equalsIgnoreCase("favored")) {
            allIntervals = __all_psi_intervals_favored;
        } else if (favoredOrAllowed.trim().equalsIgnoreCase("allowed")) {
            allIntervals = __all_psi_intervals_allowed;
        } else {
            System.out.println("Error: invalid Ramachandran region (favored/allowed) specification");
            System.exit(1);
        }

        int thisPhi = (int) Math.round(phi + 180); // Transform the angle first // offset in the vector

        return allIntervals.elementAt(thisPhi);
    }

    public Vector<myPair<Integer, Integer>> queryAllProjectedPhiIntervals(String favoredOrAllowed) {
        Vector<myPair<Integer, Integer>> allProjectedPhiIntervals = null;

        if (favoredOrAllowed.trim().equalsIgnoreCase("favored")) {
            allProjectedPhiIntervals = __all_phi_intervals_favored_projected;
        } else if (favoredOrAllowed.trim().equalsIgnoreCase("allowed")) {
            allProjectedPhiIntervals = __all_phi_intervals_allowed_projected;
        } else {
            System.out.println("Error: invalid Ramachandran region (favored/allowed) specification");
            System.exit(1);
        }

        return allProjectedPhiIntervals;
    }

    public void printAllProjectedPhiIntervals(String favoredOrAllowed) {
        Vector<myPair<Integer, Integer>> allIntervals = queryAllProjectedPhiIntervals(favoredOrAllowed);

        for (myPair<Integer, Integer> thisInterval : allIntervals) {
            System.out.print("(" + thisInterval.first() + ", " + thisInterval.second() + ")  ");
        }
        System.out.println();
    }

    public void printAllPsiIntervalsGivenPhi(double phi, String favoredOrAllowed) {
        Vector<myPair<Integer, Integer>> vp = queryAllPsiIntervalsGivenPhi(phi, favoredOrAllowed);

        System.out.print("phi: " + phi + "    #interval: " + vp.size() + "    ");
        for (myPair<Integer, Integer> p : vp) {
            System.out.print("(" + p.first() + ", " + p.second() + ")  ");
        }
        System.out.println();
    }

    public void printAllPsiIntervals(String favoredOrAllowed) {
        for (int phi = -180; phi <= 180; phi++) {
            Vector<myPair<Integer, Integer>> vp = queryAllPsiIntervalsGivenPhi(phi, favoredOrAllowed);
            System.out.print("phi: " + phi + "    #interval: " + vp.size() + "    ");
            for (myPair<Integer, Integer> p : vp) {
                System.out.print("(" + p.first() + ", " + p.second() + ")  ");
            }
            System.out.println();
        }

    }

    public void printRamaMap() {
        for (int j = 360; j >= 0; j--) {
            for (int i = 0; i <= 360; i++) {
                if (__rama_map[i][j] == __outlier) {
                    System.out.print(" ");
                } else {
                    System.out.print(__rama_map[i][j]);
                }
            }
            System.out.println();
        }
    }

    public void printRamaMapFavoredRegion() {
        for (int j = 360; j >= 0; j--) {
            for (int i = 0; i <= 360; i++) {
                if (__rama_map[i][j] == __favored) {
                    System.out.print(__rama_map[i][j]);
                } else {
                    System.out.print(" ");
                }
            }
            System.out.println();
        }
    }

    public void printRamaMapAllowedRegion() {
        for (int j = 360; j >= 0; j--) {
            for (int i = 0; i <= 360; i++) {
                if (__rama_map[i][j] == __favored) {
                    System.out.print(__rama_map[i][j]);
                } else if (__rama_map[i][j] == __allowed) {
                    System.out.print(__rama_map[i][j]);
                } else {
                    System.out.print(" ");
                }
            }
            System.out.println();
        }
    }

    public Vector<Double> chopPsiIntervals(double phi, String favoredOrAllowed, double chopSize) {
        trapInvalidDihedralValue(phi);
        Vector<myPair<Integer, Integer>> thisPsiIntervalVector = queryAllPsiIntervalsGivenPhi(phi, favoredOrAllowed);
        int maxNumberOfPoints = (int) Math.ceil(360.0 / chopSize) + 2 * thisPsiIntervalVector.size();
        Vector<Double> vd = new Vector<Double>(maxNumberOfPoints);

        for (myPair<Integer, Integer> thisPsiInterval : thisPsiIntervalVector) {
            for (double i = thisPsiInterval.first(); i < thisPsiInterval.second(); i += chopSize) {
                vd.add(new Double(i));
            }
            vd.add(new Double(thisPsiInterval.second()));
        }

        return vd;
    }

    public Vector<Double> chopPhiIntervals(String favoredOrAllowed, double chopSize) {
        Vector<myPair<Integer, Integer>> thisPhiIntervalVector = queryAllProjectedPhiIntervals(favoredOrAllowed);
        int maxNumberOfPoints = (int) Math.ceil(360.0 / chopSize) + 2 * thisPhiIntervalVector.size();
        Vector<Double> vd = new Vector<Double>(maxNumberOfPoints);

        for (myPair<Integer, Integer> thisPhiInterval : thisPhiIntervalVector) {
            for (double i = thisPhiInterval.first(); i < thisPhiInterval.second(); i += chopSize) {
                vd.add(new Double(i));
            }
            vd.add(new Double(thisPhiInterval.second()));
        }

        return vd;
    }

    public boolean isFavoredPhi(double phi) {
        trapInvalidDihedralValue(phi);
        Vector<myPair<Integer, Integer>> thisPhiIntervalVector = queryAllProjectedPhiIntervals("favored");
        if (thisPhiIntervalVector.size() == 0) {
            return false;
        }
        for (myPair<Integer, Integer> thisPhiInterval : thisPhiIntervalVector) {
            int lb = thisPhiInterval.first();
            int ub = thisPhiInterval.second();
            if (lb <= phi && phi <= ub) {
                return true;
            }
        }        
        return false;
    }

    public boolean isAllowedPhi(double phi) {
        trapInvalidDihedralValue(phi);
        Vector<myPair<Integer, Integer>> thisPhiIntervalVector = queryAllProjectedPhiIntervals("allowed");
        if (thisPhiIntervalVector.size() == 0) {
            return false;
        }
        for (myPair<Integer, Integer> thisPhiInterval : thisPhiIntervalVector) {
            int lb = thisPhiInterval.first();
            int ub = thisPhiInterval.second();
            if (lb <= phi && phi <= ub) {
                return true;
            }
        }        
        return false;
    }

    public boolean isFavoredPsi(double phi, double psi) {
        trapInvalidDihedralValue(phi);
        trapInvalidDihedralValue(psi);
        Vector<myPair<Integer, Integer>> thisPsiIntervalVector = queryAllPsiIntervalsGivenPhi(phi, "favored");
        if (thisPsiIntervalVector.size() == 0) {
            return false;
        }

        for (myPair<Integer, Integer> thisPsiInterval : thisPsiIntervalVector) {
            int lb = thisPsiInterval.first();
            int ub = thisPsiInterval.second();
            if (lb <= phi && phi <= ub) {
                return true;
            }
        }

        return false;
    }

    public boolean isAllowedPsi(double phi, double psi) {
        trapInvalidDihedralValue(phi);
        trapInvalidDihedralValue(psi);
        Vector<myPair<Integer, Integer>> thisPsiIntervalVector = queryAllPsiIntervalsGivenPhi(phi, "allowed");
        if (thisPsiIntervalVector.size() == 0) {
            return false;
        }

        for (myPair<Integer, Integer> thisPsiInterval : thisPsiIntervalVector) {
            int lb = thisPsiInterval.first();
            int ub = thisPsiInterval.second();
            if (lb <= phi && phi <= ub) {
                return true;
            }
        }

        return false;
    }


    
    
//    TODO:
//   done     (1) Implement the query to contain a vector of myPairs of all the intervals preprocessed beforehand
//        (2) Implement a chopped that returns a set of points sampled for a psi given a phi etc...vice versa
//        (3) think how to implement a good sampler of Rama for psi given PHI



    public static void main(String... args) {
        String ramaDirectory = System.getProperty("user.dir") + System.getProperty("file.separator")
                + "myRamachandran" + System.getProperty("file.separator") + "pct" + System.getProperty("file.separator")
                + "rama" + System.getProperty("file.separator");
        String ramaProFileName = ramaDirectory + "rama500-prepro.data";//"rama500-pro.data";

        int helixCounter = 1;
        String helixPdbFileName = "helix" + helixCounter + ".pdb";
        System.out.println(helixPdbFileName);
        //System.exit(1);



        Vector<myTriple<Double, Double, Double>> proRamaData = parseRamaFile(ramaProFileName);

//        for (myTriple<Double, Double, Double> d : proRamaData) {
//
//            if (d.third() > 0.0005)//0.0200)
//            System.out.println(d.first() + "  " + d.second() + "  " + d.third());
//        }

        myRama mr = new myRama(ramaProFileName, "prepro");
        //mr.printRamaMap();
        //mr.printRamaMapFavoredRegion();
        mr.printRamaMapAllowedRegion();

        mr.printAllPsiIntervals("allowed");
        mr.printAllProjectedPhiIntervals("favored");

        mr.printAllPsiIntervalsGivenPhi(-113.27877266667794, "allowed");

        Vector<Double> vd = mr.chopPsiIntervals(181, "allowed", 4.79);//   __favored_threshold, ramaDirectory, __favored_threshold)

        for (Double d : vd) {
            System.out.print(d + ", ");
        }
        System.out.println("\nsize: " + vd.size());

    }


}
