/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2012 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, August 04, 2012
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.1, August 04, 2012
 * @author        Chittaranjan Tripathy (2007-2012)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.io.*;
import java.util.*;

/**
 * Description of the class
 */
public class myPool {

    private myAtom[] __begin_pose = new myAtom[3];  // N, CA, C
    private myAtom[] __target_pose = new myAtom[3]; // N, CA, C

    private myInputDataAndParameterManager thisParameterManager = myInputDataAndParameterManager.getInstance();        
    private myLoopConfigurationManager thisLCM = null; //new myLoopConfigurationManager();


    /**
     * A random number generator that is used to model experimental errors in
     * dipolar couplings.
     */
    private static final Random rr = new Random(myMiscConstants.CRTRandomSeed);

    private static myProtein __global_fold_in_pof = null;
    private int __begin_anchor_residue_number = 0;
    private int __end_anchor_residue_number = 0;

    //myResidue __target_pose_residue = null;


    public myPool() {        
        doCleanUp();
        thisLCM = new myLoopConfigurationManager();
    }
    
    public myPool(int loopBeginResidueNumber, int loopEndResidueNumber) {        
        doCleanUp();
        updateLoopConfigurationFileToShowWhichLoopToBeComputedNow(loopBeginResidueNumber, loopEndResidueNumber);        
        thisLCM = new myLoopConfigurationManager();
    }
    
    private void doCleanUp() {
        doCleanUpsOfOutputFiles:
        {
            myInputDataAndParameterManager thisParameterManager = myInputDataAndParameterManager.getInstance();
            File bestFragmentFile = new File(thisParameterManager.getBestFragmentPdbFile());
            if (bestFragmentFile.exists()) {
                bestFragmentFile.delete();
            }

            File solutionTreeLogFile = new File(thisParameterManager.getLogFile());
            if (solutionTreeLogFile.exists()) {
                solutionTreeLogFile.delete();
            }
        }
    }
    
    private void updateLoopConfigurationFileToShowWhichLoopToBeComputedNow(int loopBeginResidueNumber, int loopEndResidueNumber) {
        String tag = "@computeNow";
        String s = tag + "(" + Integer.toString(loopBeginResidueNumber) + ", " + Integer.toString(loopEndResidueNumber) + ")";
        
        String loopConfFileName = thisParameterManager.getLoopInfoFile();

        try {
            replaceInFile(loopConfFileName, tag, s);
        } catch (FileNotFoundException e) {
            System.out.println("Error: Input file " + loopConfFileName + " not found");
            e.printStackTrace();
        } catch (IOException e) {
            System.out.println("IOException occured. The stack trace is: ");
            e.printStackTrace();
        }
    }
        
    public void replaceInFile(String fileName, String tag, String s) throws IOException, FileNotFoundException {        
        File file = new File(fileName);
        File tempFile = new File(fileName + ".tmp"); //File.createTempFile(fileName + "XX", ".tmp");
        FileWriter fw = new FileWriter(tempFile);

        Reader fr = new FileReader(file);
        BufferedReader br = new BufferedReader(fr);

        boolean flag = false;
        
        while (br.ready()) {
            String line = br.readLine();
            String line2 = myMiscUtilities.stripLineComment(line).trim();  
            if (line2.startsWith(tag)) {
                flag = true;
                line = s;
            }            
            fw.write(line + "\n");
        }
        
        if (flag == false) {
            fw.write(s + "\n");
        }

        fw.close();
        br.close();
        fr.close();

        // Finally replace the original file.
        if (file.exists()) {
            file.delete();
        }
        if (tempFile.renameTo(file)) {                    
            //System.out.println("File rename succeeded"); 
            //System.exit(1);
        } else {
            //System.out.println("File rename did NOT succeed"); 
            //System.exit(1);        
        }
    }
    
    
    
    
    

    public myAtom[] getBeginPose() {
        return __begin_pose;
    }

    public myAtom[] getTargetPose() {
        return __target_pose;
    }

    private static myDipolarCoupling findInRdcVector(final Vector<myDipolarCoupling> dcVec, int residueNumber) {
        for (myDipolarCoupling dc : dcVec) {
            if (dc.getResidueNumber() == residueNumber) {
                return dc;
            }
        }
        return null;
    }
    
    private static int[] fillInMissingRdcs(final mySseInfo thisSseId, final Vector<myDipolarCoupling> expRdcs,
            final Vector<myDipolarCoupling> backCalRdcs, Vector<myDipolarCoupling> filledRdcVec) {
        assert filledRdcVec != null : "Error: null RDC vector passed as argument";
        filledRdcVec.clear(); // clear all old data if there
        int[] expOrBackCal = new int[thisSseId.getEndResidueNumber() - thisSseId.getBeginResidueNumber() + 1];

        // We assume that expRdcs and backCalRdcs are sorted according to their ResidueNumber keys and they are the same type of RDCs
        for (int i = thisSseId.getBeginResidueNumber(); i <= thisSseId.getEndResidueNumber(); i++) {
            myDipolarCoupling dcExp = findInRdcVector(expRdcs, i);
            if (dcExp != null) {
                myDipolarCoupling dcExpClone = new myDipolarCoupling(dcExp);
                filledRdcVec.add(dcExpClone);
                expOrBackCal[i - thisSseId.getBeginResidueNumber()] = 1; // means experimental RDC is used
            } else { // use the back calculated RDC to fill in
                myDipolarCoupling ddd = expRdcs.elementAt(0);
                ddd = new myDipolarCoupling(ddd.getFirstAtomName(), i, ddd.getSecondAtomName(), i, -999, 0.1);
                filledRdcVec.add(ddd);
                expOrBackCal[i - thisSseId.getBeginResidueNumber()] = 0; // means back computed RDC is used
/*                myDipolarCoupling dcCal = findInRdcVector(backCalRdcs, i);
                if (dcCal != null) {
                    myDipolarCoupling dcCalClone = new myDipolarCoupling(dcCal);
                    filledRdcVec.add(dcCalClone);
                    expOrBackCal[i - thisSseId.getBeginResidueNumber()] = 0; // means back computed RDC is used
                } else { // no back-computed RDC found for the residue
                    System.out.println(backCalRdcs.size());
                    for (myDipolarCoupling dc : backCalRdcs)
                        System.out.println(dc.toString());
                    System.out.println("Error: While filling the missing RDCs no back-computed RDC for the residue " + i);
                    System.exit(1);
                }
*/
            }
        }
        return expOrBackCal;
    }

    private double generateSampledRdcVector(final Vector<myDipolarCoupling> rdcExpWithMissingRdcsFilled,
            /*final int[] expOrBackCal,*/ final double rmsdThreshold, Vector<myDipolarCoupling> rdcSampled, Random rr) {
//        System.out.println("rdcExpWithMissingRdcsFilled.size():  " + rdcExpWithMissingRdcsFilled.size() + "    " +
//                "rdcSampled.size(): " + rdcSampled.size() + "    " + "expOrBackCal.length: " + expOrBackCal.length);
        //assert rdcExpWithMissingRdcsFilled.size() == rdcSampled.size() && rdcExpWithMissingRdcsFilled.size() == expOrBackCal.length : "Error: The lengths of the sampled RDC vector and the filled experimental RDC vector differ";

        double sumOfSquaredDeviationThreshold = /*countExperimentalRdcs(expOrBackCal)*/ rdcExpWithMissingRdcsFilled.size() * rmsdThreshold * rmsdThreshold;
        double squaredSum = 0;
        do {
            squaredSum = 0.0;
            for (int i = 0; i < rdcExpWithMissingRdcsFilled.size(); i++) {
                if (rdcSampled.elementAt(i).getRdc() == -999) {
                    ;
                } else {
                    rdcSampled.elementAt(i).setRdc(rdcExpWithMissingRdcsFilled.elementAt(i).getRdc() + myMiscConstants.CRTGaussianEpsilon * rmsdThreshold * rr.nextGaussian());
                    squaredSum += /*expOrBackCal[i] * */ (rdcSampled.elementAt(i).getRdc() - rdcExpWithMissingRdcsFilled.elementAt(i).getRdc()) * (rdcSampled.elementAt(i).getRdc() - rdcExpWithMissingRdcsFilled.elementAt(i).getRdc());
                }
            }
        } while (squaredSum > sumOfSquaredDeviationThreshold);
        return squaredSum;
    }

    public myProtein computeLoopGivenAlignmentTensorAndRdcs(final myDipolarCouplingTable thisRdcCsaTable, String mediumName,
            Vector<myDipolarCoupling.Type> typesOfRdcsToBeUsedInTensorRefinement,
            Vector<myDipolarCoupling.Type> typesOfRdcsToBeUsedForAnalyticSolution,
            final mySseInfo thisSseId, final double Syy, final double Szz,
            final int refineCycle, final int dfsCycle, final double weightInScoringFunction, final double resolution, final boolean printResults) {

        Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMedium = thisRdcCsaTable.getRdcMapForThisMedium(mediumName);

        // Get the RDC types that will be used in analytic solutions.
        myDipolarCoupling.Type phiTypeRdc = null;
        for (myDipolarCoupling.Type t : typesOfRdcsToBeUsedForAnalyticSolution) {
            if (myDipolarCoupling.isPhiTypeRdc(t)) {
                phiTypeRdc = t;
                break;
            }
        }
        if (phiTypeRdc == null) {
            System.out.println("Error: There does not exist RDC type that can be used to compute phi using analytic method");
            System.exit(1);
        }

        myDipolarCoupling.Type psiTypeRdc = null;
        for (myDipolarCoupling.Type t : typesOfRdcsToBeUsedForAnalyticSolution) {
            if (myDipolarCoupling.isPsiTypeRdc(t)) {
                psiTypeRdc = t;
                break;
            }
        }
        if (psiTypeRdc == null) {
            System.out.println("Error: There does not exist RDC type that can be used to compute psi using analytic method");
            System.exit(1);
        }

        Vector<myDipolarCoupling> phiTypeRdcVecWithMissingRdcsFilled = new Vector<myDipolarCoupling>();
        Vector<myDipolarCoupling> psiTypeRdcVecWithMissingRdcsFilled = new Vector<myDipolarCoupling>();
/*
        if (rdcCsaTableForThisMedium.get(phiTypeRdc) == null) {
            System.out.println("NULL");
            System.exit(1);
        } else {
            System.out.println("---------" + rdcCsaTableForThisMedium.get(phiTypeRdc).size());
        }

        if (rdcCsaTableForThisMedium.get(psiTypeRdc) == null) {
            System.out.println("NULL");
            System.exit(1);
        } else {
            System.out.println("---------" + rdcCsaTableForThisMedium.get(psiTypeRdc).size());
        }
*/
        
        fillInMissingRdcs(thisSseId, rdcCsaTableForThisMedium.get(phiTypeRdc), /*rdcCsaTableForThisMediumBackCal.get(phiTypeRdc)*/null, phiTypeRdcVecWithMissingRdcsFilled);
        fillInMissingRdcs(thisSseId, rdcCsaTableForThisMedium.get(psiTypeRdc), /*rdcCsaTableForThisMediumBackCal.get(psiTypeRdc)*/null, psiTypeRdcVecWithMissingRdcsFilled);

        // Clone the vectors to be used as place holders for sampled RDCs
        Vector<myDipolarCoupling> phiTypeRdcSampled = new Vector<myDipolarCoupling>(phiTypeRdcVecWithMissingRdcsFilled.size());
        for (myDipolarCoupling dc : phiTypeRdcVecWithMissingRdcsFilled) {
            phiTypeRdcSampled.add(new myDipolarCoupling(dc));
        }

        Vector<myDipolarCoupling> psiTypeRdcSampled = new Vector<myDipolarCoupling>(psiTypeRdcVecWithMissingRdcsFilled.size());
        for (myDipolarCoupling dc : psiTypeRdcVecWithMissingRdcsFilled) {
            psiTypeRdcSampled.add(new myDipolarCoupling(dc));
        }

        // Matrix.identity(3, 3)

        try {
            //System.out.println("refineCycle = " + refineCycle); System.exit(1);

            for (int iter = 1; iter <= refineCycle; iter++) {

                inputChar.readChar();

                System.out.println("Number of DFS cycles (= #Sampled RDC Sets): " + dfsCycle);

                //phiTypeRdcRmsd = sandwich(0.05, 1.5, phiTypeRdcRmsd);
                //psiTypeRdcRmsd = sandwich(0.05, 1.0, psiTypeRdcRmsd);
                
                // for CA-C and N-HN
                //double phiTypeRdcRmsd = 0.2;//2.0; //0.2;//2.0;//0.15;//.0; 2.0;//
                //double psiTypeRdcRmsd = 1.0;

                double phiTypeRdcRmsd = thisLCM.getPhiTypeRdcRmsdThreshold(); 
                double psiTypeRdcRmsd = thisLCM.getPsiTypeRdcRmsdThreshold(); 

                //if (iter > 1) {
                    //phiTypeRdcRmsd = sandwich(0.05, 1.5, phiTypeRdcRmsd);
                    //psiTypeRdcRmsd = sandwich(0.05, 1.0, psiTypeRdcRmsd);
                    phiTypeRdcRmsd = sandwich(0.1 * myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()),
                            2.0 * myMiscConstants.phiScale * myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()), phiTypeRdcRmsd);
                    psiTypeRdcRmsd = sandwich(0.1 * myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()),
                            2.0 * myMiscConstants.psiScale * myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()), psiTypeRdcRmsd);
                //}

                System.out.println(phiTypeRdc.toString() + " RDC RMSD threshold: " + phiTypeRdcRmsd + "    " + psiTypeRdc.toString() + " RDC RMSD threshold: " + psiTypeRdcRmsd);

                myResidue firstResidue = __global_fold_in_pof.residueAt(__begin_anchor_residue_number);
                myPeptidePlane pPlane = new myPeptidePlane(firstResidue.getAtom(myAtomLabel.__N).getCoordinates(),
                        firstResidue.getAtom(myAtomLabel.__H).getCoordinates(), firstResidue.getAtom(myAtomLabel.__CA).getCoordinates());

                // computeMaximumlikelihoodFragment is mimicked here
                Matrix Rg = myForKin.getRotationWrtGlobalFrame(pPlane);
                
                boolean rightHand = true;

                long prevT = System.currentTimeMillis(), currT = System.currentTimeMillis();

                myPhiPsiSolutionTree thisPhiPsiSolutionTree = new myPhiPsiSolutionTree();  
                thisPhiPsiSolutionTree.setCurrentTreeId(0);

                int maxSearchDepthThusFar = 0;

                for (int numberOfSamplingCycle = 0; numberOfSamplingCycle < dfsCycle; numberOfSamplingCycle++) {
                    
                    thisPhiPsiSolutionTree.setCurrentTreeId(thisPhiPsiSolutionTree.getCurrentTreeId() + 1); // set the id of the current tree to be evaluated
                    
                    int myUnit = 100;
                    if (numberOfSamplingCycle % myUnit == 0) {
                        System.out.println("Number of trees completed so far: " + (numberOfSamplingCycle / myUnit) + " * " + myUnit);
                        if (numberOfSamplingCycle != 0) {
                            currT = System.currentTimeMillis();
                            System.out.println("Time taken to evaluate " + numberOfSamplingCycle + " trees: " + ((double) (currT - prevT) / 1000.0) + "s");
                            prevT = currT;
                            System.out.println("Maximum search depth: " + thisPhiPsiSolutionTree.getMaxSearchDepth());
                        }
                    }

                    /*double phiTypeSquaredRdcDev = */generateSampledRdcVector(phiTypeRdcVecWithMissingRdcsFilled, /*phiTypeExpOrBackCal,*/ phiTypeRdcRmsd, phiTypeRdcSampled, rr);
                    /*double psiTypeSquaredRdcDev = */generateSampledRdcVector(psiTypeRdcVecWithMissingRdcsFilled, /*psiTypeExpOrBackCal,*/ psiTypeRdcRmsd, psiTypeRdcSampled, rr);

                    try {                        
                        Writer phiPsiSolutionFile = null;
                        File file = new File(thisParameterManager.getLogFile());
//                        if (file.exists() && numberOfSamplingCycle == 0) { // This is now taken care of by the loop configuration manager
//                            file.delete();
//                        }
                        phiPsiSolutionFile = new BufferedWriter(new FileWriter(file, true)); //true for append mode

                        long oldTimeStamp = file.lastModified();

                        //thisPhiPsiSolutionTree.__sampled_rdc_set_number = numberOfSamplingCycle;

                        thisPhiPsiSolutionTree.phiPsiSolutionTreeRootCallerForLoop(thisSseId, phiTypeRdcSampled, psiTypeRdcSampled,
                                phiTypeRdcVecWithMissingRdcsFilled, psiTypeRdcVecWithMissingRdcsFilled,
                                Rg, Syy, Szz, pPlane, phiPsiSolutionFile, weightInScoringFunction, rightHand);

                        if (thisPhiPsiSolutionTree.getMaxSearchDepth() > maxSearchDepthThusFar) {
                            System.out.println("Max Search Depth of " + thisPhiPsiSolutionTree.getMaxSearchDepth() + " achieved for the tree with id: " + thisPhiPsiSolutionTree.getCurrentTreeId());
                            maxSearchDepthThusFar = thisPhiPsiSolutionTree.getMaxSearchDepth();
                        }

                        phiPsiSolutionFile.write("Completed computing the tree with id: " + thisPhiPsiSolutionTree.getCurrentTreeId() + "\n\n");

                        phiPsiSolutionFile.close();
                        long newTimeStamp = file.lastModified();

                        
                        // Note: Commented out for ever from July 20, 2012.
                        if (newTimeStamp - oldTimeStamp != 0) {
//                            Writer finalOutputFile = null;
//                            File finalFile = new File("phiPsiSolutionSets.txt");
//                            //if (finalFile.exists()/* && mmm == 0*/)
//                            //	finalFile.delete();
//                            finalOutputFile = new BufferedWriter(new FileWriter(finalFile, true)); //true for append mode
//
//                            System.out.println("Above solutions are for the tree with id: " + thisPhiPsiSolutionTree.getCurrentTreeId() + '\n');
//                            finalOutputFile.write("Above solutions are for the tree with id: " + thisPhiPsiSolutionTree.getCurrentTreeId() + '\n');
//                            finalOutputFile.close();
                        }                        
                        
                    } catch (IOException e) {
                        System.out.println("Error : An IOException is thrown while writing to the outputfile inside CRTMinHelix");
                        e.printStackTrace();
                        System.exit(1);
                    }
                }


//
//
//
//                sseUsingRefinedTensor = computeMaximumlikelihoodFragment(rdcCsaTableForThisMedium, rdcCsaTableForThisMediumBackCal,
//                        typesOfRdcsToBeUsedForAnalyticSolution, thisSseId, pPlane, -(Syy + Szz), Syy, Szz,
//                        phiTypeRdcRmsd, psiTypeRdcRmsd, dfsCycle, weightInScoringFunction, printResults);
//
//                if (sseUsingRefinedTensor == null) {
//                    System.out.println("Error: sseUsingRefinedTensor is null in computeSseGivenAlignmentTensorAndRdcs");
//                    System.exit(1);
//                }
///*
//                rotToPOF = myAlignmentTensorEstimator.bestFitUsingSVD(sseUsingRefinedTensor, rdcCsaTableForThisMedium,
//                        typesOfRdcsToBeUsedInTensorRefinement, rdcCsaTableForThisMediumBackCal, dummy);
//*/
//                // Note: This can be removed for longer helix in preference for the above svd
//                rotToPOF = myAlignmentTensorEstimator.bestFitUsingGridSearchForGivenAT(sseUsingRefinedTensor, rdcCsaTableForThisMedium,
//                        typesOfRdcsToBeUsedInTensorRefinement, rdcCsaTableForThisMediumBackCal, Syy, Szz, resolution);
///*
//                phiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(sseUsingRefinedTensor, rdcCsaTableForThisMedium.get(phiTypeRdc),
//                        rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()));
//                System.out.println("Phi type rdc rmsd: " + phiTypeRdcRmsd);
//
//                psiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(sseUsingRefinedTensor, rdcCsaTableForThisMedium.get(psiTypeRdc),
//                        rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()));
//                System.out.println("Psi type rdc rmsd: " + psiTypeRdcRmsd);
//*/
//                thisFragmentInPOF = new myProtein(sseUsingRefinedTensor);
//                thisFragmentInPOF.rotate(new myMatrix(rotToPOF.getArrayCopy()));
//
//                thisFragmentInPOF = myAlignmentTensorEstimator.optimizeFirstPeptidePlane(thisFragmentInPOF, rdcCsaTableForThisMedium, myDipolarCoupling.Type.N_HN,
//                        -(Syy + Szz), Syy, Szz, -myMiscConstants.maximumDeviationFromMean, myMiscConstants.maximumDeviationFromMean, myMiscConstants.stepSizeUsedInEstimatingFirstPeptidePlane);
//
//                phiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(thisFragmentInPOF, rdcCsaTableForThisMedium.get(phiTypeRdc),
//                        /*rotToPOF*/Matrix.identity(3, 3), -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()));
//                System.out.println("Phi type rdc rmsd: " + phiTypeRdcRmsd);
//
//                psiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(thisFragmentInPOF, rdcCsaTableForThisMedium.get(psiTypeRdc),
//                        /*rotToPOF*/Matrix.identity(3, 3), -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()));
//                System.out.println("Psi type rdc rmsd: " + psiTypeRdcRmsd);
//
//                System.out.println("Printing the best fragment after rotation (note: this is not returned, rather the fragment before rotation is returned");
//                thisFragmentInPOF.print();
//
//                outputSaupe = new BufferedWriter(new FileWriter(saupeListFile, true));
//                System.out.println("Alignment Tensor after Round " + iter + ":    Syy: " + Syy + "    Szz: " + Szz);
//                outputSaupe.write("Alignment Tensor after Round " + iter + ":    Syy: " + Syy + "    Szz: " + Szz + '\n');
//                System.out.println("Rhombicity = " + computeRhombicity(Syy, Szz));
//                outputSaupe.write("Rhombicity = " + computeRhombicity(Syy, Szz) + '\n');
//                outputSaupe.close();
            }
        } catch (/*IO*/Exception e) {
            e.printStackTrace();
        }

        return null;
    }


    public void analyticLoopClosure() {
        long startTime = System.currentTimeMillis();

        myDipolarCouplingTable thisRdcCsaTable = thisParameterManager.getDipolarCouplingTable();
        thisRdcCsaTable.print();
        
        // Select the RDCs to be used to compute the structure. Select medium, 
        String mediumName = thisParameterManager.parseProgramParameterFileAndGetTypesOfRdcMediaNames().firstElement();
        System.out.println("Medium name: " + mediumName);
        
        Vector<myDipolarCoupling.Type> typesOfRdcsToBeUsedForAnalyticSolution = thisParameterManager.parseProgramParameterFileAndGetTypesOfRdcsToBeUsedForAnalyticSolutions();
        Vector<myDipolarCoupling.Type> typesOfRdcsToBeUsedForRefinement = thisParameterManager.parseProgramParameterFileAndGetTypesOfRdcsToBeUsedForRefinement();
        
        System.out.println("RDC types to be used for analytic solution:");
        for (myDipolarCoupling.Type T : typesOfRdcsToBeUsedForAnalyticSolution) {
            System.out.print(T + "    ");
        }        
        System.out.println();
        System.out.println("RDC types to be used for pruning:");
        for (myDipolarCoupling.Type T : typesOfRdcsToBeUsedForRefinement) {
            System.out.print(T + "    ");
        }
        System.out.println();
        //System.exit(1);
        
        __global_fold_in_pof = thisLCM.__global_fold_in_pof;
        int beginAnchorResNum = thisLCM.getBeginResidueNumber();
        int endAnchorResNum = thisLCM.getEndResidueNumber();

        __begin_anchor_residue_number = beginAnchorResNum;
        __end_anchor_residue_number = endAnchorResNum;

        //__target_pose_residue = new myResidue(__global_fold_in_pof.residueAt(__end_anchor_residue_number));

        mySseInfo thisSseId  = new mySseInfo(beginAnchorResNum, endAnchorResNum, "L"); // TODO: work on this
        double Syy = thisLCM.getSyy(); 
        double Szz = thisLCM.getSzz();
        
        int thisDfsCycle = thisLCM.getnumberOfSearchTrees();
        System.out.println("#SearchTrees: " + thisDfsCycle);
        int thisRefineCycle = 1;//Integer.parseInt(thisParameterMap.get("numberOfOptimizationsOfPP1"));
        double thisWeight4Angles = 4.0;//new Double(thisParameterMap.get("weightForRmsDihedralDeviationTerm")).doubleValue();
        double thisResolution = 2.0;//new Double(thisParameterMap.get("gridResolutionInDegrees")).doubleValue();
        //System.out.println("Parameters for " + thisSseId.toString() + " computation: " + thisParameterMap.toString());
        boolean thisPrintResults = true; // To be used in future

        computeLoopGivenAlignmentTensorAndRdcs(thisRdcCsaTable, mediumName, typesOfRdcsToBeUsedForRefinement, typesOfRdcsToBeUsedForAnalyticSolution,
                        thisSseId, Syy, Szz, thisRefineCycle, thisDfsCycle, thisWeight4Angles, thisResolution, thisPrintResults);
    }
    
    /**
     * Given three doubles lb, ub, val, this method tests if val is between lb
     * and ub. If so, then it returns val, else it returns either lb or ub
     * depending on which one is closer to val.
     *
     * @param lb the lower bound of the interval
     * @param ub the upper bound of the interval
     * @param val the number being tested for containment in the interval
     * @return if val is between lb and ub, then it returns val, else return
     * either lb or ub depending on which one is closer to val
     */
    private double sandwich(double lb, double ub, double val) {
        if (lb > ub) { // swap
            double temp = lb;
            lb = ub;
            ub = temp;
        }
        if (val < lb) {
            val = lb;
        } else if (val > ub) {
            val = ub;
        } else ; // do nothing
        return val;
    }

}


