/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2012 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, August 04, 2012
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.1, August 04, 2012
 * @author        Chittaranjan Tripathy (2007-2012)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */

/**
 * Description of the class
 * This class defines a point in three dimensional space represented using
 * cartesian coordinates x, y and z. A set of methods are provided to use
 * the point effectively.
 */
public class myPoint implements java.lang.Comparable<myPoint>, java.io.Serializable {

    /**
     * The x, y and z coordinates of a point. They default to the origin.
     */
    protected double __x = 0;
    protected double __y = 0;
    protected double __z = 0;

    /**
     * Default constructor which does nothing other than constructing a point
     * which is the origin.
     */
    public myPoint() {
    }

    /**
     * Construct a point from an array of doubles of size three.
     *
     * @param triple the array of size three
     */
    public myPoint(final double[] triple) {
        assert triple.length == 3 : "Error: Invalid point construction.";
        __x = triple[0];
        __y = triple[1];
        __z = triple[2];
    }

    /**
     * Construct a point from the values of the given coordinates.
     *
     * @param x value of x coordinate
     * @param y value of y coordiante
     * @param z value of z coordinate
     */
    public myPoint(double x, double y, double z) {
        __x = x;
        __y = y;
        __z = z;
    }

    /**
     * A copy constructor which takes an instance of a point and constructs
     * another point with the same coordinates.
     *
     * @param p the point which is to be cloned
     */
    public myPoint(final myPoint p) {
        __x = p.getX();
        __y = p.getY();
        __z = p.getZ();
    }

    /**
     * Construct a point given a point p and a translation vector v. That is,
     * the point being constructed is the point p translated by v.
     *
     * @param p the point given
     * @param v specifies the translation
     */
    public myPoint(final myPoint p, final myVector3D v) {
        __x = p.getX() + v.getX();
        __y = p.getY() + v.getY();
        __z = p.getZ() + v.getZ();
    }

    /**
     * Compares object in the argument with this object for equality.
     *
     * @param o the Object to be compared for equality with this object
     * @return true if the specified Object is equal to this object
     */
    public boolean equals(Object o) {
	if (o == this)
	    return true;
	if (!(o instanceof myPoint))
	    return false;
        return __x == ((myPoint) o).__x && __y == ((myPoint) o).__y && __z == ((myPoint) o).__z;
    }

    /**
     * Returns a size 3 array of doubles containing the coordinates of the point.
     *
     * @return an array of size 3 containing the coordinates of the point
     */
    public double[] getXYZ() {
        double[] rep = new double[3];
        rep[0] = __x;
        rep[1] = __y;
        rep[2] = __z;
        return rep;
    }

    /**
     * Retruns the x coordinate of the point.
     *
     * @return x coordinate
     */
    public double getX() {
        return __x;
    }

    /**
     * Returns the y coordinate of the point.
     *
     * @return y coordinate
     */
    public double getY() {
        return __y;
    }

    /**
     * Returns the z coordinate of the point.
     *
     * @return z coordinate
     */
    public double getZ() {
        return __z;
    }

    /**
     * Modifies the x coordinate.
     *
     * @param x set the x coordinate to the new value x
     */
    public void setX(double x) {
        __x = x;
    }

    /**
     * Modifies the y coordinate.
     *
     * @param y set the y coordinate to the new value y
     */
    public void setY(double y) {
        __y = y;
    }

    /**
     * Modifies the z coordinate.
     *
     * @param z set the z coordinate to the new value z
     */
    public void setZ(double z) {
        __z = z;
    }

    /**
     * Convert the point to its string representation.
     *
     * @return the string representation of the point
     */
    public String toString() {
        return getX() + myMiscConstants.delimiter + getY() + myMiscConstants.delimiter + getZ();
    }

    public int compareTo(myPoint p) {
        if (__x == p.__x && __y == p.__y && __z == p.__z) {
            return 0;
        }
        return -1;
    }

    /**
     * Translate the point by an amount speficied by the vector v.
     *
     * @param v the vector that speficies the translation
     */
    public void translate(final myVector3D v) {
        __x += v.getX();
        __y += v.getY();
        __z += v.getZ();
    }

    /**
     * Create a new point that represents the translated point p by an amount
     * speficied by the vector v.
     *
     * @param p the point to be translated
     * @param v the vector that speficies the translation
     * @return a new point representing p + v
     */
    public static myPoint translate(final myPoint p, final myVector3D v) {
        return new myPoint(p.getX() + v.getX(), p.getY() + v.getY(), p.getZ() + v.getZ());
    }

    /**
     * Return the squared Euclidean distance between two points.
     *
     * @param p the first point
     * @param q the second point
     * @return squared Euclidean distance
     */
    public static double squaredDist(final myPoint p, final myPoint q) {
        double deltaX = p.getX() - q.getX();
        double deltaY = p.getY() - q.getY();
        double deltaZ = p.getZ() - q.getZ();
        return deltaX * deltaX + deltaY * deltaY + deltaZ * deltaZ;
    }

    /**
     * Return the L2 norm (Euclidean distance) between two points.
     *
     * @param p the first point
     * @param q the second point
     * @return the Euclidean distance
     */
    public static double L2Norm(final myPoint p, final myPoint q) {
        return Math.sqrt(squaredDist(p, q));
    }

    /**
     * Return the L2 norm (Euclidean distance) between two points.
     *
     * @param p the first point
     * @param q the second point
     * @return the Euclidean distance
     */
    public static double euclideanDist(final myPoint p, final myPoint q) {
        return L2Norm(p, q);
    }

    /**
     * Return the L2 norm (Euclidean distance) between two points.
     *
     * @param p the first point
     * @param q the second point
     * @return the Euclidean distance
     */
    public static double distance(final myPoint p, final myPoint q) {
        return L2Norm(p, q);
    }

    public static void main(String[] args) {
        myPoint p = new myPoint(1, 2, 3), q = new myPoint(1, 0, 3);
        System.out.println("squaredDist: " + squaredDist(p, q) + "  L2Norm: " + L2Norm(p, q));
    }
}

