/*
 * This file is part of RDC-ANALYTIC.
 * 
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2012 Bruce Donald Lab, Duke University
 * 
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation, either version 3 of the License, or (at your option) any 
 * later version.
 * 
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS 
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more 
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 * 
 * There are additional restrictions imposed on the use and distribution of this 
 * open-source code, including: (A) this header must be included in any 
 * modification or extension of the code; (B) you are required to cite our 
 * papers in any publications that use this code. The citation for the various 
 * different modules of our software, together with a complete list of 
 * requirements and restrictions are found in the document license.pdf enclosed 
 * with this distribution.
 * 
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 * 
 * <signature of Bruce Donald>, August 04, 2012
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.1, August 04, 2012
 * @author        Chittaranjan Tripathy (2007-2012)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.io.*;
import java.util.*;

/**
 * Description of the class
 */
public class myLoopConfigurationManager {

    // TODO: as in myMiscConstants file one have a parameter parser here
    public static myRama ramaGeneral = null, ramaGly = null, ramaPrePro = null, ramaPro = null;    
    public myProteinSequence __this_protein_sequence = null;    
    public myProtein __global_fold_in_pof = null;
    public Vector<myNoe> __loop_noe_vec = null; 
    public Vector<myDihedralRestraint> __talos_dihedral_restraints = null;
    
    private myProtein __global_fold_in_pof_without_loops_and_anchor = null;
    private double __clash_score_for_global_fold_in_pof_without_loops_and_anchor = Double.MAX_VALUE;
    private int __number_of_atoms_in_global_fold_in_pof_without_loops_and_anchor = 0;
       
    private int __index = -1;
    private Vector<String[]> __parameters_for_all_loops = null;        
            
    // Format is @loop(beginResidueNumber, endResidueNumber, closureDistanceThreshold, gridResolutionForPhiPsi,     
    // DepthAtWhichStericCheckerTurnsOn, phdFileContainingGlobalFold, phiTypeRdcRmsdThreshold, psiTypeRdcRmsdThreshold, Sxx, Syy);
    
    public myProtein getGlolabFoldInPofWithoutLoopsAndAnchor() { return __global_fold_in_pof_without_loops_and_anchor; }
    public double getClashScoreForGlobalFoldInPofWithoutLoopsAndAnchor() { return __clash_score_for_global_fold_in_pof_without_loops_and_anchor; }
    public int getNumberOfAtomsInGlobalFoldInPofWithoutLoopsAndAnchor() { return __number_of_atoms_in_global_fold_in_pof_without_loops_and_anchor; }
    
    public int getBeginResidueNumber() { return Integer.parseInt(__parameters_for_all_loops.elementAt(__index)[1]); }    
    public int getEndResidueNumber() { return Integer.parseInt(__parameters_for_all_loops.elementAt(__index)[2]); }    
    public double getClosureDistanceThreshold() { return Double.parseDouble(__parameters_for_all_loops.elementAt(__index)[3]);}
    //public double getRmsDistanceWrtReference() { return Double.parseDouble(__parameters_for_all_loops.elementAt(/*__index)[4]*/);}  // remove this method  
    public double getGridResolutionForPhiPsi() { return Double.parseDouble(__parameters_for_all_loops.elementAt(__index)[4]);}
    public int getDepthAtWhichStericCheckerTurnsOn() { return Integer.parseInt(__parameters_for_all_loops.elementAt(__index)[5]);}
    public String getPdbFileContainingGlobalFold() { return myInputDataAndParameterManager.getInstance().getInputDataDirectory() + (__parameters_for_all_loops.elementAt(__index)[6]);}
    public double getPhiTypeRdcRmsdThreshold() { return Double.parseDouble(__parameters_for_all_loops.elementAt(__index)[7]);}
    public double getPsiTypeRdcRmsdThreshold() { return Double.parseDouble(__parameters_for_all_loops.elementAt(__index)[8]);}
    public double getSyy() { return Double.parseDouble(__parameters_for_all_loops.elementAt(__index)[9]); }
    public double getSzz() { return Double.parseDouble(__parameters_for_all_loops.elementAt(__index)[10]); }  
    public int getnumberOfSearchTrees() { return Integer.parseInt(__parameters_for_all_loops.elementAt(__index)[11]); }

    public void setParameterForCurrentLoop(int beginResidueNumber, int endResidueNumber) {
        __index = -1;
        for (int i = 0; i < __parameters_for_all_loops.size(); i++) {
            int brNum = Integer.parseInt(__parameters_for_all_loops.elementAt(i)[1]);
            int erNum = Integer.parseInt(__parameters_for_all_loops.elementAt(i)[2]);

            if (brNum != beginResidueNumber || erNum != endResidueNumber) {
                continue;
            } else {
                __index = i;                
                return;
            }
        }        
        if (__index == -1) {
            System.out.println("Error: Loop configuration manager could not find the parameters for the loop " + beginResidueNumber + '-' + endResidueNumber);
            System.exit(1);
        }
    }
    
    public myLoopConfigurationManager() {
        int beginResidueNumber, endResidueNumber;
        readLoopConfigurationParameters:
        {
            myInputDataAndParameterManager thisParameterManager = myInputDataAndParameterManager.getInstance();        
            myPair<Integer, Integer> thisLoop = thisParameterManager.parseLoopConfigurationFileAndReturnWhichLoopToBeComputedNow();  
            if (thisLoop == null) {
                System.out.println("Error: No loop is specified to be computed now. Use @computeNow tag to specify which loop you want to compute");
                System.exit(1);
            }
            beginResidueNumber = thisLoop.first();
            endResidueNumber = thisLoop.second();            
        }
        myLoopConfigurationManagerHelper(beginResidueNumber, endResidueNumber);
        //this(50, 65);
    }
    
    public Vector<String[]> getLoopSpecifications() {
        return  __parameters_for_all_loops;
    }
    
    private void myLoopConfigurationManagerHelper(int beginResidueNumber, int endResidueNumber) {
        readLoopConfigurationParameters:
        {
            myInputDataAndParameterManager thisParameterManager = myInputDataAndParameterManager.getInstance();        
            __parameters_for_all_loops = thisParameterManager.parseLoopConfigurationFileAndReturnLoopAndParameterSpecifications();  
            setParameterForCurrentLoop(beginResidueNumber, endResidueNumber);
        }
        
        ramachandramMaps:
        {
            String ramaDirectory = System.getProperty("user.dir") + System.getProperty("file.separator")
                    + "myRamachandran" + System.getProperty("file.separator") + "pct" + System.getProperty("file.separator")
                    + "rama" + System.getProperty("file.separator");
            System.out.println("Rama dir: " + ramaDirectory);
            String ramaGeneralFileName = ramaDirectory + "rama500-general-nosec.data";
            String ramaGlyFileName = ramaDirectory + "rama500-gly-sym-nosec.data";
            String ramaPreProFileName = ramaDirectory + "rama500-prepro.data";
            String ramaProFileName = ramaDirectory + "rama500-pro.data";

            ramaGeneral = new myRama(ramaGeneralFileName, "general");
            ramaGly = new myRama(ramaGlyFileName, "gly");
            ramaPrePro = new myRama(ramaPreProFileName, "prepro");
            ramaPro = new myRama(ramaProFileName, "pro");
        }

        readTheProteinSequence:
        {
            myInputDataAndParameterManager thisParameterManager = myInputDataAndParameterManager.getInstance();
            String inputFile = thisParameterManager.getSeqFile();

            myProteinSequence thisProteinSeq = new myProteinSequence(inputFile);
            System.out.println("Printing the protein sequence");
            System.out.println(thisProteinSeq);
            __this_protein_sequence = thisProteinSeq;                       
        }

        readTheProteinInPOF:
        {
            Vector<myProtein> vp = new Vector<myProtein>();
            myPdbParser pParser = new myPdbParser(getPdbFileContainingGlobalFold());
            while (pParser.hasNextProtein()) {
                vp.add(pParser.nextProtein());
            }
            __global_fold_in_pof = vp.elementAt(0);
        }

        helperToExtractMoreStuffsFromProteins();        
        
        readNOEsForLoop:
        {
            myInputDataAndParameterManager thisParameterManager = myInputDataAndParameterManager.getInstance();                        
            __loop_noe_vec = myNoe.parseNoeFileInXplorFormatAndGetRestraintsForGivenRange(thisParameterManager.getNoeFile(), this.getBeginResidueNumber(), this.getEndResidueNumber());
            if (__loop_noe_vec != null) {
                System.out.println("Printing the NOE restraints");
                myNoe.printNoeVector(__loop_noe_vec);
            }
        }                
        
        readTalosDihedralRestraints:
        {            
            myInputDataAndParameterManager thisParameterManager = myInputDataAndParameterManager.getInstance();                         
            __talos_dihedral_restraints = myDihedralRestraint.parseTalosRestraintFileAndGetRestraintsForGivenRange
                    (thisParameterManager.getTalosDihedralRestraintFile(), this.getBeginResidueNumber(), this.getEndResidueNumber());  
            if (__talos_dihedral_restraints != null) {
                System.out.println("Printing the dihedral restraints");
                myDihedralRestraint.printTalosRestraintVector(__talos_dihedral_restraints);
            }
        }                        
        
    }
    
    
    private void helperToExtractMoreStuffsFromProteins() {
        extractMoreStuffsFromProteins:
        {
            myProtein thisP = new myProtein(__global_fold_in_pof);
            for (int i = getBeginResidueNumber(); i <= getEndResidueNumber(); i++) {
                thisP.removeResidue(i);
            }
            __global_fold_in_pof_without_loops_and_anchor = thisP;
            //System.out.println("**--**");
            //thisP.print();
            //System.out.println("**--**");

            // Check for steric clash and return the clash score
            myStericChecker sc = new myStericChecker(1.0, 0.8, 0.4);
            __clash_score_for_global_fold_in_pof_without_loops_and_anchor = sc.checkStericClash(__global_fold_in_pof_without_loops_and_anchor);
            __number_of_atoms_in_global_fold_in_pof_without_loops_and_anchor = __global_fold_in_pof_without_loops_and_anchor.numberOfAtoms();

            System.out.println("Clash score for the global fold: " + __clash_score_for_global_fold_in_pof_without_loops_and_anchor);
            System.out.println("Number of atoms in the global fold: " + __number_of_atoms_in_global_fold_in_pof_without_loops_and_anchor);
            //__global_fold_in_pof_without_loops_and_anchor.print();
            //System.exit(1);
        }        
        return;
    }
    
    
    
    
    
    ////////////// These methods below not in use anymore /////////////
    /*
    public String getTheCurrentLoopParameters(int beginResNum, int endResNum, String argument) {
        // Format is @loop(beginResidueNumber, endResidueNumber, closureDistanceThreshold, rmsDistanceWrtReference, gridResolutionForPhiPsi, __depth_at_which_steric_checker_turns_on
        // phdFileContainingGlobalFold, phiTypeRdcRmsdThreshold, psiTypeRdcRmsdThreshold, Sxx, Syy);

        for (int i = 0; i < __parameters_for_all_loops.size(); i++) {
            int brNum = Integer.parseInt(__parameters_for_all_loops.elementAt(i)[0]);
            int erNum = Integer.parseInt(__parameters_for_all_loops.elementAt(i)[1]);

            if (brNum != beginResNum || erNum != endResNum) {
                continue;
            } else {
                String[] line = __parameters_for_all_loops.elementAt(i);
                switch (argument.trim().toLowerCase()) {
                    case "beginResidueNumber":
                        return line[0];
                    case "endResidueNumber":
                        return line[1];
                    case "closureDistanceThreshold":
                        return line[2];
                    case "rmsDistanceWrtReference":
                        return line[3];
                    case "gridResolutionForPhiPsi":
                        return line[4];
                    case "phdFileContainingGlobalFold":
                        return line[5];
                    case "phiTypeRdcRmsdThreshold":
                        return line[6];
                    case "psiTypeRdcRmsdThreshold":
                        return line[7];
                    case "Sxx":
                        return line[8];
                    case "Syy":
                        return line[9];
                    default:
                        System.out.println("Error: The parameter(s) for the loop " + line[0] + '-' + line[1] + " are incorrect. Exiting...");
                        System.exit(1);
                }
            }
        }
        return null;
    }
    */
    
    
    public static void main(String... args) {
        myLoopConfigurationManager mlcm = new myLoopConfigurationManager();
        
    }
}
