/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2012 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, August 04, 2012
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.1, August 04, 2012
 * @author        Chittaranjan Tripathy (2007-2012)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.io.*;
import java.util.*;

/**
 * Main driver class.
 */
public class Pool {

    /**
     * Constructor does not do anything special.
     */
    public Pool() {
    }

    /**
     * Check for a valid option.
     *
     * @param s string that specifies an option
     * @return return true if the option is valid
     */
    public static boolean validOption(String s) {
        return s.startsWith("-");
    }

    /**
     * Parse the command line arguments and invoke Pool.
     *
     * @param args an array of strings specifying the command line parameters
     */
    public void parseCommandLineArgumentsAndInvokePool(String[] args) {
        String masterDir = null;
        String directoryArchitectureFile = null;
        int beginAnchorResidueNumber = -1;
        int endAnchorResidueNumber = -1;

        if (args.length == 0) {
            System.out.println("To learn more about this command, please type: " + new Exception().getStackTrace()[0].getClassName() + " -help");
        }
                
        // Parse the command line arguments        
        for (int i = 0; i < args.length && validOption(args[i]);) {
            String thisArgument = args[i++];

            if (thisArgument.equals("-help") || thisArgument.equals("-verbose")) {
                String helpString = "Usage: " + new Exception().getStackTrace()[0].getClassName() + " <options> \nwhere possible options include: \n"
                        + "-masterdir <file name>                 Specify the master directory name which contains all the input files and the output will also be redirected to this directory  " + '\n'
                        + "-directoryarchitecturefile <file name> Specify the directory architecture file name which contains the information about the input and output directories and files under the master directory" + '\n'
                        + "-beginanchorresiduenumber              Specify the begin anchor residue number" + '\n'
                        + "-endanchorresiduenumber                Specify the end anchor residue number" + '\n'
                        + "-verbose                               Print a synopsis of standard options and return" + '\n'
                        + "-help                                  Print a synopsis of standard options and return" + '\n';
                System.out.println(helpString);
                return;
            } else if (thisArgument.equals("-masterdir")) {
                if (i < args.length) {
                    masterDir = args[i++];
                    if (validOption(masterDir)) {
                        System.out.println("Error: incorrect master directory name or missing argument(s)");
                        System.exit(1);
                    }
                } else {
                    System.out.println("Error: master directory name is not supplied");
                    System.exit(1);
                }
            } else if (thisArgument.equals("-directoryarchitecturefile")) {
                if (i < args.length) {
                    directoryArchitectureFile = args[i++];
                    if (validOption(directoryArchitectureFile)) {
                        System.out.println("Error: incorrect directory architecture file name or missing argument(s)");
                        System.exit(1);
                    }
                } else {
                    System.out.println("Error: directory architecture file name is not supplied");
                    System.exit(1);
                }
            } else if (thisArgument.equals("-beginanchorresiduenumber")) {
                if (i < args.length) {
                    beginAnchorResidueNumber = Integer.parseInt(args[i++]);
                    assert beginAnchorResidueNumber > 0 : "Error: the begin anchor residue number must be greater than zero";
                } else {
                    System.out.println("Error: the begin anchor residue number not supplied");
                    System.exit(1);
                }
            } else if (thisArgument.equals("-endanchorresiduenumber")) {
                if (i < args.length) {
                    endAnchorResidueNumber = Integer.parseInt(args[i++]);
                    assert endAnchorResidueNumber > 0 : "Error: the end anchor residue number must be greater than zero";
                } else {
                    System.out.println("Error: the end anchor residue number not supplied");
                    System.exit(1);
                }
            } else {
                System.out.println("Error: incorrect argument specification: " + thisArgument);
                System.exit(1);
            }
        }

        if (masterDir == null) {
            System.out.println("Error: master directory name is not supplied");
            System.exit(1);
        }

        if (directoryArchitectureFile == null) {
            System.out.println("Warning: directory architecture file name is not supplied. It is defaulted to dirArch.txt");
            directoryArchitectureFile = "dirArch.txt";
        }

        if (beginAnchorResidueNumber == -1 && endAnchorResidueNumber == -1) {
            // No begin and end anchor position specified. The positions must be read from the input parameter files
            myInputDataAndParameterManager mdpm = myInputDataAndParameterManager.getInstance();
            mdpm.setMasterDirectory(masterDir);
            mdpm.setDirectoryArchitectureFile(directoryArchitectureFile);
            myPool pool = new myPool();
            pool.analyticLoopClosure();
        } else if (beginAnchorResidueNumber > 0 && endAnchorResidueNumber > 0) {
            myInputDataAndParameterManager mdpm = myInputDataAndParameterManager.getInstance();
            mdpm.setMasterDirectory(masterDir);
            mdpm.setDirectoryArchitectureFile(directoryArchitectureFile);
            myPool pool = new myPool(beginAnchorResidueNumber, endAnchorResidueNumber);
            pool.analyticLoopClosure();
        } else {
            System.out.println("Error: incorrect loop anchor specification");
            System.exit(1);
        }
    }

    public static void main(String... args) {
        Pool thisPool = new Pool();
        thisPool.parseCommandLineArgumentsAndInvokePool(args);
    }
}
