/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2012 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, August 04, 2012
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.1, August 04, 2012
 * @author        Chittaranjan Tripathy (2007-2012)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */

/**
 * Description of the class
 */
public class myVector3D extends myPoint implements java.io.Serializable {

    /**
     * Default constructor which does nothing other than constructing a vector
     * of length zero.
     */
    public myVector3D() {
        super();
    }

    /**
     * Construct a vector from an array of doubles of size three.
     *
     * @param triple the array of size three
     */
    public myVector3D(final double[] triple) {
        super(triple);
    }

    /**
     * Construct a vector from the values of the given coordinates.
     *
     * @param x value of x coordinate
     * @param y value of y coordiante
     * @param z value of z coordinate
     */
    public myVector3D(double x, double y, double z) {
        super(x, y, z);
    }

    /**
     * A copy constructor which takes an instance of a vector and constructs
     * another vector with the same coordinates.
     *
     * @param v the vector which is to be cloned
     */
    public myVector3D(final myVector3D v) {
        __x = v.getX();
        __y = v.getY();
        __z = v.getZ();
    }

    /**
     * Construct a vector given the head and tail coordinates.
     *
     * @param tail the point specifying the coordinates of the tail
     * @param head the point specifying the coordinates of the head
     */
    public myVector3D(final myPoint tail, final myPoint head) {
        __x = head.getX() - tail.getX();
        __y = head.getY() - tail.getY();
        __z = head.getZ() - tail.getZ();
    }

    /**
     * Compute the cross (vector) product of two vectors.
     *
     * @param v1 first vector
     * @param v2 second vector
     * @return cross product of v1 and v2
     */
    public static myVector3D crossProduct(final myVector3D v1, final myVector3D v2) {
        // The 3D vector cross product of vectors v1 and v2 is defined as
        // (v1_y * v2_z - v1_z * v2_y) * i +
        // (v1_z * v2_x - v1_x * v2_z) * j +
        // (v1_x * v2_y - v1_y * v2_x) * k
        return new myVector3D(v1.getY() * v2.getZ() - v1.getZ() * v2.getY(),
                v1.getZ() * v2.getX() - v1.getX() * v2.getZ(),
                v1.getX() * v2.getY() - v1.getY() * v2.getX());
    }

    /**
     * Compute the inner (dot) product of two vectors.
     *
     * @param v1 first vector
     * @param v2 second vector
     * @return the inner product of v1 and v2
     */
    public static double innerProduct(final myVector3D v1, final myVector3D v2) {
        return v1.getX() * v2.getX() + v1.getY() * v2.getY() + v1.getZ() * v2.getZ();
    }

    /**
     * Compute the inner (dot) product of two vectors.
     *
     * @param v1 first vector
     * @param v2 second vector
     * @return the inner (dot) product of v1 and v2
     */
    public static double dotProduct(final myVector3D v1, final myVector3D v2) {
        return innerProduct(v1, v2);
    }

    /**
     * Compute the L2 (Euclidean) norm (i.e., length) of the vector
     *
     * @return the length of the vector
     */
    public double norm() {
        return Math.sqrt(__x * __x + __y * __y + __z * __z);
    }

    /**
     * Compute the square of the length of the vector.
     *
     * @return the square of the length of the vector
     */
    public double SquaredNorm() {
        return __x * __x + __y * __y + __z * __z;
    }

    /**
     * Reverse the vector, i.e., v = -v.
     */
    public void reverse() {
        __x = -__x;
        __y = -__y;
        __z = -__z;
    }

    /**
     * Return a new vector which is the reverse of the vector given as argument.
     *
     * @param v the vector which reverse is to be constructed
     * @return reverse of the vector
     */
    public static myVector3D reverse(final myVector3D v) {
        return new myVector3D(-v.getX(), -v.getY(), -v.getZ());
    }

    /**
     * Return the cosine of the angle between two vectors.
     * 
     * @param v1 the first vector
     * @param v2 the second vector
     * @return cosine of the angle between v1 and v2
     */
    public static double cosineOfAngle(final myVector3D v1, final myVector3D v2) {
        return innerProduct(v1, v2) / (v1.norm() * v2.norm());
    }

    /**
     * Return the consine of the angle between this vector and the vector
     * supplied as argument.
     *
     * @param v the vector which makes some angle with this vector
     * @return the cosine of the angle between this vector and v
     */
    public double cosineOfAngle(final myVector3D v) {
        return innerProduct(this, v) / (this.norm() * v.norm());
    }

    /**
     * Return the angle in radians between two vectors where the angle lies in
     * the interval [0, pi].
     *
     * @param v1 the first vector
     * @param v2 the second vector
     * @return the angle in radians in the interval [0, pi]
     */
    public static double angle(final myVector3D v1, final myVector3D v2) {
        double cosOfAngle = cosineOfAngle(v1, v2);
        if (cosOfAngle < -1.0) {
            cosOfAngle = -1.0;
        }
        if (cosOfAngle > 1.0) {
            cosOfAngle = 1.0;
        }
        return Math.acos(cosOfAngle);
    }

    /**
     * Return the angle in radians between this vector and the vector
     * supplied as argument where the angle lies in the interval [0, pi].
     *
     * @param v the vector which makes some angle with this vector
     * @return the angle in radians in the interval [0, pi]
     */
    public double angle(final myVector3D v) {
        double cosOfAngle = this.cosineOfAngle(v);
        if (cosOfAngle < -1.0) {
            cosOfAngle = -1.0;
        }
        if (cosOfAngle > 1.0) {
            cosOfAngle = 1.0;
        }
        return Math.acos(cosOfAngle);
    }

    /**
     * Add two vectors and return the sum as a new vector.
     *
     * @param v1 the first vector
     * @param v2 the second vector
     * @return a new vector which is the sum v1 + v2
     */
    public static myVector3D add(final myVector3D v1, final myVector3D v2) {
        return new myVector3D(v1.getX() + v2.getX(), v1.getY() + v2.getY(), v1.getZ() + v2.getZ());
    }

    /**
     * Add the vector v to this vector.
     *
     * @param v the vector to be added to this vector
     */
    public void add(final myVector3D v) {
        __x += v.getX();
        __y += v.getY();
        __z += v.getZ();
    }

    /**
     * Subtract v2 from v1.
     *
     * @param v1 the first vector
     * @param v2 the second vector
     * @return a new vectow which is v1 - v2
     */
    public static myVector3D sub(final myVector3D v1, final myVector3D v2) {
        return new myVector3D(v1.getX() - v2.getX(), v1.getY() - v2.getY(), v1.getZ() - v2.getZ());
    }

    /**
     * Subtract the vector v from this vector.
     *
     * @param v the vector to be subtracted from this vector
     */
    public void sub(final myVector3D v) {
        __x -= v.getX();
        __y -= v.getY();
        __z -= v.getZ();
    }

    /**
     * Compute the cosine of the angle this vector makes with x-axis.
     *
     * @return the cosine of the angle this vector makes with x-axis
     */
    public double cosAlpha() {
        return cosineOfAngle(this, new myVector3D(1, 0, 0));
    }

    /**
     * Compute the cosine of the angle this vector makes with y-axis.
     *
     * @return the cosine of the angle this vector makes with y-axis
     */
    public double cosBeta() {
        return cosineOfAngle(this, new myVector3D(0, 1, 0));
    }

    /**
     * Compute the cosine of the angle this vector makes with z-axis.
     *
     * @return the cosine of the angle this vector makes with z-axis
     */
    public double cosGamma() {
        return cosineOfAngle(this, new myVector3D(0, 0, 1));
    }

    /**
     * Rescale the vector by a multiplicative factor.
     *
     * @param v the vector to be rescaled
     * @param c the multiplicative scaling factor
     * @return a new vector which represents c * v
     */
    public static myVector3D rescale(final myVector3D v, final double c) {
        return new myVector3D(v.getX() * c, v.getY() * c, v.getZ() * c);
    }

    /**
     * Rescale this vector by a multiplicative factor.
     *
     * @param c the multiplicative scaling factor
     */
    public void rescale(final double c) {
        __x *= c;
        __y *= c;
        __z *= c;
    }

    /**
     * Rotate the vector by an amount specified by a rotation matrix.
     *
     * @param v the vector to be rotated
     * @param R the rotation matrix
     * @return a new vector which represents R(v)
     */
    public static myVector3D rotate(final myVector3D v, final Matrix R) {
        return new myVector3D(R.times(v.getXYZ()));
    }

    /**
     * Normalize the vector, i.e., make it a unit vector.
     *
     * @param v the vector to be normalized
     * @return a new vector which is a unit vector along the direction of v
     */
    public static myVector3D normalize(final myVector3D v) {
        double lengthOfThisVector = v.norm();
        if (lengthOfThisVector == 0)
            return new myVector3D(0, 0, 0);
        return new myVector3D(v.getX() / lengthOfThisVector, v.getY() / lengthOfThisVector, v.getZ() / lengthOfThisVector);
    }

    /**
     * Project the first vector onto the second vector.
     *
     * @param toBeProjected the first vector
     * @param projectedOnTo the second vector
     * @return the projection of the first vector on the second vector
     */
    public static myVector3D projectionOntoVector(final myVector3D toBeProjected, final myVector3D projectedOnTo) {
        double scalingFactor = dotProduct(toBeProjected, projectedOnTo) / projectedOnTo.SquaredNorm();
        myVector3D v = new myVector3D(projectedOnTo);
        v.rescale(scalingFactor);
        return v;
    }

    /**
     * Project the vector onto a hyperplane.
     *
     * @param v the vector to be projected
     * @param normalToHyperplane the normal to the hyperplane
     * @return the projection of v onto the hyperplane
     */
    public static myVector3D projectionOntoHyperplane(final myVector3D v, final myVector3D normalToHyperplane) {
        return sub(v, projectionOntoVector(v, normalToHyperplane));
    }
}
