/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2012 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, August 04, 2012
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.1, August 04, 2012
 * @author        Chittaranjan Tripathy (2007-2012)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.io.*;
import java.util.*;

/**
 * Description of the class
 */
public class myPdbParser {

    private int __line_number = 0;
    private String __input_file_name = null;
    private BufferedReader __buffered_reader = null;
    private States __curr_state = States.Header;
    private int __is_model_open = 0; // 0: MODEL found, 1: ENDMDL found, 2: Expect one more MODEL to be there in PDB
    private myProtein __next_protein = null;

    /**
     * The different states where the parser DFA can be in.
     */
    public enum States {

        Header, Remark, Model, Endmdl, Ter, End, Atom, Hetatm, Conect, Master
    }

    /**
     * PDB row type.
     */
    public enum LineType {

        HEADER, DBREF, SEQRES, ATOM, HETATM, MASTER, ENDMDL, OTHER, TER, MODEL, END, CONECT
    }

    /**
     * If set to true then the pdb file being parsed uses old pdb naming convention
     * for atoms. By default it is set to false and hence new pdb naming convention
     * is assumed.
     */
    boolean __read_old_format_pdb = false;

    /**
     * This data structure contains the mapping between old and new atom names.
     */
    private static Map<String, Map<String, String>> __fall_backs = new TreeMap<String, Map<String, String>>();

    /**
     * This static block initializes the mapping between the old and new atom names.
     */
    static {
        {
            // ALA fall back atoms
            Map<String, String> __ala_fall_back = new TreeMap<String, String>();
            __ala_fall_back.put("HN", "H");
            __ala_fall_back.put("1HB", "HB1");
            __ala_fall_back.put("2HB", "HB2");
            __ala_fall_back.put("3HB", "HB3");
            __fall_backs.put("ALA", __ala_fall_back);
        }

        {
            // ARG fall back atoms
            Map<String, String> __arg_fall_back = new TreeMap<String, String>();
            __arg_fall_back.put("HN", "H");
            __arg_fall_back.put("1HB", "HB2");
            __arg_fall_back.put("2HB", "HB3");
            __arg_fall_back.put("1HD", "HD2");
            __arg_fall_back.put("2HD", "HD3");
            __arg_fall_back.put("1HG", "HG2");
            __arg_fall_back.put("2HG", "HG3");
            __arg_fall_back.put("1HH1", "HH11");
            __arg_fall_back.put("2HH1", "HH12");
            __arg_fall_back.put("1HH2", "HH21");
            __arg_fall_back.put("2HH2", "HH22");
            __fall_backs.put("ARG", __arg_fall_back);
        }

        {
            // ASN fall back atoms
            Map<String, String> __asn_fall_back = new TreeMap<String, String>();
            __asn_fall_back.put("HN", "H");
            __asn_fall_back.put("1HB", "HB2");
            __asn_fall_back.put("2HB", "HB3");
            __asn_fall_back.put("1HD2", "HD21");
            __asn_fall_back.put("2HD2", "HD22");
            __fall_backs.put("ASN", __asn_fall_back);
        }

        {
            // ASP fall back atoms
            Map<String, String> __asp_fall_back = new TreeMap<String, String>();
            __asp_fall_back.put("HN", "H");
            __asp_fall_back.put("1HB", "HB2");
            __asp_fall_back.put("2HB", "HB3");
            __fall_backs.put("ASP", __asp_fall_back);
        }

        {
            // CYS fall back atoms
            Map<String, String> __cys_fall_back = new TreeMap<String, String>();
            __cys_fall_back.put("HN", "H");
            __cys_fall_back.put("1HB", "HB2");
            __cys_fall_back.put("2HB", "HB3");
            __fall_backs.put("CYS", __cys_fall_back);
        }

        {
            // GLN fall back atoms
            Map<String, String> __gln_fall_back = new TreeMap<String, String>();
            __gln_fall_back.put("HN", "H");
            __gln_fall_back.put("1HB", "HB2");
            __gln_fall_back.put("2HB", "HB3");
            __gln_fall_back.put("1HG", "HG2");
            __gln_fall_back.put("2HG", "HG3");
            __gln_fall_back.put("1HE2", "HE21");
            __gln_fall_back.put("2HE2", "HE22");
            __fall_backs.put("GLN", __gln_fall_back);
        }

        {
            // GLU fall back atoms
            Map<String, String> __glu_fall_back = new TreeMap<String, String>();
            __glu_fall_back.put("HN", "H");
            __glu_fall_back.put("1HB", "HB2");
            __glu_fall_back.put("2HB", "HB3");
            __glu_fall_back.put("1HG", "HG2");
            __glu_fall_back.put("2HG", "HG3");
            __fall_backs.put("GLU", __glu_fall_back);
        }

        {
            // GLY fall back atoms
            Map<String, String> __gly_fall_back = new TreeMap<String, String>();
            __gly_fall_back.put("HN", "H");
            __gly_fall_back.put("HA1", "HA2");
            __gly_fall_back.put("HA2", "HA3");
            __fall_backs.put("GLY", __gly_fall_back);
        }

        {
            // HIS fall back atoms
            Map<String, String> __his_fall_back = new TreeMap<String, String>();
            __his_fall_back.put("HN", "H");
            __his_fall_back.put("1HB", "HB2");
            __his_fall_back.put("2HB", "HB3");
            __his_fall_back.put("1HD", "HD1");
            __his_fall_back.put("2HD", "HD2"); // note: HIS is different so come back here later
            __fall_backs.put("HIS", __his_fall_back);
        }

        {
            // ILE fall back atoms
            Map<String, String> __ile_fall_back = new TreeMap<String, String>();
            __ile_fall_back.put("HN", "H");
            __ile_fall_back.put("1HG1", "HG12");
            __ile_fall_back.put("2HG1", "HG13");
            __ile_fall_back.put("1HG2", "HG21");
            __ile_fall_back.put("2HG2", "HG22");
            __ile_fall_back.put("3HG2", "HG23");
            __ile_fall_back.put("1HD1", "HD11");
            __ile_fall_back.put("2HD1", "HD12");
            __ile_fall_back.put("3HD1", "HD13");
            __fall_backs.put("ILE", __ile_fall_back);
        }

        {
            // LEU fall back atoms
            Map<String, String> __leu_fall_back = new TreeMap<String, String>();
            __leu_fall_back.put("HN", "H");
            __leu_fall_back.put("1HB", "HB2");
            __leu_fall_back.put("2HB", "HB3");
            __leu_fall_back.put("1HD1", "HD11");
            __leu_fall_back.put("2HD1", "HD12");
            __leu_fall_back.put("3HD1", "HD13");
            __leu_fall_back.put("1HD2", "HD21");
            __leu_fall_back.put("2HD2", "HD22");
            __leu_fall_back.put("3HD2", "HD23");
            __fall_backs.put("LEU", __leu_fall_back);
        }

        {
            // LYS fall back atoms
            Map<String, String> __lys_fall_back = new TreeMap<String, String>();
            __lys_fall_back.put("HN", "H");
            __lys_fall_back.put("1HB", "HB2");
            __lys_fall_back.put("2HB", "HB3");
            __lys_fall_back.put("1HG", "HG2");
            __lys_fall_back.put("2HG", "HG3");
            __lys_fall_back.put("1HD", "HD2");
            __lys_fall_back.put("2HD", "HD3");
            __lys_fall_back.put("1HE", "HE2");
            __lys_fall_back.put("2HE", "HE3");
            __lys_fall_back.put("1HZ", "HZ1");
            __lys_fall_back.put("2HZ", "HZ2");
            __lys_fall_back.put("3HZ", "HZ3");
            __fall_backs.put("LYS", __lys_fall_back);
        }

        {
            // MET fall back atoms
            Map<String, String> __met_fall_back = new TreeMap<String, String>();
            __met_fall_back.put("HN", "H");
            __met_fall_back.put("1HB", "HB2");
            __met_fall_back.put("2HB", "HB3");
            __met_fall_back.put("1HG", "HG2");
            __met_fall_back.put("2HG", "HG3");
            __met_fall_back.put("1HE", "HE1");
            __met_fall_back.put("2HE", "HE2");
            __met_fall_back.put("3HE", "HE3");
            __fall_backs.put("MET", __met_fall_back);
        }

        {
            // PHE fall back atoms
            Map<String, String> __phe_fall_back = new TreeMap<String, String>();
            __phe_fall_back.put("HN", "H");
            __phe_fall_back.put("1HB", "HB2");
            __phe_fall_back.put("2HB", "HB3");
            __fall_backs.put("PHE", __phe_fall_back);
        }

        {
            // PRO fall back atoms
            Map<String, String> __pro_fall_back = new TreeMap<String, String>();
            __pro_fall_back.put("1HB", "HB2");
            __pro_fall_back.put("2HB", "HB3");
            __pro_fall_back.put("1HG", "HG2");
            __pro_fall_back.put("2HG", "HG3");
            __pro_fall_back.put("1HD", "HD2");
            __pro_fall_back.put("2HD", "HD3");
            __fall_backs.put("PRO", __pro_fall_back);
        }

        {
            // SER fall back atoms
            Map<String, String> __ser_fall_back = new TreeMap<String, String>();
            __ser_fall_back.put("HN", "H");
            __ser_fall_back.put("1HB", "HB2");
            __ser_fall_back.put("2HB", "HB3");
            __fall_backs.put("SER", __ser_fall_back);
        }

        {
            // THR fall back atoms
            Map<String, String> __thr_fall_back = new TreeMap<String, String>();
            __thr_fall_back.put("HN", "H");
            __thr_fall_back.put("1HG2", "HG21");
            __thr_fall_back.put("2HG2", "HG22");
            __thr_fall_back.put("3HG2", "HG23");
            __fall_backs.put("THR", __thr_fall_back);
        }

        {
            // TRP fall back atoms
            Map<String, String> __trp_fall_back = new TreeMap<String, String>();
            __trp_fall_back.put("HN", "H");
            __trp_fall_back.put("1HB", "HB2");
            __trp_fall_back.put("2HB", "HB3");
            __fall_backs.put("TRP", __trp_fall_back);
        }

        {
            // TYR fall back atoms
            Map<String, String> __tyr_fall_back = new TreeMap<String, String>();
            __tyr_fall_back.put("HN", "H");
            __tyr_fall_back.put("1HB", "HB2");
            __tyr_fall_back.put("2HB", "HB3");
            __fall_backs.put("TYR", __tyr_fall_back);
        }

        {
            // VAL fall back atoms
            Map<String, String> __val_fall_back = new TreeMap<String, String>();
            __val_fall_back.put("HN", "H");
            __val_fall_back.put("1HG1", "HG11");
            __val_fall_back.put("2HG1", "HG12");
            __val_fall_back.put("3HG1", "HG13");
            __val_fall_back.put("1HG2", "HG21");
            __val_fall_back.put("2HG2", "HG22");
            __val_fall_back.put("3HG2", "HG23");
            __fall_backs.put("VAL", __val_fall_back);
        }
    }

    /**
     * Return the line type of the line provided in the argument.
     *
     * @param line whose type (type of PDB row) is to be determined
     * @return the type of the line which is a PDB row
     */
    public LineType lineType(String line) {

        if (line.trim().isEmpty()) {
            return LineType.OTHER;
        }
        int i = 0;
        for (; i < line.length(); i++) {
            if (Character.isWhitespace(line.charAt(i))) {
                break;
            }
        }
        String recordName = line.substring(0, i);

        if (recordName.equals("DBREF")) {
            return LineType.DBREF;
        } else if (recordName.equals("SEQRES")) {
            return LineType.SEQRES;
        } else if (recordName.equals("ATOM")) {
            return LineType.ATOM;
        } else if (recordName.equals("HETATM")) {
            return LineType.HETATM;
        } else if (recordName.equals("MASTER")) {
            return LineType.MASTER;
        } else if (recordName.equals("ENDMDL")) {
            return LineType.ENDMDL;
        } else if (recordName.equals("END")) {
            return LineType.END;
        } else if (recordName.equals("HEADER") || recordName.equals("TITLE") || recordName.equals("COMPND") ||
                recordName.equals("SOURCE") || recordName.equals("KEYWDS") || recordName.equals("EXPDTA") ||
                recordName.equals("AUTHOR") || recordName.equals("REVDAT") || recordName.equals("JRNL") ||
                recordName.equals("REMARK") || recordName.equals("HELIX") || recordName.equals("SHEET") ||
                recordName.equals("SITE") || recordName.equals("CRYST1") || recordName.equals("ORIGX1") ||
                recordName.equals("ORIGX2") || recordName.equals("ORIGX3") || recordName.equals("SCALE1") ||
                recordName.equals("SCALE2") || recordName.equals("SCALE3") || recordName.equals("SEQADV") ||
                recordName.equals("TURN") || recordName.equals("FORMUL") || recordName.equals("HETNAM") ||
                recordName.equals("SSBOND") || recordName.equals("MODRES") || 
                recordName.equals("HET") || recordName.equals("CISPEP")) {
            return LineType.HEADER;
        } else if (recordName.equals("TER")) {
            return LineType.TER;
        } else if (recordName.equals("MODEL")) {
            return LineType.MODEL;
        } else if (recordName.equals("CONECT")) {
            return LineType.CONECT;
        } else {
            return LineType.OTHER; // Notify error condition.
        }
    }

    /**
     * Default constructor which constructs a parser object.
     */
    public myPdbParser() {
    }
    
    /**
     * Supply the parser with the the PDB file to be parsed.
     *
     * @param inputFileName the PDB file to be parsed
     */
    public void setInputFile(String inputFileName) {
        __line_number = 0;
        __input_file_name = null;
        __buffered_reader = null;
        __curr_state = States.Header;
        __is_model_open = 0; // 0: MODEL found, 1: ENDMDL found, 2: Expect one more MODEL to be there in PDB
        __next_protein = null;
        
        __input_file_name = inputFileName;
        try {
            __buffered_reader = new BufferedReader(new FileReader(__input_file_name));
        } catch (FileNotFoundException e) {
            System.out.println(inputFileName + ":" + "Error: Input file not found");
            e.printStackTrace();
        } 
        __next_protein = scanNextProtein();
    }

    /**
     * Constructor that constructs a parser object that provides a handle to the
     * file being parsed.
     *
     * @param inputFileName the file being parsed
     */
    public myPdbParser(String inputFileName) {
        __input_file_name = inputFileName;
        try {
            __buffered_reader = new BufferedReader(new FileReader(__input_file_name));
        } catch (FileNotFoundException e) {
            System.out.println(inputFileName + ":" + "Error: Input file not found");
            e.printStackTrace();
        } 
        __next_protein = scanNextProtein();
    }

    /**
     * Constructor that constructs a parser object that provides a handle to the
     * file being parsed.
     *
     * @param inputFileName the file being parsed
     * @param setOldFormatPdbRead if set to true the file uses old pdb naming
     * of atoms
     */
    public myPdbParser(String inputFileName, boolean setOldFormatPdbRead) {
        __read_old_format_pdb = setOldFormatPdbRead;

        __input_file_name = inputFileName;
        try {
            __buffered_reader = new BufferedReader(new FileReader(__input_file_name));
        } catch (FileNotFoundException e) {
            System.out.println(inputFileName + ":" + "Error: Input file not found");
            e.printStackTrace();
        } 
        __next_protein = scanNextProtein();
    }

    /**TODO
     * Constructor that constructs a parser object that provides a handle to the
     * file being parsed.
     *
     * @param inputFileName the file being parsed
     */
    List<String> __list_of_strings_as_pdb = null;
    Iterator<String> __list_iterator = null;
    
    public myPdbParser(List<String> inputPdbFileContentInList) {
        __list_of_strings_as_pdb = inputPdbFileContentInList;
        __list_iterator = __list_of_strings_as_pdb.iterator();
        __input_file_name = "<NOT Valid> as reading from a list of strings representing a PDB";
//        try {
//            __buffered_reader = new BufferedReader(new FileReader(__input_file_name));
//        } catch (FileNotFoundException e) {
//            System.out.println(inputFileName + ":" + "Error: Input file not found");
//            e.printStackTrace();
//        } catch (IOException e) {
//            e.printStackTrace();
//        }
        __next_protein = scanNextProtein();
    }
    
    /**
     * Return the line number of the line in the PDB file which is being parsed.
     *
     * @return the line number (in the PDB file) of the current line being parsed
     */
    public int getLineNumber() {
        return __line_number;
    }

    /**
     * Return the name of the PDB file being parsed.
     *
     * @return the name of the PDB file being parsed
     */
    public String getInputFileName() {
        return __input_file_name;
    }

    
    
    private String nextLineFromListOfString() {
        String padding = "                                                                                "; // 80 spaces for padding       
        
        do {
            String currLine = null;
            if (__list_iterator.hasNext()) {
                currLine = __list_iterator.next();
            }
            __line_number++;
        
            //System.out.println("currLineRead: " + currLine);
            if (currLine == null) {
                currLine = "EOF";
            } else if (lineType(currLine).equals(LineType.OTHER)) { // means you read a blank line (or an unrecognized line) somewhere in the PDB file so eat this and continue with next iteration
                continue;
            }

            if (currLine.length() < 80) {
                currLine += padding.substring(0, 80 - currLine.length());
            } else if (currLine.length() > 80) {
                currLine = currLine.substring(0, 80);
            } else; // do nothing as the string length is 80

            assert currLine.length() == 80 : "Error: The pdb row is not 80 character long";

            if (__read_old_format_pdb == true) {
                currLine = convertOldPdbFormatToNewPdbFormat(currLine);
            }
            
            return currLine;
        } while (true);

    }
    
    /**
     * Read the next line from the input file.
     *
     * @return the line read
     */
    private String nextLine() {        
        if (__list_of_strings_as_pdb != null) {
            return nextLineFromListOfString();            
        }       
        
        String padding = "                                                                                "; // 80 spaces for padding
        String currLine = null;

        do {
            try {
                currLine = __buffered_reader.readLine();
            } catch (IOException e) {
                e.printStackTrace();
            }
            __line_number++;

            //System.out.println("currLineRead: " + currLine);
            if (currLine == null) {
                currLine = "EOF";
            } else if (lineType(currLine).equals(LineType.OTHER)) { // means you read a blank line (or an unrecognized line) somewhere in the PDB file so eat this and continue with next iteration
                continue;
            }

            if (currLine.length() < 80) {
                currLine += padding.substring(0, 80 - currLine.length());
            } else if (currLine.length() > 80) {
                currLine = currLine.substring(0, 80);
            } else; // do nothing as the string length is 80

            assert currLine.length() == 80 : "Error: The pdb row is not 80 character long";

            if (__read_old_format_pdb == true) {
                currLine = convertOldPdbFormatToNewPdbFormat(currLine);
            }
            
            return currLine;
        } while (true);
    }

    /**
     * Convert a pdb atom row from old format to new format by changing (only) the
     * name of the pdb atom.
     *
     * @param row pdb row for an atom
     * @return the pdb row in new format
     */
    public String convertOldPdbFormatToNewPdbFormat(String row) {
        LineType ltype = lineType(row);
        if (ltype.equals(LineType.ATOM)) {
            String atomName = row.substring(12, 16).trim();
            String residueName = row.substring(17, 20).trim();

            String newAtomName = __fall_backs.get(residueName).get(atomName);
            if (newAtomName != null) {
                if (newAtomName.length() == 1) {
                    newAtomName = " " + newAtomName + "  ";
                } else if (newAtomName.length() == 2) {
                    newAtomName = " " + newAtomName + " ";
                } else if (newAtomName.length() == 3) {
                    newAtomName = " " + newAtomName;
                } else;

                String newRow = row.substring(0, 12) + newAtomName + row.substring(16, row.length());
                return newRow;
            } else {
                return row;
            }
        } else {
            return row;
        }
    }


    /**
     * Return true if the PDB file being parsed still have more model remaining to be parsed.
     *
     * @return true if the PDB file being parsed still have more model remaining to be parsed.
     */
    public boolean hasNextProtein() {
        return __next_protein != null;
    }

    /**
     * Return the next PDB model parsed from the input file supplied previously.
     *
     * @return the parsed protein representation of the PDB model
     */
    public myProtein nextProtein() {
        if (__next_protein == null) {
            return null;
        }
        myProtein p = __next_protein;
        __next_protein = scanNextProtein();
        return p;
    }

    /**
     * This method implements a deterministic finite Automaton (DFA). It scans 
     * one model at a time and returns the corresponding protein structure.
     * Note that in the usual sense a parser parses a CFL. Owing to the simplicity
     * of the PDB format, we found that a DFA with suitable flags can do the
     * same job since the stack depth is atmost one (MODEL->ENDMDL).
     *
     * @return the parsed protein
     */
    private myProtein scanNextProtein() {
        myProtein thisProtein = new myProtein();
        String currLine = null;
        char chainId;
        Vector<String> block = new Vector<String>();
        LineType ltype = null;

        BreakFromTheWhileLoopAndReturnProtein:
        while (true) {
            switch (__curr_state) {
                case Header:
                    currLine = nextLine();
                    ltype = lineType(currLine);
                    switch (ltype) {
                        case HEADER:
                        case DBREF:
                        case SEQRES:
                            thisProtein.setHeader(thisProtein.getHeader() + currLine + '\n');
                            __curr_state = States.Header;
                            break;
                        case MODEL:
                            int modelNum = Integer.parseInt(currLine.substring(10, 14).trim());
                            System.out.println("Reading Model: " + modelNum);
                            __is_model_open = 1;                                
                            __curr_state = States.Model;
                            break;
                        case ATOM:
                            if (!block.isEmpty()) {
                                System.out.println(getInputFileName() + ":" + getLineNumber() + ":Error: Invalid row format");
                                System.exit(1);
                            } else {
                                block.add(currLine);
                            }
                            __curr_state = States.Atom;
                            break;
                        case HETATM:
                            if (!block.isEmpty()) {
                                System.out.println(getInputFileName() + ":" + getLineNumber() + ":Error: Invalid row format");
                                System.exit(1);
                            } else {
                                block.add(currLine);
                            }
                            __curr_state = States.Hetatm;
                            break;
                        default:
                            System.out.println(getInputFileName() + ":" + getLineNumber() + ":Error: Invalid row format");
                            System.exit(1);
                    }
                    break;

                case Remark:
                    currLine = nextLine();
                    ltype = lineType(currLine);
                    break;

                case Model:
                    currLine = nextLine();
                    ltype = lineType(currLine);
                    switch (ltype) {
                        case ATOM:
                            if (!block.isEmpty()) {
                                System.out.println(getInputFileName() + ":" + getLineNumber() + ":Error: Invalid row format");
                                System.exit(1);
                            } else {
                                block.add(currLine);
                            }
                            __curr_state = States.Atom;
                            break;
                        case HETATM:
                            if (!block.isEmpty()) {
                                System.out.println(getInputFileName() + ":" + getLineNumber() + ":Error: Invalid row format");
                                System.exit(1);
                            } else {
                                block.add(currLine);
                            }
                            __curr_state = States.Hetatm;
                            break;
                        default:
                            System.out.println(getInputFileName() + ":" + getLineNumber() + ":Error: Invalid row format");
                            System.exit(1);
                    }
                    break;

                case Endmdl:
                    if (__is_model_open == 0) {
                        System.out.println(getInputFileName() + ":" + getLineNumber() + ":Error: There is no MODEL that matches with this ENDMDL");
                        System.exit(1);
                    } else if (__is_model_open == 1) {
                        __is_model_open = 2;
                        thisProtein.synchronizeProtein();
                        return thisProtein;
                    } else if (__is_model_open == 2) {
                        currLine = nextLine();
                        ltype = lineType(currLine);
                        __is_model_open = 0;
                    }

                    switch (ltype) {
                        case MODEL:
                            int modelNum = Integer.parseInt(currLine.substring(10, 14).trim());
                            System.out.println("Reading Model: " + modelNum);
                            __is_model_open = 1;
                            __curr_state = States.Model;
                            break;
                        case CONECT:
                            __curr_state = States.Conect;                            
                            interpretConnect(currLine, thisProtein);
                            break;
                        case MASTER:
                            __curr_state = States.Master;
                            System.out.println(getInputFileName() + ":" + getLineNumber() + ":TODO: MASTER row is yet to be handled. So skipping MASTER");
                            break;
                        case END:
                            __curr_state = States.End;
                            break;
                        default:
                            System.out.println(getInputFileName() + ":" + getLineNumber() + ":Error: Invalid row format");
                            System.exit(1);
                    }
                    break;

                case Ter:
                    currLine = nextLine();
                    ltype = lineType(currLine);

                    assert block.size() == 0 : "Error: Invalid TER state";
                    switch (ltype) {
                        case ENDMDL:
                            __curr_state = States.Endmdl;
                            break;
                        case END:
                            __curr_state = States.End;
                            break;
                        case ATOM:
                            block.add(currLine);
                            __curr_state = States.Atom;
                            break;
                        case HETATM:
                            block.add(currLine);
                            __curr_state = States.Hetatm;
                            break;
                        case MASTER:
                            __curr_state = States.Master;
                            System.out.println(getInputFileName() + ":" + getLineNumber() + ":TODO: MASTER row is yet to be handled. So skipping MASTER");
                            break;                            
                        default:
                            System.out.println(getInputFileName() + ":" + getLineNumber() + ":Error: Invalid row format");
                            System.exit(1);
                    }
                    break;

                case Master:
                    currLine = nextLine();
                    ltype = lineType(currLine);

                    switch (ltype) {
                        case END:
                            __curr_state = States.End;
                            break;
                        default:
                            System.out.println(getInputFileName() + ":" + getLineNumber() + ":Error: END is expected");
                            System.exit(1);
                        //break;
                    }
                    break;

                case End:
                    currLine = nextLine();
                    ltype = lineType(currLine);

                    System.out.println("currLine: " + currLine);
                    System.out.println("Success! The PDB file has been parsed successfully " + __input_file_name);
                    break BreakFromTheWhileLoopAndReturnProtein;

                case Atom:
                    currLine = nextLine();
                    ltype = lineType(currLine);

                    switch (ltype) {
                        case ATOM: // Either completes reading a residue block or still reading a block
                            if (!block.isEmpty()) {
                                if (Integer.parseInt(currLine.substring(22, 26).trim()) != Integer.parseInt(block.elementAt(0).substring(22, 26).trim())) { // completed reading a residue block
                                    chainId = block.elementAt(0).charAt(21);
                                    thisProtein.setChainId(chainId);
                                    myResidue r = new myResidue(block);
                                    thisProtein.addResidue(r);
                                    block.clear();
                                    block.add(currLine);
                                } else { // keep reading the current residue block
                                    block.add(currLine);
                                }
                            } else { // keep reading the current residue block
                                block.add(currLine);
                            }
                            __curr_state = States.Atom;
                            break;
                        case TER: // End reading this current residue block
                            // The content of the block is now is a residue
                            myResidue r = new myResidue(block);
                            thisProtein.addResidue(r);
                            block.clear();
                            __curr_state = States.Ter;
                            break;
                        case HETATM:
                            myResidue r2 = new myResidue(block);
                            thisProtein.addResidue(r2);
                            block.clear();
                            block.add(currLine);
                            __curr_state = States.Hetatm;                                                    
                            break;
                        default:  // TODO: check whether a TER/ENDMDL has to be there between ATOM and HETATM
                            System.out.println(getInputFileName() + ":" + getLineNumber() + ":Error: Missing TER");
                            System.exit(1);
                            break;
                    }
                    break;

                case Hetatm:
                    currLine = nextLine();
                    ltype = lineType(currLine);

                    switch (ltype) {
                        case ATOM: // End reading this current residue block
                            // The content of the block is now is a residue
                            myResidue r = new myResidue(block);
                            thisProtein.addResidue(r);
                            block.clear();
                            block.add(currLine);
                            __curr_state = States.Atom;
                            break;
                        case TER:
                            thisProtein.addResidue(new myResidue(block));
                            block.clear();
                            __curr_state = States.Ter;
                            break;
                        case HETATM: // Either completes reading a residue block or still reading a block
                            if (!block.isEmpty()) {
                                if (Integer.parseInt(currLine.substring(22, 26).trim()) != Integer.parseInt(block.elementAt(0).substring(22, 26).trim())) { // completed reading a residue block
                                    chainId = block.elementAt(0).charAt(21);
                                    thisProtein.setChainId(chainId);
                                    thisProtein.addResidue(new myResidue(block));
                                    block.clear();
                                    block.add(currLine);
                                } else { // keep reading the current residue block
                                    block.add(currLine);
                                }
                            } else { // keep reading the current residue block
                                block.add(currLine);
                            }
                            __curr_state = States.Hetatm;
                            break;
                        case ENDMDL: // End reading this current residue block
                            // The content of the block is now is a residue
                            thisProtein.addResidue(new myResidue(block));
                            block.clear();
                            __curr_state = States.Endmdl;
                            break;
                        case CONECT:
                            thisProtein.addResidue(new myResidue(block));
                            block.clear();
                            __curr_state = States.Conect;                            
                            interpretConnect(currLine, thisProtein);
                            //System.exit(1);
                            break;
                        case MASTER:
                            __curr_state = States.Master;
                            System.out.println(getInputFileName() + ":" + getLineNumber() + ":TODO: MASTER row is yet to be handled. So skipping MASTER");
                            break;
                        default:
                            System.out.println(__curr_state + "   " + ltype);
                            System.out.println(getInputFileName() + ":" + getLineNumber() + ":Error: TER or ENDMDL is expected");
                            System.exit(1);
                            break;
                    }
                    break;
                    
                case Conect:
                    currLine = nextLine();
                    ltype = lineType(currLine);
                    
                    switch (ltype) {
                        case CONECT:
                            __curr_state = States.Conect;                            
                            interpretConnect(currLine, thisProtein);
                            break;
                        case END:
                            __curr_state = States.End;
                            break;
                        case MASTER:   
                            __curr_state = States.Master;
                            System.out.println(getInputFileName() + ":" + getLineNumber() + ":TODO: MASTER row is yet to be handled. So skipping MASTER");
                            break;
                        default:
                            System.out.println(getInputFileName() + ":" + getLineNumber() + ":Error: MASTER/END is expected");
                            System.exit(1);
                        //break;
                            
                    }                    
                    break;
                    
                default:
                    System.out.println(getInputFileName() + ":" + getLineNumber() + ":Error: Invalid row format");
                    System.exit(1);
            }
        }

        if (thisProtein.numberOfAtoms() == 0) {        
            __next_protein = null;            
        }
        else {
            __next_protein = thisProtein;
            __next_protein.synchronizeProtein();
        }
        return __next_protein;
    }
    
    /**
     * Parse the CONNECT bond information provided in the PDB file.
     *
     * @param connectRow a PDB row for that represents CONNECT bond information
     * @param p add the CONNECT information to p
     */
    private void interpretConnect(String connectRow, myProtein p) {
        assert connectRow.substring(0, 6).trim().equals("CONECT") == true : "Error: Invalid CONECT row";
        int[] conn = {0, 0, 0, 0};
        int hostAtomSerialNum = 0;
        
        String thisAtom = connectRow.substring(6, 11).trim();
        if (!thisAtom.isEmpty())
            hostAtomSerialNum = Integer.parseInt(thisAtom);
            
        String bondedAtom1 = connectRow.substring(11, 16).trim();
        if (!bondedAtom1.isEmpty())
            conn[0] = Integer.parseInt(bondedAtom1);
        
        String bondedAtom2 = connectRow.substring(16, 21).trim();
        if (!bondedAtom2.isEmpty())
            conn[1] = Integer.parseInt(bondedAtom2);       
        
        String bondedAtom3 = connectRow.substring(21, 26).trim();
        if (!bondedAtom3.isEmpty())
            conn[2] = Integer.parseInt(bondedAtom3);       
        
        String bondedAtom4 = connectRow.substring(26, 31).trim();
        if (!bondedAtom4.isEmpty())
            conn[3] = Integer.parseInt(bondedAtom4);
        
        assert hostAtomSerialNum != 0 : "Error: Invalid CONECT row";
        
        myAtom hostAtom = null; 
        for (myResidue r : p) {
            for (myAtom a : r) {
                if (a.getSerialNumber() == hostAtomSerialNum)
                    hostAtom = a;
            }
        }        
        myAtom bAtom1 = null;
        if (conn[0] != 0) {
            for (myResidue r : p) {
                for (myAtom a : r) {
                    if (a.getSerialNumber() == conn[0]) {
                        bAtom1 = a;
                    }
                }
            }
        }        
        myAtom bAtom2 = null;
        if (conn[1] != 0) {
            for (myResidue r : p) {
                for (myAtom a : r) {
                    if (a.getSerialNumber() == conn[1]) {
                        bAtom2 = a;
                    }
                }
            }
        }          
        myAtom bAtom3 = null;
        if (conn[2] != 0) {
            for (myResidue r : p) {
                for (myAtom a : r) {
                    if (a.getSerialNumber() == conn[2]) {
                        bAtom3 = a;
                    }
                }
            }
        }        
        myAtom bAtom4 = null;
        if (conn[3] != 0) {
            for (myResidue r : p) {
                for (myAtom a : r) {
                    if (a.getSerialNumber() == conn[3]) {
                        bAtom4 = a;
                    }
                }
            }
        }          
        
        // Note addBond() method is reflexive, so it adds bond to both the atoms
        if (bAtom1 != null)
            hostAtom.addBond(bAtom1);
        if (bAtom2 != null)
            hostAtom.addBond(bAtom2);
        if (bAtom3 != null)
            hostAtom.addBond(bAtom3);
        if (bAtom3 != null)
            hostAtom.addBond(bAtom3);   
        if (bAtom4 != null)
            hostAtom.addBond(bAtom4);        
        return;
    }

    /**
     * The main method tests this class.
     *
     * @param args
     */
    public static void main(String... args) {
        //myPdbParser myP = new myPdbParser("2I5O_Modified.pdb");
        myPdbParser myP = new myPdbParser("1H3D.pdb");//("3ICB.pdb");//("1HYA.pdb");
        //myPdbParser myP = new myPdbParser();
        //myP.setInputFile("3ICB.pdb"); 
        //myP.setInputFile("1HYA.pdb");
        int iterNum = 1;
        //System.out.println(myP.hasNextProtein());
        while (myP.hasNextProtein()) {
            myProtein p = myP.nextProtein();

            //if (p == null) System.out.println(")))))))))))))))))))))))))))))))))))))))))))))))))))))");

            System.out.println("Number of Residues: " + p.numberOfResidues() + " Number of Atoms: " + p.numberOfAtoms());
            System.out.println("There are " + p.getSequence().size() + " residues in the following order: " + p.getSequence().toString());
            System.out.println("IterNum: " + iterNum);
            iterNum++;
//            //System.out.println("Printing the header: --------------------\n" + p.getHeader() + "------------------------------------------");
//            System.out.println("Has residue 45? " + p.hasResidue(45));
//            System.out.println("Chain Id: " + p.getChainId());
//            //System.out.println("Good bye... parsed the PDB file!!!");
//            //System.exit(1);
//            System.out.println("Printing the protein residue by residue");
//            System.out.println("---------------------------------------");
            for (myResidue r : p) {
                System.out.println("Residue Name: " + r.getResidueName() + " Residue Number: " + r.getResidueNumber() + " #Atoms: " + r.numberOfAtoms() + " #Bonds :" + r.numberOfBonds());
                for (myAtom a : r) {
                    a.print();
                    if (a.getElementType().equals("CA"))
                        System.out.println(a.getVdwRadius());
                //break;
                }
            }
        }
    }
}





