/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2012 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, August 04, 2012
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.1, August 04, 2012
 * @author        Chittaranjan Tripathy (2007-2012)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.util.*;

/**
 * Description of the class
 */
public class myAminoAcidResidueMap {

    /**
     * An instance of this class keeps track of new residue types.
     */
    public static class triple {

        public char __1_letter_code;
        public String __3_letter_code;
        public String __acronym;

        triple() {
        }

        triple(char ch, String s1, String s2) {
            __1_letter_code = ch;
            __3_letter_code = s1;
            __acronym = s2;
        }
    }

    /**
     * The set of all residue types that are available for use.
     */
    public static Vector<triple> __IUPAC_amino_acid_residue_map = new Vector<triple>(20);

    /**
     * A static initialization block to initialize the residue map once and for all.
     */
    static {
        __IUPAC_amino_acid_residue_map.add(new triple('A', "ALA", "Alanine"));
        __IUPAC_amino_acid_residue_map.add(new triple('R', "ARG", "Arginine"));
        __IUPAC_amino_acid_residue_map.add(new triple('N', "ASN", "Asparagine"));
        __IUPAC_amino_acid_residue_map.add(new triple('D', "ASP", "Aspartic acid"));
        __IUPAC_amino_acid_residue_map.add(new triple('C', "CYS", "Cystine"));
        __IUPAC_amino_acid_residue_map.add(new triple('Q', "GLN", "Glutamine"));
        __IUPAC_amino_acid_residue_map.add(new triple('E', "GLU", "Glutamine")); //Glutamate
        __IUPAC_amino_acid_residue_map.add(new triple('G', "GLY", "Glycine"));
        __IUPAC_amino_acid_residue_map.add(new triple('H', "HIS", "Histidine"));
        __IUPAC_amino_acid_residue_map.add(new triple('I', "ILE", "Isoleucine"));
        __IUPAC_amino_acid_residue_map.add(new triple('L', "LEU", "Leucine"));
        __IUPAC_amino_acid_residue_map.add(new triple('K', "LYS", "Lysine"));
        __IUPAC_amino_acid_residue_map.add(new triple('M', "MET", "Methionine"));
        __IUPAC_amino_acid_residue_map.add(new triple('F', "PHE", "Phenylalanine"));
        __IUPAC_amino_acid_residue_map.add(new triple('P', "PRO", "Proline"));
        __IUPAC_amino_acid_residue_map.add(new triple('S', "SER", "Serine"));
        __IUPAC_amino_acid_residue_map.add(new triple('T', "THR", "Threonine"));
        __IUPAC_amino_acid_residue_map.add(new triple('W', "TRP", "Tryptophan"));
        __IUPAC_amino_acid_residue_map.add(new triple('Y', "TYR", "Tyrosine"));
        __IUPAC_amino_acid_residue_map.add(new triple('V', "VAL", "Valine"));
    }

    /**
     * This default constructor is made private to ensure that instantiation of
     * this class is entirely forbidden. All members of this class are static.
     */
    private myAminoAcidResidueMap() {
    }

    /**
     * Test whether the argument character represents a valid amino acid residue
     * in one letter format using IUPAC nomenclature. If so then return true.
     *
     * @param ch represents a valid amion acid residue in one letter format
     * @return true if the argument is a valid amino acid residue in one letter format
     */
    public static boolean validResidueName(char ch) {
        ch = Character.toUpperCase(ch);
        // Find whether ch is a valid 1 letter code for an amino acid residue.
        for (triple t : __IUPAC_amino_acid_residue_map) {
            if (t.__1_letter_code == ch) {
                return true;
            }
        }
        return false;
    }

    /**
     * Test whether the argument string represents a valid amino acid residue
     * in three letter format using IUPAC nomenclature. If so then return true.
     *
     * @param s represents a valid amion acid residue in three letter format
     * @return true if the argument is a valid amino acid residue in three letter format
     */
    public static boolean validResidueName(String s) {
        s = s.trim().toUpperCase();
        assert s.length() == 3 : "Error: Invalid amino acid residue 3 letter code";
        // Find whether s is a valid 3 letter code for an amino acid residue.  
        for (triple t : __IUPAC_amino_acid_residue_map) {
            if (t.__3_letter_code.equalsIgnoreCase(s)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Return the three letter amino acid code given the one letter code.
     *
     * @param ch one letter amino acid code
     * @return the three letter code if there exists such a code, or else
     * return a string representing illegal code
     */
    public static String getThreeLetterCode(char ch) {
        ch = Character.toUpperCase(ch);
        // Find whether ch is a valid 1 letter code for an amino acid residue.
        for (triple t : __IUPAC_amino_acid_residue_map) {
            if (t.__1_letter_code == ch) {
                return t.__3_letter_code;
            }
        }
        return myMiscConstants.illegal;
    }

    /**
     * Return the one letter amino acid code given the three letter code.
     *
     * @param s three letter amino acid code
     * @return the one letter code if there exists such a code, or else
     * return a character representing illegal code
     */
    public static char getOneLetterCode(String s) {
        s = s.trim().toLowerCase();
        assert s.length() == 3 : "Error: Invalid amino acid residue 3 letter code";
        // Find whether s is a valid 3 letter code for an amino acid residue.
        for (triple t : __IUPAC_amino_acid_residue_map) {
            if (t.__3_letter_code.equalsIgnoreCase(s)) {
                return t.__1_letter_code;
            }
        }
        return myMiscConstants.illegalAAName;
    }

    /**
     * Below we define the set of atoms that the amino acid residues can have.
     */
    public static String[] __ALA_ATOMS = { //10 atoms
        myAtomLabel.__N,
        myAtomLabel.__CA,
        myAtomLabel.__C,
        myAtomLabel.__O,
        myAtomLabel.__CB,
        myAtomLabel.__H,
        myAtomLabel.__HA,
        myAtomLabel.__HB1,
        myAtomLabel.__HB2,
        myAtomLabel.__HB3
    };

    public static String[] __ARG_ATOMS = { //24 atoms
        myAtomLabel.__N,
        myAtomLabel.__CA,
        myAtomLabel.__C,
        myAtomLabel.__O,
        myAtomLabel.__CB,
        myAtomLabel.__CG,
        myAtomLabel.__CD,
        myAtomLabel.__NE,
        myAtomLabel.__CZ,
        myAtomLabel.__NH1,
        myAtomLabel.__NH2,
        myAtomLabel.__H,
        myAtomLabel.__HA,
        myAtomLabel.__HB2,
        myAtomLabel.__HB3,
        myAtomLabel.__HG2,
        myAtomLabel.__HG3,
        myAtomLabel.__HD2,
        myAtomLabel.__HD3,
        myAtomLabel.__HE,
        myAtomLabel.__HH11,
        myAtomLabel.__HH12,
        myAtomLabel.__HH21,
        myAtomLabel.__HH22
    };

    public static String[] __ASN_ATOMS = { //14 atoms
        myAtomLabel.__N,
        myAtomLabel.__CA,
        myAtomLabel.__C,
        myAtomLabel.__O,
        myAtomLabel.__CB,
        myAtomLabel.__CG,
        myAtomLabel.__OD1,
        myAtomLabel.__ND2,
        myAtomLabel.__H,
        myAtomLabel.__HA,
        myAtomLabel.__HB2,
        myAtomLabel.__HB3,
        myAtomLabel.__HD21,
        myAtomLabel.__HD22
    };

    public static String[] __ASP_ATOMS = { //12 atoms
        myAtomLabel.__N,
        myAtomLabel.__CA,
        myAtomLabel.__C,
        myAtomLabel.__O,
        myAtomLabel.__CB,
        myAtomLabel.__CG,
        myAtomLabel.__OD1,
        myAtomLabel.__OD2,
        myAtomLabel.__H,
        myAtomLabel.__HA,
        myAtomLabel.__HB2,
        myAtomLabel.__HB3
    };

    public static String[] __CYS_ATOMS = { //11 atoms
        myAtomLabel.__N,
        myAtomLabel.__CA,
        myAtomLabel.__C,
        myAtomLabel.__O,
        myAtomLabel.__CB,
        myAtomLabel.__SG,
        myAtomLabel.__H,
        myAtomLabel.__HA,
        myAtomLabel.__HB2,
        myAtomLabel.__HB3,
        myAtomLabel.__HG
    };

    public static String[] __GLN_ATOMS = { //17 atoms
        myAtomLabel.__N,
        myAtomLabel.__CA,
        myAtomLabel.__C,
        myAtomLabel.__O,
        myAtomLabel.__CB,
        myAtomLabel.__CG,
        myAtomLabel.__CD,
        myAtomLabel.__OE1,
        myAtomLabel.__NE2,
        myAtomLabel.__H,
        myAtomLabel.__HA,
        myAtomLabel.__HB2,
        myAtomLabel.__HB3,
        myAtomLabel.__HG2,
        myAtomLabel.__HG3,
        myAtomLabel.__HE21,
        myAtomLabel.__HE22
    };

    public static String[] __GLU_ATOMS = { //15 atoms
        myAtomLabel.__N,
        myAtomLabel.__CA,
        myAtomLabel.__C,
        myAtomLabel.__O,
        myAtomLabel.__CB,
        myAtomLabel.__CG,
        myAtomLabel.__CD,
        myAtomLabel.__OE1,
        myAtomLabel.__OE2,
        myAtomLabel.__H,
        myAtomLabel.__HA,
        myAtomLabel.__HB2,
        myAtomLabel.__HB3,
        myAtomLabel.__HG2,
        myAtomLabel.__HG3
    };

    public static String[] __GLY_ATOMS = { //7 atoms
        myAtomLabel.__N,
        myAtomLabel.__CA,
        myAtomLabel.__C,
        myAtomLabel.__O,
        myAtomLabel.__H,
        myAtomLabel.__HA2,
        myAtomLabel.__HA3
    };

    public static String[] __HIS_ATOMS = { //18 atoms
        myAtomLabel.__N,
        myAtomLabel.__CA,
        myAtomLabel.__C,
        myAtomLabel.__O,
        myAtomLabel.__CB,
        myAtomLabel.__CG,
        myAtomLabel.__ND1,
        myAtomLabel.__CD2,
        myAtomLabel.__CE1,
        myAtomLabel.__NE2,
        myAtomLabel.__H,
        myAtomLabel.__HA,
        myAtomLabel.__HB2,
        myAtomLabel.__HB3,
        myAtomLabel.__HD1,
        myAtomLabel.__HD2,
        myAtomLabel.__HE1
        //AtomLabel.__HE2 // imidazole tautomerism
    };

    public static String[] __ILE_ATOMS = { //19 atoms
        myAtomLabel.__N,
        myAtomLabel.__CA,
        myAtomLabel.__C,
        myAtomLabel.__O,
        myAtomLabel.__CB,
        myAtomLabel.__CG1,
        myAtomLabel.__CG2,
        myAtomLabel.__CD1,
        myAtomLabel.__H,
        myAtomLabel.__HA,
        myAtomLabel.__HB,
        myAtomLabel.__HG12,
        myAtomLabel.__HG13,
        myAtomLabel.__HG21,
        myAtomLabel.__HG22,
        myAtomLabel.__HG23,
        myAtomLabel.__HD11,
        myAtomLabel.__HD12,
        myAtomLabel.__HD13
    };

    public static String[] __LEU_ATOMS = { //19 atoms
        myAtomLabel.__N,
        myAtomLabel.__CA,
        myAtomLabel.__C,
        myAtomLabel.__O,
        myAtomLabel.__CB,
        myAtomLabel.__CG,
        myAtomLabel.__CD1,
        myAtomLabel.__CD2,
        myAtomLabel.__H,
        myAtomLabel.__HA,
        myAtomLabel.__HB2,
        myAtomLabel.__HB3,
        myAtomLabel.__HG,
        myAtomLabel.__HD11,
        myAtomLabel.__HD12,
        myAtomLabel.__HD13,
        myAtomLabel.__HD21,
        myAtomLabel.__HD22,
        myAtomLabel.__HD23
    };

    public static String[] __LYS_ATOMS = { //22 atoms
        myAtomLabel.__N,
        myAtomLabel.__CA,
        myAtomLabel.__C,
        myAtomLabel.__O,
        myAtomLabel.__CB,
        myAtomLabel.__CG,
        myAtomLabel.__CD,
        myAtomLabel.__CE,
        myAtomLabel.__NZ,
        myAtomLabel.__H,
        myAtomLabel.__HA,
        myAtomLabel.__HB2,
        myAtomLabel.__HB3,
        myAtomLabel.__HG2,
        myAtomLabel.__HG3,
        myAtomLabel.__HD2,
        myAtomLabel.__HD3,
        myAtomLabel.__HE2,
        myAtomLabel.__HE3,
        myAtomLabel.__HZ1,
        myAtomLabel.__HZ2,
        myAtomLabel.__HZ3
    };

    public static String[] __MET_ATOMS = { //17 atoms
        myAtomLabel.__N,
        myAtomLabel.__CA,
        myAtomLabel.__C,
        myAtomLabel.__O,
        myAtomLabel.__CB,
        myAtomLabel.__CG,
        myAtomLabel.__SD,
        myAtomLabel.__CE,
        myAtomLabel.__H,
        myAtomLabel.__HA,
        myAtomLabel.__HB2,
        myAtomLabel.__HB3,
        myAtomLabel.__HG2,
        myAtomLabel.__HG3,
        myAtomLabel.__HE1,
        myAtomLabel.__HE2,
        myAtomLabel.__HE3
    };

    public static String[] __PHE_ATOMS = { //20 atoms
        myAtomLabel.__N,
        myAtomLabel.__CA,
        myAtomLabel.__C,
        myAtomLabel.__O,
        myAtomLabel.__CB,
        myAtomLabel.__CG,
        myAtomLabel.__CD1,
        myAtomLabel.__CD2,
        myAtomLabel.__CE1,
        myAtomLabel.__CE2,
        myAtomLabel.__CZ,
        myAtomLabel.__H,
        myAtomLabel.__HA,
        myAtomLabel.__HB2,
        myAtomLabel.__HB3,
        myAtomLabel.__HD1,
        myAtomLabel.__HD2,
        myAtomLabel.__HE1,
        myAtomLabel.__HE2,
        myAtomLabel.__HZ
    };

    public static String[] __PRO_ATOMS = { //14 atoms
        myAtomLabel.__N,
        myAtomLabel.__CA,
        myAtomLabel.__C,
        myAtomLabel.__O,
        myAtomLabel.__CB,
        myAtomLabel.__CG,
        myAtomLabel.__CD,
        myAtomLabel.__HA,
        myAtomLabel.__HB2,
        myAtomLabel.__HB3,
        myAtomLabel.__HG2,
        myAtomLabel.__HG3,
        myAtomLabel.__HD2,
        myAtomLabel.__HD3
    };

    public static String[] __SER_ATOMS = { //11 atoms
        myAtomLabel.__N,
        myAtomLabel.__CA,
        myAtomLabel.__C,
        myAtomLabel.__O,
        myAtomLabel.__CB,
        myAtomLabel.__OG,
        myAtomLabel.__H,
        myAtomLabel.__HA,
        myAtomLabel.__HB2,
        myAtomLabel.__HB3,
        myAtomLabel.__HG
    };

    public static String[] __THR_ATOMS = { //14 atoms
        myAtomLabel.__N,
        myAtomLabel.__CA,
        myAtomLabel.__C,
        myAtomLabel.__O,
        myAtomLabel.__CB,
        myAtomLabel.__OG1,
        myAtomLabel.__CG2,
        myAtomLabel.__H,
        myAtomLabel.__HA,
        myAtomLabel.__HB,
        myAtomLabel.__HG1,
        myAtomLabel.__HG21,
        myAtomLabel.__HG22,
        myAtomLabel.__HG23
    };

    public static String[] __TRP_ATOMS = { //24 atoms
        myAtomLabel.__N,
        myAtomLabel.__CA,
        myAtomLabel.__C,
        myAtomLabel.__O,
        myAtomLabel.__CB,
        myAtomLabel.__CG,
        myAtomLabel.__CD1,
        myAtomLabel.__CD2,
        myAtomLabel.__NE1,
        myAtomLabel.__CE2,
        myAtomLabel.__CE3,
        myAtomLabel.__CZ2,
        myAtomLabel.__CZ3,
        myAtomLabel.__CH2,
        myAtomLabel.__H,
        myAtomLabel.__HA,
        myAtomLabel.__HB2,
        myAtomLabel.__HB3,
        myAtomLabel.__HD1,
        myAtomLabel.__HE1,
        myAtomLabel.__HE3,
        myAtomLabel.__HZ2,
        myAtomLabel.__HZ3,
        myAtomLabel.__HH2
    };

    public static String[] __TYR_ATOMS = { //21 atoms
        myAtomLabel.__N,
        myAtomLabel.__CA,
        myAtomLabel.__C,
        myAtomLabel.__O,
        myAtomLabel.__CB,
        myAtomLabel.__CG,
        myAtomLabel.__CD1,
        myAtomLabel.__CD2,
        myAtomLabel.__CE1,
        myAtomLabel.__CE2,
        myAtomLabel.__CZ,
        myAtomLabel.__OH,
        myAtomLabel.__H,
        myAtomLabel.__HA,
        myAtomLabel.__HB2,
        myAtomLabel.__HB3,
        myAtomLabel.__HD1,
        myAtomLabel.__HD2,
        myAtomLabel.__HE1,
        myAtomLabel.__HE2,
        myAtomLabel.__HH
    };

    public static String[] __VAL_ATOMS = { //16 atoms
        myAtomLabel.__N,
        myAtomLabel.__CA,
        myAtomLabel.__C,
        myAtomLabel.__O,
        myAtomLabel.__CB,
        myAtomLabel.__CG1,
        myAtomLabel.__CG2,
        myAtomLabel.__H,
        myAtomLabel.__HA,
        myAtomLabel.__HB,
        myAtomLabel.__HG11,
        myAtomLabel.__HG12,
        myAtomLabel.__HG13,
        myAtomLabel.__HG21,
        myAtomLabel.__HG22,
        myAtomLabel.__HG23
    };

    /**
     * The set of all atoms in the set of residue types that are available for use.
     */
    public static Map<String, String[]> __residue_atom_map = new TreeMap<String, String[]>();

    /**
     * A static initialization block to initialize the atom map for the residues once and for all.
     */
    static {
        __residue_atom_map.put("ALA", __ALA_ATOMS);
        __residue_atom_map.put("ARG", __ARG_ATOMS);
        __residue_atom_map.put("ASN", __ASN_ATOMS);
        __residue_atom_map.put("ASP", __ASP_ATOMS);
        __residue_atom_map.put("CYS", __CYS_ATOMS);
        __residue_atom_map.put("GLN", __GLN_ATOMS);
        __residue_atom_map.put("GLU", __GLU_ATOMS);
        __residue_atom_map.put("GLY", __GLY_ATOMS);
        __residue_atom_map.put("HIS", __HIS_ATOMS);
        __residue_atom_map.put("ILE", __ILE_ATOMS);
        __residue_atom_map.put("LEU", __LEU_ATOMS);
        __residue_atom_map.put("LYS", __LYS_ATOMS);
        __residue_atom_map.put("MET", __MET_ATOMS);
        __residue_atom_map.put("PHE", __PHE_ATOMS);
        __residue_atom_map.put("PRO", __PRO_ATOMS);
        __residue_atom_map.put("SER", __SER_ATOMS);
        __residue_atom_map.put("THR", __THR_ATOMS);
        __residue_atom_map.put("TRP", __TRP_ATOMS);
        __residue_atom_map.put("TYR", __TYR_ATOMS);
        __residue_atom_map.put("VAL", __VAL_ATOMS);
    }
}




