/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2012 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, August 04, 2012
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.1, August 04, 2012
 * @author        Chittaranjan Tripathy (2007-2012)
 * @email         chittu@cs.duke.edu
 * @author        Lincong Wang (2001-2005)
 * @email         wlincong@cs.dartmouth.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.text.NumberFormat;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.Locale;
import java.text.FieldPosition;
import java.io.PrintWriter;
import java.io.BufferedReader;
import java.io.StreamTokenizer;

/**
 * Description of the class
 */
public class Matrix implements Cloneable, java.io.Serializable {

    private double[][] A;    //a 2-by-2 matrix
    private int m, n;        //row m and column n.

   /** Construct an m-by-n matrix of zeros. 
   @param m    Number of rows.
   @param n    Number of colums.
   */
    public Matrix (int m, int n) {
	this.m = m;
	this.n = n;
	A = new double[m][n];
    }

   /** Construct an m-by-n constant matrix.
   @param m    Number of rows.
   @param n    Number of colums.
   @param s    Fill the matrix with this scalar value.
   */
   public Matrix (int m, int n, double s) {
      this.m = m;
      this.n = n;
      A = new double[m][n];
      for (int i = 0; i < m; i++) {
         for (int j = 0; j < n; j++) {
            A[i][j] = s;
         }
      }
   }

   /** Construct a matrix from a 2-D array.
   @param A    Two-dimensional array of doubles.
   @exception  IllegalArgumentException All rows must have the same length
   */
   public Matrix (double[][] A) {
      m = A.length;
      n = A[0].length;
      for (int i = 0; i < m; i++) {
         if (A[i].length != n) {
            throw new IllegalArgumentException("All rows must have the same length.");
         }
      }
      this.A = A;
   }

   /** Construct a matrix quickly without checking arguments.
   @param A    Two-dimensional array of doubles.
   @param m    Number of rows.
   @param n    Number of colums.
   */
   public Matrix (double[][] A, int m, int n) {
      this.A = A;
      this.m = m;
      this.n = n;
   }

   /** Construct a matrix from a one-dimensional packed array
   @param vals One-dimensional array of doubles, packed by columns (ala Fortran).
   @param m    Number of rows.
   @exception  IllegalArgumentException Array length must be a multiple of m.
   */
   public Matrix (double vals[], int m) {
      this.m = m;
      n = (m != 0 ? vals.length/m : 0);
      if (m*n != vals.length) {
         throw new IllegalArgumentException("Array length must be a multiple of m.");
      }
      A = new double[m][n];
      for (int i = 0; i < m; i++) {
         for (int j = 0; j < n; j++) {
            A[i][j] = vals[i+j*m];
         }
      }
   }

   /** 
       Make a deep copy of a matrix
   */
   public Matrix copy () {
      Matrix X = new Matrix(m,n);
      double[][] C = X.getArray();
      for (int i = 0; i < m; i++) {
         for (int j = 0; j < n; j++) {
            C[i][j] = A[i][j];
         }
      }
      return X;
   }
      
    /** 
     * make a 3x3  rotation matrix with a rotation angle theta and along an axis
     * Please note that only axis only take +/-x, y, z
     * A method by Lincong Wang
     @param theta the rotation angle
     @param axis  the axis along which the rotation is made
     @return the 3x3 matrix
    */
    public static Matrix rotationMat(double theta, String axis){
	double [][] a = new double[3][3];
	if (axis.equals("+x")){
	    a[0][0] = 1.0;
	    a[0][1] = 0.0;
	    a[0][2] = 0.0;
	    a[1][0] = 0.0;
	    a[1][1] = Math.cos(theta);
	    a[1][2] = Math.sin(theta);
	    a[2][0] = 0.0;
	    a[2][1] = -Math.sin(theta);
	    a[2][2] = Math.cos(theta);
	} else if (axis.equals("-x")){
	    a[0][0] = 1.0;
	    a[0][1] = 0.0;
	    a[0][2] = 0.0;
	    a[1][0] = 0.0;
	    a[1][1] = Math.cos(theta);
	    a[1][2] = -Math.sin(theta);
	    a[2][0] = 0.0;
	    a[2][1] = Math.sin(theta);
	    a[2][2] = Math.cos(theta);
	}else if (axis.equals("+y")){
	    a[0][0] = Math.cos(theta);
	    a[0][1] = 0.0;
	    a[0][2] = -Math.sin(theta);
	    a[1][0] = 0.0;
	    a[1][1] = 1.0;
	    a[1][2] = 0.0;
	    a[2][0] = Math.sin(theta);
	    a[2][1] = 0.0;
	    a[2][2] = Math.cos(theta);
	}else if (axis.equals("-y")){
	    a[0][0] = Math.cos(theta);
	    a[0][1] = 0.0;
	    a[0][2] = Math.sin(theta);
	    a[1][0] = 0.0;
	    a[1][1] = 1.0;
	    a[1][2] = 0.0;
	    a[2][0] = -Math.sin(theta);
	    a[2][1] = 0.0;
	    a[2][2] = Math.cos(theta);
	}else if (axis.equals("+z")){
	    a[0][0] = Math.cos(theta);
	    a[0][1] = Math.sin(theta);
	    a[0][2] = 0.0;
	    a[1][0] = -Math.sin(theta);
	    a[1][1] = Math.cos(theta);
	    a[1][2] = 0.0;
	    a[2][0] = 0.0;
	    a[2][1] = 0.0;
	    a[2][2] = 1.0;
	}else if (axis.equals("-z")){
	    a[0][0] = Math.cos(theta);
	    a[0][1] = -Math.sin(theta);
	    a[0][2] = 0.0;
	    a[1][0] = Math.sin(theta);
	    a[1][1] = Math.cos(theta);
	    a[1][2] = 0.0;
	    a[2][0] = 0.0;
	    a[2][1] = 0.0;
	    a[2][2] = 1.0;
	}
	return (new Matrix(a,3,3));
    }

    /**
     * A method for generating an euler Matrix from three Euler angles.
     * The convention, refer to the Method of methemetical physics by CalTec guys
     @param alpha the alpha angle
     @param beta the beta angle
     @param gamma the gamma angle
     @return an Euler matrix
     * Wrote by Lincong
     */
    public static Matrix eulerMat(double alpha, double beta, double gamma) {
        double[][] mat = new double[3][3];
        mat[0][0] = Math.cos(alpha) * Math.cos(beta) * Math.cos(gamma) - Math.sin(alpha) * Math.sin(gamma);
        mat[0][1] = Math.sin(alpha) * Math.cos(beta) * Math.cos(gamma) + Math.cos(alpha) * Math.sin(gamma);
        mat[0][2] = -Math.sin(beta) * Math.cos(gamma);
        mat[1][0] = -Math.cos(alpha) * Math.cos(beta) * Math.sin(gamma) - Math.sin(alpha) * Math.cos(gamma);
        mat[1][1] = -Math.sin(alpha) * Math.cos(beta) * Math.sin(gamma) + Math.cos(alpha) * Math.cos(gamma);
        mat[1][2] = Math.sin(gamma) * Math.sin(beta);
        mat[2][0] = Math.sin(beta) * Math.cos(alpha);
        mat[2][1] = Math.sin(alpha) * Math.sin(beta);
        mat[2][2] = Math.cos(beta);
        return (new Matrix(mat));//though we have two sets but two Rotational matrix should be the same
    }

//    /* Method for solving the quadratic equation generated from the
//      * NOE distance as shown in the above.
//      @param coefs coefs generated by the above methods 
//      @param solutions for returning all the real solutions if there are
//      @return true 
//      */
//     public boolean quadraticSolve(double[] coefs, double[] solutions){
// 	double c = coefs[0]; //the equation a x^2 + b x + c = 0 
// 	double b = coefs[1];
// 	double a = coefs[2];
// 	if (b * b - 4 * a * c < 0.0)
// 	    return false;
// 	if( a != 0.0){
// 	    solutions[0] = 0.50*(-b + Math.sqrt(b *b - 4 *a *c )) / a;
// 	    solutions[1] = 0.50*(-b - Math.sqrt(b *b - 4 *a *c )) / a;
// 	    return true;
// 	}else if( b != 0.0){
// 	     solutions[0] = -c / b;
// 	     solutions[1] = -c / b;
// 	}else return false;
// 	return true;
//     }

//     /**
//      * Compute a rotation Matrix R between two given planes with each
//      * specified by two vectors: $ u2 = R u1, v2 = R v1 $  
//      @param u1 one vector specify plane 1
//      @param v1 the other vector specify plane 1
//      @param u2 one vector specify plane 2
//      @param v2 the other vector specify plane 2
//      @param mat return the computed matrix. 
//      @return true if such rotation matrix can be computed.
//      * Wrote by Lincong
//      */
//     public boolean rot(double [] u1, double [] v1, double [] u2, double [] v2, double [][] mat){
// 	double dz = u1[0] * v1[1] - u1[1] * v1[0];
// 	double dy = u1[0] * v1[2] - u1[2] * v1[0];
// 	double dx = u1[1] * v1[2] - u1[2] * v1[1];
// 	double e1 = 0.0, e0 = 0.0;
// 	double f1 = 0.0, f0 = 0.0;
// 	double [] coefs = new double[3];
// 	double [] solutions = new double[2];
// 	double x1 = 0.0, y1 = 0.0, z1 = 0.0;
// 	double x2 = 0.0, y2 = 0.0, z2 = 0.0;
// 	if ( Math.abs(dz) <= Const.eps && Math.abs(dy) <= Const.eps && Math.abs(dx) <= Const.eps)
// 	    return false;
// 	else if ( Math.abs(dz) > Const.eps ){
// 	    e1 = (v1[2] * u1[1] - u1[2] * v1[1]) / dz;
// 	    e0 = (v1[1] * u2[0] - u1[1] * v2[0]) / dz;
// 	    f1 = (v1[0] * u1[2] - u1[0] * v1[2]) / dz;
// 	    f0 = (v2[0] * u1[0] - u2[0] * v1[0]) / dz;
// 	    coefs[2] = e1*e1 + f1 * f1 +1.00;
// 	    coefs[1] = 2.0 * e1*e0 + 2.0 * f1 * f0;
// 	    coefs[0] = e0*e0 + f0 * f0 - 1.00;
// 	    if (quadraticSolve(coefs, solutions)){
// 		z1 = solutions[0];
// 		x1 = e1 * z1 + e0;
// 		y1 = f1 * z1 + f0;
// 		z2 = solutions[1];
// 		x2 = e1 * z2 + e0;
// 		y2 = f1 * z2 + f0;
// 		return true;
// 	    }else return false;
// 	}
//     }

   /** Clone the Matrix object.
   */
   public Object clone () {
      return this.copy();
   }

   /** Access the internal two-dimensional array.
   @return     Pointer to the two-dimensional array of matrix elements.
   */
   public double[][] getArray () {
      return A;
   }
 
   /** Copy the internal two-dimensional array.
   @return     Two-dimensional array copy of matrix elements.
   */

   public double[][] getArrayCopy () {
      double[][] C = new double[m][n];
      for (int i = 0; i < m; i++) {
         for (int j = 0; j < n; j++) {
            C[i][j] = A[i][j];
         }
      }
      return C;
   }


   /** Get row dimension.
   @return     m, the number of rows.
   */
   public int getRowDimension () {
      return m;
   }

   /** Get column dimension.
   @return     n, the number of columns.
   */
   public int getColumnDimension () {
      return n;
   }

   /** Get a single element.
   @param i    Row index.
   @param j    Column index.
   @return     A(i,j)
   @exception  ArrayIndexOutOfBoundsException
   */
   public double get (int i, int j) {
      return A[i][j];
   }

   /** Set a single element.
   @param i    Row index.
   @param j    Column index.
   @param s    A(i,j).
   @exception  ArrayIndexOutOfBoundsException
   */
   public void set (int i, int j, double s) {
      A[i][j] = s;
   }

   /** Matrix transpose.
   @return    A'
   */
   public Matrix transpose () {
      Matrix X = new Matrix(n,m);
      double[][] C = X.getArray();
      for (int i = 0; i < m; i++) {
         for (int j = 0; j < n; j++) {
            C[j][i] = A[i][j];
         }
      }
      return X;
   }

    /** -a of every elements
     */
   public Matrix uminus () {
      Matrix X = new Matrix(m,n);
      double[][] C = X.getArray();
      for (int i = 0; i < m; i++) {
         for (int j = 0; j < n; j++) {
            C[i][j] = -A[i][j];
         }
      }
      return X;
   }

   /** C = A + B
   @param B    another matrix
   @return     A + B
   */
   public Matrix plus (Matrix B) {
      checkMatrixDimensions(B);
      Matrix X = new Matrix(m,n);
      double[][] C = X.getArray();
      for (int i = 0; i < m; i++) {
         for (int j = 0; j < n; j++) {
            C[i][j] = A[i][j] + B.A[i][j];
         }
      }
      return X;
   }

   /** C = A - B
   @param B    another matrix
   @return     A - B
   */

   public Matrix minus (Matrix B) {
      checkMatrixDimensions(B);
      Matrix X = new Matrix(m,n);
      double[][] C = X.getArray();
      for (int i = 0; i < m; i++) {
         for (int j = 0; j < n; j++) {
            C[i][j] = A[i][j] - B.A[i][j];
         }
      }
      return X;
   }

   /** Multiply a matrix by a scalar, C = s*A
   @param s    scalar
   @return     s*A
   */
   public Matrix times (double s) {
      Matrix X = new Matrix(m,n);
      double[][] C = X.getArray();
      for (int i = 0; i < m; i++) {
         for (int j = 0; j < n; j++) {
            C[i][j] = s*A[i][j];
         }
      }
      return X;
   }

   /** Linear algebraic matrix multiplication, A * B
   @param B    another matrix
   @return     Matrix product, A * B
   @exception  IllegalArgumentException Matrix inner dimensions must agree.
   */
   public Matrix times (Matrix B) {
      if (B.m != n) {
         throw new IllegalArgumentException("Matrix inner dimensions must agree.");
      }
      Matrix X = new Matrix(m,B.n);
      double[][] C = X.getArray();
      double[] Bcolj = new double[n];
      for (int j = 0; j < B.n; j++) {
         for (int k = 0; k < n; k++) {
            Bcolj[k] = B.A[k][j];
         }
         for (int i = 0; i < m; i++) {
            double[] Arowi = A[i];
            double s = 0;
            for (int k = 0; k < n; k++) {
               s += Arowi[k]*Bcolj[k];
            }
            C[i][j] = s;
         }
      }
      return X;
   }

   public double[] times (double[] B) {
       int len = B.length;
       if ( len != n) {
	   throw new IllegalArgumentException("Matrix inner dimensions must agree.");
       }
       double[] vec = new double[m];
       for (int i = 0; i<m; i++)
	   vec[i] = 0.0;
       for (int j = 0; j < m; j++) {
	   for (int k = 0; k < n; k++) {
	       vec[j] += A[j][k] * B[k];
	   }
       }
       return vec;
   }

 /** One norm
   @return    maximum column sum.
   */

   public double norm1 () {
      double f = 0;
      for (int j = 0; j < n; j++) {
         double s = 0;
         for (int i = 0; i < m; i++) {
            s += Math.abs(A[i][j]);
         }
         f = Math.max(f,s);
      }
      return f;
   }

   public double trace () {
      double t = 0;
      for (int i = 0; i < Math.min(m,n); i++) {
         t += A[i][i];
      }
      return t;
   }
     /** Print the matrix to stdout.   Line the elements up in columns
     * with a Fortran-like 'Fw.d' style format.
   @param w    Column width.
   @param d    Number of digits after the decimal.
   */
    public void print (int w, int d) {
      print(new PrintWriter(System.out,true),w,d); }

   /** Print the matrix to the output stream.   Line the elements up in
     * columns with a Fortran-like 'Fw.d' style format.
   @param output Output stream.
   @param w      Column width.
   @param d      Number of digits after the decimal.
   */

   public void print (PrintWriter output, int w, int d) {
      DecimalFormat format = new DecimalFormat();
      format.setDecimalFormatSymbols(new DecimalFormatSymbols(Locale.US));
      format.setMinimumIntegerDigits(1);
      format.setMaximumFractionDigits(d);
      format.setMinimumFractionDigits(d);
      format.setGroupingUsed(false);
      print(output,format,w+2);
   }

    public void print (NumberFormat format, int width) {
	print(new PrintWriter(System.out,true),format,width); }

   // DecimalFormat is a little disappointing coming from Fortran or C's printf.
   // Since it doesn't pad on the left, the elements will come out different
   // widths.  Consequently, we'll pass the desired column width in as an
   // argument and do the extra padding ourselves.

   /** Print the matrix to the output stream.  Line the elements up in columns.
     * Use the format object, and right justify within columns of width
     * characters.
     * Note that is the matrix is to be read back in, you probably will want
     * to use a NumberFormat that is set to US Locale.
   @param output the output stream.
   @param format A formatting object to format the matrix elements 
   @param width  Column width.
   @see java.text.DecimalFormat#setDecimalFormatSymbols
   */

   public void print (PrintWriter output, NumberFormat format, int width) {
      output.println();  // start on new line.
      for (int i = 0; i < m; i++) {
         for (int j = 0; j < n; j++) {
            String s = format.format(A[i][j]); // format the number
            int padding = Math.max(1,width-s.length()); // At _least_ 1 space
            for (int k = 0; k < padding; k++)
               output.print(' ');
            output.print(s);
         }
         output.println();
      }
      output.println();   // end with blank line.
   }
    /** Singular Value Decomposition
   @return     SingularValueDecomposition
   @see SingularValueDecomposition
   */
    public SingularValueDecomposition svd () {
      return new SingularValueDecomposition(this);
   }

   /** Generate identity matrix
   @param m    Number of rows.
   @param n    Number of colums.
   @return     An m-by-n matrix with ones on the diagonal and zeros elsewhere.
   */
   public static Matrix identity (int m, int n) {
      Matrix A = new Matrix(m,n);
      double[][] X = A.getArray();
      for (int i = 0; i < m; i++) {
         for (int j = 0; j < n; j++) {
            X[i][j] = (i == j ? 1.0 : 0.0);
         }
      }
      return A;
   }

 /** Generate identity matrix
   @param m    Number of rows.
   @param n    Number of colums.
   @return     An m-by-n matrix with ones on the diagonal and zeros elsewhere.
   */
   public static double det (double[][] a) {
       double det = a[0][0] * a[1][1] * a[2][2] + a[0][1] * a[1][2] * a[2][0] + a[0][2] * a[1][0] * a[2][1]
	   -a[0][2] * a[1][1] * a[2][0] - a[0][0] * a[1][2] * a[2][1]  -a[0][1] * a[1][0] * a[2][2];
       return det;
   }

/* ------------------------
   Private Methods
 * ------------------------ */

   /** Check if size(A) == size(B) **/

   private void checkMatrixDimensions (Matrix B) {
      if (B.m != m || B.n != n) {
         throw new IllegalArgumentException("Matrix dimensions must agree.");
      }
   }
}
