package rdcPanda;

///////////////////////////////////////////////////////////////////////////////////////////////
//	PhiPsi.java
//
//	  Version:           0.1
//
//
//	  authors:
// 	  initials            name                      organization               email
//	 ---------   -----------------------        ------------------------    ------------------
//	  LW            Lincong Wang                  Dartmouth College       wlincong@cs.dartmouth.edu
//
///////////////////////////////////////////////////////////////////////////////////////////////


/*
	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.
	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	Lesser General Public License for more details.
	
	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
	USA
	
	Contact Info:
		Bruce Donald
		Duke University
		Department of Computer Science
		Levine Science Research Center (LSRC)
		Durham
		NC 27708-0129 
		USA
		brd@cs.duke.edu
	
	If you use or publish any results derived from the use of this program please cite:
	J. Zeng, J. Boyles, C. Tripathy, L. Wang, A. Yan, P. Zhou and B.R. Donald. 
	"High-Resolution Protein Structure Determination Starting with a Global Fold 
	Calculated from Exact Solutions to the RDC Equations." Submitted For Review.

	Copyright (C) 2009 Jianyang (Michael) Zeng, Lincong Wang and Bruce R. Donald		
	<signature of Bruce Donald>, June 2008 and January 2009
	Bruce Donald, Professor of Computer Science
 */


import java.io. *;
import java.util. *;

// TODO: Auto-generated Javadoc
/** * 
 * 
   Various methods for exact solution and systematic search
 * including the computation of rotation to the frist peptide plane
 * the DFS-search and solver of the quartic equation and the computation 
 * of \phi and \psi angles from two vectors in consecutive planse etc.
 * 
 *  Written by Lincong Wang (2001-2005).
*/

public class PhiPsi implements Cloneable
{
    
    /** The residue no. */
    private int residueNo;  //Though Phi and Psi are between two residues 
    
    /** The residue. */
    private String residue; //We count it as belonging to the first of the two
    
    /** The phi. */
    private double phi;
    
    /** The psi. */
    private double psi;
    
    /** The rama_score. */
    private double rama_score=0.0; 
    
    /** The Constant dirCosCH. */
    static final double[] dirCosCH = Const.dirCosCHcnt; //for convenience
    
    /** The Constant dirCosNH. */
    static final double[] dirCosNH = Const.dirCosNHcnt;

    /** The phi lower. */
    private double phiLower=0.0;
    
    /** The phi upper. */
    private  double phiUpper=0.0;
    
    /** The psi lower. */
    private double psiLower=0.0;
    
    /** The psi upper. */
    private  double psiUpper=0.0;
    
    /**
     * Instantiates a new phi psi.
     * 
     * @param phiT the phi t
     * @param psiT the psi t
     * @param r_score the r_score
     */
    public PhiPsi(double phiT, double psiT, double r_score )
    {
    	residueNo = 0;
    	residue = null;
        phi = phiT;
        psi = psiT;
        rama_score=r_score;
        phiLower=0.0;
        phiUpper=0.0;
        psiLower=0.0;
        psiUpper=0.0;
    }	
    
    /**
     * Instantiates a new phi psi.
     */
    public PhiPsi(){
	residueNo = 0;
	residue = null;
        phi = 0.0;
        psi = 0.0;
        phiLower=0.0;
        phiUpper=0.0;
        psiLower=0.0;
        psiUpper=0.0;
    }	
    
    /**
     * Instantiates a new phi psi.
     * 
     * @param No the no
     */
    public PhiPsi(int No){
	residueNo = No;
	residue = ""; 
	phi = 0.0;
        psi = 0.0;
        phiLower=0.0;
        phiUpper=0.0;
        psiLower=0.0;
        psiUpper=0.0;
    }
    
    /**
     * Instantiates a new phi psi.
     * 
     * @param no the no
     * @param re the re
     * @param x the x
     * @param y the y
     */
    public PhiPsi(int no, String re, double x, double y){
    	residueNo = no;
    	residue = re;
        phi = x;
        psi = y;
        phiLower=0.0;
        phiUpper=0.0;
        psiLower=0.0;
        psiUpper=0.0;
    }
    
    /**
     * Instantiates a new phi psi.
     * 
     * @param no the no
     * @param re the re
     * @param x the x
     * @param y the y
     * @param phi_lo the phi_lo
     * @param phi_up the phi_up
     * @param psi_lo the psi_lo
     * @param psi_up the psi_up
     */
    public PhiPsi(int no, String re, double x, double y, 
    		double phi_lo, double phi_up, double psi_lo, double psi_up)
    {
    	residueNo = no;
    	residue = re;
        phi = x;
        psi = y;
        phiLower=phi_lo;
        phiUpper=phi_up;
        psiLower=psi_lo;
        psiUpper=psi_up;
    }
    
    //getting the values	
    /**
     * Gets the residue no.
     * 
     * @return the residue no
     */
    public int getResidueNo(){
	return residueNo;
    }	
    
    /**
     * Gets the residue.
     * 
     * @return the residue
     */
    public String getResidue(){
	return residue;
    }	
    
    /**
     * Gets the phi.
     * 
     * @return the phi
     */
    public double getPhi(){
	return phi;
    }
    
    /**
     * Sets the phi.
     * 
     * @param f1 the new phi
     */
    public void setPhi(double f1){
    	 phi=f1;
        }
    
    /**
     * Gets the psi.
     * 
     * @return the psi
     */
    public double getPsi(){
	return psi;
    }
    
    /**
     * Sets the psi.
     * 
     * @param f1 the new psi
     */
    public void setPsi(double f1){
   	 psi=f1;
       }
    
    /**
     * Gets the rama score.
     * 
     * @return the rama score
     */
    public double getRamaScore(){
    	return rama_score;
        }
    
    /**
     * Gets the phi lower.
     * 
     * @return the phi lower
     */
    public double getPhiLower(){
    	return phiLower;
        }
    
    /**
     * Gets the phi upper.
     * 
     * @return the phi upper
     */
    public double getPhiUpper(){
    	return phiUpper;
        }
    
    /**
     * Gets the psi lower.
     * 
     * @return the psi lower
     */
    public double getPsiLower(){
    	return psiLower;
        }
    
    /**
     * Gets the psi upper.
     * 
     * @return the psi upper
     */
    public double getPsiUpper(){
    	return psiUpper;
        }
    
    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    public String toString(){
  	String desc = String.valueOf(residueNo) + "  " + residue
  	    +"  " + String.valueOf(phi*180/Math.PI) + "  " 
	    + String.valueOf(psi*180.0/Math.PI);
	return desc;	
    }
    
    /**
     * The Class PPComparator.
     */
    public static class PPComparator implements Comparator{
	
	/* (non-Javadoc)
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(Object o1, Object o2){
	    PhiPsi n1 = (PhiPsi)o1;
	    PhiPsi n2 = (PhiPsi)o2;
	    int d1 = n1.getResidueNo();
	    int d2 = n2.getResidueNo();
	    if (d1 < d2)
		return -1;
	    else if (d1 > d2)
		return 1;
	    else return 0;
	}
    }
   
   /**
    * Prints the array.
    * 
    * @param n1 the n1
    */
   public void printArray(double[] n1){
        for (int i=0; i<n1.length; i++)
            System.out.print(n1[i]+"  ");
        System.out.println();
    }

    /**
     * calculate the length between two vectors v1 and v2
     * The returned angle is the [0,Pi] region.
     * 
     * @param v1 the v1
     * 
     * @return the double
     */
    public double length(double[] v1){
	double v1Len = Math.sqrt(v1[0]*v1[0] + v1[1]*v1[1] + v1[2]*v1[2]);
	return v1Len;
    }
    
    /**
     * calculate the angles between two vectors v1 and v2
     * The returned angle is in the [0,Pi] range.
     * 
     * @param v1 the v1
     * @param v2 the v2
     * 
     * @return the double
     */
    public double interAngle(double[] v1, double[] v2){
	double v1Len = Math.sqrt(v1[0]*v1[0] + v1[1]*v1[1] + v1[2]*v1[2]);
	double v2Len = Math.sqrt(v2[0]*v2[0] + v2[1]*v2[1] + v2[2]*v2[2]);
	double c = (v1[0]*v2[0]+v1[1]*v2[1]+v1[2]*v2[2]) / (v1Len * v2Len);

	if ( c > 0 && Math.abs(c - 1.00) < Const.eps)
	    return 0.0;
	if ( c < 0 && Math.abs(c + 1.00) < Const.eps)
	    return Math.PI;
	return Math.acos(c);  // 	return acos [0, PI] so should be OK
    }

    /**
     * cal the directioal cosines of the vector vec.
     * 
     * @param vec the vec
     * 
     * @return the double[]
     */
    public double[] dirCos(double [] vec) {
	double len = 0.0;
	double [] dirs = new double[vec.length];
	for (int i=0; i<vec.length; i++)
	    len += vec[i] * vec[i];
	for (int j=0; j<vec.length; j++)
	    dirs[j] = vec[j] / Math.sqrt(len);
	return dirs;
    }

   /**
    * Compute the internuclear vector between two atoms.
    * 
    * @param n1 the coordinate for atom 1
    * @param n2 the coordinate for atom 2
    * 
    * @return a vector from n1->n2
    */
    public double[] internuclearVec(double[] n1, double[] n2){
        return new double[]{n2[0]-n1[0], n2[1]-n1[1], n2[2]-n1[2]};
    }

 /**
  * Compute the internuclear vector between two atoms.
  * 
  * @param n1 the coordinate for atom 1
  * @param n2 the coordinate for atom 2
  * 
  * @return a vector from n1->n2
  */
    public double internuclearDistance (double[] n1, double[] n2){
	return (Math.sqrt((n2[0]-n1[0])*(n2[0]-n1[0])+(n2[1]-n1[1])*(n2[1]-n1[1])+(n2[2]-n1[2])*(n2[2]-n1[2])));
    }

    /**
     * adding two vectors.
     * 
     * @param n1 the coordinate (direction cosine) for vector n1
     * @param n2 the coordinate (direction cosine) for vector n2
     * 
     * @return a vector from n1+n2
     */
    public double[] addCoords(double[] n1, double[] n2){
        return new double[]{n2[0]+n1[0], n2[1]+n1[1], n2[2]+n1[2]};
    }

    /**
     * Compute the global rotation matrix for rotating a coordinate
     * frame such as the PDB frame where the two vectors NH(i)->N(i) (nToh)
     * and N(i)->CA(i) (nToca) defined into the following frame:
     * +Z-axis along the bond NH(i)->N(i), "->" means from nuclear NH(i) to nuclear N(i).
     * +Y-axis in the peptide plane i,and the angle between +Y and the N(i)->CA(i) < 90 degree.
     * +X-axis is defined based on the right-handedness.
     * 
     * @param nToh  the direction cosine of N->NH vector
     * @param nToca the direction cosine of N->CA vector
     * 
     * @return All the rotation Matrix
     */
    public Matrix RgCal(final double[] nToh, final double[] nToca){
        double[] n2h  = dirCos(nToh); //cal directional cosines
        double[] n2ca = dirCos(nToca);
        double theta = Math.PI - interAngle(nToh, nToca);
        double xNH   = n2h[0],   yNH = n2h[1],   zNH = n2h[2];
        double xNCA  = n2ca[0], yNCA = n2ca[1], zNCA = n2ca[2];
        double[][] mat = new double[3][3];
        Matrix A = new Matrix(3,3);

        double sinTheta = 9.99;
        double alpha1 = 9.99, beta1 = 9.99, gamma1 = 9.99;
        if ( Math.abs(zNH - 1.0) < Const.eps) { //when the NH vector is on +Z direction
            if ( Math.abs(zNCA - 1.0) > Const.eps && Math.abs(zNCA + 1.0) > Const.eps ){
                sinTheta  = Math.sqrt(1.0 - zNCA * zNCA); //under such condition cosTheta = zNCA
                mat[0][0] = yNCA / sinTheta;
                mat[0][1] = xNCA / sinTheta;
                mat[0][2] = 0.0;
                mat[1][0] = -xNCA / sinTheta;
                mat[1][1] = yNCA / sinTheta;
                mat[1][2] = 0.0;
                mat[2][0] = 0.0;
                mat[2][1] = 0.0;
                mat[2][2] = 1.0;
                return (new Matrix(mat)).transpose(); //return directly the transposed matrix
            } else { //impossible for normal PDB file
                System.out.println("zNCA is 1 or -1");
                System.exit(0);
            }
        }else if ( Math.abs(zNH + 1.0) < Const.eps) { //when the NH vector is on +Z direction
        	 if ( Math.abs(zNCA - 1.0) > Const.eps && Math.abs(zNCA + 1.0) > Const.eps ){
                 sinTheta  = Math.sqrt(1.0 - zNCA * zNCA); //under such condition cosTheta = zNCA
           
        	mat[0][0] = yNCA / sinTheta;
            mat[0][1] = xNCA  / sinTheta;
            mat[0][2] = 0.0;
            mat[1][0] = xNCA / sinTheta;
            mat[1][1] = yNCA / sinTheta;
            mat[1][2] = 0.0;
            mat[2][0] = 0.0;
            mat[2][1] = 0.0;
            mat[2][2] = 1.0;
            return (new Matrix(mat)).transpose();
        	  } else { //impossible for normal PDB file
                  System.out.println("zNCA is 1 or -1");
                  System.exit(0);
              }
        }else {
            beta1 = Math.acos(zNH);
            double sinAlpha = yNH / Math.sin(beta1);
            double cosAlpha = xNH / Math.sin(beta1);
            if ( cosAlpha >= 0 )
                alpha1 = Math.asin(sinAlpha);
            else alpha1 = Math.PI-Math.asin(sinAlpha);
            double sinGamma = (zNCA+Math.cos(beta1)*Math.cos(theta))/(Math.sin(theta)*Math.sin(beta1));
            double cosGamma = (yNCA*Math.cos(alpha1)-xNCA*Math.sin(alpha1))/Math.sin(theta);
            if (cosGamma >= 0 )
                gamma1 = Math.asin(sinGamma);
            else gamma1 = Math.PI-Math.asin(sinGamma);
        }
        //make an Euler rotation matrix from three Euler angles
        Matrix mm = A.eulerMat(alpha1, beta1, gamma1);
        //invert the +z axis: point along the NH->N direction for the sake of easy math
        Matrix MM = A.rotationMat(Math.PI, "+y");
        return (MM.times(mm));
    }

     /* compute global rotation (Rg) with two noncolinear vectors, Rg
      * is the rotation matrix for rotating the PDB coordinate system
      * into the following system:
      * +z-axis along the bond NH(i)->N(i), "->" means from atom N(i) to atom NH(i). 
      * +y-axis in the peptide plane i,and the angle  between +y and the N(i)->CA(i) < 90 degree.  
      * +x-axis is defined based on the right-handedness.
      @param nToh the directional cosine of N->NH vector
      @param nToca the directional cosine of N->CA vector
      @param  rightHand to compare with the rotation matrix from the SVD method. 
      @return All the rotational Matrix, not just one
     */
    /**
      * Rg cal.
      * 
      * @param nToh the n toh
      * @param nToca the n toca
      * @param rightHand the right hand
      * 
      * @return the matrix
      */
     public Matrix RgCal(double[] nToh, double[] nToca, boolean rightHand){
	double[] n2h = dirCos(nToh); //cal directional cosines
	double[] n2ca = dirCos(nToca);
	double theta = Math.PI - interAngle(nToh, nToca);
	double xNH = n2h[0], yNH = n2h[1], zNH = n2h[2];
	double xNCA = n2ca[0], yNCA = n2ca[1], zNCA = n2ca[2];
	double[][] mat = new double[3][3];
	double sinTheta = 9.99;
	Matrix A = new Matrix(3,3);
	double alpha1=9.99,  beta1=9.99,  gamma1=9.99;
        if ( Math.abs(zNH - 1.0) < Const.eps) { //when the NH vector is on +Z direction
            if ( Math.abs(zNCA - 1.0) > Const.eps && Math.abs(zNCA + 1.0) > Const.eps ){
		sinTheta = Math.sqrt(1.0-zNCA*zNCA); //under such condition cosTheta = zNCA
		mat[0][0] = yNCA / sinTheta;
		mat[0][1] = xNCA / sinTheta;
		mat[0][2] = 0.0;
		mat[1][0] = -xNCA / sinTheta;
		mat[1][1] = yNCA / sinTheta;
		mat[1][2] = 0.0;
		mat[2][0] = 0.0;
		mat[2][1] = 0.0;
		mat[2][2] = 1.0;
		return (new Matrix(mat)).transpose();
	    } else { //impossible for normal PDB file
		System.out.println("zNCA is 1 or -1");
		System.exit(0);
	    }
	}else if ( Math.abs(zNH + 1.0) < Const.eps) { //when the NH vector is on +Z direction
	    mat[0][0] = -yNCA / sinTheta;
	    mat[0][1] = xNCA / sinTheta;
	    mat[0][2] = 0.0;
	    mat[1][0] = xNCA / sinTheta;
	    mat[1][1] = yNCA / sinTheta;
	    mat[1][2] = 0.0;
	    mat[2][0] = 0.0;
	    mat[2][1] = 0.0;
	    mat[2][2] = 1.0;
	    return (new Matrix(mat)).transpose(); //return directly need transpose
	}else{
	    beta1 = Math.acos(zNH);  //Should be minus
	    double sinAlpha = yNH / Math.sin(beta1);
	    double cosAlpha = xNH / Math.sin(beta1);
	    if (( cosAlpha > 0) || (cosAlpha == 0) )
		alpha1 = Math.asin(sinAlpha);
	    else if ( cosAlpha < 0)
		alpha1 = Math.PI - Math.asin(sinAlpha);
	    double sinGamma = (zNCA + Math.cos(beta1) * Math.cos(theta)) / (Math.sin(theta)*Math.sin(beta1));
	    double cosGamma = (yNCA * Math.cos(alpha1) - xNCA*Math.sin(alpha1)) / Math.sin(theta);
	    if (( cosGamma > 0) || (cosGamma == 0) )
		gamma1 = Math.asin(sinGamma);
	    else if ( cosGamma < 0 )
		gamma1 = Math.PI - Math.asin(sinGamma);
	}//make a euler rotation matrix from three Euler angles
	Matrix MM = A.rotationMat(Math.PI, "+y");
 	Matrix mm = A.eulerMat(alpha1, beta1, gamma1);
	if (rightHand)
	    return MM.times(mm);
	return (Const.mLeftHand.times(MM.times(mm)));
    }

     /**
      * Method for solving the quadratic equation generated from the
      * NOE distance as shown in the above.
      * 
      * @param coefs coefs generated by the above methods
      * @param solutions for returning all the real solutions if there are
      * 
      * @return true
      */
    public boolean quadraticSolve(double[] coefs, double[] solutions){
	double c = coefs[0]; //the equation a x^2 + b x + c = 0 
	double b = coefs[1];
	double a = coefs[2];
	if (b * b - 4 * a * c < 0.0)
	    return false;
	if( a != 0.0){
	    solutions[0] = 0.50*(-b + Math.sqrt(b *b - 4 *a *c )) / a;
	    solutions[1] = 0.50*(-b - Math.sqrt(b *b - 4 *a *c )) / a;
	    return true;
	}else if( b != 0.0){
	     solutions[0] = -c / b;
	     solutions[1] = -c / b;
	}else return false;
	return true;
    }

    /**
     * Compute the Phi angles from the NOE distances, given the coordinates for ca and nh.
     * 
     * @param d  the intra noe distance bewteen NH(i) and HA(i) of residue i+1
     * 
     * @return a set of solutions for \phi angles: if the set empty ==> no solution
     */
    public Vector phiCalNoe(double d){
	double e0 = Const.e0 - d * d;
	double[] coefs = {Const.e1 + e0, 2 * Const.e2, e0 - Const.e1};
	double [] solutions = new double[2];
	boolean hasSln = quadraticSolve(coefs, solutions); //solve the quartic equation
	Vector phiVec = new Vector();
	if (hasSln){
	    double t = 0.0;
	    double sinPhi = 0.0;
	    double cosPhi = 0.0;
	    double phi = 0.0;
	    for (int i = 0; i < solutions.length; i++){
		t = solutions[i];
		sinPhi = 2 * t / (1 + t*t);
		cosPhi = (1- t*t) / (1 + t*t);
		if ( cosPhi >= 0 )
		    phi = Math.asin(sinPhi);
		else phi = Math.PI - Math.asin(sinPhi);
		while (phi > Math.PI)
		    phi -= 2.0*Math.PI;
		phiVec.add(new Double(phi));
	    }
	}
	return phiVec;
    }

    /**
     * Compute the Phi angles from the NOE distances, given the coordinates for ca and nh.
     * 
     * @param d  the intra noe distance bewteen NH(i) and HA(i) of residue i+1
     * 
     * @return a set of solutions for \psi angles: if the set empty ==> no solution
     */
    public Vector psiCalNoe(double d){
	double e0 = Const.ee0 - d * d;
	double[] coefs = {Const.ee1 + e0, 2 * Const.ee2,  e0 - Const.ee1};
	double[] solutions = new double[2];
	boolean hasSln = quadraticSolve(coefs, solutions); //solve the quartic equation
	Vector psiVec = new Vector();
	if (hasSln){
	    double t = 0.0;
	    double sinPhi = 0.0;
	    double cosPhi = 0.0;
	    double psi = 0.0;
	    for (int i = 0; i < solutions.length; i++){
		t = solutions[i];
		sinPhi = 2 * t / (1 + t*t);
		cosPhi = (1- t*t) / (1 + t*t);
		if ( cosPhi >= 0 )
		    psi = Math.asin(sinPhi);
		else psi = Math.PI - Math.asin(sinPhi);
		while (psi > Math.PI)
		    psi -= 2.0*Math.PI;
		psiVec.add(new Double(psi));
	    }
	}
	return psiVec;
    }

    /**
     * Testing the methods for computing the \phi and \psi angle from NOE disances.
     */
    public void testNOEs(){
	int i = 0, j=0, k=0;
    	double [] n1 = {0.0, 0.0, 0.0};         //A coordinate frame in the 1st peptide plane
	double [] nh1 = {0.0, 0.0, -Const.dN2H}; 
    	double [] ca1 = {0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA*Math.sin(Const.theta1)};
	Vector phiPsiVec = new Vector();
	int resolution = 2;
	double alpha, beta, gamma;
	Matrix rg = Matrix.identity(3, 3); //RgCal(nh1, ca1, true); //
// 	for (i=0; i<10; i++)
//             phiPsiVec.add(new PhiPsi(i, "ALA", phiAve, psiAve));
//         Vector pdbVec = modelBuild(phiPsiVec, n1, nh1, ca1);
// 	rg.print(10,10);
	double d =  2.5328777490734073;//2.773382777221239;//2.5328777490734073;//2.3208091451615305; 2.7899133800509675; //
	Vector phiVec = phiCalNoe(d);
	System.out.println( "# Solutions for Phi: "+phiVec.size());
	if (phiVec.isEmpty())
	    System.out.println("No Solutions for Phi");
	else {
	    for (i=0; i<phiVec.size(); i++)
		System.out.println( ((Double)phiVec.elementAt(i)).doubleValue()+"  "
				    +((Double)phiVec.elementAt(i)).doubleValue() / Const.cst);
	}

// 	Matrix rgInv = rg.transpose();
 	double phi = 0.0; 
// 	Matrix r2yInv = rg.rotationMat(phi, "-y"); 
// 	Matrix matT = rgInv.times(Const.r1x9yInv.times(r2yInv));
// 	//Compute the HA coordinate residue (i)
// 	double [] coordHA = new double[]{0.0, 0.0, Const.dCA2HA};
// 	double [] cntVec = Const.rHA2HA1Inv.times(coordHA);
// 	double[] caToHAVec = matT.times(cntVec);
// 	coordHA = addCoords(caToHAVec, ca1);
// 	//Compute the CO coordinate residue (i)
// 	Matrix matCnt  = matT.times(Const.r3xInv);
// 	double [] coordCO   =  new double[]{0.0, 0.0, Const.dCA2CO};
// 	double [] caToCOVec = matCnt.times(coordCO);
// 	coordCO = addCoords(caToCOVec,  ca1); 

	d =  3.2679550115635974;
	Vector  psiVec = psiCalNoe(d);
	System.out.println( "# Solutions for Psi: "+psiVec.size());
	if (psiVec.isEmpty())
	    System.out.println("No Solutions for Psi");
	else {
	    for (i=0; i<psiVec.size(); i++)
		System.out.println( ((Double)psiVec.elementAt(i)).doubleValue()+"  "
				    +((Double)psiVec.elementAt(i)).doubleValue() / Const.cst);
	}
    }

    /**
     * Noe dis vs phi.
     */
    public void noeDisVsPhi(){
	Matrix r2yInv = new Matrix(3, 3);
	Matrix matT =   new Matrix(3, 3);
	double [] n1 = {0.0, 0.0, 0.0};         //A coordinate frame in the 1st peptide plane
	double [] nh1 = {0.0, 0.0, -Const.dN2H}; 
    	double [] ca1 = {0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA * Math.sin(Const.theta1)};
 	Matrix rg = Matrix.identity(3,3);
	Matrix rgInv = rg.transpose();
	double [] coordHA = new double[]{0.0, 0.0, Const.dCA2HA};
	double [] cntVec = Const.rHA2HA1Inv.times(coordHA);
	double phi = 0.0; 
	double psi = 0.0;
	double[] caToHAVec = new double[3];
	double [] interNucVec = new double[3];
	double noeDis = 0.0;
	for (int i=0; i<360; i++){
	    phi = i * Math.PI / 180 - Math.PI;
	    r2yInv = rg.rotationMat(phi, "-y"); 
	    matT = rgInv.times(Const.r1x9yInv.times(r2yInv));
	    caToHAVec =  matT.times(cntVec);
	    coordHA = addCoords(caToHAVec, ca1);
	    interNucVec = internuclearVec(nh1, coordHA);
	    noeDis = length(interNucVec);
	    System.out.println(phi+"  "+noeDis);
	}
    }

    /**
     * A method for computing the distance dN(i, i+1), between the NH(i) and NH(i+1)
     * begin with an frame defined in the peptide plane i.
     * 
     * @param phi     The backbone \phi angle
     * 
     * @return the distance dna between HN(i) and HA(i)
     */    
    public double dnaIntra (double phi){
	Matrix mat = Matrix.identity(3,3);
	double [] ca = {0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA * Math.sin(Const.theta1)};
  	Matrix r2yInv = mat.rotationMat(phi, "-y"); //transpose == inverse == rotate in opposite direction
	double [] caToHAVec =  Const.r1x9yInv.times(r2yInv.times(Const.cntHAVec));
	double[] coordHA = addCoords(caToHAVec, ca);
	double[] interNucVec = new double[]{coordHA[0], coordHA[1], coordHA[2] + Const.dN2H};
	return (length(interNucVec));
    }

    /**
     * A method for computing the distance dN(i, i+1), between the NH(i) and NH(i+1)
     * begin with an frame defined in the peptide plane i.
     * 
     * @param psi     The backbone \psi angle
     * @param ha      the coordinate of HA (i)
     * @param co      the coordinate of CO (i)
     * 
     * @return the distance dan between HA(i) and HN(i+1)
     */    
    public double danSeq (double[] ha, double [] co, double psi){
	//Compute the N coordinate residue (i+1), assuming phi = 0
	Matrix mat = new Matrix(3, 3);
	Matrix r4zInv = mat.rotationMat(psi+Math.PI, "-z"); 
	mat = Const.r1x9y3xInv.times(r4zInv.times(Const.r5xInv));
	double [] coordN  = new double[]{0.0, Const.dCO2N, 0.0};
	double [] co1ToNVec = mat.times(coordN);
	coordN = addCoords(co, co1ToNVec);

	//Compute the NH coordinate residue (i+1)
	double [] nh = {0.0, 0.0, -Const.dN2H}; 
	mat = mat.times(Const.r7x6yInv);
	double [] nToNHVec2 = mat.times(nh);
	nh  = addCoords(coordN, nToNHVec2);

	double[] interNucVec =  new double[]{-ha[0], -ha[1],  -Const.dN2H - ha[2]}; //HA --> HN
	return (length(interNucVec));
    }

    /**
     * A method for computing the distance dN(i, i+1), between the NH(i) and NH(i+1)
     * begin with an frame defined in the peptide plane i.
     * 
     * @param phi     The backbone \phi angle
     * @param psi     The backbone \psi angle
     * 
     * @return Pdb object consisting of atoms: NH, CA, CO, HA and O of residue (i).
     * Please note that the coordinates of N, NH and CA of residue (i+1) are returned by
     * overwriting the original array for them: This may be BUG prone.
     */    
    public double dNNseq (double phi, double psi){
	double [] nh1= {0.0, 0.0, -Const.dN2H}; 
    	double [] ca1= {0.0, Const.dN2CA*Math.cos(Const.theta1), Const.dN2CA*Math.sin(Const.theta1)};
	Matrix rg = Matrix.identity(3,3);
  	Matrix r2yInv = rg.rotationMat(phi, "-y"); //transpose == inverse == rotate in opposite direction
	Matrix r4zInv = rg.rotationMat(psi+Math.PI, "-z"); 
 	Matrix matT = Const.r1x9yInv.times(r2yInv.times(Const.r3xInv));
	//Compute the CO coordinate of residue (i)
	double [] coordCO   =  new double[]{0.0, 0.0, Const.dCA2CO};
	double [] caToCOVec = matT.times(coordCO);
	coordCO = addCoords(caToCOVec,  ca1); 
	//Compute the N coordinate of  residue (i+1)
	double [] coordN  = new double[]{0.0, Const.dCO2N, 0.0};
	matT = matT.times(r4zInv.times(Const.r5xInv));
	double [] co1ToNVec = matT.times(coordN);
	coordN = addCoords(coordCO, co1ToNVec);
	//Compute the NH coordinate of residue (i+1)
	double [] coordNH = new double[]{0.0, 0.0,  -Const.dN2H};
	matT = matT.times(Const.r7x6yInv);
	double [] nToNHVec2 = matT.times(coordNH);
	coordNH = addCoords(coordN, nToNHVec2);
	double dnn = length(internuclearVec(nh1, coordNH));
	return dnn;
    }

    /**
     * Noe dis vs psi.
     */
    public void noeDisVsPsi(){
	double [] n1 = {0.0, 0.0, 0.0};         //A coordinate frame in the 1st peptide plane
	double [] nh1 = {0.0, 0.0, -Const.dN2H}; 
    	double [] ca1 = {0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA * Math.sin(Const.theta1)};
 	Matrix rg = Matrix.identity(3,3);
	Matrix rgInv = rg.transpose();
	double phi = 0.0; 
	Matrix r2yInv = rg.rotationMat(phi, "-y"); 
	Matrix matT = rgInv.times(Const.r1x9yInv.times(r2yInv));
	//Compute the HA coordinate residue (i)
	double [] coordHA = new double[]{0.0, 0.0, Const.dCA2HA};
	double [] cntVec = Const.rHA2HA1Inv.times(coordHA);
	double[] caToHAVec = matT.times(cntVec);
	coordHA = addCoords(caToHAVec, ca1);
	//Compute the CO coordinate residue (i)
	Matrix matCnt  = matT.times(Const.r3xInv);
	double [] coordCO   =  new double[]{0.0, 0.0, Const.dCA2CO};
	double [] caToCOVec = matCnt.times(coordCO);
	coordCO = addCoords(caToCOVec,  ca1); 

	double psi = 0.0;
	Matrix r4zInv =  new Matrix(3, 3);
	double [] interNucVec = new double[3];
	double noeDis = 0.0;
	double [] coordN  = new double[]{0.0, Const.dCO2N, 0.0};
	double [] coordCA = new double[]{0.0, Const.dN2CA, 0.0};
	double [] coordN2  = new double[]{0.0, Const.dCO2N, 0.0};
	double [] coordCA2 = new double[]{0.0, Const.dN2CA, 0.0};
	double [] co1ToNVec = new double[3];
	double [] nToNHVec2 = new double[3];
	double [] nToCAVec = new double[3];
	double [] coordNH  = new double[3];
	for (int i=0; i<720; i++){
	    psi = i * Math.PI / 360 - Math.PI;
	    //Compute the N coordinate residue (i+1)
	    r4zInv = rg.rotationMat(psi+Math.PI, "-z"); 
	    matT = matCnt.times(r4zInv.times(Const.r5xInv));
	    co1ToNVec = matT.times(coordN);
	    coordN2 = addCoords(coordCO, co1ToNVec);

	    //Compute the CA coordinate residue (i+1)
	    matT = matT.times(Const.r6yInv.times(Const.r7xInv));
	    //   	System.out.println(" CA(i+1) -> "+coordCA[0]+"  "+coordCA[1]+"  "+coordCA[2]);
	    //Compute the NH coordinate residue (i+1)
	    nToNHVec2 = matT.times(nh1);
	    coordNH = addCoords(coordN2, nToNHVec2);

	    interNucVec = internuclearVec(coordHA, coordNH);
	    noeDis = length(interNucVec);
	    System.out.println(psi+"  "+noeDis);
	}
    }

  /**
   * A method for computing the distance Ca(i, i+1), between the
   * CA(i) and CA(i+1) begin with an frame defined in the peptide
   * plane i. In FACT, with standard bond angles and lengths, dCaCa is
   * a constant.
   * 
   * @param phi     The backbone \phi angle
   * @param psi     The backbone \psi angle
   * 
   * @return the distance dCaCa.
   */    
    public double dCaCa(double phi, double psi){
	double [] nh1= {0.0, 0.0, -Const.dN2H}; 
    	double [] ca1= {0.0, Const.dN2CA*Math.cos(Const.theta1), Const.dN2CA*Math.sin(Const.theta1)};
	Matrix rg = Matrix.identity(3,3);
  	Matrix r2yInv = rg.rotationMat(phi, "-y"); //transpose == inverse == rotate in opposite direction
	Matrix r4zInv = rg.rotationMat(psi+Math.PI, "-z"); 
 	Matrix matT = Const.r1x9yInv.times(r2yInv.times(Const.r3xInv));
	//Compute the CO coordinate of residue (i)
	double [] coordCO   =  new double[]{0.0, 0.0, Const.dCA2CO};
	double [] caToCOVec = matT.times(coordCO);
	coordCO = addCoords(caToCOVec,  ca1); 
	//Compute the N coordinate of  residue (i+1)
	double [] coordN  = new double[]{0.0, Const.dCO2N, 0.0};
	matT = matT.times(r4zInv.times(Const.r5xInv));
	double [] co1ToNVec = matT.times(coordN);
	coordN = addCoords(coordCO, co1ToNVec);
	//Compute the CA coordinate residue (i+1)
	double [] coordCA = new double[]{0.0, Const.dN2CA, 0.0};
	double [] nToCAVec = matT.times(Const.r8zInv.times(Const.r1xInv.times(coordCA)));
	coordCA = addCoords(coordN, nToCAVec);
	double dnn = length(internuclearVec(ca1, coordCA));
	return dnn;
    }

    /**
     * Compute the CA directions of residues 2 and 3 specified by polar
     * angles (theta, phi), given the CA coordinates for residue 1 and
     * 3 and the theta polar angle for residue 2.  See Math derivation
     * for the detail; Be careful to return all the solutions.
     * 
     * @param ca1 the coordinate of the CA nucleus of residue i
     * @param ca3 the coordinate of the CA nucleus of residue i+2
     * @param theta2 the polar theta angle of CA2CA vector between residues i and i+1.
     * @param solutions 	double[][] allAngles = new double[2][3];
     * 
     * @return true, if dir ca ca
     */
    public boolean dirCaCa(double [] ca1, double[] ca3, double theta2, double[][] solutions){
	double c1 = (ca3[0] - ca1[0]) / Const.dCA2CA;
	double c2 = (ca3[1] - ca1[1]) / Const.dCA2CA;
	double c3 = (ca3[2] - ca1[2]) / Const.dCA2CA;
	double cosTheta2 = Math.cos(theta2);
	double sinTheta2 = Math.sin(theta2);
	if (Math.abs(c3 - cosTheta2) > 1.0){
// 	    System.out.println("kk01");
	    return false;
	}
	double theta3 = Math.acos(c3 - cosTheta2); //for polar angle, theta is in [0. pi] 
	double sinTheta3 = Math.sin(theta3);
	if (sinTheta2 == 0.0 || sinTheta3 == 0.0){
// 	    System.out.println("kk02");
	    return false;
	}
	double[] phi0Arr = new double[2];
	double[] phi1Arr = new double[2];

	double sinPhi1 = 0.0;
	double cosPhi1 = 0.0;
	double phiCnt  = 0.0;
	double sinPhiCnt = 0.0;
	double cosPhiCnt = 0.0;
	double c = 1.0;
	double sinPhi0 = 0.0;
	double cosPhi0 = 0.50*(c1*c1 + c2*c2 - sinTheta2 * sinTheta2 - sinTheta3 * sinTheta3) / (sinTheta2 * sinTheta3);
	double c1Cal = 0.0;
 	double c2Cal = 0.0;
// 	double ca3Cal = 0.0;
// 	double[] ca2Cal = new double[3];
// 	double[] ca2caDir = new double[3];
	double phi2 = 0.0, phi3 = 0.0;
	if (Math.abs(cosPhi0) > 1.0){
// 	    System.out.println("kk03");
	    return false;
	}
	phi0Arr[0] = Math.acos(cosPhi0); //2 solutions for Phi0
	phi0Arr[1] = -phi0Arr[0];
	for (int j = 0; j < 2; j++){
	    sinPhi0 = Math.sin(phi0Arr[j]);
// 	    cosPhi0 = Math.cos(phi0Arr[j]);
	    c = Math.sqrt( (sinTheta2 * cosPhi0 + sinTheta3) * (sinTheta2 * cosPhi0 + sinTheta3) 
			   + sinTheta2 * sinTheta2 * sinPhi0 * sinPhi0);
	    if ( c == 0.0 || Math.abs(c1 / c) > 1.0){
		System.out.println("kk04");
		return false;
	    }
	    sinPhiCnt = (sinTheta2 * cosPhi0 + sinTheta3) / c; //compute phiC
	    cosPhiCnt = sinTheta2 * sinPhi0 / c;
	    if ( cosPhiCnt >= 0 )
		phiCnt = Math.asin(sinPhiCnt);
	    else if ( cosPhiCnt < 0)
		phiCnt = Math.PI - Math.asin(sinPhiCnt);

	    phi1Arr[0] = Math.asin(c1/c);
	    phi1Arr[1] = Math.PI - phi1Arr[0];
	    for (int k=0; k<2; k++){
		phi2 = phiCnt - phi1Arr[k] + phi0Arr[j]; //phi2
		phi3 = phiCnt - phi1Arr[k];   //phi3
		if (phi2 < 0.0)
		    phi2 += 2.0*Math.PI;
		if (phi3 < 0.0)
		    phi3 += 2.0*Math.PI;
// 		ca2caDir[0] = Const.dCA2CA * Math.sin(theta2) * Math.cos(phi2);
// 		ca2caDir[1] = Const.dCA2CA * Math.sin(theta2) * Math.sin(phi2);
// 		ca2caDir[2] = Const.dCA2CA * Math.cos(theta2);
// 		ca2Cal = addCoords(ca1, ca2caDir);
// // 		printArray(ca2);
// //  		printArray(ca2Cal);
//  		ca3Cal = Const.dCA2CA * (sinTheta2 * Math.sin(phi2)) + ca1[1]; // + sinTheta3 * Math.sin(phi3)) + ca1[1];
//  		System.out.println("  -phi2- "+phi2 / Const.cst);
//  		System.out.println( ca2[1]+"  -y- "+ca3Cal);
// 		ca3Cal = Const.dCA2CA * (sinTheta2 * Math.cos(phi2)) + ca1[0];// + sinTheta3 * Math.cos(phi3)) + ca1[0];
//  		System.out.println( ca2[0]+"  -x- "+ca3Cal);
// 		ca3Cal = Const.dCA2CA * cosTheta2 + ca1[2];// + sinTheta3 * Math.cos(phi3)) + ca1[0];
//  		System.out.println( ca2[2]+"  -z- "+ca3Cal);
// 		System.out.println(c2+"  --** "+c2Cal);
//  		c1Cal = sinTheta2 * Math.cos(phi2) + sinTheta3 * Math.cos(phi3);
// 		System.out.println(c1+" ---  "+c1Cal);
 		c2Cal = sinTheta2 * Math.sin(phi2) + sinTheta3 * Math.sin(phi3);
		if (Math.abs(c2Cal - c2 ) < Const.eps){
		    solutions[j] [0] = phi2;
		    solutions[j] [1] = theta3;
		    solutions[j] [2] = phi3;
		}
	    }
	}
	return true;
    } 

    /**
     * Ca by next peptide plane.
     * 
     * @param n1 the n1
     * @param rg the rg
     * 
     * @return the double[]
     */
    public double[] caByNextPeptidePlane(double [] n1, Matrix rg){
	Matrix rgInv = rg.transpose();
	double [] n2coCnt = {0.0, 0.0, Const.dCO2N};
	double [] n2co = rgInv.times(Const.r9z1xAlphaInv.times(n2coCnt));
	double [] coCal = addCoords(n1, n2co);
	double [] co2caCnt = {0.0, 0.0, Const.dCA2CO};
	Matrix mat =  rgInv.times(Const.r9z1xAlphaInv.times(Const.r6zAlphaInv.times(Const.r2xAlphaInv)));
	double[] co2ca = mat.times(co2caCnt);
	return addCoords(coCal, co2ca);
    }

    /**
     * Test dir ca ca.
     */
    public void testDirCaCa(){
	int i = 0, j = 0;
	double [] n = {0.0, 0.0, 0.0};         //A coordinate frame in the 1st peptide plane
	double [] nh = {0.0, 0.0, -Const.dN2H}; 
    	double [] ca = {0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA * Math.sin(Const.theta1)};
	ModelRdc mdc = new ModelRdc();
	//build an initial  helix model with average \phi and \psi angles.
	int firstResidueNo =   1;
	int lastResidueNo  =  11;
	int N = lastResidueNo - firstResidueNo;
	Vector phiPsiVec = new Vector();
        for (i=firstResidueNo; i<lastResidueNo; i++)
            phiPsiVec.add(new PhiPsi(i, "ALA",  Const.phiAveHelix, Const.psiAveHelix));
        Vector pdbVec = mdc.modelBuild(phiPsiVec, n, nh, ca);

	double [] amide1 = new double[3];
	double [] nh1 = new double[3];
	double [] ca1 = new double[3];
	double [] co1 = new double[3];
	double [] amide2 = new double[3];
	double [] nh2 = new double[3];
	double [] ca2 = new double[3];
	double [] amide3 = new double[3];
	double [] nh3 = new double[3];
	double [] ca3 = new double[3];
	Cartesian cc = new Cartesian();
	String atom = "";

	Pdb pp = (Pdb)pdbVec.elementAt(1); 
	String resid = pp.getResidue();
	Vector atomVec = pp.getAtomVec();
	for (j=0; j<atomVec.size(); j++){
	    cc = (Cartesian)atomVec.elementAt(j);
	    atom = cc.getAtom();
	    if (atom.equals("N"))
		amide1 = cc.getXYZ();
	    else if (atom.equals("CA"))
		ca1 = cc.getXYZ();
	    else if (atom.equals("C"))
		co1 = cc.getXYZ();
	    else if (atom.equals("H"))
		nh1 = cc.getXYZ();
	}
// 	printArray(coCal);
// 	printArray(co1);

	double[] nToNHVec = internuclearVec(amide1, nh1);
	double[] nToCAVec = internuclearVec(amide1, ca1);
	Matrix rg = RgCal(nToNHVec, nToCAVec);

	pp = (Pdb)pdbVec.elementAt(2); 
	resid = pp.getResidue();
	atomVec = pp.getAtomVec();
	for (j=0; j<atomVec.size(); j++){
	    cc = (Cartesian)atomVec.elementAt(j);
	    atom = cc.getAtom();
	    if (atom.equals("N"))
		amide2 = cc.getXYZ();
	    else if (atom.equals("CA"))
		ca2 = cc.getXYZ();
	    else if (atom.equals("H"))
		nh2 = cc.getXYZ();
	}

	pp = (Pdb)pdbVec.elementAt(3); 
	resid = pp.getResidue();
	atomVec = pp.getAtomVec();
	for (j=0; j<atomVec.size(); j++){
	    cc = (Cartesian)atomVec.elementAt(j);
	    atom = cc.getAtom();
	    if (atom.equals("N"))
		amide3 = cc.getXYZ();
	    else if (atom.equals("CA"))
		ca3 = cc.getXYZ();
	    else if (atom.equals("H"))
		nh3 = cc.getXYZ();
	}
// 	nToNHVec = internuclearVec(amide1, nh1);
// 	nToCAVec = internuclearVec(amide1, ca1);
// 	rg = RgCal(nToNHVec, nToCAVec);
	double[] ca2caVec = internuclearVec(ca1, ca2);
// 	System.out.println(length(ca2caVec));
	double theta2 = Math.acos(ca2caVec[2] / Const.dCA2CA);

	double sinPhiCnt = ca2caVec[1] / (Const.dCA2CA * Math.sin(theta2)); //compute phiC
	double cosPhiCnt = ca2caVec[0] / (Const.dCA2CA * Math.sin(theta2)); //compute phiC
	double phiCnt = 0.0;
	if ( cosPhiCnt >= 0 )
	    phiCnt = Math.asin(sinPhiCnt);
	else if ( cosPhiCnt < 0)
	    phiCnt = Math.PI - Math.asin(sinPhiCnt);
	if (phiCnt < 0.0)
	    phiCnt += 2.0* Math.PI;
// 	System.out.println( phiCnt / Const.cst+"   "+ ( (2.0*Math.PI - phiCnt) / Const.cst));

	ca2caVec = internuclearVec(ca2, ca3);
// 	System.out.println(length(ca2caVec));
	double theta3 = Math.acos(ca2caVec[2] / Const.dCA2CA);
// 	System.out.println(" Theta3 = "+theta3 / Const.cst);
// 	System.out.println(length(ca2caVec));
	double[][] solutions = new double[2][3];
// 	Matrix rg = Matrix.identity(3, 3);
	boolean hasSolutions = dirCaCa(ca1, ca3, theta2, solutions);
	double phi2;
// 	double theta3;
	double phi3;
	double [] ca2Cal = new double[3];
	double [] ca3Cal = new double[3];
	double [] ca2caDir = new double[3];
// 	    for(j=0; j<3; j++){
// 		System.out.print(solutions[i][j]/ Const.cst+"  ");
// 	    System.out.println();
// 	System.out.println(Const.sinPsiC+"  "+Const.cosPsiC+"  "+Const.psiC);
	for (i=0; i<2; i++){
	    phi2 = solutions[i][0];
	    theta3 = solutions[i][1];
	    phi3 = solutions[i][2];
// 	    System.out.println( (theta2/Const.cst)+"  "+(phi2/Const.cst)+"  "+(theta3/Const.cst)+"  "+(phi3/Const.cst));
	    ca2caDir[0] = Const.dCA2CA * Math.sin(theta2) * Math.cos(phi2);
	    ca2caDir[1] = Const.dCA2CA * Math.sin(theta2) * Math.sin(phi2);
	    ca2caDir[2] = Const.dCA2CA * Math.cos(theta2);
	    ca2Cal = addCoords(ca1, ca2caDir);
// 	    System.out.println(" **");
// 	    printArray(ca2);
// 	    printArray(ca2Cal);
// 	    System.out.println(" **");
	    ca2caDir[0] = Const.dCA2CA * Math.sin(theta3) * Math.cos(phi3);
	    ca2caDir[1] = Const.dCA2CA * Math.sin(theta3) * Math.sin(phi3);
	    ca2caDir[2] = Const.dCA2CA * Math.cos(theta3);
	    ca3Cal = addCoords(ca2Cal, ca2caDir);
// 	    System.out.println(" ***");
// 	    printArray(ca3);
// 	    printArray(ca3Cal);
// 	    System.out.println(" ***");
	}
    }

 /**
  * *********************************************************************************
  * This method compute the \Phi and \Psi pair from the coordinates of
  * CA(i) and CA(i+1) EXACTLY assuming the six angles are known.
  * The math is similar to those for computing \Phi and \Psi from
  * two vectors in consequtive peptide plane.
  * See the detail for the Math derivation
  * ****************************************************
  * 
  * @param rg the rotation matrix from a unique global frame to a
  * local frame specified in the peptide plane i
  * @param solutions 	double[][] solutions = new double[2][3]; //since only two pairs
  * @param cacaVec the caca vec
  * 
  * @return  the solutions in the array solutions, and true is there are solutions
  * The six angles are assumed to be constants
  */    
    public boolean phiPsiByCa(Matrix rg, double[] cacaVec, double[][] solutions){
	double sinTheta3 = Math.sin(Const.theta3);
	double cosTheta3 = Math.cos(Const.theta3);
	double [] dirC = Const.r9y1x.times(rg.times(cacaVec));
	double cx = dirC[0];
	double cy = dirC[1];
	double cz = dirC[2];
	double ax = Const.dirA[0]; //Defined for convenince
	double ay = Const.dirA[1];
	double az = Const.dirA[2];
	double bx = Const.dirB[0];
	double by = Const.dirB[1];
	double bz = Const.dirB[2];
// 	System.out.println(cx+" LEFT  "+cy+"   "+cz);
// 	System.out.println(bx+"  "+by+"   "+bz);
	double cyCal = ay - bz * sinTheta3 - bx * cosTheta3 * Math.sin(Const.psiAveHelix) 
	    - by * cosTheta3 * Math.cos(Const.psiAveHelix);
// 	System.out.println(cyCal+" ***  "+cy);
// 	System.out.println(Const.cosPsiC +" &&*  "+cy);
	double d = cosTheta3 * Math.sqrt(bx * bx + by * by);
	if (  Math.abs( (ay - bz * sinTheta3 - cy) / d ) > 1.0)
	    return false;
	double psi1 = Math.asin((ay - bz * sinTheta3 - cy) / d) - Const.psiC;
	if (psi1 < (-Math.PI))
	    psi1 += 2*Math.PI;
	else if (psi1 > Math.PI)
	    psi1 -= 2*Math.PI;

	double psi2 = Math.PI - Math.asin((ay - bz * sinTheta3 - cy) / d) - Const.psiC;
	if (psi2 < (-Math.PI))
	    psi2 += 2*Math.PI;
	else if (psi2 > Math.PI)
	    psi2 -= 2*Math.PI;
	double[] psiArr = {psi1, psi2};
	double psi = 0.0, phi1, phi2;
	double a, b;
	double sinPhi0, cosPhi0, phi0 =0.0;
	double [] phi1Arr = new double[2];
	double [] phiArr = new double[2];
	double phi = 0.0;
	double czCal = 0.0;
	for (int i= 0; i< 2; i++){
	    psi  = psiArr[i];
	    a = ax - bx * Math.cos(psi) + by * Math.sin(psi);
	    b = az - bx * sinTheta3 * Math.sin(psi) - by * sinTheta3 * Math.cos(psi) + bz * cosTheta3;
	    d = Math.sqrt(a*a + b*b);
	    if ( d == 0.0 || (Math.abs( cx / d ) > 1.0))
		return false;
	    sinPhi0 = a / d;
	    cosPhi0 = b / d;
	    if ( cosPhi0 >= 0.0 )
		phi0 = Math.asin(sinPhi0);
	    else if ( cosPhi0 < 0)
		phi0 = Math.PI - Math.asin(sinPhi0);
// 	    System.out.println(" ..,, "+phi0/Const.cst);
	    phi1Arr[0] = Math.asin( cx / d );
	    phi1Arr[1] = Math.PI - phi1Arr[0];
	    
	    phi1 = phi1Arr[0] - phi0;
	    if (phi1 < (-Math.PI))
		phi1 += 2*Math.PI;
	    else if (phi1 > Math.PI)
		phi1 -= 2*Math.PI;

	    phi2 = phi1Arr[1] - phi0;
	    if (phi2 < (-Math.PI))
		phi2 += 2*Math.PI;
	    else if (phi2 > Math.PI)
		phi2 -= 2*Math.PI;
	
	    phiArr[0] = phi1;
	    phiArr[1] = phi2;
	    for (int j=0; j<2; j++){
		phi = phiArr[j];
// 		//check the solutions using an equation other than that used to compute phi
		czCal = -ax * Math.sin(phi) + az * Math.cos(phi) +
		    bx * (Math.sin(phi) * Math.cos(psi) - sinTheta3 * Math.cos(phi) * Math.sin(psi)) -
		    by * (Math.sin(phi) * Math.sin(psi) + sinTheta3 * Math.cos(phi) * Math.cos(psi)) +
		    bz * cosTheta3 * Math.cos(phi);
		if ( Math.abs(czCal - cz) < Const.eps){
		    solutions[i][0] = phi;
		    solutions[i][1] = psi;
		}
	    }
	}
  	return true;
    }

    /**
     * Compute the turns (8 - 11) by randomly changing the.
     * 
     * @param no1 the first residue of the fragment (turn and loop)
     * @param no2 the last residue of the fragment (turn and loop)
     * @param h1Vec resonance assignment sorted by H1 CS shift.
     * @param asgVec the merged resonacs assigement and HN and HA NOEs
     * @param rdcVec1 the CH RDC
     * @param rdcVec2 the NH RDC
     * @param a1  the constant used to convert the intensity to distance
     * @param a2  the constant used to convert the intensity to distance
     * @param pdbVec1 the structure for the beginning side of turn or loop
     * @param pdbVec2 the structure for the end side of turn or loop
     * @param Syy the diagnolized Saupe elements
     * @param Szz the diagnolized Saupe elements
     * 
     * @return  the refined Pdb vector
     */
    public Vector turnByCaCa(final Vector h1Vec, final Vector asgVec, Vector pdbVec1, Vector pdbVec2,  
			      final Vector rdcVec1, final Vector rdcVec2, double Syy, double Szz, int no1, int no2,
			      double a1, double a2){
	int i = 0, j = 0, k = 0;
	PhiPsi ff = new PhiPsi();
	double [] amide1 = new double[3];
	double [] nh1 = new double[3];
	double [] ca1 = new double[3];
	double [] amide2 = new double[3];
	double [] nh2 = new double[3];
	double [] ca2 = new double[3];
	double [] amide3 = new double[3];
	double [] nh3 = new double[3];
	double [] ca3 = new double[3];
	Cartesian cc = new Cartesian();
	String atom = "";
	//Extract the n, nh, ca coordinates of the first residue of turn, which the pdbVec1 have.
	int index = Collections.binarySearch(pdbVec1, new Pdb(no1), new Pdb.PdbComparator());
	if (index < 0){
	    System.out.println("Error in the PDB file 1");
	    return new Vector();
	}
	Pdb pp = (Pdb)pdbVec1.elementAt(index); 
	String resid = pp.getResidue();
	Vector atomVec = pp.getAtomVec();
	for (j=0; j<atomVec.size(); j++){
	    cc = (Cartesian)atomVec.elementAt(j);
	    atom = cc.getAtom();
	    if (atom.equals("N"))
		amide1 = cc.getXYZ();
	    else if (atom.equals("CA"))
		ca1 = cc.getXYZ();
	    else if (atom.equals("H"))
		nh1 = cc.getXYZ();
	}
	double[] nToNHVec = internuclearVec(amide1, nh1);
	double[] nToCAVec = internuclearVec(amide1, ca1);
	Matrix rg = RgCal(nToNHVec, nToCAVec);
	//The following number is good for 8-11 link
// 	Vector phiPsiVec = new Vector();
// 	phiPsiVec.add(new PhiPsi(no1, "ALA", -0.10373781727925674, -1.9178514182707769));
// 	no1++;
// 	phiPsiVec.add(new PhiPsi(no1, "ALA", -0.3585402544986122,  1.829438488320328));
// 	no1++;
// 	phiPsiVec.add(new PhiPsi(no1, "ALA", -0.40474021889760364,  0.9255922185428955));
// 	ModelRdc mdc = new ModelRdc();
// 	boolean isLink = true;
//         Vector pdbVecN = mdc.modelBuild(phiPsiVec, amide1, nh1, ca1, isLink);
// 	pp.print(pdbVecN);
// 	Collections.sort(pdbVec1, new Pdb.PdbComparator());
//  	pp.print(pdbVec1);

// 	Vector phiPsiVec = new Vector();  //one solutions for 45-46 with angle difference = 8.0
// 	phiPsiVec.add(new PhiPsi(no1, "ALA",-1.9000371400831886,  1.5642633527957572));
// 	no1++;
// 	phiPsiVec.add(new PhiPsi(no1, "ALA",-0.01961083782735104,  0.46109969497819425));
// 	no1++;
// 	phiPsiVec.add(new PhiPsi(no1, "ALA",-2.794852490690973,  -1.0384929379249943));
// 	ModelRdc mdc = new ModelRdc();
// 	boolean isLink = true;
//         Vector pdbVecN = mdc.modelBuild(phiPsiVec, amide1, nh1, ca1, isLink);
// 	pp.print(pdbVecN);
// 	Collections.sort(pdbVec1, new Pdb.PdbComparator());
//  	pp.print(pdbVec1);

//Good for 45-48
	Vector phiPsiVec = new Vector();
	phiPsiVec.add(new PhiPsi(no1, "ALA",-1.8690487073077084,  1.19895573617919));
	no1++;
	phiPsiVec.add(new PhiPsi(no1, "ALA",0.4860507350995962,  -0.35489809607846956));
	no1++;
	phiPsiVec.add(new PhiPsi(no1, "ALA",-1.9907688476363932,  -1.1374529481996267));
				 
	ModelRdc mdc = new ModelRdc();
	boolean isLink = true;
        Vector pdbVecN = mdc.modelBuild(phiPsiVec, amide1, nh1, ca1, isLink);
// 	pp.print(pdbVecN);
	Collections.sort(pdbVec1, new Pdb.PdbComparator());
//  	pp.print(pdbVec1);

	Matrix rgInv = rg.transpose();
	//Extract the n, nh, ca coordinates of the first residue of turn, which the pdbVec1 have.
	index = Collections.binarySearch(pdbVec2, new Pdb(no2), new Pdb.PdbComparator());
	if (index < 0){
	    System.out.println("Error in the PDB file 2");
	    return new Vector();
	}
	pp = (Pdb)pdbVec2.elementAt(index); 
	resid  = pp.getResidue();
	atomVec = pp.getAtomVec();
	for (j=0; j<atomVec.size(); j++){
	    cc = (Cartesian)atomVec.elementAt(j);
	    atom = cc.getAtom();
	    if (atom.equals("N"))
		amide3 = cc.getXYZ();
	    else if (atom.equals("CA"))
		ca3 = cc.getXYZ();
	    else if (atom.equals("H"))
		nh3 = cc.getXYZ();
	}

	double[] nToNHVec4 = internuclearVec(amide3, nh3);
	double[] nToCAVec4 = internuclearVec(amide3, ca3);
	Matrix rg2 = RgCal(nToNHVec4, nToCAVec4);
	double [] ca3CalCnt = caByNextPeptidePlane(amide3, rg2);
	double [] ca3Toca4DirT = internuclearVec(ca3CalCnt, ca3);
	double dd = length(ca3Toca4DirT);
	double [] ca3Toca4Dir = {ca3Toca4DirT[0] * Const.dCA2CA/ dd, ca3Toca4DirT[1] * Const.dCA2CA/ dd, ca3Toca4DirT[2] * Const.dCA2CA/ dd};
 	System.out.println(dd);
// 	printArray(ca3);
// 	printArray(ca3CalCnt);
	double resolution = 0.5;
	double theta1 = 0.0;
	double phi1 = 0.0;
	double theta2 = 0.0;
	double phi2 = 0.0;
	double theta3 = 0.0;
	double phi3 = 0.0;
	double [] ca1Cal = new double[3];
	double [] ca1Dir = new double[3];
	double [] ca2Cal = new double[3];
	double [] ca4Cal = new double[3];
	double [] ca2Dir = new double[3];
	double [] ca3Cal = new double[3];
	double [] ca3Dir = new double[3];
	boolean hasSolutions;
	double[][] solutions = new double[2][3];
	boolean hasPhiPsi;
	double[][] phiPsiSolutions1 = new double[2][2];
	double[][] phiPsiSolutions2 = new double[2][2];
	double[][] phiPsiSolutions3 = new double[2][2];
	int ss = 0;
	double phi1BB = 0.0; //the backbone phi and psi angles
	double psi1BB = 0.0;
	double [] nh2Dir = new double[3];
	double [] nh3Dir = new double[3];
	double [] nh4Dir = new double[3];
	double [] nh2Coord = new double[3];
	double [] nh3Coord = new double[3];
	double [] nh4Coord = new double[3];
	double [] ha1Dir = new double[3];
	double [] ha2Dir = new double[3];
	double [] ha3Dir = new double[3];
	double [] ha1Coord = new double[3];
	double [] ha2Coord = new double[3];
	double [] ha3Coord = new double[3];

	double phi2BB = 0.0; //the backbone phi and psi angles
	double psi2BB = 0.0;
	double phi3BB = 0.0; //the backbone phi and psi angles
	double psi3BB = 0.0;
	Matrix mat1 = new Matrix(3, 3);
	Matrix mat2 = new Matrix(3, 3);
	Matrix r2yInv1 = new Matrix(3, 3);
	Matrix r4zInv1 = new Matrix(3, 3);
	Matrix r2yInv2 = new Matrix(3, 3);
	Matrix r4zInv2 = new Matrix(3, 3);
	Matrix r2yInv3 = new Matrix(3, 3);
	Matrix r4zInv3 = new Matrix(3, 3);
// 	double[][] restraints = extractRestraints(no1, no2, h1Vec, asgVec, rdcVec1, rdcVec2, a1, a2);
	for (k=0; k < (180 / resolution); k++){
	    theta2 =  k * resolution * Math.PI / 180.0;
	    hasSolutions = dirCaCa(ca1, ca3CalCnt, theta2, solutions); //compute polar angles for other two ca2ca vectors
	    if (hasSolutions){  //now we have a set of solutions for polar angles for residue 1, 2, 3
// 		System.out.println(" nnn "+theta2);
		for (ss = 0; ss<2; ss++){
		    phi2 = solutions[ss][0];
//  		    System.out.println(" n* "+phi2);
		    ca2Dir[0] = Const.dCA2CA * Math.sin(theta2) * Math.cos(phi2); //ca2ca dir from residue 8 -> 9
		    ca2Dir[1] = Const.dCA2CA * Math.sin(theta2) * Math.sin(phi2);
		    ca2Dir[2] = Const.dCA2CA * Math.cos(theta2);
		    hasPhiPsi = phiPsiByCa(rg, ca2Dir, phiPsiSolutions1); //compute phi_8, psi_8
		    if(hasPhiPsi){
			ca2Cal = addCoords(ca1, ca2Dir);  //ca of residue 9
			theta3 = solutions[ss][1];
			phi3   = solutions[ss][2];
// 			System.out.println(" n** "+theta3+"  "+phi3);
			ca3Dir[0] = Const.dCA2CA * Math.sin(theta3) * Math.cos(phi3);//ca2ca dir from residue 9 -> 10
			ca3Dir[1] = Const.dCA2CA * Math.sin(theta3) * Math.sin(phi3);
			ca3Dir[2] = Const.dCA2CA * Math.cos(theta3);
			ca3Cal = addCoords(ca2Cal, ca3Dir);  //ca of residue 10
			ca4Cal = addCoords(ca3Cal, ca3Toca4Dir);  //ca of residue 10
//   			printArray(ca4Cal);
			for (int tt = 0; tt < 2; tt++){
			    phi1BB = phiPsiSolutions1[tt][0];
			    psi1BB = phiPsiSolutions1[tt][1];
// 			    r2yInv1 = rg.rotationMat(phi1BB, "-y"); //transpose == inverse == rotate in opposite direction
// 			    r4zInv1 = rg.rotationMat(psi1BB+Math.PI, "-z"); 
// 			    nh2Dir = rgInv.times(Const.r1x9yInv.times(r2yInv1.times(Const.r3xInv.times(r4zInv1.times(Const.matRNH)))));
			    mat1 = newRG(phi1BB, psi1BB, rg);
			    hasPhiPsi = phiPsiByCa(mat1, ca3Dir, phiPsiSolutions2);  //compute phi_9, psi_9
			    if(hasPhiPsi){
// 				System.out.println(" n** "+theta3+"  "+phi3);
				for (int qq = 0; qq < 2; qq++){
				    phi2BB = phiPsiSolutions2[qq][0];
				    psi2BB = phiPsiSolutions2[qq][1];
				    //r2yInv2 = rg.rotationMat(phi2BB, "-y"); //transpose == inverse == rotate in opposite direction
				    //r4zInv2 = rg.rotationMat(psi2BB+Math.PI, "-z"); 
				    //nh3Dir = (mat1.transpose()).times(Const.r1x9yInv.times(r2yInv2.times(Const.r3xInv.times(r4zInv2.times(Const.matRNH)))));
				    mat2 = newRG(phi2BB, psi2BB, mat1);
				    hasPhiPsi = phiPsiByCa(mat2, ca3Toca4Dir, phiPsiSolutions3);  //compute phi_10, psi_10
				    if(hasPhiPsi){
					for (int u= 0; u < 2; u++){
					    phi3BB = phiPsiSolutions3[u][0];
					    psi3BB = phiPsiSolutions3[u][1];
					    r2yInv3 = rg.rotationMat(phi3BB, "-y"); //transpose == inverse == rotate in opposite direction
					    r4zInv3 = rg.rotationMat(psi3BB+Math.PI, "-z"); 
					    nh4Dir = (mat2.transpose()).times(Const.r1x9yInv.times(r2yInv3.times(Const.r3xInv.times(r4zInv3.times(Const.matRNH)))));
					    if (interAngle(nh4Dir, nToNHVec4) < 0.50 * Const.cst){
						System.out.println(phi1BB+"  "+psi1BB);
						System.out.println(phi2BB+"  "+psi2BB);
						System.out.println(phi3BB+"  "+psi3BB);
						System.out.println(interAngle(nh4Dir, nToNHVec4) / Const.cst);
					    }//else {System.out.println(" Still error?");}
					}
				    }
				}
			    }
			}
		    }
		}
	    }
	}
	return new Vector();
    }
    
    /**
     * refine the psi and phi angles based on rdc rmsd and.
     * 
     * @param no1 the first residue of the fragment (turn and loop)
     * @param no2 the last residue of the fragment (turn and loop)
     * @param h1Vec resonance assignment sorted by H1 CS shift.
     * @param asgVec the merged resonance assigement and HN and HA NOEs
     * @param rdcVec1 the CH RDC
     * @param rdcVec2 the NH RDC
     * @param a1  the constant used to convert the intensity to distance
     * @param a2  the constant used to convert the intensity to distance
     * @param pdbVec1 the structure for the beginning side of turn or loop
     * @param pdbVec2 the structure for the end side of turn or loop
     * @param Syy the diagnolized Saupe elements
     * @param Szz the diagnolized Saupe elements
     * 
     * @return  the refined Pdb vector
     */
   /* public Vector phipsiRefineFromVDW()
    {
    	
    }*/
    		
    		
    		
    		
   /**
     * Compute the loop (18 - 23) by randomly changing the 18, 19
     * backbone dihedral angles.
     @param no1 the first residue of the fragment (turn and loop)
     @param no2 the last residue of the fragment (turn and loop)
     @param h1Vec resonance assignment sorted by H1 CS shift.
     @param asgVec the merged resonance assigement and HN and HA NOEs
     @param rdcVec1 the CH RDC
     @param rdcVec2 the NH RDC
     @param a1  the constant used to convert the intensity to distance
     @param a2  the constant used to convert the intensity to distance
     @param pdbVec1 the structure for the beginning side of turn or loop
     @param pdbVec2 the structure for the end side of turn or loop
     @param Syy the diagnolized Saupe elements
     @param Szz the diagnolized Saupe elements
     @return  the refined Pdb vector
     */
    public Vector longTurn(final Vector h1Vec, final Vector asgVec, Vector pdbVec1, Vector pdbVec2, 
			   Vector rdcVec1, Vector rdcVec2, double Syy, double Szz, int no1, int no2,
			   double a1, double a2){
	int i = 0, j = 0, k = 0;
	PhiPsi ff = new PhiPsi();
	double [] amide1 = new double[3];
	double [] nh1 = new double[3];
	double [] ca1 = new double[3];
	double [] amide3 = new double[3];
	double [] nh3 = new double[3];
	double [] ca3 = new double[3];
	Cartesian cc = new Cartesian();
	String atom = "";

	//Extract the n, nh, ca coordinates of the first residue of turn, which the pdbVec1 have.
	int index = Collections.binarySearch(pdbVec1, new Pdb(no1), new Pdb.PdbComparator());
	if (index < 0){
	    System.out.println("Error in the PDB file 1");
	    return new Vector();
	}
	Pdb pp = (Pdb)pdbVec1.elementAt(index); 
	String resid = pp.getResidue();
	Vector atomVec = pp.getAtomVec();
	for (j=0; j<atomVec.size(); j++){
	    cc = (Cartesian)atomVec.elementAt(j);
	    atom = cc.getAtom();
	    if (atom.equals("N"))
		amide1 = cc.getXYZ();
	    else if (atom.equals("CA"))
		ca1 = cc.getXYZ();
	    else if (atom.equals("H"))
		nh1 = cc.getXYZ();
	}

	Vector phiPsiVec = new Vector();
	phiPsiVec.add(new PhiPsi(no1, "ALA", -2.3459797068036274,  -1.0074042391114535));
	no1++;
	phiPsiVec.add(new PhiPsi(no1, "ALA", -1.1344640137963142,  -2.896526774209493));
	no1++;
	phiPsiVec.add(new PhiPsi(no1, "ALA",-0.6536351104162039,  -1.8651676752987254));
	no1++;
	phiPsiVec.add(new PhiPsi(no1, "ALA", -2.827096526295202,  -1.541907976458233));
	no1++;
	phiPsiVec.add(new PhiPsi(no1, "ALA", 1.4819369120738939,  1.896621252342314));

	ModelRdc mdc = new ModelRdc();
	boolean isLink = true;
        Vector pdbVecN = mdc.modelBuild(phiPsiVec, amide1, nh1, ca1, isLink);
	pp.print(pdbVecN);
	pp.print(pdbVec1);

	double[] nToNHVec = internuclearVec(amide1, nh1);
	double[] nToCAVec = internuclearVec(amide1, ca1);
	Matrix rg = RgCal(nToNHVec, nToCAVec);
	Matrix rgInv = rg.transpose();
	//Extract the n, nh, ca coordinates of the first residue of end (pdbVec2).
	index = Collections.binarySearch(pdbVec2, new Pdb(no2), new Pdb.PdbComparator());
	if (index < 0){
	    System.out.println("Error in the PDB file 2");
	    return new Vector();
	}
	pp = (Pdb)pdbVec2.elementAt(index);
	resid  = pp.getResidue();
	atomVec = pp.getAtomVec();
	for (j=0; j<atomVec.size(); j++){
	    cc = (Cartesian)atomVec.elementAt(j);
	    atom = cc.getAtom();
	    if (atom.equals("N"))
		amide3 = cc.getXYZ();
	    else if (atom.equals("CA"))
		ca3 = cc.getXYZ();
	    else if (atom.equals("H"))
		nh3 = cc.getXYZ();
	}

	double[] nToNHVec4 = internuclearVec(amide3, nh3);
	double[] nToCAVec4 = internuclearVec(amide3, ca3);
	double ca18To23 = length(internuclearVec(ca1, ca3));
	Matrix rg3 = RgCal(nToNHVec4, nToCAVec4);
	double [] caCal = caByNextPeptidePlane(amide3, rg3); //the computed ca for residue 22
	double [] caTocaDirT = internuclearVec(caCal, ca3);
	double dd = length(caTocaDirT);
	double [] ca3Toca4Dir = {caTocaDirT[0] * Const.dCA2CA/dd, caTocaDirT[1] * Const.dCA2CA/dd, caTocaDirT[2] * Const.dCA2CA/dd};
	double [] ca22Cal = {ca3[0] - ca3Toca4Dir[0], ca3[1] - ca3Toca4Dir[1], ca3[2] - ca3Toca4Dir[2]};
	double ca18To22 = length(internuclearVec(ca1, ca22Cal));
	System.out.println(ca18To23+" CA DIS  "+ca18To22);
// 	printArray(caCal);
// 	printArray(ca22Cal);
// 	System.out.println(dd);
// 	printArray(ca3);
// 	printArray(ca3CalCnt);
// 	double resolution = 5;
	double theta1 = 0.0;
	double phi1 = 0.0;
	double theta2 = 0.0;
	double phi2 = 0.0;
	double theta3 = 0.0;
	double phi3 = 0.0;
	double [] ca1Cal = new double[3];
	double [] ca1Dir = new double[3];
	double [] ca2Cal = new double[3];
	double [] ca4Cal = new double[3];
	double [] ca2Dir = new double[3];
	double [] ca3Cal = new double[3];
	double [] ca3Dir = new double[3];
	boolean hasSolutions;
	double[][] solutions = new double[2][3];
	boolean hasPhiPsi;

	double[][] phiPsiSolutions  = new double[2][2];
	double[][] phiPsiSolutions1 = new double[2][2];
	double[][] phiPsiSolutions2 = new double[2][2];
	double[][] phiPsiSolutions3 = new double[2][2];
	int ss = 0;
	double phiBB = 0.0; //the backbone phi and psi angles
	double psiBB = 0.0;
	double phi1BB = 0.0; //the backbone phi and psi angles
	double psi1BB = 0.0;
	double [] nh2Dir = new double[3];
	double [] nh3Dir = new double[3];
	double [] nh4Dir = new double[3];
	double [] nh2Coord = new double[3];
	double [] nh3Coord = new double[3];
	double [] nh4Coord = new double[3];
	double [] ha1Dir = new double[3];
	double [] ha2Dir = new double[3];
	double [] ha3Dir = new double[3];
	double [] ha1Coord = new double[3];
	double [] ha2Coord = new double[3];
	double [] ha3Coord = new double[3];

	double phi2BB = 0.0; //the backbone phi and psi angles
	double psi2BB = 0.0;
	double phi3BB = 0.0; //the backbone phi and psi angles
	double psi3BB = 0.0;
	Matrix mat1 = new Matrix(3, 3);
	Matrix mat2 = new Matrix(3, 3);
	Matrix r2yInv1 = new Matrix(3, 3);
	Matrix r4zInv1 = new Matrix(3, 3);
	Matrix r2yInv2 = new Matrix(3, 3);
	Matrix r4zInv2 = new Matrix(3, 3);
	Matrix r2yInv3 = new Matrix(3, 3);
	Matrix r4zInv3 = new Matrix(3, 3);
	double theta19, phi19, bbPhi19, bbPsi19;
	Matrix rg19 = new Matrix(3, 3);
	Matrix rg20 = new Matrix(3, 3);
	double[] ca19 = new double[3];
	double[] ca20 = new double[3];
	double[] caDir20 = new double[3];
// 	double[][] restraints = extractRestraints(no1, no2, h1Vec, asgVec, rdcVec1, rdcVec2, a1, a2);
	long seed = 738578932;
	Random rr = new Random(seed);
	int nCycle = 1;//28 * 1024;//*1024;
	double[] ca2caTmp = new double[3];
	double ca2caDis;
	for (i=0; i < nCycle; i++){
	    theta19 = Math.PI * rr.nextDouble(); //theta19 and phi19 determine the ca(18)->ca(19) vector
	    phi19 = 2.0 * Math.PI * rr.nextDouble(); //residue 19 is a proline
	    ca19[0] = ca1[0] + Const.dCA2CA * Math.sin(theta19) * Math.cos(phi19);
	    ca19[1] = ca1[1] + Const.dCA2CA * Math.sin(theta19) * Math.sin(phi19);
	    ca19[2] = ca1[2] + Const.dCA2CA * Math.cos(theta19);
	    ca2caTmp = internuclearVec(ca19, ca22Cal);
	    ca2caDis = length(ca2caTmp);
// 	    if (ca2caDis < 3.0 * Const.dCA2CA) 
// 		System.out.println("ca19 to ca22 = "+ca2caDis);
	    hasPhiPsi = phiPsiByCa(rg, internuclearVec(ca1, ca19), phiPsiSolutions); 	
	    if (hasPhiPsi){
// 		System.out.println(" nnn "+theta19+"  "+ phi19);    
		for (int ii = 0; ii < 2; ii++){ 
		    phiBB = phiPsiSolutions[ii][0]; //phiPsi angles of residue 18.
		    psiBB = phiPsiSolutions[ii][1];
		    rg19  = newRG(phiBB, psiBB, rg);
		    bbPhi19 = -65.0 * Const.cst;
		    bbPsi19 = -Math.PI+Math.PI * 2.0 * rr.nextDouble();
		    caDir20 = ca2caDirCal(bbPhi19, bbPsi19, ca19, rg19);//
		    ca20 = addCoords(ca19, caDir20);
// 		    ca2caTmp = internuclearVec(ca19, ca20);
// 		    ca2caDis = length(ca2caTmp);
// 		    System.out.println("ca19 to ca20 = "+ca2caDis);
		    ca2caTmp = internuclearVec(ca20, ca22Cal);
		    ca2caDis = length(ca2caTmp);
// 		    if (ca2caDis < 2.0 * Const.dCA2CA)
// 			System.out.println("20 to ca22 = "+ca2caDis);
		    rg20 = newRG(bbPhi19, bbPsi19, rg19);
		    theta2 = Math.PI * rr.nextDouble(); //theta21 for  ca(20)->ca(21) vector
		    hasSolutions = dirCaCa(ca20, ca22Cal, theta2, solutions); //compute polar angles for other two ca2ca vectors
		    if (hasSolutions){  //now we have a set of solutions for polar angles for residue 1, 2, 3
//  			System.out.println(" nnn "+theta2);
			for (ss = 0; ss<2; ss++){
			    phi2 = solutions[ss][0];
			    //  		    System.out.println(" n* "+phi2);
			    ca2Dir[0] = Const.dCA2CA * Math.sin(theta2) * Math.cos(phi2); //ca2ca dir from residue 8 -> 9
			    ca2Dir[1] = Const.dCA2CA * Math.sin(theta2) * Math.sin(phi2);
			    ca2Dir[2] = Const.dCA2CA * Math.cos(theta2);
			    hasPhiPsi = phiPsiByCa(rg20, ca2Dir, phiPsiSolutions1); //compute phi_8, psi_8
			    if(hasPhiPsi){
				ca2Cal = addCoords(ca1, ca2Dir);  //ca of residue 9
				theta3 = solutions[ss][1];
				phi3   = solutions[ss][2];
				// 			System.out.println(" n** "+theta3+"  "+phi3);
				ca3Dir[0] = Const.dCA2CA * Math.sin(theta3) * Math.cos(phi3);//ca2ca dir from residue 9 -> 10
				ca3Dir[1] = Const.dCA2CA * Math.sin(theta3) * Math.sin(phi3);
				ca3Dir[2] = Const.dCA2CA * Math.cos(theta3);
				ca3Cal = addCoords(ca2Cal, ca3Dir);  //ca of residue 10
				ca4Cal = addCoords(ca3Cal, ca3Toca4Dir);  //ca of residue 10
				//   			printArray(ca4Cal);
				for (int tt = 0; tt < 2; tt++){
				    phi1BB = phiPsiSolutions1[tt][0];
				    psi1BB = phiPsiSolutions1[tt][1];
				    //r2yInv1 = rg.rotationMat(phi1BB, "-y"); //transpose == inverse == rotate in opposite direction
				    //r4zInv1 = rg.rotationMat(psi1BB+Math.PI, "-z"); 
				    // nh2Dir = rgInv.times(Const.r1x9yInv.times(r2yInv1.times(Const.r3xInv.times(r4zInv1.times(Const.matRNH)))));
				    mat1 = newRG(phi1BB, psi1BB, rg20); //rg21
				    hasPhiPsi = phiPsiByCa(mat1, ca3Dir, phiPsiSolutions2);  //compute phi_9, psi_9
				    if(hasPhiPsi){
					// 				System.out.println(" n** "+theta3+"  "+phi3);
					for (int qq = 0; qq < 2; qq++){
					    phi2BB = phiPsiSolutions2[qq][0];
					    psi2BB = phiPsiSolutions2[qq][1];
					    //r2yInv2 = rg.rotationMat(phi2BB, "-y"); //transpose == inverse == rotate in opposite direction
					    //r4zInv2 = rg.rotationMat(psi2BB+Math.PI, "-z"); 
					    //nh3Dir = (mat1.transpose()).times(Const.r1x9yInv.times(r2yInv2.times(Const.r3xInv.times(r4zInv2.times(Const.matRNH)))));
					    mat2 = newRG(phi2BB, psi2BB, mat1);
					    hasPhiPsi = phiPsiByCa(mat2, ca3Toca4Dir, phiPsiSolutions3);  //compute phi_10, psi_10
					    if(hasPhiPsi){
						for (int u= 0; u < 2; u++){
						    phi3BB = phiPsiSolutions3[u][0];
						    psi3BB = phiPsiSolutions3[u][1];
						    r2yInv3 = rg.rotationMat(phi3BB, "-y"); //transpose == inverse == rotate in opposite direction
						    r4zInv3 = rg.rotationMat(psi3BB+Math.PI, "-z"); 
						    nh4Dir = (mat2.transpose()).times(Const.r1x9yInv.times(r2yInv3.times(Const.r3xInv.times(r4zInv3.times(Const.matRNH)))));
						    if (interAngle(nh4Dir, nToNHVec4) < 1.0 * Const.cst){
							System.out.println(phiBB+"  "+psiBB);
							System.out.println(bbPhi19+"  "+bbPsi19);
							System.out.println(phi1BB+"  "+psi1BB);
							System.out.println(phi2BB+"  "+psi2BB);
							System.out.println(phi3BB+"  "+psi3BB);
							System.out.println(interAngle(nh4Dir, nToNHVec4) / Const.cst);
						    }//else {System.out.println(" Still error?"+ interAngle(nh4Dir, nToNHVec4) / Const.cst);}
						}
					    }
					}
				    }
				}
			    }
			}
		    }//
		}
	    }
	}
	return new Vector();
    }

  /**
   * A method for computing the the Vector CA(i)-->CA(i+1), between the
   * CA(i) and CA(i+1) begin with an frame defined in the peptide
   * plane i. In FACT, with standard bond angles and lengths, dCaCa is
   * a constant.
   * 
   * @param phi     The backbone \phi angle
   * @param psi     The backbone \psi angle
   * @param rg the rotation matrix to the peptide plane i
   * @param ca1 the ca1
   * 
   * @return the Ca(i)-->Ca(i+1) vector.
   */    
    public double[] ca2caDirCal(double phi, double psi, double[] ca1, Matrix rg){
	Matrix rgInv = rg.transpose();
  	Matrix r2yInv = rg.rotationMat(phi, "-y"); 
	Matrix r4zInv = rg.rotationMat(psi+Math.PI, "-z"); 
	
 	Matrix matT = rgInv.times(Const.r1x9yInv.times(r2yInv.times(Const.r3xInv)));
	//Compute the CO coordinate of residue (i)
	double [] caToCOVec = matT.times(Const.ca2coVec);
	double [] coordCO = addCoords(caToCOVec,  ca1); 
	//Compute the N coordinate of residue (i+1)
	matT = matT.times(r4zInv.times(Const.r5xInv));
	double [] co1ToNVec = matT.times(Const.co2nVec);
	double [] coordN = addCoords(coordCO, co1ToNVec);
	//Compute the CA coordinate residue (i+1)
	double [] nToCAVec = matT.times(Const.r6y7x8z1xInv.times(Const.coordCA));
	double [] coordCA = addCoords(coordN, nToCAVec);
	double [] ca2ca = internuclearVec(ca1, coordCA);
// 	System.out.println("CA2CA length = "+length(ca2ca));
	return ca2ca;
    }


    /**
     * There is another method for computing these angles, see the backup file.
     */
    public void dNNtest(){
	double dnn = 0.0;
	double phi = 0.0, psi = 0.0;
	for (int i=0; i<360; i++){
	    phi = i * Math.PI / 360 - Math.PI;
	    for (int j=0; j<360; j++){
		psi = j * Math.PI / 360 - Math.PI;
// 		dnn = dNNseq(phi, psi);
		dnn = dCaCa(phi, psi);
		System.out.println(phi+"  "+psi+"  "+dnn);
	    }
	}
    }

    /**
     * Test phi psi by ca.
     * 
     * @param pdbVec the pdb vec
     */
    public void testPhiPsiByCa(Vector pdbVec){
	int i = 0, j = 0;
	double [] n = {0.0, 0.0, 0.0};         //A coordinate frame in the 1st peptide plane
	double [] nh = {0.0, 0.0, -Const.dN2H}; 
    	double [] ca = {0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA * Math.sin(Const.theta1)};
	ModelRdc mdc = new ModelRdc();
	//build an initial  helix model with average \phi and \psi angles.
	int firstResidueNo =   1;
	int lastResidueNo  =  11;
	int N = lastResidueNo - firstResidueNo;
	Vector phiPsiVec = new Vector();
        for (i=firstResidueNo; i<lastResidueNo; i++)
            phiPsiVec.add(new PhiPsi(i, "ALA",  Const.phiAveHelix, Const.psiAveBeta));
        pdbVec = mdc.modelBuild(phiPsiVec, n, nh, ca);

	double phi =  Const.phiAveHelix;
	double psi =  Const.psiAveHelix;
	Matrix mat = new Matrix(3, 3);
	Matrix r2yInv = mat.rotationMat(phi, "-y"); //transpose == inverse == rotate in opposite direction
	Matrix r4zInv = mat.rotationMat(psi+Math.PI, "-z"); 
	double [] dirA = r2yInv.times(Const.dirA);
	double [] dirB = r2yInv.times(Const.r3xInv.times(r4zInv.times(Const.dirB)));
// 	System.out.println( (dirA[0]+dirB[0])+" right "+(dirA[1]+dirB[1])+"  "+(dirA[2]+dirB[2]));
// 	System.out.println(Const.dirC[0]" left  "+Const.dirC[2]+"   "+Const.dirC[2]);
	double [] ca2co = Const.r1x9yInv.times(dirA);
	double [] coCal = addCoords(ca, ca2co);
	double [] amide1 = new double[3];
	double [] nh1 = new double[3];
	double [] ca1 = new double[3];
	double [] co1 = new double[3];
	double [] amide2 = new double[3];
	double [] nh2 = new double[3];
	double [] ca2 = new double[3];
	Cartesian cc = new Cartesian();
	String atom = "";

	Pdb pp = (Pdb)pdbVec.elementAt(1); 
	String resid = pp.getResidue();
	Vector atomVec = pp.getAtomVec();
	for (j=0; j<atomVec.size(); j++){
	    cc = (Cartesian)atomVec.elementAt(j);
	    atom = cc.getAtom();
	    if (atom.equals("N"))
		amide1 = cc.getXYZ();
	    else if (atom.equals("CA"))
		ca1 = cc.getXYZ();
	    else if (atom.equals("C"))
		co1 = cc.getXYZ();
	    else if (atom.equals("H"))
		nh1 = cc.getXYZ();
	}
// 	printArray(coCal);
// 	printArray(co1);
	pp = (Pdb)pdbVec.elementAt(2); 
	resid = pp.getResidue();
	atomVec = pp.getAtomVec();
	for (j=0; j<atomVec.size(); j++){
	    cc = (Cartesian)atomVec.elementAt(j);
	    atom = cc.getAtom();
	    if (atom.equals("N"))
		amide2 = cc.getXYZ();
	    else if (atom.equals("CA"))
		ca2 = cc.getXYZ();
	    else if (atom.equals("H"))
		nh2 = cc.getXYZ();
	}
	double[] nToNHVec = internuclearVec(amide2, nh2);
	double[] nToCAVec = internuclearVec(amide2, ca2);
	Matrix rg = RgCal(nToNHVec, nToCAVec);
// 	double[] ca1Cal = caByNextPeptidePlane(amide2, rg);
// 	printArray(ca1Cal);
// 	printArray(ca1);

// 	double [] ca2Nseq = internuclearVec(amide2, ca1);
// 	double [] ca2NHseq = internuclearVec(nh2, ca1);
// 	System.out.println( length(ca2Nseq)+"  "+length(ca2NHseq));
// // 	double [] co2n = Const.r1x9yInv.times(r2yInv.times(Const.r3xInv.times(r4zInv.times(Const.dirB1))));
// // 	double [] nCal = addCoords(coCal, co2n);
// // 	printArray(nCal);
// // 	printArray(amide2);

// // 	double [] n2ca = Const.r1x9yInv.times(r2yInv.times(Const.r3xInv.times(r4zInv.times(Const.dirB2))));
// // 	double [] caCal = addCoords(nCal, n2ca);

// // 	nToNHVec = internuclearVec(amide1, nh1);
// // 	nToCAVec = internuclearVec(amide1, ca1);
// // 	rg = RgCal(nToNHVec, nToCAVec);
// 	double[] ca2caVec = internuclearVec(ca1, ca2);
// // 	System.out.println(length(ca2caVec));
// 	double[][] solutions = new double[2][3];
// // 	Matrix rg = Matrix.identity(3, 3);
// 	boolean hasSolutions = phiPsiByCa(rg, ca2caVec, solutions);
// // 	System.out.println(Const.sinPsiC+"  "+Const.cosPsiC+"  "+Const.psiC);
// 	for (i=0; i<2; i++){
// 	    for(j=0; j<2; j++)
// 		System.out.print(solutions[i][j] / Const.cst+"  ");
// 	    System.out.println();
// 	}
    }

    /**
     * Test the computation of psi1, phi2, psi2 given the peptide
     * plane for residue 1 an d3 and phi1 without knowing the relative
     * position between two planes.
     */
    public void nh2ca2Cal(){
	int i = 0, j = 0;
	double [] n = {0.0, 0.0, 0.0};         //A coordinate frame in the 1st peptide plane
	double [] nh = {0.0, 0.0, -Const.dN2H}; 
    	double [] ca = {0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA * Math.sin(Const.theta1)};
	ModelRdc mdc = new ModelRdc();
	//build an initial  helix model with average \phi and \psi angles.
	int firstResidueNo =   1;
	int lastResidueNo  =  11;
	int N = lastResidueNo - firstResidueNo;
	Vector phiPsiVec = new Vector();
        for (i=firstResidueNo; i<lastResidueNo; i++)
            phiPsiVec.add(new PhiPsi(i, "ALA",  Const.phiAveHelix, Const.psiAveBeta));
	double phi1 = 90.0 * Const.cst;
	double psi1 = -33.0 * Const.cst;
	double phi2 = 71.0 * Const.cst;
	double psi2 = -62.33 * Const.cst;
	double phi3 = -190.55 * Const.cst;
	double psi3 = 60.0 * Const.cst;
  	phiPsiVec.setElementAt(new PhiPsi(firstResidueNo+1, "ALA", phi1, psi1), 1);// + 40.0 * Const.cst), 1);
  	phiPsiVec.setElementAt(new PhiPsi(firstResidueNo+2, "ALA", phi2, psi2), 2);
  	phiPsiVec.setElementAt(new PhiPsi(firstResidueNo+3, "ALA", phi3, psi3), 3);
        Vector pdbVec = mdc.modelBuild(phiPsiVec, n, nh, ca);
//         Vector pdbVec = mdc.modelBuild(phiArr, psiArr, n, nh, ca, 1, true);
 	Pdb pp = new Pdb(); 
// 	Vector ppAll =	pp.PhiPsiTotalPdb(pdbVec);
	// 	pp.print(pdbVec);
	double [] amide1 = new double[3];
	double [] nh1 = new double[3];
	double [] ca1 = new double[3];
	double [] ha1 = new double[3];
	double [] o1  = new double[3];
	double [] co1 = new double[3];
	double [] amide2 = new double[3];
	double [] nh2 = new double[3];
	double [] ca2 = new double[3];
	double [] ha2 = new double[3];
	double [] co2 = new double[3];
	double [] o2 = new double[3];
	double [] cb2 = new double[3];
	double [] amide3 = new double[3];
	double [] nh3 = new double[3];
	double [] ca3 = new double[3];
	double [] amide4 = new double[3];
	double [] nh4 = new double[3];
	double [] ca4 = new double[3];
	Cartesian cc = new Cartesian();
	String atom = "";

	pp = (Pdb)pdbVec.elementAt(1); 
	String resid = pp.getResidue();
	Vector atomVec = pp.getAtomVec();
	for (j=0; j<atomVec.size(); j++){
	    cc = (Cartesian)atomVec.elementAt(j);
	    atom = cc.getAtom();
	    if (atom.equals("N"))
		amide1 = cc.getXYZ();
	    else if (atom.equals("CA"))
		ca1 = cc.getXYZ();
	    else if (atom.equals("HA"))
		ha1 = cc.getXYZ();
	    else if (atom.equals("C"))
		co1 = cc.getXYZ();
	    else if (atom.equals("O"))
		o1 = cc.getXYZ();
	    else if (atom.equals("H"))
		nh1 = cc.getXYZ();
	}
	double[] nToNHVec = internuclearVec(amide1, nh1);
	double[] nToCAVec = internuclearVec(amide1, ca1);
	Matrix rg1 = RgCal(nToNHVec, nToCAVec);
	Matrix rg1Inv = rg1.transpose();
	pp = (Pdb)pdbVec.elementAt(2); 
	resid = pp.getResidue();
	atomVec = pp.getAtomVec();
	for (j=0; j<atomVec.size(); j++){
	    cc = (Cartesian)atomVec.elementAt(j);
	    atom = cc.getAtom();
	    if (atom.equals("N"))
		amide2 = cc.getXYZ();
	    else if (atom.equals("CA"))
		ca2 = cc.getXYZ();
	    else if (atom.equals("CB"))
		cb2 = cc.getXYZ();
	    else if (atom.equals("HA"))
		ha2 = cc.getXYZ();
	    else if (atom.equals("C"))
		co2 = cc.getXYZ();
	    else if (atom.equals("O"))
		o2 = cc.getXYZ();
	    else if (atom.equals("H"))
		nh2 = cc.getXYZ();
	}
	double[] nToNHVec2 = internuclearVec(amide2, nh2);
	double[] nToCAVec2 = internuclearVec(amide2, ca2);
	Matrix rg2 = RgCal(nToNHVec2, nToCAVec2);

	double [] n2nhCnt = {0.0, 0.0, -Const.dN2H};
    	double [] n2caCnt = {0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA * Math.sin(Const.theta1)};

	Matrix r2y = rg1.rotationMat(phi1, "+y"); 
	Matrix r4z = rg1.rotationMat(psi1, "+z");//psi2+Math.PI, "-z"); 
	Matrix r2yInv = rg1.rotationMat(phi1, "-y"); 
	Matrix r4zInv = rg1.rotationMat(psi1, "-z");//psi2+Math.PI, "-z"); 
	Matrix rg2Cal = Const.rot1Inv.times(r4z.times(Const.r3x).times(r2y.times(Const.r9y1x.times(rg1))));
	double[] n2nhCal2 = rg2Cal.transpose().times(n2nhCnt);
	double[] n2caCal2 = rg2Cal.transpose().times(n2caCnt);
// 	printArray(nToNHVec2);
// 	printArray(n2nhCal2);
// 	printArray(nToCAVec2);
// 	printArray(n2caCal2);
// 	rg2.print(12,14);
// 	rg2Cal.print(12,14);
	Matrix rg1Cal =Const.r1x9yInv.times(r2yInv.times(Const.r3xInv).times(r4zInv.times(Const.rot1.times(rg2))));
// 	rg1.print(12,14);
// 	rg1Cal.print(12,14);

 	Matrix L1 = Const.r1x9yInv.times(r2yInv.times(Const.r3xInv));
	nToNHVec2 = rg1Inv.times(L1.times(r4zInv.times(Const.Cv)));	
   	double [] n2nh = dirCos(internuclearVec(amide2, nh2));
// 	printArray(n2nh);
// 	printArray(nToNHVec2);
   	double [] n2ca = dirCos(internuclearVec(amide2, ca2));
	double [] coordCA = new double[]{0.0, 1.0, 0.00};
	double [] n2CAVec = rg1Inv.times(L1.times(r4zInv.times(Const.Cu)));	
// 	printArray(n2ca);
// 	printArray(n2CAVec);

	pp = (Pdb)pdbVec.elementAt(3); 
	resid = pp.getResidue();
	atomVec = pp.getAtomVec();
	for (j=0; j<atomVec.size(); j++){
	    cc = (Cartesian)atomVec.elementAt(j);
	    atom = cc.getAtom();
	    if (atom.equals("N"))
		amide3 = cc.getXYZ();
	    else if (atom.equals("CA"))
		ca3 = cc.getXYZ();
	    else if (atom.equals("H"))
		nh3 = cc.getXYZ();
	}
	double[] nToNHVec3 = internuclearVec(amide3, nh3);
	double[] nToCAVec3 = internuclearVec(amide3, ca3);
	Matrix rg3 = RgCal(nToNHVec3, nToCAVec3);
	Matrix rg3Inv = rg3.transpose();

	pp = (Pdb)pdbVec.elementAt(4); 
	resid = pp.getResidue();
	atomVec = pp.getAtomVec();
	for (j=0; j<atomVec.size(); j++){
	    cc = (Cartesian)atomVec.elementAt(j);
	    atom = cc.getAtom();
	    if (atom.equals("N"))
		amide4 = cc.getXYZ();
	    else if (atom.equals("CA"))
		ca4 = cc.getXYZ();
	    else if (atom.equals("H"))
		nh4 = cc.getXYZ();
	}
	double[] nToNHVec4 = internuclearVec(amide4, nh4);
	double[] nToCAVec4 = internuclearVec(amide4, ca4);
	Matrix rg4 = RgCal(nToNHVec4, nToCAVec4);
	Matrix rg4Inv = rg4.transpose();
	Vector solutionVec = new Vector();
	boolean hasSolution = phiPsiByPP14(ca1, amide4, ca4, nh4, rg1, rg4, solutionVec);
	int slnNo = solutionVec.size() / 3;
	double[] solution1 = new double[2];
	double[] solution2 = new double[2];
	double[] solution3 = new double[2];
	for (i=0; i< slnNo; i += 3){
	    solution1 =(double[])solutionVec.elementAt(i);
	    solution2 =(double[])solutionVec.elementAt(i+1);
	    solution3 =(double[])solutionVec.elementAt(i+2);
	    printArray(solution1);
	    System.out.println(phi1+"  "+psi1);
	    printArray(solution2);
	    System.out.println(phi2+"  "+psi2);
	    printArray(solution3);
	    System.out.println(phi3+"  "+psi3);
	}
	    
// 	Matrix r2yInv2 = rg1.rotationMat(phi2, "-y"); 
// 	Matrix r4zInv2 = rg1.rotationMat(psi2, "-z");//psi2+Math.PI, "-z"); 
// 	Matrix matA = Const.rot1.times(Const.r1x9yInv);
// 	double [] nh3Cal = rg1Inv.times(L1.times(r4zInv.times(matA.times(r2yInv2.times(Const.r3xInv.times(r4zInv2.times(Const.Cv)))))));
//    	double [] n2nh3 = dirCos(internuclearVec(amide3, nh3));
// //  	printArray(n2nh3);
// //  	printArray(nh3Cal);
//   	double [] n2ca3 = dirCos(internuclearVec(amide3, ca3));
// 	double[] n2ca3Cal =rg1Inv.times(L1.times(r4zInv.times(matA.times(r2yInv2.times(Const.r3xInv.times(r4zInv2.times(Const.Cu)))))));
// //  	printArray(n2ca3);
// //  	printArray(n2ca3Cal);
// 	double [] u3 = L1.transpose().times(rg1.times(n2ca3));
// 	double [] v3 = L1.transpose().times(rg1.times(n2nh3));
// 	Matrix rot3 = RgCal(v3,u3);
// 	Matrix rot1 = RgCal(Const.Cv, Const.Cu);
// 	Matrix rot = rot3.transpose().times(rot1);
// 	double [] u3Cal = rot.times(Const.Cu);
// 	double [] v3Cal = rot.times(Const.Cv);
// // 	printArray(v3);
// // 	printArray(v3Cal);
// // 	printArray(u3);
// // 	printArray(u3Cal);
// 	Matrix rCal = r4zInv.times(matA.times(r2yInv2.times(Const.r3xInv.times(r4zInv2))));
// //  	rot.print(12, 15);
// // 	rCal.print(12, 15);
// 	double [][] r = rot.getArray();
// 	double [][] a = matA.getArray();
// 	double r22Cal = a[2][0] * Math.cos(Const.theta3) * Math.sin(phi2)
// 	    + a[2][2] * Math.cos(Const.theta3) * Math.cos(phi2) 
// 	    - a[2][1] * Math.sin(Const.theta3);
// // 	System.out.println(r[2][2]+" r22  "+r22Cal);
// 	double r20Cal = (a[2][0] * Math.cos(phi2) - a[2][2] * Math.sin(phi2)) * Math.cos(psi2)
// 	    + Math.sin(psi2) * (a[2][0] * Math.sin(Const.theta3) * Math.sin(phi2) 
// 				+a[2][1]*Math.cos(Const.theta3)+a[2][2] *Math.sin(Const.theta3) *Math.cos(phi2));
// // 	System.out.println(r[2][0]+" r20  "+r20Cal);
// 	double r12Cal = (a[1][0] * Math.cos(Const.theta3) *Math.sin(phi2) - a[1][1] * Math.sin(Const.theta3)
// 			 + a[1][2]*Math.cos(Const.theta3) * Math.cos(phi2)) * Math.cos(psi1)
// 	    + Math.sin(psi1) * (a[0][0] * Math.cos(Const.theta3) * Math.sin(phi2) 
// 				-a[0][1]*Math.sin(Const.theta3)+a[0][2] *Math.cos(Const.theta3) *Math.cos(phi2));
// // 	System.out.println(r[1][2]+" r12  "+r12Cal);
//  	Vector solutionVec = phiPsiByPP13(nToCAVec, nToNHVec, nToCAVec3, nToNHVec3, phi1);
// 	double [] solutions = new double[3];
// 	for (i=0; i< solutionVec.size(); i++){
// 	    solutions = (double[])solutionVec.elementAt(i);
// 	    printArray(solutions);
// 	    psi1 = solutions[0];
// 	    phi2 = solutions[1];
// 	    psi2 = solutions[2];
// //  	    System.out.println("psi1 = "+psi1+", phi2= "+phi2+", psi2= "+psi2);	
// //  	    System.out.println("psi1 = "+psi1/ Const.cst+", phi2= "+phi2/ Const.cst+", psi2= "+psi2/Const.cst);	
// 	    r2yInv = rg1.rotationMat(phi1, "-y"); 
// 	    r4zInv = rg1.rotationMat(psi1, "-z");//psi2+Math.PI, "-z"); 
// 	    L1 = Const.r1x9yInv.times(r2yInv.times(Const.r3xInv));
// 	    r2yInv2 = rg1.rotationMat(phi2, "-y"); 
// 	    r4zInv2 = rg1.rotationMat(psi2, "-z");//psi2+Math.PI, "-z"); 
// 	    nh3Cal = rg1Inv.times(L1.times(r4zInv.times(matA.times(r2yInv2.times(Const.r3xInv.times(r4zInv2.times(Const.Cv)))))));
// 	    n2ca3Cal =rg1Inv.times(L1.times(r4zInv.times(matA.times(r2yInv2.times(Const.r3xInv.times(r4zInv2.times(Const.Cu)))))));
// // 	    printArray(nh3Cal);
// // 	    printArray(dirCos(nToNHVec3));
// // 	    printArray(n2ca3Cal);
// // 	    printArray(dirCos(nToCAVec3));
// 	}
    }
    
    /**
     * computation of psi1, phi2, psi2 given the peptide planes for
     * residue 1 an d3 and phi1 without knowing the relative  position
     * between two planes. The peptide plane specified by  N2NH and N2CA
     * vectors
     * 
     * @param u1 the n2ca vector of residue 1
     * @param v1 the n2nh vector of residue 1
     * @param n2ca3 the n2ca vector of residue 3
     * @param n2nh3 the n2nh vector of residue 3
     * @param phi1 the phi angle of residue 1.
     * 
     * @return the vector
     */
    public Vector phiPsiByPP13(double[] u1, double[] v1, double[] n2ca3, double[] n2nh3, double phi1){
	double sinTheta3 = Math.sin(Const.theta3);
	double cosTheta3 = Math.cos(Const.theta3);
	Matrix rg1 = RgCal(v1, u1);
	Matrix rg1Inv = rg1.transpose();
	Matrix r2yInv = rg1.rotationMat(phi1, "-y"); 
 	Matrix L1 = Const.r1x9yInv.times(r2yInv.times(Const.r3xInv));
	double [] u3 = L1.transpose().times(rg1.times(n2ca3));
	double [] v3 = L1.transpose().times(rg1.times(n2nh3));
	Matrix rot3 = RgCal(v3,u3);
	Matrix rot1 = RgCal(Const.Cv, Const.Cu);
	Matrix rot = rot3.transpose().times(rot1);
	double [][] r = rot.getArray();
	double [][] a = Const.matA.getArray();
	double [] psi1All = new double[2];
	double [] psi2All = new double[2];
	double [] phi2All = new double[2];
	double a1 = a[2][0] * cosTheta3; //a * sinx + b * cosx = c;
	double b1 = a[2][2] * cosTheta3;
	double c1 = r[2][2] + a[2][1] * sinTheta3;
	if (!Maths.sinAngles(a1, b1, c1, phi2All)){
//  	    System.out.println("KK 01");
	    return new Vector();	
	}
// 	double psi2a = psi2All[0]; //two solutions for psi_2
// 	double psi2b = psi2All[1];
//  	System.out.println(psi2a/Const.cst+"  "+psi2b/Const.cst);
 	double psi2 = 0.0, phi2 = 0.0, psi1 = 0.0;
	double b01, b11, b21;
	double r22Cal = 0.0, r01Cal = 0.0, r21Cal = 0.0;
	double a2, b2, c2;
	Vector solutionVec = new Vector();
	for(int i=0; i<phi2All.length; i++){
	    phi2 = phi2All[i];
	    a1 = a[2][0] * sinTheta3 * Math.sin(phi2)+a[2][1]*cosTheta3+a[2][2] *sinTheta3 *Math.cos(phi2);
	    b1 = a[2][0] * Math.cos(phi2) - a[2][2] * Math.sin(phi2);
	    c1 = r[2][0];
	    if (!Maths.sinAngles(a1, b1, c1, psi2All)){
//  		System.out.println("KK 02");
		return new Vector();
	    }
	    for (int j = 0; j<psi2All.length; j++){
		psi2 = psi2All[j];
		b11 = cosTheta3 * Math.cos(psi2);
		b01 = sinTheta3 * Math.sin(phi2) * Math.cos(psi2) - Math.cos(phi2) *  Math.sin(psi2);
		b21 = sinTheta3 * Math.cos(phi2) * Math.cos(psi2) + Math.sin(phi2) *  Math.sin(psi2);
		r21Cal = a[2][0] * b01 + a[2][1] * b11 + a[2][2] * b21;
		if (Math.abs(r21Cal - r[2][1]) < Const.eps){
		    a2 = a[0][0] * cosTheta3 * Math.sin(phi2) - a[0][1] * sinTheta3 
			+ a[0][2] * cosTheta3 * Math.cos(phi2);
		    b2 = a[1][0] * cosTheta3 * Math.sin(phi2) - a[1][1] * sinTheta3 
			+ a[1][2] * cosTheta3 * Math.cos(phi2);
		    c2 = r[1][2];
		    if (!Maths.sinAngles(a2, b2, c2, psi1All)){
//  			System.out.println("KK 03");
			return new Vector();
		    }
		    for (int k=0; k<psi1All.length; k++){
			psi1 = psi1All[k];
			r01Cal = b01 * (a[0][0] * Math.cos(psi1) - a[1][0] * Math.sin(psi1))
			    + b11 * (a[0][1] * Math.cos(psi1) - a[1][1] * Math.sin(psi1))
			    + b21 * (a[0][2] * Math.cos(psi1) - a[1][2] * Math.sin(psi1));
			if (Math.abs(r01Cal - r[0][1]) < Const.eps){
			    solutionVec.add(new double[]{psi1, phi2, psi2});
//  			    System.out.println("psi1 = "+psi1+", phi2= "+phi2+", psi2= "+psi2);	
			}
		    }		    
		}
	    }
	}
	return solutionVec;
    }

  /**
   * Method for generating coefs for the quartic equation for computing CH . suppose
   * \phi is computed from an CH RDC.
   * 
   * @param rdc Rdc data of CH vector
   * @param Syy diagonilized Saupe elements computed by SVD with both CH and NH RDCs.
   * @param Szz diagonilized Saupe elements
   * @param mat the elements of the Matrix M as appeared in the math derivation
   * @param coefs the five coefs of the quartic equation and others for checking the final solution
   * 
   * @return false if no quartic equation can be generated, otherwise, true.
   */	 
    public boolean quarticCoefPhi(double Syy, double Szz, double rdc, double[][] mat, double[] coefs){
	double a = -Syy - 2.0*Szz; //Sxx=-Syy-Szz, a1 = Sxx-Szz, from the RDC equation
	double b = Syy - Szz;      //b shoulb never be zero
	double r = rdc - Szz;      //CH RDCs
	double ca = 0.0, cb = 0.0, c1 = 0.0, c2 = 0.0, c3 = 0.0, c4=0.0, e0 = 0.0, e1 = 0.0;
	//Cx = dirCos[0], Cy = dirCos[1], Cz = dirCos[2].
	if (mat[1][2] != 0){
	    ca = dirCosCH[1] * mat[0][2] / mat[1][2];
	    cb = dirCosCH[1] * mat[2][2] / mat[1][2];
	    c1 = mat[0][0] - mat[0][2] * mat[1][0] / mat[1][2];
	    c2 = mat[0][1] - mat[0][2] * mat[1][1] / mat[1][2];
	    c3 = mat[2][0] - mat[2][2] * mat[1][0] / mat[1][2];
	    c4 = mat[2][1] - mat[2][2] * mat[1][1] / mat[1][2];
	}else{
	    System.out.println("mat[1][2] == 0.0");
	    return false;
	}
	double c0 = dirCosCH[0] * dirCosCH[0] + dirCosCH[2] * dirCosCH[2]; //C0 = Cx^2+Cz^2
	double d5 = 2*c2*ca + 2*c4*cb;
	double d4 = 2*c1*ca + 2*c3*cb;
	double d3 = 2*c1*c2 + 2*c3*c4;
	double d2 = c2*c2 + c4*c4;
	double d1 = c1*c1 + c3*c3;
	double d0 = ca*ca + cb*cb - c0;
	//To check that the above coefficients are computed correctly, specific residue
// 	System.out.println("IS xy OK "+ (d1*x*x + d2*y*y+d3*x*y + d4*x + d5*y + d0));
	if (b == 0.0 ){
	    System.out.println(" Syy == Szz !!");
	    return false;
	}else{
	    e0 = d0 + d2 * r / b; 
	    e1 = d1 - d2 * a / b;
	    coefs[0] = e1*e1 + a* d3 * d3 / b; //The five coeffs [0,4] for the quartic equation=>f4
	    coefs[1] = 2*e1*d4 +  2 * a * d3 * d5 / b;  //=>f3
	    coefs[2] = d4*d4+2*e1*e0 + (a*d5 *d5 - r*d3*d3) / b; //=>f2
	    coefs[3] = 2*d4*e0 - 2 * r * d3 * d5 / b; //=>f1
	    coefs[4] = e0*e0 - r*d5*d5 / b;   //corresponds to f0 in the file
	    coefs[5] = a;          //They are returned for convenience
	    coefs[6] = b;
	    coefs[7] = r;
	    coefs[8] = d0;
	    coefs[9] = d1;
	    coefs[10] = d2;
	    coefs[11] = d3;
	    coefs[12] = d4;
	    coefs[13] = d5;
	}
	//To check that the coefs above are computed correctly
// 	System.out.println("IS X OK "+ (coefs[0]*x*x*x*x + coefs[1]*x*x*x + coefs[2]*x*x + coefs[3]*x + coefs[4]));
	return true;
    }

    /**
     * Method for generating coefs for the quartic equation. suppose \psi is computed from an NH RDC.
     * 
     * @param rdc  Rdc data of NH vector
     * @param Syy diagonilized Saupe elements computed by SVD with both CH and NH RDCs.
     * @param Szz diagonilized Saupe elements
     * @param mat the elements of the Matrix M as appeared in the math derivation
     * @param coefs the five coefs of the quartic equation and others for checking the final solution
     * 
     * @return false if no quartic equation can be generated, otherwise. true.
     */	 
    public boolean quarticCoefPsi(double Syy, double Szz, double rdc, double[][] mat, double[] coefs){
	double a = -Syy - 2.0*Szz; //Sxx=-Syy-Szz, a1 = Sxx-Szz, from the RDC equation
	double b = Syy - Szz;
	double r = rdc - Szz;     //CH RDCs
//  	System.out.println(Math.sqrt(r/a)+"  "+Math.sqrt(r/b)); //major and minor axes
	//use the matrix element directly
	double ca = 0.0, cb = 0.0, c1 = 0.0, c2 = 0.0, c3 = 0.0, c4=0.0, e0 = 0.0, e1 = 0.0;
	//Cx = dirCos[0], Cy = dirCos[1], Cz = dirCos[2].
	if (mat[2][2] != 0){
	    ca = dirCosNH[2] * mat[0][2] / mat[2][2];
	    cb = dirCosNH[2] * mat[1][2] / mat[2][2];
	    c1 = mat[0][0] - mat[0][2] * mat[2][0] / mat[2][2];
	    c2 = mat[0][1] - mat[0][2] * mat[2][1] / mat[2][2];
	    c3 = mat[1][0] - mat[1][2] * mat[2][0] / mat[2][2];
	    c4 = mat[1][1] - mat[1][2] * mat[2][1] / mat[2][2];
	}else{
	    System.out.println(" mat[2][2] = 0.0");
	    return false;
	}
	double c0 = dirCosNH[0] * dirCosNH[0] + dirCosNH[1] * dirCosNH[1]; //C0 = Cx^2+Cy^2
	double d5 = 2*c2*ca + 2*c4*cb;
	double d4 = 2*c1*ca + 2*c3*cb;
	double d3 = 2*c1*c2 + 2*c3*c4;
	double d2 = c2*c2 + c4*c4;
	double d1 = c1*c1 + c3*c3;
	double d0 = ca*ca + cb*cb - c0;
// 	System.out.println("IS xy OK "+ (d1*x*x + d2*y*y+d3*x*y + d4*x + d5*y + d0));
	if (b == 0.0 ){
	    System.out.println(" Syy == Szz !!");
	    return false;
	}else{
	    e0 = d0 + d2 * r / b; 
	    e1 = d1 - d2 * a / b;
	    coefs[0] = e1*e1 + a* d3 * d3 / b;  //The five coeffs [0,4] for the quartic equation=>f4
	    coefs[1] = 2*e1*d4 +  2 * a * d3 * d5 / b;  //=>f3
	    coefs[2] = d4*d4 + 2*e1*e0 + (a*d5 *d5 - r*d3*d3) / b; //=>f2
	    coefs[3] = 2*d4*e0 - 2 * r * d3 * d5 / b; //=>f1
	    coefs[4] = e0*e0 - r*d5*d5 / b;   //corresponds to f0 in the file
	    coefs[5] = a;          //They are returned for convenience
	    coefs[6] = b;
	    coefs[7] = r;
	    coefs[8] = d0;
	    coefs[9] = d1;
	    coefs[10] = d2;
	    coefs[11] = d3;
	    coefs[12] = d4;
	    coefs[13] = d5;
	}
// 	System.out.println("IS X OK "+ (coefs[0]*x*x*x*x + coefs[1]*x*x*x + coefs[2]*x*x + coefs[3]*x + coefs[4]));
	return true;
    }
    
    /**
     * Method is for computing \Phi or \Psi using Two RDC in one medium by solving a Quartic eqaution
     * It differs from  the method for solving RDCs in two media .
     * solve the quartic equation by eigenValues method (cf. NR book),
     * which is at least 2-fold slower than the other method, and much
     * more slower than the exact method, return: an array of solutions
     * 
     * @param coefs coefs generated by the coef generating methods
     * 
     * @return all the real solutions.
     */
    public Vector quarticSolve(double[] coefs){
	double c1 = coefs[0];
	double c2 = coefs[1];
	double c3 = coefs[2];
	double c4 = coefs[3];
	double c0 = coefs[4];
	double a  = coefs[5];
	double b  = coefs[6];
	double r  = coefs[7];
	double d0 = coefs[8];
	double d1 = coefs[9];
	double d2 = coefs[10];
	double d3 = coefs[11];
	double d4 = coefs[12];
	double d5 = coefs[13];
	if ( b == 0.0 ){
	    System.out.println(" b == 0!!");
	    System.exit(0);
	}
// 	System.out.println("Coeefs: "+c1+"  "+c2+"  "+c3+"  "+c4+"  "+c0+"   "+d1+"   "+d2+"   "+d3+"   "+d4);
	double xr = 0.0, yr = 0.0, zr = 0.0;
	double x = 0.0, y = 0.0, z = 0.0;
	double r1Cal=0.0, r2Cal = 0.0;
	final int [] indexI  = {1, 1, 1,-1, -1,1, -1,-1};
	final int fourFold = 8; //For convenience
	final double eps1 = 1.0E-8;
	final double eps = 1.0E-7;
	Vector rootVec = new Vector(); //a root vector in PDB Class format
	Vector realVec = new Vector();
	if( c1 != 0.0){
	    double[][] mArr = { {-c2/c1, -c3/c1, -c4/c1, -c0/c1},
                                {1.0,    0.0,    0.0,    0.0},
                                {0.0,    1.0,    0.0,    0.0},
                                {0.0,    0.0,    1.0,    0.0}};
	    Matrix mm = new Matrix(mArr);
	    EigenvalueDecomposition eigs = new  EigenvalueDecomposition (mm, false);
	    double [] realValues = eigs.getRealEigenvalues();
//   	    for (int k =0; k<realValues.length; k++)   
// 		System.out.println(realValues[k]);
  	    double [] imageValues = eigs.getImagEigenvalues();
	    //Only return the real solutions. 
 	    for (int k =0; k<imageValues.length; k++){
		if (Math.abs(imageValues[k] - 0.0) < eps1)
		    realVec.add(new Double(realValues[k]));
//   		System.out.println(realValues[k]+"  "+imageValues[k]);
	    }
	    for (int i=0; i<realVec.size(); i++){
		xr = ((Double)realVec.elementAt(i)).doubleValue();
		yr = (r - a * xr * xr) / b;
		if (yr < 0.0){
//  		    System.out.println(" yr < 0, no solutions" + yr);
		    return rootVec;
// 		    System.exit(0);
		}else{
		    yr = Math.sqrt(yr);
		    if ( (xr*xr+yr*yr) <= 1.0){
			zr = Math.sqrt(1.0 - xr*xr - yr*yr);
			for (int m = 0; m < fourFold; m += 2){ //generate all the possible solutions.
			    x = xr;
			    y = indexI[m] * yr;
			    z = indexI[m+1] * zr;
			    //Check the solutions
 			    r2Cal = d1 * x*x + d2 * y*y + d3 * x*y + d4 * x + d5 * y; 
			    r1Cal = a * x*x + b * y*y;
			    if ( Math.abs(r1Cal - r ) < eps && Math.abs(r2Cal + d0 ) < eps){
				rootVec.add(new double[]{x, y, z});
//     				System.out.println(x+" ...<<  "+y+"  "+z);
			    }// else if ( Math.abs(r1Cal - r ) > eps) {
// 			    	System.out.println (Math.abs(r1Cal - r )+" ... "+ Math.abs(r2Cal + d0 ));}
			}
		    }
		}
	    }
	}
	return rootVec;
    }

    /**
     * Compute the Phi angles. This is a test method to see if it can
     * reproduce the directional cosines of the CH vector and the \phi angle.
     * 
     * @param rdc  Rdc data of CH vector
     * @param rg  the rotation matrix rotate a POF to a frame defined in the peptine plane $i$
     * @param Syy diagonilized Saupe elements computed by SVD with both CH and NH RDCs.
     * @param Szz diagonilized Saupe elements
     * 
     * @return a set of solutions for \phi angles: if the set empty ==> no solution
     */
    public Vector phiCal(double Syy, double Szz, double rdc, Matrix rg){
	Matrix M = Const.r9y1x.times(rg); //The "M" Matrix as appeared in the Math derivation
	double [][] mat = M.getArray();
 	Matrix r2y     = new Matrix(3, 3);  //M.rotationMat(phiAve, "+y"); 
	Vector phiVec  = new Vector();
	double[] coefs = new double[14]; //used for returning the coefficents from the quartic equation.
	boolean coefFlag = quarticCoefPhi(Syy,  Szz,  rdc,  mat, coefs);
	if(!coefFlag){       //can not compute coefficients
//  	    System.out.println("can not compute coefficients for the quartic equation");
	    return phiVec;
	}else{
	    Vector rootVec = quarticSolve(coefs);
//  	    System.out.println(rootVec.size());
	    if (rootVec.isEmpty()){  //no solution
//   		System.out.println("no real solution for Phi from the quartic equation "+rdc);
		return phiVec;
	    }else{ //Compute the Phi angle
		double x = 0.0, y = 0.0, z = 0.0;
		double [] bondVector = new double[3]; //three directional cosines for a bind vector
		double tmp1 = 0.0, tmp2 = 0.0, sinPhi = 0.0, cosPhi = 0.0, phi = 0.0;
		double cSquare = dirCosCH[0] * dirCosCH[0] + dirCosCH[2] * dirCosCH[2]; //Cx^2+Cz^2
// 		Matrix mat1 = new Matrix(3, 3);
		double[] bondDir = new double[3];
		final double eps = 1.0E-7;
		for (int i=0; i<rootVec.size(); i++){
		    bondVector = (double[])(rootVec.elementAt(i));
		    x = bondVector[0];
		    y = bondVector[1];
		    z = bondVector[2];
		    tmp1 = dirCosCH[0] * (mat[0][0] * x + mat[0][1] * y + mat[0][2] * z) 
			+ dirCosCH[2] * (mat[2][0] * x + mat[2][1] * y + mat[2][2] * z);

		    tmp2 = dirCosCH[2] * (mat[0][0] * x + mat[0][1] * y + mat[0][2] * z) 
			- dirCosCH[0] * (mat[2][0] * x + mat[2][1] * y + mat[2][2] * z);
		    if ( cSquare != 0.0){
			cosPhi = tmp1 / cSquare;
			sinPhi = tmp2 / cSquare;
			if ( cosPhi >= 0.0 )
			    phi = Math.asin(sinPhi);
			else if ( cosPhi < 0 )
			    phi = Math.PI - Math.asin(sinPhi);
		    } else {
			System.out.println("both Cx and Cz are zero");
			System.exit(1);
		    }
		    if (phi > Math.PI)
			phi = phi - 2.0 * Math.PI;
		    r2y = M.rotationMat(phi, "+y");  //The computed phi angle: and check the solutions
		    bondDir = (Const.rHA2HA1).times(r2y.times(M.times(bondVector)));
		    if(Math.abs(bondDir[0] - 0.0) < eps && Math.abs(bondDir[1] - 0.0) < eps 
		       && Math.abs(bondDir[2] - 1.0) < eps){
			// 			    System.out.println(x+" ...  "+y+"  "+z);
			//   			    System.out.println("Phi Values: "+phi/cst);
			phiVec.add(new Double(phi));
		    }
		}
		return  phiVec;
	    }
	}
    }

    /**
     * Compute the Phi angles. This is a test method to see if it can
     * reproduce the directional cosines of the CH vector and the \phi angle.
     * 
     * @param rdc  Rdc data of CH vector
     * @param Syy diagonilized Saupe elements computed by SVD with both CH and NH RDCs.
     * @param Szz diagonilized Saupe elements
     * @param Rg the rg
     * @param isHelix the is helix
     * 
     * @return a set of solutions for \phi angles: if the set empty ==> no solution
     */
    public Vector phiCal(double Syy, double Szz, double rdc, Matrix Rg, boolean isHelix){
	Matrix M = Const.r9y1x.times(Rg); //The "M" Matrix as appeared in the Math derivation
	double [][] mat = M.getArray();
 	Matrix r2y     = new Matrix(3, 3);  //M.rotationMat(phiAve, "+y"); 
	Vector phiVec  = new Vector();
	double phiHigh = 0.0;
	double phiLow = 0.0;
	if(isHelix){
	    phiHigh = Const.phiHighHelix;
	    phiLow  = Const.phiLowHelix;
	}else{
	    phiHigh = Const.phiHighBeta;
	    phiLow  = Const.phiLowBeta;
	}
	double[] coefs = new double[14]; //used for returning the coefficents from the quartic equation.
	boolean coefFlag = quarticCoefPhi(Syy,  Szz,  rdc,  mat, coefs);
	if(!coefFlag){       //can not compute coefficients
// 	    System.out.println("can not compute coefficients for the quartic equation");
	    return phiVec;
	}else{
	    Vector rootVec = quarticSolve(coefs);
//  	    System.out.println(rootVec.size());
	    if (rootVec.isEmpty()){  //no solution
//  		System.out.println("no real solution for Phi  from the quartic equation "+rdc);
		return phiVec;
	    }else{ //Compute the Phi angle
		double x = 0.0, y = 0.0, z = 0.0;
		double [] bondVector = new double[3]; //three directional cosines for a bind vector
		double tmp1 = 0.0, tmp2 = 0.0, sinPhi = 0.0, cosPhi = 0.0, phi = 0.0;
		double cSquare = dirCosCH[0] * dirCosCH[0] + dirCosCH[2] * dirCosCH[2]; //Cx^2+Cz^2
// 		Matrix mat1 = new Matrix(3, 3);
		double[] bondDir = new double[3];
		final double eps = 1.0E-7;
		for (int i=0; i<rootVec.size(); i++){
		    bondVector = (double[])(rootVec.elementAt(i));
		    x = bondVector[0];
		    y = bondVector[1];
		    z = bondVector[2];
		    tmp1 = dirCosCH[0] * (mat[0][0] * x + mat[0][1] * y + mat[0][2] * z) 
			+ dirCosCH[2] * (mat[2][0] * x + mat[2][1] * y + mat[2][2] * z);

		    tmp2 = dirCosCH[2] * (mat[0][0] * x + mat[0][1] * y + mat[0][2] * z) 
			- dirCosCH[0] * (mat[2][0] * x + mat[2][1] * y + mat[2][2] * z);
		    if ( cSquare != 0.0){
			cosPhi = tmp1 / cSquare;
			sinPhi = tmp2 / cSquare;
			if ( cosPhi >= 0.0 )
			    phi = Math.asin(sinPhi);
			else if ( cosPhi < 0 )
			    phi = Math.PI - Math.asin(sinPhi);
		    } else {
			System.out.println("both Cx and Cz are zero");
			System.exit(1);
		    }
		    if (phi > Math.PI)
			phi = phi - 2.0*Math.PI;
  		    if ( phi < phiHigh && phi > phiLow ){
			r2y = M.rotationMat(phi, "+y");  //The computed phi angle: and check the solutions
			bondDir = (Const.rHA2HA1).times(r2y.times(M.times(bondVector)));
 			if(Math.abs(bondDir[0] - 0.0) < eps && Math.abs(bondDir[1] - 0.0) < eps 
			   && Math.abs(bondDir[2] - 1.0) < eps){
// 			    System.out.println(x+" ...  "+y+"  "+z);
//   			    System.out.println("Phi Values: "+phi/cst);
			    phiVec.add(new Double(phi));
 			}
  		    }
		}
		return  phiVec;
	    }
	}
    }
    
    /**
     * Compute the Phi angles for loop fragment.
     * 
     * @param rdc  Rdc data of CH vector
     * @param Syy diagonilized Saupe elements computed by SVD with both CH and NH RDCs.
     * @param Szz diagonilized Saupe elements
     * @param Rg the rg
     * 
     * @return a set of solutions for \phi angles: if the set empty ==> no solution
     */
    public Vector phiCalLoop(double Syy, double Szz, double rdc, Matrix Rg)
    {
		Matrix M = Const.r9y1x.times(Rg); //The "M" Matrix as appeared in the Math derivation
		double [][] mat = M.getArray();
	 	Matrix r2y     = new Matrix(3, 3);  //M.rotationMat(phiAve, "+y"); 
		Vector phiVec  = new Vector();
		double phiHigh = 360.0;
		double phiLow = -360;
	
		double[] coefs = new double[14]; //used for returning the coefficents from the quartic equation.
		boolean coefFlag = quarticCoefPhi(Syy,  Szz,  rdc,  mat, coefs);
		if(!coefFlag)
		{       //can not compute coefficients
			return phiVec;
		}else
		{
		    Vector rootVec = quarticSolve(coefs);
		    if (rootVec.isEmpty())
		    {  //no solution
		    	//  		System.out.println("no real solution for Phi  from the quartic equation "+rdc);
		    	return phiVec;
		    }else
		    { //Compute the Phi angle
				double x = 0.0, y = 0.0, z = 0.0;
				double [] bondVector = new double[3]; //three directional cosines for a bind vector
				double tmp1 = 0.0, tmp2 = 0.0, sinPhi = 0.0, cosPhi = 0.0, phi = 0.0;
				double cSquare = dirCosCH[0] * dirCosCH[0] + dirCosCH[2] * dirCosCH[2]; //Cx^2+Cz^2
				// 		Matrix mat1 = new Matrix(3, 3);
				double[] bondDir = new double[3];
				final double eps = 1.0E-7;
				for (int i=0; i<rootVec.size(); i++)
				{
				    bondVector = (double[])(rootVec.elementAt(i));
				    x = bondVector[0];
				    y = bondVector[1];
				    z = bondVector[2];
				    tmp1 = dirCosCH[0] * (mat[0][0] * x + mat[0][1] * y + mat[0][2] * z) 
					+ dirCosCH[2] * (mat[2][0] * x + mat[2][1] * y + mat[2][2] * z);
		
				    tmp2 = dirCosCH[2] * (mat[0][0] * x + mat[0][1] * y + mat[0][2] * z) 
					- dirCosCH[0] * (mat[2][0] * x + mat[2][1] * y + mat[2][2] * z);
				    if ( cSquare != 0.0)
				    {
						cosPhi = tmp1 / cSquare;
						sinPhi = tmp2 / cSquare;
						if ( cosPhi >= 0.0 )
						    phi = Math.asin(sinPhi);
						else if ( cosPhi < 0 )
						    phi = Math.PI - Math.asin(sinPhi);
					} 
				    else 
				    {
						System.out.println("both Cx and Cz are zero");
						System.exit(1);
					 }
				    if (phi > Math.PI)
					phi = phi - 2.0*Math.PI;
		  		    if ( phi < phiHigh && phi > phiLow )
		  		    {
						r2y = M.rotationMat(phi, "+y");  //The computed phi angle: and check the solutions
						bondDir = (Const.rHA2HA1).times(r2y.times(M.times(bondVector)));
			 			if(Math.abs(bondDir[0] - 0.0) < eps && Math.abs(bondDir[1] - 0.0) < eps 
						   && Math.abs(bondDir[2] - 1.0) < eps)
			 			{		
						    phiVec.add(new Double(phi));
			 			}
		  		    }
				}
				return  phiVec;
		    }
		}
    }

    /**
     * Compute the Phi angles. This is a test method to see if it can
     * reproduce the directional cosines of the CH vector and the \phi angle.
     * 
     * @param rdc  Rdc data of CH vector
     * @param Syy diagonilized Saupe elements computed by SVD with both CH and NH RDCs.
     * @param Szz diagonilized Saupe elements
     * @param Rg the rg
     * @param isHelix the is helix
     * @param curResNo the cur res no
     * @param vecTalos the vec talos
     * 
     * @return a set of solutions for \phi angles: if the set empty ==> no solution
     */
    public Vector phiCal(double Syy, double Szz, double rdc, Matrix Rg, boolean isHelix, int curResNo, Vector vecTalos){
	Matrix M = Const.r9y1x.times(Rg); //The "M" Matrix as appeared in the Math derivation
	double [][] mat = M.getArray();
 	Matrix r2y     = new Matrix(3, 3);  //M.rotationMat(phiAve, "+y"); 
	Vector phiVec  = new Vector();
	double phiHigh = 0.0;
	double phiLow = 0.0;
	double phiRamHigh=0.0;
	double phiRamLower=0.0;
	if(isHelix){
		phiRamHigh = Const.phiHighHelix;
		phiRamLower  = Const.phiLowHelix;
	}else{
		phiRamHigh = Const.phiHighBeta;
		phiRamLower  = Const.phiLowBeta;
	}
	boolean isInTalos=false;
	for (int i=0; i<vecTalos.size();i++)
	{
		PhiPsi phipsi=(PhiPsi)vecTalos.elementAt(i);
		int resNo_temp=phipsi.getResidueNo();
		if(resNo_temp==curResNo)
		{
			phiHigh=Math.min( (phipsi.getPhiUpper())*Const.cst,phiRamHigh);
			//phiHigh=phiHigh*Const.cst;
			phiLow=Math.max( (phipsi.getPhiLower())*Const.cst,phiRamLower);
			//phiLow=phiLow*Const.cst;
			isInTalos=true;
			break;
		}		
	}//for (int i=0; i<vecTalos.size();i++)
	
	if(!isInTalos)
	{
		phiHigh=phiRamHigh;
		phiLow=phiRamLower;	
	}
	
	double[] coefs = new double[14]; //used for returning the coefficents from the quartic equation.
	boolean coefFlag = quarticCoefPhi(Syy,  Szz,  rdc,  mat, coefs);
	if(!coefFlag){       //can not compute coefficients
// 	    System.out.println("can not compute coefficients for the quartic equation");
	    return phiVec;
	}else{
	    Vector rootVec = quarticSolve(coefs);
//  	    System.out.println(rootVec.size());
	    if (rootVec.isEmpty()){  //no solution
//  		System.out.println("no real solution for Phi  from the quartic equation "+rdc);
		return phiVec;
	    }else{ //Compute the Phi angle
		double x = 0.0, y = 0.0, z = 0.0;
		double [] bondVector = new double[3]; //three directional cosines for a bind vector
		double tmp1 = 0.0, tmp2 = 0.0, sinPhi = 0.0, cosPhi = 0.0, phi = 0.0;
		double cSquare = dirCosCH[0] * dirCosCH[0] + dirCosCH[2] * dirCosCH[2]; //Cx^2+Cz^2
// 		Matrix mat1 = new Matrix(3, 3);
		double[] bondDir = new double[3];
		final double eps = 1.0E-7;
		for (int i=0; i<rootVec.size(); i++){
		    bondVector = (double[])(rootVec.elementAt(i));
		    x = bondVector[0];
		    y = bondVector[1];
		    z = bondVector[2];
		    tmp1 = dirCosCH[0] * (mat[0][0] * x + mat[0][1] * y + mat[0][2] * z) 
			+ dirCosCH[2] * (mat[2][0] * x + mat[2][1] * y + mat[2][2] * z);

		    tmp2 = dirCosCH[2] * (mat[0][0] * x + mat[0][1] * y + mat[0][2] * z) 
			- dirCosCH[0] * (mat[2][0] * x + mat[2][1] * y + mat[2][2] * z);
		    if ( cSquare != 0.0){
			cosPhi = tmp1 / cSquare;
			sinPhi = tmp2 / cSquare;
			if ( cosPhi >= 0.0 )
			    phi = Math.asin(sinPhi);
			else if ( cosPhi < 0 )
			    phi = Math.PI - Math.asin(sinPhi);
		    } else {
			System.out.println("both Cx and Cz are zero");
			System.exit(1);
		    }
		    if (phi > Math.PI)
			phi = phi - 2.0*Math.PI;
  		    if ( phi < phiHigh && phi > phiLow ){
			r2y = M.rotationMat(phi, "+y");  //The computed phi angle: and check the solutions
			bondDir = (Const.rHA2HA1).times(r2y.times(M.times(bondVector)));
 			if(Math.abs(bondDir[0] - 0.0) < eps && Math.abs(bondDir[1] - 0.0) < eps 
			   && Math.abs(bondDir[2] - 1.0) < eps){
// 			    System.out.println(x+" ...  "+y+"  "+z);
//   			    System.out.println("Phi Values: "+phi/cst);
			    phiVec.add(new Double(phi));
 			}
  		    }
		}
		return  phiVec;
	    }
	}
    }


    /**
     * Compute the Psi angles. This is a test method to see if it can
     * reproduce the directional cosines of the CH vector and the \phi angle.
     * 
     * @param rdc  Rdc data of NH vector
     * @param phi  the computed \phi value for the same residue
     * @param rg  the rotation matrix rotate a POF to a frame defined in the peptine plane $i$
     * @param Syy diagonilized Saupe elements computed by SVD with both CH and NH RDCs.
     * @param Szz diagonilized Saupe elements
     * 
     * @return a set of solutions for \psi angles: if the set empty ==> no solution
     */
    public Vector psiCal(double Syy, double Szz, double rdc, double phi, Matrix rg){
	Matrix r2y = rg.rotationMat(phi, "+y"); 
	//The Matrix M as appeared in the Mth derivation
	Matrix M = Const.r3x.times(r2y.times(Const.r9y1x.times(rg))); 
	double [][] mat = M.getArray();
	double[] coefs = new double[14];
	Vector psiVec = new Vector();
	final double eps = 1.0E-7;
	boolean coefFlag = quarticCoefPsi(Syy,  Szz,  rdc,  mat,  coefs);
	if(!coefFlag){ //can not compute coefficients
	    System.out.println("can not compute coefficients for the quartic equation");
	    return psiVec;
	}else{
	    Vector rootVec = quarticSolve(coefs);
// 	    System.out.println(rootVec.size());
	    if ( rootVec.isEmpty() ){  //no solution
//   		System.out.println("no real solution for Psi from the quartic equation  "+rdc);
		return psiVec;
	    }else{
		double x = 0.0, y = 0.0, z = 0.0;
		double [] bondVector = new double[3]; //three directional cosines for a bind vector
		double tmp1 = 0.0, tmp2 = 0.0, sinPsi = 0.0, cosPsi = 0.0,  psi=0.0;
		double cSquare = dirCosNH[0] * dirCosNH[0] + dirCosNH[1] * dirCosNH[1];
// 		Matrix mat1 = new Matrix(3,3);
		Matrix rPsiz = new Matrix(3,3);
		double[] bondDir = new double[3];
// 		final double eps = 1.0E-8;
		for (int i=0; i<rootVec.size(); i++){
		    bondVector = (double[])(rootVec.elementAt(i)); //IN this case it is NH vector
		    x = bondVector[0];
		    y = bondVector[1];
		    z = bondVector[2];
		    tmp1 = dirCosNH[0] * (mat[0][0] * x + mat[0][1] * y + mat[0][2] * z) 
			+ dirCosNH[1] * (mat[1][0] * x + mat[1][1] * y + mat[1][2] * z);
		    tmp2 = dirCosNH[1] * (mat[0][0] * x + mat[0][1] * y + mat[0][2] * z) 
			- dirCosNH[0] * (mat[1][0] * x + mat[1][1] * y + mat[1][2] * z);
		    if ( cSquare != 0.0){
			sinPsi = tmp2 / cSquare;
			cosPsi = -tmp1 / cSquare;
			if ( cosPsi >= 0.0 )
			    psi = Math.asin(sinPsi);
			else if ( cosPsi < 0 )
			    psi = Math.PI - Math.asin(sinPsi);
		    }else{
			System.out.println("Both Cx or Cz are zero");
			System.exit(1);
		    }
		    if (psi < -Math.PI)  //in order to bring \Psi angle to be in [0, PI] for Beta, Sheet
			psi = psi + 2.0*Math.PI;
		    //  			System.out.println("Psi Values: "+ (psi / Const.cst));
		    rPsiz = M.rotationMat(psi + Math.PI, "+z");  //check the solutions before squaring
		    bondDir = (Const.r7x6y5x).times(rPsiz.times(M.times(bondVector)));			
		    if(Math.abs(bondDir[0] - 0.0) < eps && Math.abs(bondDir[1] - 0.0) < eps 
		       && Math.abs(bondDir[2] + 1.0) < eps){
			//System.out.println(bondDir[0]+" Kack  "+bondDir[1]+"   "+bondDir[2]);
			//  			    System.out.println("Psi Values: "+psi/cst);
			psiVec.add(new Double(psi));
		    }
		}
		return  psiVec;
	    }
	}
    }

   /**
    * Compute the Psi angles. This is a test method to see if it can
    * reproduce the directional cosines of the CH vector and the \phi angle.
    * 
    * @param rdc  Rdc data of NH vector
    * @param phi  the computed \phi value for the same residue
    * @param rg  the rotation matrix rotate a POF to a frame defined in the peptine plane $i$
    * @param Syy diagonilized Saupe elements computed by SVD with both CH and NH RDCs.
    * @param Szz diagonilized Saupe elements
    * @param isHelix the is helix
    * 
    * @return a set of solutions for \psi angles: if the set empty ==> no solution
    */
    public Vector psiCal(double Syy, double Szz, double rdc, double phi, Matrix rg, boolean isHelix){
	Matrix r2y = rg.rotationMat(phi, "+y"); 
	//The Matrix M as appeared in the Mth derivation
	Matrix M = Const.r3x.times(r2y.times(Const.r9y1x.times(rg)));
	double [][] mat = M.getArray();
	double[] coefs = new double[14];
	Vector psiVec = new Vector();
	final double eps = 1.0E-7;
	double psiHigh = 0.0;
	double psiLow = 0.0;
	if(isHelix){
	    psiHigh = Const.psiHighHelix;
	    psiLow  = Const.psiLowHelix;
	}else{
	    psiHigh = Const.psiHighBeta;
	    psiLow  = Const.psiLowBeta;
	}
	boolean coefFlag = quarticCoefPsi(Syy,  Szz,  rdc,  mat,  coefs);
	if(!coefFlag){ //can not compute coefficients
	    System.out.println("can not compute coefficients for the quartic equation");
	    return psiVec;
	}else{
	    Vector rootVec = quarticSolve(coefs);
// 	    System.out.println(rootVec.size());
	    if ( rootVec.isEmpty() ){  //no solution
//   		System.out.println("no real solution for Psi from the quartic equation  "+rdc);
		return psiVec;
	    }else{
		double x = 0.0, y = 0.0, z = 0.0;
		double [] bondVector = new double[3]; //three directional cosines for a bind vector
		double tmp1 = 0.0, tmp2 = 0.0, sinPsi = 0.0, cosPsi = 0.0,  psi=0.0;
		double cSquare = dirCosNH[0] * dirCosNH[0] + dirCosNH[1] * dirCosNH[1];
// 		Matrix mat1 = new Matrix(3,3);
		Matrix rPsiz = new Matrix(3,3);
		double[] bondDir = new double[3];
// 		final double eps = 1.0E-8;
		for (int i=0; i<rootVec.size(); i++){
		    bondVector = (double[])(rootVec.elementAt(i)); //IN this case it is NH vector
		    x = bondVector[0];
		    y = bondVector[1];
		    z = bondVector[2];
		    tmp1 = dirCosNH[0] * (mat[0][0] * x + mat[0][1] * y + mat[0][2] * z) 
			 + dirCosNH[1] * (mat[1][0] * x + mat[1][1] * y + mat[1][2] * z);
		    tmp2 = dirCosNH[1] * (mat[0][0] * x + mat[0][1] * y + mat[0][2] * z) 
			 - dirCosNH[0] * (mat[1][0] * x + mat[1][1] * y + mat[1][2] * z);
		    if ( cSquare != 0.0){
			sinPsi = tmp2 / cSquare;
			cosPsi = -tmp1 / cSquare;
			if ( cosPsi >= 0.0 )
			    psi = Math.asin(sinPsi);
			else if ( cosPsi < 0 )
			    psi = Math.PI - Math.asin(sinPsi);
		    }else{
			System.out.println("Both Cx or Cz are zero");
			System.exit(1);
		    }
		    if (psi < -Math.PI)  //in order to bring \Psi angle to be in [0, PI] for Beta, Sheet
			psi = psi + 2.0*Math.PI;
//  			System.out.println("Psi Values: "+ (psi / Const.cst));
		    //here we prune bad angles using ramachandran plot
 		    if ( psi < psiHigh && psi > psiLow){
			rPsiz = M.rotationMat(psi + Math.PI, "+z");  //check the solutions before squaring
			bondDir = (Const.r7x6y5x).times(rPsiz.times(M.times(bondVector)));			
			if(Math.abs(bondDir[0] - 0.0) < eps && Math.abs(bondDir[1] - 0.0) < eps 
			   && Math.abs(bondDir[2] + 1.0) < eps){
			    //System.out.println(bondDir[0]+" Kack  "+bondDir[1]+"   "+bondDir[2]);
//  			    System.out.println("Psi Values: "+psi/cst);
			    psiVec.add(new Double(psi));
 			}
		    }
		}
		return  psiVec;
	    }
	}
    }
    
    /**
     * Compute the Psi angles for loops.
     * 
     * @param rdc  Rdc data of NH vector
     * @param phi  the computed \phi value for the same residue
     * @param rg  the rotation matrix rotate a POF to a frame defined in the peptine plane $i$
     * @param Syy diagonilized Saupe elements computed by SVD with both CH and NH RDCs.
     * @param Szz diagonilized Saupe elements
     * 
     * @return a set of solutions for \psi angles: if the set empty ==> no solution
     */
    public Vector psiCalLoop(double Syy, double Szz, double rdc, double phi, Matrix rg){
	Matrix r2y = rg.rotationMat(phi, "+y"); 
	//The Matrix M as appeared in the Mth derivation
	Matrix M = Const.r3x.times(r2y.times(Const.r9y1x.times(rg)));
	double [][] mat = M.getArray();
	double[] coefs = new double[14];
	Vector psiVec = new Vector();
	final double eps = 1.0E-7;
	double psiHigh = 360.0;
	double psiLow = -360.0;
	
	boolean coefFlag = quarticCoefPsi(Syy,  Szz,  rdc,  mat,  coefs);
	if(!coefFlag){ //can not compute coefficients
	    System.out.println("can not compute coefficients for the quartic equation");
	    return psiVec;
	}else{
	    Vector rootVec = quarticSolve(coefs);
// 	    System.out.println(rootVec.size());
	    if ( rootVec.isEmpty() ){  //no solution
//   		System.out.println("no real solution for Psi from the quartic equation  "+rdc);
		return psiVec;
	    }else{
		double x = 0.0, y = 0.0, z = 0.0;
		double [] bondVector = new double[3]; //three directional cosines for a bind vector
		double tmp1 = 0.0, tmp2 = 0.0, sinPsi = 0.0, cosPsi = 0.0,  psi=0.0;
		double cSquare = dirCosNH[0] * dirCosNH[0] + dirCosNH[1] * dirCosNH[1];
// 		Matrix mat1 = new Matrix(3,3);
		Matrix rPsiz = new Matrix(3,3);
		double[] bondDir = new double[3];
// 		final double eps = 1.0E-8;
		for (int i=0; i<rootVec.size(); i++){
		    bondVector = (double[])(rootVec.elementAt(i)); //IN this case it is NH vector
		    x = bondVector[0];
		    y = bondVector[1];
		    z = bondVector[2];
		    tmp1 = dirCosNH[0] * (mat[0][0] * x + mat[0][1] * y + mat[0][2] * z) 
			 + dirCosNH[1] * (mat[1][0] * x + mat[1][1] * y + mat[1][2] * z);
		    tmp2 = dirCosNH[1] * (mat[0][0] * x + mat[0][1] * y + mat[0][2] * z) 
			 - dirCosNH[0] * (mat[1][0] * x + mat[1][1] * y + mat[1][2] * z);
		    if ( cSquare != 0.0){
			sinPsi = tmp2 / cSquare;
			cosPsi = -tmp1 / cSquare;
			if ( cosPsi >= 0.0 )
			    psi = Math.asin(sinPsi);
			else if ( cosPsi < 0 )
			    psi = Math.PI - Math.asin(sinPsi);
		    }else{
			System.out.println("Both Cx or Cz are zero");
			System.exit(1);
		    }
		    if (psi < -Math.PI)  //in order to bring \Psi angle to be in [0, PI] for Beta, Sheet
			psi = psi + 2.0*Math.PI;
//  			System.out.println("Psi Values: "+ (psi / Const.cst));
		    //here we prune bad angles using ramachandran plot
 		    if ( psi < psiHigh && psi > psiLow){
			rPsiz = M.rotationMat(psi + Math.PI, "+z");  //check the solutions before squaring
			bondDir = (Const.r7x6y5x).times(rPsiz.times(M.times(bondVector)));			
			if(Math.abs(bondDir[0] - 0.0) < eps && Math.abs(bondDir[1] - 0.0) < eps 
			   && Math.abs(bondDir[2] + 1.0) < eps){
			    //System.out.println(bondDir[0]+" Kack  "+bondDir[1]+"   "+bondDir[2]);
//  			    System.out.println("Psi Values: "+psi/cst);
			    psiVec.add(new Double(psi));
 			}
		    }
		}
		return  psiVec;
	    }
	}
    }
    
    /**
     * Compute the Psi angles. This is a test method to see if it can
     * reproduce the directional cosines of the CH vector and the \phi angle.
     * 
     * @param rdc  Rdc data of NH vector
     * @param phi  the computed \phi value for the same residue
     * @param rg  the rotation matrix rotate a POF to a frame defined in the peptine plane $i$
     * @param Syy diagonilized Saupe elements computed by SVD with both CH and NH RDCs.
     * @param Szz diagonilized Saupe elements
     * @param isHelix the is helix
     * @param curResNo the cur res no
     * @param vecTalos the vec talos
     * 
     * @return a set of solutions for \psi angles: if the set empty ==> no solution
     */
    public Vector psiCal(double Syy, double Szz, double rdc, double phi, Matrix rg, boolean isHelix,int curResNo, Vector vecTalos){
	Matrix r2y = rg.rotationMat(phi, "+y"); 
	//The Matrix M as appeared in the Mth derivation
	Matrix M = Const.r3x.times(r2y.times(Const.r9y1x.times(rg)));
	double [][] mat = M.getArray();
	double[] coefs = new double[14];
	Vector psiVec = new Vector();
	final double eps = 1.0E-7;
	double psiHigh = 0.0;
	double psiLow = 0.0;
	double psiRamHigh=0.0;
	double psiRamLower=0.0;
	if(isHelix){
		psiRamHigh = Const.psiHighHelix;
		psiRamLower  = Const.psiLowHelix;
	}else{
		psiRamHigh = Const.psiHighBeta;
		psiRamLower  = Const.psiLowBeta;
	}
	
	boolean isInTalos=false;
	for (int i=0; i<vecTalos.size();i++)
	{
		PhiPsi phipsi=(PhiPsi)vecTalos.elementAt(i);
		int resNo_temp=phipsi.getResidueNo();
		if(resNo_temp==curResNo)
		{
			psiHigh=Math.min( (phipsi.getPsiUpper()+0.0)*Const.cst,psiRamHigh);
			//psiHigh=psiHigh*Const.cst;
			psiLow=Math.max( (phipsi.getPsiLower()-0.0)*Const.cst,psiRamLower);
			//psiLow=psiLow*Const.cst;
			isInTalos=true;
			break;
		}		
	}//for (int i=0; i<vecTalos.size();i++)
	if(!isInTalos)
	{
		psiHigh=psiRamHigh;
		psiLow=psiRamLower;	
	}
	
	
	
	boolean coefFlag = quarticCoefPsi(Syy,  Szz,  rdc,  mat,  coefs);
	if(!coefFlag){ //can not compute coefficients
	    System.out.println("can not compute coefficients for the quartic equation");
	    return psiVec;
	}else{
	    Vector rootVec = quarticSolve(coefs);
// 	    System.out.println(rootVec.size());
	    if ( rootVec.isEmpty() ){  //no solution
//   		System.out.println("no real solution for Psi from the quartic equation  "+rdc);
		return psiVec;
	    }else{
		double x = 0.0, y = 0.0, z = 0.0;
		double [] bondVector = new double[3]; //three directional cosines for a bind vector
		double tmp1 = 0.0, tmp2 = 0.0, sinPsi = 0.0, cosPsi = 0.0,  psi=0.0;
		double cSquare = dirCosNH[0] * dirCosNH[0] + dirCosNH[1] * dirCosNH[1];
// 		Matrix mat1 = new Matrix(3,3);
		Matrix rPsiz = new Matrix(3,3);
		double[] bondDir = new double[3];
// 		final double eps = 1.0E-8;
		for (int i=0; i<rootVec.size(); i++){
		    bondVector = (double[])(rootVec.elementAt(i)); //IN this case it is NH vector
		    x = bondVector[0];
		    y = bondVector[1];
		    z = bondVector[2];
		    tmp1 = dirCosNH[0] * (mat[0][0] * x + mat[0][1] * y + mat[0][2] * z) 
			 + dirCosNH[1] * (mat[1][0] * x + mat[1][1] * y + mat[1][2] * z);
		    tmp2 = dirCosNH[1] * (mat[0][0] * x + mat[0][1] * y + mat[0][2] * z) 
			 - dirCosNH[0] * (mat[1][0] * x + mat[1][1] * y + mat[1][2] * z);
		    if ( cSquare != 0.0){
			sinPsi = tmp2 / cSquare;
			cosPsi = -tmp1 / cSquare;
			if ( cosPsi >= 0.0 )
			    psi = Math.asin(sinPsi);
			else if ( cosPsi < 0 )
			    psi = Math.PI - Math.asin(sinPsi);
		    }else{
			System.out.println("Both Cx or Cz are zero");
			System.exit(1);
		    }
		    if (psi < -Math.PI)  //in order to bring \Psi angle to be in [0, PI] for Beta, Sheet
			psi = psi + 2.0*Math.PI;
//  			System.out.println("Psi Values: "+ (psi / Const.cst));
		    //here we prune bad angles using ramachandran plot
 		    if ( psi < psiHigh && psi > psiLow){
			rPsiz = M.rotationMat(psi + Math.PI, "+z");  //check the solutions before squaring
			bondDir = (Const.r7x6y5x).times(rPsiz.times(M.times(bondVector)));			
			if(Math.abs(bondDir[0] - 0.0) < eps && Math.abs(bondDir[1] - 0.0) < eps 
			   && Math.abs(bondDir[2] + 1.0) < eps){
			    //System.out.println(bondDir[0]+" Kack  "+bondDir[1]+"   "+bondDir[2]);
//  			    System.out.println("Psi Values: "+psi/cst);
			    psiVec.add(new Double(psi));
 			}
		    }
		}
		return  psiVec;
	    }
	}
    }

  /**
   * Method for generating coefs for the quartic equation for
   * computing phi_i from NH RDCs for residue i by working backwards
   * from the residue i+1. The method is identical to that for the
   * forward method but the coefficients (the matrix M and Cs)  differ.
   * 
   * @param rdc Rdc data of NH vector
   * @param Syy diagonilized Saupe elements computed by SVD with both CH and NH RDCs.
   * @param Szz diagonilized Saupe elements
   * @param mat the elements of the Matrix M as appeared in the math derivation
   * @param coefs the five coefs of the quartic equation and others for checking the final solution
   * 
   * @return false if no quartic equation can be generated, otherwise, true.
   */	 
    public boolean coefPhiBackward(double Syy, double Szz, double rdc, double[][] mat, double[] coefs){
	double a = -Syy - 2.0*Szz; //Sxx=-Syy-Szz, a1 = Sxx-Szz, from the RDC equation
	double b = Syy - Szz;      //b shoulb never be zero
	double r = rdc - Szz;      //CH RDCs
	double ca = 0.0, cb = 0.0, c1 = 0.0, c2 = 0.0, c3 = 0.0, c4=0.0, e0 = 0.0, e1 = 0.0;
	//Cx = 0.0, Cy = -Const.dN2H *Math.sin(Const.alpha5), Cz = Const.dN2H *Math.cos(Const.alpha5);
	double cy = Const.phiCnt[1]; // -Math.sin(Const.alpha5);
	double cz = Const.phiCnt[2]; //Math.cos(Const.alpha5);
	double x = 0.3913496747972713,  y = -0.44891450738046385,  z = -0.8033188639011445;
	if (mat[1][2] != 0){
	    ca = cy * mat[0][2] / mat[1][2];
	    cb = cy * mat[2][2] / mat[1][2];
	    c1 = mat[0][0] - mat[0][2] * mat[1][0] / mat[1][2];
	    c2 = mat[0][1] - mat[0][2] * mat[1][1] / mat[1][2];
	    c3 = mat[2][0] - mat[2][2] * mat[1][0] / mat[1][2];
	    c4 = mat[2][1] - mat[2][2] * mat[1][1] / mat[1][2];
	}else{
	    System.out.println("mat[1][2] == 0.0");
	    return false;
	}
	double c0 = cz * cz; //C0 = Cx^2+Cz^2 and Cx = 0, Cz = Const.dN2H *  Math.cos(Const.alpha5)
	double d5 = 2*c2*ca + 2*c4*cb;
	double d4 = 2*c1*ca + 2*c3*cb;
	double d3 = 2*c1*c2 + 2*c3*c4;
	double d2 = c2*c2 + c4*c4;
	double d1 = c1*c1 + c3*c3;
	double d0 = ca*ca + cb*cb - c0;
	//To check that the above coefficients are computed correctly, specific residue
//  	System.out.println("IS xy OK "+ (d1*x*x + d2*y*y+d3*x*y + d4*x + d5*y + d0));
	if (b == 0.0 ){
	    System.out.println(" Syy == Szz !!");
	    return false;
	}else{
	    e0 = d0 + d2 * r / b; 
	    e1 = d1 - d2 * a / b;
	    coefs[0] = e1*e1 + a* d3 * d3 / b; //The five coeffs [0,4] for the quartic equation=>f4
	    coefs[1] = 2*e1*d4 +  2 * a * d3 * d5 / b;  //=>f3
	    coefs[2] = d4*d4+2*e1*e0 + (a*d5 *d5 - r*d3*d3) / b; //=>f2
	    coefs[3] = 2*d4*e0 - 2 * r * d3 * d5 / b; //=>f1
	    coefs[4] = e0*e0 - r*d5*d5 / b;   //corresponds to f0 in the file
	    coefs[5] = a;          //They are returned for convenience
	    coefs[6] = b;
	    coefs[7] = r;
	    coefs[8] = d0;
	    coefs[9] = d1;
	    coefs[10] = d2;
	    coefs[11] = d3;
	    coefs[12] = d4;
	    coefs[13] = d5;
	}
	//To check that the coefs above are computed correctly
//  	System.out.println("IS X OK "+ (coefs[0]*x*x*x*x + coefs[1]*x*x*x + coefs[2]*x*x + coefs[3]*x + coefs[4]));
	return true;
    }

    /**
     * Method for generating coefs for the quartic equation. suppose
     * \psi_i is computed from an = CH RDC of residue i.
     * 
     * @param rdc  Rdc data of CH vector of residue i
     * @param Syy diagonilized Saupe elements computed by SVD with both CH and NH RDCs.
     * @param Szz diagonilized Saupe elements
     * @param mat the elements of the Matrix M as appeared in the math derivation
     * @param coefs the five coefs of the quartic equation and others for checking the final solution
     * 
     * @return false if no quartic equation can be generated, otherwise. true.
     */	 
    public boolean coefPsiBackward(double Syy, double Szz, double rdc, double[][] mat, double[] coefs){
	double a = -Syy - 2.0*Szz; //Sxx=-Syy-Szz, a1 = Sxx-Szz, from the RDC equation
	double b = Syy - Szz;
	double r = rdc - Szz;     //CH RDCs
//  	System.out.println(Math.sqrt(r/a)+"  "+Math.sqrt(r/b)); //major and minor axes
	//use the matrix element directly
	double ca = 0.0, cb = 0.0, c1 = 0.0, c2 = 0.0, c3 = 0.0, c4=0.0, e0 = 0.0, e1 = 0.0;
// 	cx = Const.psiCnt[0], double cy = Const.psiCnt[1], cz = Const.psiCnt[2];
// 	double x = 0.16456652307757394, y =   -0.0909976281067454, z = 0.9821595039305504;
// 	double x=-0.8854602755021399, y= 0.291864654083072, z = -0.3616284338995532;
	if (mat[2][2] != 0){
	    ca = Const.psiCntUnit[2] * mat[0][2] / mat[2][2];
	    cb = Const.psiCntUnit[2] * mat[1][2] / mat[2][2];
	    c1 = mat[0][0] - mat[0][2] * mat[2][0] / mat[2][2];
	    c2 = mat[0][1] - mat[0][2] * mat[2][1] / mat[2][2];
	    c3 = mat[1][0] - mat[1][2] * mat[2][0] / mat[2][2];
	    c4 = mat[1][1] - mat[1][2] * mat[2][1] / mat[2][2];
	}else{
	    System.out.println(" mat[2][2] = 0.0");
	    return false;
	}
	double c0 = Const.psiCntUnit[0] * Const.psiCntUnit[0] + Const.psiCntUnit[1] * Const.psiCntUnit[1]; //C0 = Cx^2+Cy^2
	double d5 = 2*c2*ca + 2*c4*cb;
	double d4 = 2*c1*ca + 2*c3*cb;
	double d3 = 2*c1*c2 + 2*c3*c4;
	double d2 = c2*c2 + c4*c4;
	double d1 = c1*c1 + c3*c3;
	double d0 = ca*ca + cb*cb - c0;
//   	System.out.println("IS xy OK "+ (d1*x*x + d2*y*y+d3*x*y + d4*x + d5*y + d0));
	if (b == 0.0 ){
	    System.out.println(" Syy == Szz !!");
	    return false;
	}else{
	    e0 = d0 + d2 * r / b; 
	    e1 = d1 - d2 * a / b;
	    coefs[0] = e1*e1 + a* d3 * d3 / b;  //The five coeffs [0,4] for the quartic equation=>f4
	    coefs[1] = 2*e1*d4 +  2 * a * d3 * d5 / b;  //=>f3
	    coefs[2] = d4*d4 + 2*e1*e0 + (a*d5 *d5 - r*d3*d3) / b; //=>f2
	    coefs[3] = 2*d4*e0 - 2 * r * d3 * d5 / b; //=>f1
	    coefs[4] = e0*e0 - r*d5*d5 / b;   //corresponds to f0 in the file
	    coefs[5] = a;          //They are returned for convenience
	    coefs[6] = b;
	    coefs[7] = r;
	    coefs[8] = d0;
	    coefs[9] = d1;
	    coefs[10] = d2;
	    coefs[11] = d3;
	    coefs[12] = d4;
	    coefs[13] = d5;
	}
//   	System.out.println("IS X OK2 "+ (coefs[0]*x*x*x*x + coefs[1]*x*x*x + coefs[2]*x*x + coefs[3]*x + coefs[4]));
	return true;
    }

    /**
     * Compute the Phi angles. This is a test method to see if it can
     * reproduce the directional cosines of the CH vector and the \phi angle.
     * 
     * @param rdc  Rdc data of CH vector
     * @param rg  the rotation matrix rotate a POF to a frame defined in the peptine plane $i+1$
     * @param Syy diagonilized Saupe elements computed by SVD with both CH and NH RDCs.
     * @param Szz diagonilized Saupe elements
     * 
     * @return a set of solutions for \phi angles: if the set empty ==> no solution
     */
    public Vector psiCalBackward(double Syy, double Szz, double rdc, Matrix rg)
    {
		Matrix M = Const.r2x6z1x9zAlpha.times(rg); //The "M" Matrix as appeared in the Math derivation
		double [][] mat = M.getArray();
		Vector psiVec  = new Vector();
		double[] coefs = new double[14]; //used for returning the coefficents from the quartic equation.
		boolean coefFlag = coefPsiBackward(Syy,  Szz,  rdc,  mat, coefs);

		if(!coefFlag)
		{       //can not compute coefficients
	 	    System.out.println("can not compute coefficients for the quartic equation");
		    return psiVec;
		}else
		{
		    Vector rootVec = quarticSolve(coefs);
	
		    if ( rootVec.isEmpty() )
		    {  //no solution
		    	return psiVec;
		    }else
		    {
				double x = 0.0, y = 0.0, z = 0.0;
				double [] bondVector = new double[3]; //three directional cosines for a bind vector
				double tmp1 = 0.0, tmp2 = 0.0, sinPsi = 0.0, cosPsi = 0.0,  psi=0.0;
				double cSquare = Const.psiCntUnit[0] * Const.psiCntUnit[0] + Const.psiCntUnit[1] * Const.psiCntUnit[1];
	
				Matrix rPsiz = new Matrix(3,3);
				double[] bondDirCal = new double[3];
				double[] bondDir    = new double[3];
		 		final double eps = 1.0E-8;
				for (int i=0; i<rootVec.size(); i++)
				{
				    bondVector = (double[])(rootVec.elementAt(i)); //IN this case it is NH vector
				    x = bondVector[0];
				    y = bondVector[1];
				    z = bondVector[2];
				    tmp1 = Const.psiCntUnit[0] * (mat[0][0] * x + mat[0][1] * y + mat[0][2] * z) 
					 + Const.psiCntUnit[1] * (mat[1][0] * x + mat[1][1] * y + mat[1][2] * z);
				    tmp2 = Const.psiCntUnit[1] * (mat[0][0] * x + mat[0][1] * y + mat[0][2] * z) 
					 - Const.psiCntUnit[0] * (mat[1][0] * x + mat[1][1] * y + mat[1][2] * z);
				    if ( cSquare != 0.0){
					sinPsi = tmp2 / cSquare;
					cosPsi = -tmp1 / cSquare;
					if ( cosPsi >= 0.0 )
					    psi = Math.asin(sinPsi);
					else if ( cosPsi < 0 )
					    psi = Math.PI - Math.asin(sinPsi);
				    }else
				    {
						System.out.println("Both Cx or Cz are zero");
						System.exit(1);
				    }
				    if (psi <= -Math.PI)  //in order to bring \Psi angle to be in [0, PI] for Beta, Sheet
					psi = psi + 2.0*Math.PI;
				    if (psi > Math.PI)  //in order to bring \Psi angle to be in [0, PI] for Beta, Sheet
					psi = psi - 2.0*Math.PI;
	
				    rPsiz = M.rotationMat(psi + Math.PI, "-z");  //check the solutions before squaring
				    bondDirCal = rPsiz.times(Const.psiCntUnit);
				    bondDir = M.times(bondVector);			
				    if(Math.abs(bondDir[0] - bondDirCal[0]) < eps && Math.abs(bondDir[1] - bondDirCal[1]) < eps 
				    		&& Math.abs(bondDir[2] - bondDirCal[2]) < eps)
				    {
				
				    	psiVec.add(new Double(psi));
				    }//else System.out.println(" no pass "+rdc);
				}
				return  psiVec;
		    }
		}
    }

    /**
     * Compute the Psi angles. This is a test method to see if it can
     * reproduce the directional cosines of the CH vector and the \phi angle.
     * 
     * @param rdc  Rdc data of NH vector
     * @param rg  the rotation matrix rotate a POF to a frame defined in the peptine plane $i$
     * @param Syy diagonilized Saupe elements computed by SVD with both CH and NH RDCs.
     * @param Szz diagonilized Saupe elements
     * @param psi the psi
     * 
     * @return a set of solutions for \psi angles: if the set empty ==> no solution
     */
    public Vector phiCalBackward(double Syy, double Szz, double rdc, double psi, Matrix rg)
    {
		Matrix r2z = rg.rotationMat(psi+Math.PI, "+z"); 
	
		//Matrix matPhi = rg.rotationMat(Const.phiAveHelix, "+y"); 
		Matrix M = Const.r8y3xAlpha.times(r2z.times(Const.r2x6z1x9zAlpha.times(rg))); 	
		double [][] mat = M.getArray();
		double[] coefs = new double[14];
		Vector phiVec = new Vector();
		final double eps = 1.0E-7;
		boolean coefFlag = coefPhiBackward(Syy,  Szz,  rdc,  mat,  coefs);
	
		double cz = Const.phiCnt[2];
		Matrix r2y = new Matrix(3, 3);
		if(!coefFlag)
		{   //can not compute coefficients
		    System.out.println("can not compute coefficients for the quartic equation");
		    return phiVec;
		}else
		{
		    Vector rootVec = quarticSolve(coefs);	
		    if (rootVec.isEmpty())
		    {  //no solution	
		    	return phiVec;
	        }else
	        { //Compute the Phi angle
				double x = 0.0, y = 0.0, z = 0.0;
				double [] bondVector = new double[3]; //three directional cosines for a bind vector
				double tmp1 = 0.0, tmp2 = 0.0, sinPhi = 0.0, cosPhi = 0.0, phi = 0.0;
		
				double[] bondDir    = new double[3];
		       
				for (int i=0; i<rootVec.size(); i++)
				{
				    bondVector = (double[])(rootVec.elementAt(i));
				    x = bondVector[0];
				    y = bondVector[1];
				    z = bondVector[2];
				    tmp1 = mat[2][0] * x + mat[2][1] * y + mat[2][2] * z;
				    tmp2 = mat[0][0] * x + mat[0][1] * y + mat[0][2] * z;
				    cosPhi = tmp1 / cz;
				    sinPhi = tmp2 / cz;
				    if ( cosPhi >= 0.0 )
					phi = Math.asin(sinPhi);
				    else if ( cosPhi < 0 )
					phi = Math.PI - Math.asin(sinPhi);
				    if (phi <= -Math.PI)
					phi = phi + 2.0 * Math.PI;
				    if (phi > Math.PI)
					phi = phi - 2.0 * Math.PI;
				    r2y = M.rotationMat(phi, "+y");  //The computed phi angle: and check the solutions
				    bondDir = r2y.times(M.times(bondVector));
				    if(Math.abs(bondDir[0] - Const.phiCnt[0]) < eps && Math.abs(bondDir[1] - Const.phiCnt[1]) < eps 
				    		&& Math.abs(bondDir[2] - Const.phiCnt[2]) < eps)
				    {
				
				    	phiVec.add(new Double(phi));
				    }
				}
				return  phiVec;
		    }
		}
    }

    /**
     * Phi psi backward test.
     * 
     * @param rdcNH the rdc nh
     * @param rdcCH the rdc ch
     */
    public void phiPsiBackwardTest(Vector rdcNH, Vector rdcCH){
	int i = 0, j = 0;
	double []  n = {0.0, 0.0, 0.0};         //A coordinate frame in the 1st peptide plane
	double [] nh = {0.0, 0.0, -Const.dN2H}; 
    	double [] ca = {0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA * Math.sin(Const.theta1)};
	ModelRdc mdc = new ModelRdc();
	//build an initial  helix model with average \phi and \psi angles.
	int firstResidueNo =  25;
	int lastResidueNo  =  35;
	Vector helixChRdc = new Vector();
	Vector helixNhRdc = new Vector();
	Dipolar dd = new Dipolar();
	int residueNo = 0;
	for (i = 0; i< rdcCH.size(); i++){
	    dd = (Dipolar) rdcCH.elementAt(i);
	    residueNo = dd.getResidueNo();
	    if (residueNo > 24 && residueNo < 34)
		helixChRdc.add(dd);
	}
	for (i = 0; i< rdcNH.size(); i++){
	    dd = (Dipolar) rdcNH.elementAt(i);
	    residueNo = dd.getResidueNo();
	    if (residueNo > 24 && residueNo < 34)
		helixNhRdc.add(dd);
	}
	Collections.sort(helixChRdc, new Dipolar.rdcComparator());
	Collections.sort(helixNhRdc, new Dipolar.rdcComparator());
	int N = lastResidueNo - firstResidueNo;
	Vector phiPsiVec = new Vector();
        for (i=firstResidueNo; i<lastResidueNo; i++)
            phiPsiVec.add(new PhiPsi(i, "ALA",  Const.phiAveHelix, Const.psiAveHelix));

	double phi1 = 90.0 * Const.cst;
	double psi1 = -33.0 * Const.cst;
	double phi2 = 71.0 * Const.cst;
	double psi2 = 62.33 * Const.cst;
	double phi3 = 109.55 * Const.cst;
	double psi3 = -60.0 * Const.cst;
  	phiPsiVec.setElementAt(new PhiPsi(firstResidueNo+2, "ALA", phi1, psi1), 2);// + 40.0 * Const.cst), 1);
  	phiPsiVec.setElementAt(new PhiPsi(firstResidueNo+3, "ALA", phi2, psi2), 3);
  	phiPsiVec.setElementAt(new PhiPsi(firstResidueNo+4, "ALA", phi3, psi3), 4);

        Vector pdbVec1 = mdc.modelBuild(phiPsiVec, n, nh, ca);
	double[] saupe = new double[4];
	PdbRdc pdr = new PdbRdc();
	double[] rdc1Cal  = new double[pdbVec1.size() - 1];
	double[] rdc2Cal  = new double[pdbVec1.size() - 1];
	Matrix mm = pdr.bestFit(pdbVec1, helixNhRdc, helixChRdc, rdc1Cal, rdc2Cal, saupe);
	Pdb pp = new Pdb();
	Vector pdbVec = pp.newPdb(pdbVec1, mm);
	printArray(saupe);
	double Syy = saupe[0];
	double Szz = saupe[1];
	double [] amide1 = new double[3];
	double [] nh1 = new double[3];
	double [] ca1 = new double[3];
	double [] ha1 = new double[3];
	double [] co1 = new double[3];
	double [] amide2 = new double[3];
	double [] nh2 = new double[3];
	double [] ca2 = new double[3];
	double [] ha2 = new double[3];
	double [] co2 = new double[3];
	double [] cb2 = new double[3];
	double [] amide3 = new double[3];
	double [] nh3 = new double[3];
	double [] ca3 = new double[3];
	double [] ha3 = new double[3];
	double [] co3 = new double[3];
	Cartesian cc = new Cartesian();
	String atom = "";

	pp = (Pdb)pdbVec.elementAt(5); 
	String resid = pp.getResidue();
	Vector atomVec = pp.getAtomVec();
	for (j=0; j<atomVec.size(); j++){
	    cc = (Cartesian)atomVec.elementAt(j);
	    atom = cc.getAtom();
	    if (atom.equals("N"))
		amide1 = cc.getXYZ();
	    else if (atom.equals("CA"))
		ca1 = cc.getXYZ();
	    else if (atom.equals("HA"))
		ha1 = cc.getXYZ();
	    else if (atom.equals("C"))
		co1 = cc.getXYZ();
	    else if (atom.equals("H"))
		nh1 = cc.getXYZ();
	}
	double[] nToNHVec = internuclearVec(amide1, nh1);
	double[] nToCAVec = internuclearVec(amide1, ca1);
	Matrix rg = RgCal(nToNHVec, nToCAVec);

// 	printArray(coCal);
// 	printArray(co1);
	pp = (Pdb)pdbVec.elementAt(4); 
	resid = pp.getResidue();
	atomVec = pp.getAtomVec();
	for (j=0; j<atomVec.size(); j++){
	    cc = (Cartesian)atomVec.elementAt(j);
	    atom = cc.getAtom();
	    if (atom.equals("N"))
		amide2 = cc.getXYZ();
	    else if (atom.equals("CA"))
		ca2 = cc.getXYZ();
	    else if (atom.equals("CB"))
		cb2 = cc.getXYZ();
	    else if (atom.equals("HA"))
		ha2 = cc.getXYZ();
	    else if (atom.equals("C"))
		co2 = cc.getXYZ();
	    else if (atom.equals("H"))
		nh2 = cc.getXYZ();
	}
	double[] ca2ha = dirCos(internuclearVec(ca2, ha2));
	double[] n2nh = dirCos(internuclearVec(amide2, nh2));
	double rdcNHCal = n2nh[0] * n2nh[0] * (-Syy-Szz) + n2nh[1]  * n2nh[1] * Syy + n2nh[2] * n2nh[2] * Szz;
	double rdcCal = ca2ha[0] * ca2ha[0] * (-Syy-Szz) + ca2ha[1] * ca2ha[1] *Syy +ca2ha[2] *ca2ha[2] * Szz;
 	System.out.println(" RDC = "+rdcCal+"  "+rdcNHCal+"  "+rdcNHCal * Const.nhRatio);
// 	printArray(ca2ha);
	double[] CsCal = Const.r2x6z1x9zAlpha.times(rg.times(ca2ha));
	Matrix rzPsiInv = rg.rotationMat(Const.psiAveHelix + Math.PI,  "-z");
	double [] c2Cal = rzPsiInv.times(Const.psiCntUnit);
// 	printArray(c2Cal);
// 	printArray(CsCal);
	double rdcNhVal = rdcNHCal;// / Const.nhRatio; //6.673605338661637
	double rdcChVal = rdcCal;//24.106632302099943;//4.288418196614507;//5.768650884607913;
 	System.out.println(" psi = "+psi3);
	Vector psiVec = psiCalBackward(Syy, Szz, rdcChVal, rg);
	if (psiVec.isEmpty())
	    System.out.println ( "No Solution" );
	for (i=0; i < psiVec.size(); i++)
	    System.out.println ( (Double)(psiVec.elementAt(i)));

 	System.out.println(" phi = "+phi3);
	Vector phiVec =  phiCalBackward(Syy, Szz, rdcNhVal, psi3, rg);
	if (phiVec.isEmpty())
	    System.out.println ("No Solution" );
	System.out.println ();
	for (i=0; i < phiVec.size(); i++)
	    System.out.println (( (Double)phiVec.elementAt(i)).doubleValue() / Const.cst);
// 	Matrix r2z = rg.rotationMat(Const.psiAveHelix+Math.PI, "+z"); 
 	double[] nhVec2 = dirCos(internuclearVec(amide2, nh2));
// 	Matrix matPhi = rg.rotationMat(Const.phiAveHelix, "+y"); 
// 	Matrix M = Const.r8y3xAlpha.times(r2z.times(Const.r2x6z1x9zAlpha.times(rg))); 
// 	double [] csCal =  matPhi.times(M.times(nhVec2));
// 	System.out.println(" csCal ");
// 	printArray(csCal);
// 	printArray(Const.phiCnt);
	printArray(nhVec2);

// 	pp = (Pdb)pdbVec.elementAt(8); 
// 	resid = pp.getResidue();
// 	atomVec = pp.getAtomVec();
// 	for (j=0; j<atomVec.size(); j++){
// 	    cc = (Cartesian)atomVec.elementAt(j);
// 	    atom = cc.getAtom();
// 	    if (atom.equals("N"))
// 		amide3 = cc.getXYZ();
// 	    else if (atom.equals("CA"))
// 		ca3 = cc.getXYZ();
// 	    else if (atom.equals("HA"))
// 		ha3 = cc.getXYZ();
// 	    else if (atom.equals("C"))
// 		co3 = cc.getXYZ();
// 	    else if (atom.equals("H"))
// 		nh3 = cc.getXYZ();
// 	}
// // 	double[] nh1Cal = rg.times(nToNHVec);
// // 	double[] ca1Cal = rg.times(nToCAVec);
// // 	printArray(nh1Cal);
// // 	printArray(ca1Cal);
// 	Matrix rgInv = rg.transpose();
    }

    /**
     * Testing the Computation of backbone structure backwards.
     * 
     * @param pdbVec2 the pdb vec2
     */
    public void backwardCoords(Vector pdbVec2){
	int i = 0, j = 0;
	double [] n = {0.0, 0.0, 0.0};         //A coordinate frame in the 1st peptide plane
	double [] nh = {0.0, 0.0, -Const.dN2H}; 
    	double [] ca = {0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA * Math.sin(Const.theta1)};
	ModelRdc mdc = new ModelRdc();
	//build an initial  helix model with average \phi and \psi angles.
	int firstResidueNo =   1;
	int lastResidueNo  =  11;
	int N = lastResidueNo - firstResidueNo;
	Vector phiPsiVec = new Vector();
        for (i=firstResidueNo; i<lastResidueNo; i++)
            phiPsiVec.add(new PhiPsi(i, "ALA",  Const.phiAveHelix, Const.psiAveBeta));
        Vector pdbVec = mdc.modelBuild(phiPsiVec, n, nh, ca);
// 	Pdb pp = new Pdb(); 
// 	System.out.println(pp);
// 	pp.print(pdbVec);
	double [] amide1 = new double[3];
	double [] nh1 = new double[3];
	double [] ca1 = new double[3];
	double [] ha1 = new double[3];
	double [] o1  = new double[3];
	double [] co1 = new double[3];
	double [] amide2 = new double[3];
	double [] nh2 = new double[3];
	double [] ca2 = new double[3];
	double [] ha2 = new double[3];
	double [] co2 = new double[3];
	double [] o2 = new double[3];
	double [] cb2 = new double[3];
	double [] amide3 = new double[3];
	double [] nh3 = new double[3];
	double [] ca3 = new double[3];
	double [] ha3 = new double[3];
	double [] co3 = new double[3];
	Cartesian cc = new Cartesian();
	String atom = "";

	Pdb pp = (Pdb)pdbVec.elementAt(10); 
	String resid = pp.getResidue();
	Vector atomVec = pp.getAtomVec();
	for (j=0; j<atomVec.size(); j++){
	    cc = (Cartesian)atomVec.elementAt(j);
	    atom = cc.getAtom();
	    if (atom.equals("N"))
		amide1 = cc.getXYZ();
	    else if (atom.equals("CA"))
		ca1 = cc.getXYZ();
	    else if (atom.equals("HA"))
		ha1 = cc.getXYZ();
	    else if (atom.equals("C"))
		co1 = cc.getXYZ();
	    else if (atom.equals("O"))
		o1 = cc.getXYZ();
	    else if (atom.equals("H"))
		nh1 = cc.getXYZ();
	}
// 	printArray(coCal);
// 	printArray(co1);
	pp = (Pdb)pdbVec.elementAt(9); 
	resid = pp.getResidue();
	atomVec = pp.getAtomVec();
	for (j=0; j<atomVec.size(); j++){
	    cc = (Cartesian)atomVec.elementAt(j);
	    atom = cc.getAtom();
	    if (atom.equals("N"))
		amide2 = cc.getXYZ();
	    else if (atom.equals("CA"))
		ca2 = cc.getXYZ();
	    else if (atom.equals("CB"))
		cb2 = cc.getXYZ();
	    else if (atom.equals("HA"))
		ha2 = cc.getXYZ();
	    else if (atom.equals("C"))
		co2 = cc.getXYZ();
	    else if (atom.equals("O"))
		o2 = cc.getXYZ();
	    else if (atom.equals("H"))
		nh2 = cc.getXYZ();
	}
	double nNhCaHaNext [][] = mdc.nNhCaHaByBackward(Const.phiAveHelix, Const.psiAveBeta, amide1, nh1, ca1);
	printArray(amide2);
	printArray(nNhCaHaNext[0]);
	printArray(nh2);
	printArray(nNhCaHaNext[1]);
	printArray(ca2);
	printArray(nNhCaHaNext[2]);
	printArray(ha2);
	printArray(nNhCaHaNext[3]);

	pp = (Pdb)pdbVec.elementAt(8); 
	resid = pp.getResidue();
	atomVec = pp.getAtomVec();
	for (j=0; j<atomVec.size(); j++){
	    cc = (Cartesian)atomVec.elementAt(j);
	    atom = cc.getAtom();
	    if (atom.equals("N"))
		amide3 = cc.getXYZ();
	    else if (atom.equals("CA"))
		ca3 = cc.getXYZ();
	    else if (atom.equals("HA"))
		ha3 = cc.getXYZ();
	    else if (atom.equals("C"))
		co3 = cc.getXYZ();
	    else if (atom.equals("H"))
		nh3 = cc.getXYZ();
	}

	double[] nToNHVec = internuclearVec(amide1, nh1);
	double[] nToCAVec = internuclearVec(amide1, ca1);
	Matrix rg = RgCal(nToNHVec, nToCAVec);
// 	double[] nh1Cal = rg.times(nToNHVec);
// 	double[] ca1Cal = rg.times(nToCAVec);
// 	printArray(nh1Cal);
// 	printArray(ca1Cal);
	Matrix rgInv = rg.transpose();
	double[] coToNVec = internuclearVec(co2, amide1);
	double[] coToOVec = internuclearVec(co2, o2);
	double[] nToCOVec = internuclearVec(amide1, co2);
	double alpha9 = pp.PhiPsiCalPDB(coToNVec, nToNHVec,  nToCAVec);
	System.out.println(" alpha9 = " + alpha9 / Const.cst+"  "+Const.alpha1/Const.cst);

// 	double deltaO = -Math.PI - pp.PhiPsiCalPDB(nToNHVec, coToNVec,  coToOVec);
// 	Matrix matDeltaO = rg.rotationMat(deltaO, "-z");
// 	double thetaO = pp.interAngle(coToNVec,  coToOVec) - 0.50 * Math.PI;
// 	Matrix matThetaO = rg.rotationMat(thetaO, "-x");

	Matrix ra9zInv = rg.rotationMat(alpha9, "-z");
	double [] n2coCnt = {0.0, 0.0, Const.dCO2N};
	
	double[] n2co = rgInv.times(Const.r9zAlphaInv.times(Const.r1xAlphaInv.times(n2coCnt)));
	double [] coCal = addCoords(amide1, n2co);
	printArray(co2);
	printArray(coCal);

// 	System.out.println(" deltaO = " + deltaO / Const.cst);
	double [] co2oCnt  = {0.0, Const.dCO2O, 0.0};
	double [] co2o  = rgInv.times(Const.r9zAlphaInv.times(Const.r1xAlphaInv.times(Const.matOInv.times(co2oCnt))));
	double [] o2Cal = addCoords(coCal, co2o);
	printArray(o2);
	printArray(o2Cal);
// 	System.out.println(" thetaO = " + thetaO / Const.cst);

	double [] co2caCnt = {0.0, 0.0, Const.dCA2CO};
	Matrix mat =  rgInv.times(Const.r9zAlphaInv.times(Const.r1xAlphaInv.times(Const.r6zAlphaInv.times(Const.r2xAlphaInv))));
	double[] co2ca = mat.times(co2caCnt);
	double[] caCal = addCoords(coCal, co2ca);
	printArray(ca2);
	printArray(caCal);
	Matrix rzPsiInv = mat.rotationMat(Const.psiAveBeta+Math.PI, "-z");
	mat = mat.times(rzPsiInv.times(Const.r3xAlphaInv));
	double[] ca2nCnt = {0.0, Const.dN2CA, 0.0};
	double[] ca2n = mat.times(ca2nCnt);
	double [] nCal = addCoords(caCal, ca2n);
	printArray(amide2);
	printArray(nCal);
	
	double[] haCnt = {0.0, 0.0, Const.dCA2HA};
	Matrix matHA = mat.times(Const.matHAInv);
	double[] ha2Cal = addCoords(caCal, matHA.times(haCnt));
	printArray(ha2);
	printArray(ha2Cal);

	double[] cbCnt = {0.0, 0.0, Const.dCA2CB};
	Matrix matCB = mat.times(Const.matCBInv);
	double[] cb2Cal = addCoords(caCal, matCB.times(cbCnt));
	printArray(cb2);
	printArray(cb2Cal);

	Matrix ryPhiInv = mat.rotationMat(Const.phiAveHelix, "-y");
// 	Matrix ryPhiInv = mat.rotationMat(Const.phiAveHelix+Math.PI, "-y");
	Matrix ra10Inv = rg.rotationMat(Math.PI, "-y");
	mat = mat.times(Const.r8yAlphaInv.times(ryPhiInv.times(ra10Inv.times(Const.r5xAlphaInv))));
	double[] n2nhCnt = {0.0, 0.0, -Const.dN2H};
	double[] n2nh = mat.times(n2nhCnt);
	double [] nhCal = addCoords(nCal, n2nh);
// 	System.out.println(Const.alpha5  / Const.cst);
	printArray(nh2);
	printArray(nhCal);
// 	Pdb ppCal = mdc.coordByBackward(Const.phiAveHelix, Const.psiAveHelix, amide1, nh1, ca1, 10, "ALA");
// 	System.out.println(ppCal);
// 	System.out.println((Pdb)pdbVec.elementAt(9)); 
	int no = 24;
	int index =  Collections.binarySearch(pdbVec2, new Pdb(no),  new Pdb.PdbComparator());
	Pdb pp24 = (Pdb) pdbVec2.elementAt(index);
	resid = pp24.getResidue();
	atomVec = pp24.getAtomVec();
	for (j=0; j<atomVec.size(); j++){
	    cc = (Cartesian)atomVec.elementAt(j);
	    atom = cc.getAtom();
	    if (atom.equals("N"))
		amide1 = cc.getXYZ();
	    else if (atom.equals("CA"))
		ca1 = cc.getXYZ();
	    else if (atom.equals("HA"))
		ha1 = cc.getXYZ();
	    else if (atom.equals("C"))
		co1 = cc.getXYZ();
	    else if (atom.equals("O"))
		o1 = cc.getXYZ();
	    else if (atom.equals("H"))
		nh1 = cc.getXYZ();
	}
	Pdb ppCal = mdc.coordByBackward(Const.phiAveHelix, Const.psiAveHelix, amide1, nh1, ca1, no-1, "ALA");
	pdbVec2.add(ppCal);
// 	pp.print(pdbVec2);
    }

   /**
    * A recusive function to compute all the backbone (phi, psi) angles for an $N$-residue fragment
    * This is a version for real computation, NOT for collecting running statistics.
    * 
    * @param rdcArr1  an array of CH RDCs for the fragment
    * @param rdcArr2  an array of NH RDCs for the fragment
    * @param rg  a matrix to rotate the POF to a coodinate system defined in the first peptide plane.
    * @param Syy Saupe elements computed from both NH and CH rdcs
    * @param Szz  Saupe elements computed from both NH and CH rdcs
    * @param i    the current depth of the depth-first search. Begin the call with i = 0;
    * @param N the total depth of the depth-first search.
    * @param ppS a temporary array for saving intermediate (\phi, \psi) angles.
    * @param ppVec an array for saving and returning final (\phi, \psi) angles.
    * @param rightHand make sure the computed rg has the same handedness as the rotating matrix by the SVD method
    * @param isHelix the is helix
    * 
    * @return true, if phi psi chain
    */
    public boolean phiPsiChain (double[] rdcArr1, double[] rdcArr2, Matrix rg, double Syy, double Szz, int i,
				final int N, double[] ppS, Vector ppVec, boolean rightHand, boolean isHelix){
	int j, k;
	double u, v; //for CH and NH RDC respectively
	Vector phiVec = new Vector();
	Vector psiVec = new Vector();
	double phi = 0.0, psi = 0.0;
	Matrix mat = new Matrix(3,3);
	final double ratio = Const.nhRatio;
	if (i > N - 1){
	    u = rdcArr1[i];
	    phiVec = phiCal(Syy, Szz, u, rg, isHelix);
	    if (!phiVec.isEmpty()){
		for (int m = 0; m < phiVec.size(); m++){
		    // System.out.print( ((Double)phiVec.elementAt(m)).doubleValue() / Const.cst+", **  ");
		    for (j=0; j<2*N; j++){
			//  System.out.println((ppS[j]/Const.cst));
			ppVec.addElement(new Double(ppS[j]));
		    }
		    ppVec.addElement(phiVec.elementAt(m));
		    // 	     ppVec.addElement(new Double(Const.psiAveBeta));  //This one is for \beta sheet
		    if(isHelix)
			ppVec.addElement(new Double(Const.psiAveHelix)); //This one is for helix
		    else 
			ppVec.addElement(new Double(Const.psiAveBeta)); //This one is for helix
		    // 	     System.out.println(Const.psiAveHelix / Const.cst );
		}
		return true;
	    }else return false;
	}else{
	    u = rdcArr1[i];
//  	    System.out.println("CH :"+u);
	    phiVec = phiCal(Syy, Szz, u, rg, isHelix);
//  	    for (int m = 0; m < phiVec.size(); m++)
//  		System.out.print( ((Double)phiVec.elementAt(m)).doubleValue() / Const.cst+",   ");
	    if (!phiVec.isEmpty()){
		for (j =0; j<phiVec.size(); j++){
		    phi = ((Double)phiVec.elementAt(j)).doubleValue();
//  		    ppVec.add(phiVec.elementAt(j));   //add phi at level $i$
		    v = rdcArr2[i+1] / ratio; //
// 		    System.out.println("NH :"+rdcArr2[i+1]);
		    psiVec = psiCal(Syy, Szz, v, phi,  rg, isHelix);
//  		    for (int mm = 0; mm < psiVec.size(); mm++)
//  			System.out.print( ( (Double)psiVec.elementAt(mm)).doubleValue() / Const.cst+" ,, ");
// 		    System.out.println("  ");
		    if (!psiVec.isEmpty()){
			for (k =0; k<psiVec.size(); k++){
			    psi = ((Double)psiVec.elementAt(k)).doubleValue();
			    ppS[2*i] = phi;
			    ppS[2*i+1] = psi;
			    mat = newRG(phi, psi, rg, rightHand); //compute a new rotation matrix rg
			    phiPsiChain(rdcArr1, rdcArr2, mat, Syy, Szz, i+1, N, ppS, ppVec, rightHand, isHelix);
			}
		    }
		}
	    } //when either no feasible solution for \Phi or though \phi has solution but all the \Phi and Psi
	      //solutions leading from it failed so...
	}
	return false;
    }
    
    /**
     * A recusive function to compute all the backbone (phi, psi) angles for an $N$-residue fragment
     * This is a version for real computation, NOT for collecting running statistics.
     * 
     * @param rdcArr1  an array of CH RDCs for the fragment
     * @param rdcArr2  an array of NH RDCs for the fragment
     * @param rg  a matrix to rotate the POF to a coodinate system defined in the first peptide plane.
     * @param Syy Saupe elements computed from both NH and CH rdcs
     * @param Szz  Saupe elements computed from both NH and CH rdcs
     * @param i    the current depth of the depth-first search. Begin the call with i = 0;
     * @param N the total depth of the depth-first search.
     * @param ppS a temporary array for saving intermediate (\phi, \psi) angles.
     * @param ppVec an array for saving and returning final (\phi, \psi) angles.
     * @param rightHand make sure the computed rg has the same handedness as the rotating matrix by the SVD method
     * @param isHelix the is helix
     * @param firstResNo the first res no
     * @param vecTalos the vec talos
     * 
     * @return true, if phi psi chain talos
     */
     public boolean phiPsiChainTalos(double[] rdcArr1, double[] rdcArr2, Matrix rg, double Syy, double Szz, int i,
 				final int N, double[] ppS, Vector ppVec, boolean rightHand, boolean isHelix,int firstResNo, Vector vecTalos){
 	int j, k;
 	double u, v; //for CH and NH RDC respectively
 	Vector phiVec = new Vector();
 	Vector psiVec = new Vector();
 	double phi = 0.0, psi = 0.0;
 	Matrix mat = new Matrix(3,3);
 	int curResNo=firstResNo+i;
 	final double ratio = Const.nhRatio;
 	if (i > N - 1){
 	    u = rdcArr1[i];
 	    phiVec = phiCal(Syy, Szz, u, rg, isHelix,curResNo,vecTalos);
 	    if (!phiVec.isEmpty()){
 		for (int m = 0; m < phiVec.size(); m++){
 		    // System.out.print( ((Double)phiVec.elementAt(m)).doubleValue() / Const.cst+", **  ");
 		    for (j=0; j<2*N; j++){
 			//  System.out.println((ppS[j]/Const.cst));
 			ppVec.addElement(new Double(ppS[j]));
 		    }
 		    ppVec.addElement(phiVec.elementAt(m));
 		    // 	     ppVec.addElement(new Double(Const.psiAveBeta));  //This one is for \beta sheet
 		    if(isHelix)
 			ppVec.addElement(new Double(Const.psiAveHelix)); //This one is for helix
 		    else 
 			ppVec.addElement(new Double(Const.psiAveBeta)); //This one is for helix
 		    // 	     System.out.println(Const.psiAveHelix / Const.cst );
 		}
 		return true;
 	    }else return false;
 	}else{
 	    u = rdcArr1[i];
//   	    System.out.println("CH :"+u);
 	    phiVec = phiCal(Syy, Szz, u, rg, isHelix,curResNo,vecTalos);
//   	    for (int m = 0; m < phiVec.size(); m++)
//   		System.out.print( ((Double)phiVec.elementAt(m)).doubleValue() / Const.cst+",   ");
 	    if (!phiVec.isEmpty()){
 		for (j =0; j<phiVec.size(); j++){
 		    phi = ((Double)phiVec.elementAt(j)).doubleValue();
//   		    ppVec.add(phiVec.elementAt(j));   //add phi at level $i$
 		    v = rdcArr2[i+1] / ratio; //
//  		    System.out.println("NH :"+rdcArr2[i+1]);
 		    psiVec = psiCal(Syy, Szz, v, phi,  rg, isHelix,curResNo,vecTalos);
//   		    for (int mm = 0; mm < psiVec.size(); mm++)
//   			System.out.print( ( (Double)psiVec.elementAt(mm)).doubleValue() / Const.cst+" ,, ");
//  		    System.out.println("  ");
 		    if (!psiVec.isEmpty()){
 			for (k =0; k<psiVec.size(); k++){
 			    psi = ((Double)psiVec.elementAt(k)).doubleValue();
 			    ppS[2*i] = phi;
 			    ppS[2*i+1] = psi;
 			    mat = newRG(phi, psi, rg, rightHand); //compute a new rotation matrix rg
 			    phiPsiChainTalos(rdcArr1, rdcArr2, mat, Syy, Szz, i+1, N, ppS, ppVec, rightHand, isHelix,firstResNo, vecTalos);
 			}
 		    }
 		}
 	    } //when either no feasible solution for \Phi or though \phi has solution but all the \Phi and Psi
 	      //solutions leading from it failed so...
 	}
 	return false;
     }

    /**
     * The method right now may have ERROR!
     * To extract all the sequential restraints (both RDCs and NOEs)
     * involving the backbone nuclei (HA and NH) located in the turn or the loop.
     * 
     * @param begin the first residue of the fragment (turn and loop)
     * @param end the last residue of the fragment (turn and loop)
     * @param h1Vec resonance assignment sorted by H1 CS shift.
     * @param asgVec the merged resonacs assigement and HN and HA NOEs
     * @param rdcVec1 the CH RDC
     * @param rdcVec2 the NH RDC
     * @param a1  the constant used to convert the intensity to distance
     * @param a2  the constant used to convert the intensity to distance
     * 
     * @return a double array, and the elements of each row is {chRdc, nhRdc, dna, dan, dnn}
     * where chRdc is for residue i and nhRDc for residue i+1
     */
    public double[][] extractRestraints(int begin, int end, final Vector h1Vec, final Vector asgVec, 
					Vector rdcVec1, Vector rdcVec2, double a1, double a2){
	int i = 0, j = 0;
	int lastNo = end + 2;
	int index = -1;
	double chRdc = Const.noDataIndicator;
	double nhRdc = Const.noDataIndicator;
	double intensity = -1.00;
	double dna = 0.0;  //the intraNOEs, dna(i, i), between HN(i) and HA(i)
	double dan = 0.0;  //the seqNOEs, dan(i, i), between HA(i) and HN(i+1)
	double dnn = 0.0;  //the seqNOEs, dnn(i, i), between HA(i) and HN(i+1)
	Peak pk = new Peak();
	int numberOfResidues = end - begin + 2; 
	double [][] restraints = new double[numberOfResidues][5]; 
	for (i=0; i<numberOfResidues; i++)
	    for (j=0; j<5; j++)
		restraints[i][j] = Const.noDataIndicator;
	int cnt = 0;
	String resid ="";
	Assign asg = new Assign();
	double[] noe = new double[1];
	boolean hasNOE = false;
	for(i = begin; i < lastNo; i++){
	    index = Collections.binarySearch(asgVec, new Assign(i), new Assign.assignComparator());
	    if (index > -1){
		asg = (Assign)asgVec.elementAt(index);
		resid = asg.getResidueType();
	    }else continue;
	    //CH RDC or Dna NOE for computing phi_i angle
	    index = Collections.binarySearch(rdcVec1, new Dipolar(i), new Dipolar.rdcComparator());
// 	    System.out.println(i+"  "+index);
	    if (index > -1){
		chRdc = ((Dipolar)rdcVec1.elementAt(index)).getRdc();
	    }else chRdc = Const.noDataIndicator;
	    if(resid.equalsIgnoreCase("GLY"))
		hasNOE = pk.noeSearch(h1Vec, asgVec, i, "HN", i, "HA2", noe); //not find, intensity =-1.00
	    else hasNOE= pk.noeSearch(h1Vec, asgVec, i, "HN", i, "HA", noe); //not find, intensity =-1.00
	    if (hasNOE){
		dna = Math.pow(a2 / (noe[0] - a1), 0.25);
// 		System.out.println(noe[0]+", dna "+dna);
		if (dna < Const.noeLowerLimit)
		    dna = Const.noeLowerLimit;
		else if (dna > Const.dnaLimit)
		    dna = Const.dnaLimit;
	    }else dna = Const.noDataIndicator;
	    //NH RDC of residue (i+1) and Dan NOE of residue (i, i+1) for computing psi_i angle
	    index = Collections.binarySearch(rdcVec2, new Dipolar(i+1), new Dipolar.rdcComparator());
	    if (index > -1){
		nhRdc = ((Dipolar)rdcVec2.elementAt(index)).getRdc();
	    }else nhRdc = Const.noDataIndicator;
// 	    System.out.println((i+1)+"  "+index);
	    hasNOE = false;
	    if(resid.equalsIgnoreCase("GLY"))
		hasNOE  = pk.noeSearch(h1Vec, asgVec, i+1, "HN", i, "HA2", noe); //not find, intensity =-1.00
	    else hasNOE = pk.noeSearch(h1Vec, asgVec, i+1, "HN", i, "HA", noe); //not find, intensity =-1.00
	    if (hasNOE){
		dan = Math.pow(a2 / (noe[0] - a1), 0.25);
// 		System.out.println(noe[0]+", dan "+dan);
		if (dan < Const.noeLowerLimit)
		    dan = Const.noeLowerLimit;
		else if (dan > Const.danLimit)
		    dan = Const.danLimit;
	    } else dan = Const.noDataIndicator;
	    //the dNN NOEs
	    hasNOE = pk.noeSearch(h1Vec, asgVec, i, "HN", i+1, "HN", noe); //not find, intensity =-1.00
	    if (hasNOE){
		dnn = Math.pow(a2 / (noe[0] - a1), 0.25);
// 		System.out.println(intensity+", dnn "+dnn);
		if (dnn < Const.noeLowerLimit)
		    dnn = Const.noeLowerLimit;
		else if (dnn > Const.noeUpLimit)
		    dnn = Const.noeUpLimit;
	    } else dnn = Const.noDataIndicator;
	    restraints[cnt][0] = chRdc;
	    restraints[cnt][1] = nhRdc;
	    restraints[cnt][2] = dna;
	    restraints[cnt][3] = dan;
	    restraints[cnt][4] = dnn;
	    cnt++;
	}
	return restraints;
    }

    /**
     * To extract all the sequential restraints (both RDCs and NOEs)
     * involving the backbone nuclei (HA and NH) located in the turn or the loop.
     * 
     * @param begin the first residue of the fragment (turn and loop) such as 22.
     * @param end the last residue of the fragment (turn and loop) such as 18
     * @param h1Vec resonance assignment sorted by H1 CS shift.
     * @param asgVec the merged resonacs assigement and HN and HA NOEs
     * @param rdcVec1 the CH RDC
     * @param rdcVec2 the NH RDC
     * @param a1  the constant used to convert the intensity to distance
     * @param a2  the constant used to convert the intensity to distance
     * 
     * @return a double array, and the elements of each row is {chRdc,
     * nhRdc, dna, dan, dnn}, where dan and dnn are between residue HA(i) <-->
     * HN(i+1) and HN(i) <--> HN(i+1), the double array begin with residue "begin"
     */
    public double[][] restraintsBackward(int begin, int end, final Vector h1Vec, final Vector asgVec, 
					 Vector rdcVec1, Vector rdcVec2, double a1, double a2){
	int i = 0, j = 0;
	int previousNo = begin; //such as 23
	int lastNo = end - 1; //such as 17
	int index = -1;
	double chRdc = Const.noDataIndicator;
	double nhRdc = Const.noDataIndicator;
	double dna = 0.0;  //the intraNOEs, dna(i, i), between HN(i) and HA(i)
	double dan = 0.0;  //the seqNOEs, dan(i, i), between HA(i) and HN(i+1)
	double dnn = 0.0;  //the seqNOEs, dnn(i, i), between HA(i) and HN(i+1)
	Peak pk = new Peak();
	int numberOfResidues = begin - end + 1;  //six residues (17-23)
	double [][] restraints = new double[numberOfResidues][5]; 
	for (i=0; i<numberOfResidues; i++)
	    for (j=0; j<5; j++)
		restraints[i][j] = Const.noDataIndicator;
	int cnt = 0;
	String resid ="";
	Assign asg = new Assign();
	boolean hasNOE = false;
	double [] noe = new double[1]; //the noe intensity
	for(i = previousNo; i > lastNo; i--){
	    index = Collections.binarySearch(asgVec, new Assign(i), new Assign.assignComparator());
	    if (index > -1){
		asg = (Assign)asgVec.elementAt(index);
		resid = asg.getResidueType();
	    }else continue;
	    //CH RDC
	    index = Collections.binarySearch(rdcVec1, new Dipolar(i), new Dipolar.rdcComparator());
	    if (index > -1){
		chRdc = ((Dipolar)rdcVec1.elementAt(index)).getRdc();
	    }else chRdc = Const.noDataIndicator;
	    //Dna NOE
	    if(resid.equalsIgnoreCase("GLY"))
		hasNOE = pk.noeSearch(h1Vec, asgVec, i, "HN", i,"HA2", noe);//not find,intensity =-1.00
	    else hasNOE = pk.noeSearch(h1Vec, asgVec, i,"HN", i, "HA", noe);//not find, intensity =-1.00
	    if (hasNOE){
		dna = Math.pow(a2 / (noe[0] - a1), 0.25);
//  		System.out.println(noe[0]+", dna "+dna);
		if (dna < Const.noeLowerLimit)
		    dna = Const.noeLowerLimit;
		else if (dna > Const.dnaLimit) // larger than the maximum is meaningless
		    dna = Const.dnaLimit;
	    }else dna = Const.noDataIndicator;
	    //NH RDC
	    index = Collections.binarySearch(rdcVec2, new Dipolar(i), new Dipolar.rdcComparator());
	    if (index > -1){
		nhRdc = ((Dipolar)rdcVec2.elementAt(index)).getRdc();
	    }else nhRdc = Const.noDataIndicator;
	    //Dan NOE: HA(i)<--)HN(i+1)
	    if(resid.equalsIgnoreCase("GLY"))
		hasNOE = pk.noeSearch(h1Vec, asgVec, i+1,"HN",i,"HA2", noe); //not find, intensity =-1.00
	    else hasNOE =pk.noeSearch(h1Vec, asgVec, i+1,"HN",i, "HA", noe); //not find, intensity =-1.00
	    if (hasNOE){
		dan = Math.pow(a2 / (noe[0] - a1), 0.25);
//  		System.out.println(noe[0]+", dan "+dan);
		if (dan < Const.noeLowerLimit)
		    dan = Const.noeLowerLimit;
		else if (dan > Const.danLimit)
		    dan = Const.danLimit;
	    } else dan = Const.noDataIndicator;
	    //the dNN NOEs HN(i)<--)HN(i+1)
	    hasNOE = pk.noeSearch(h1Vec, asgVec, i+1, "HN",i, "HN", noe); //not find, intensity =-1.00
	    if (hasNOE){
		dnn = Math.pow(a2 / (noe[0] - a1), 0.25);
//  		System.out.println(noe[0]+", dnn "+dnn);
		if (dnn < Const.noeLowerLimit)
		    dnn = Const.noeLowerLimit;
		else if (dnn > Const.noeUpLimit)
		    dnn = Const.noeUpLimit;
	    } else dnn = Const.noDataIndicator;
	    restraints[cnt][0] = chRdc;
	    restraints[cnt][1] = nhRdc;
	    restraints[cnt][2] = dna;
	    restraints[cnt][3] = dan;
	    restraints[cnt][4] = dnn;
	    cnt++;
	}
	return restraints;
    }

   /**
    * A recusive function to compute all the backbone (phi, psi) angles for an $m$-residue fragment
    * This is a version for real computation, NOT for collecting running statistics.
    * 
    * @param asgVec  has the resonance assignment and NOE peaklist
    * @param restraints a double array, and the elements of each row is {chRdc, nhRdc, dna, dan, dnn}
    * @param rg  a matrix to rotate the POF to a coodinate system defined in the first peptide plane.
    * @param Syy  Saupe elements computed from both NH and CH rdcs
    * @param Szz  Saupe elements computed from both NH and CH rdcs
    * @param depth    the current depth of the depth-first search. Begin with i = 0;
    * @param N    the total depth of the depth-first search, the same as the number of residues
    * @param ppS  a temporary array for saving intermediate (\phi, \psi) angles.
    * @param ppVec an array for returning final (\phi, \psi) angles.
    * @param ca2 the coordinate of the CA nucleus of the end side of the loop
    * We need a method to search all the possible NOEs based two proton CSs, which can be used for
    * identify possible sequential NOE assignment.
    * // 	double dnaCal = 0.0, danCal = 0.0, dnnCal = 0.0;
    * // 	double [] noeScore = new double[5];
    * // 	for (i=0; i<noeScore.length; i++)
    * // 	    noeScore[i] = 0.0;
    * @param n0 the n0
    * @param nh0 the nh0
    * @param ca0 the ca0
    * @param nToNHVec2 the n to nh vec2
    * @param nToCAVec2 the n to ca vec2
    * 
    * @return true, if phi psi by rdc noe
    */
    public boolean phiPsiByRdcNoe (Vector asgVec, double[][] restraints, double Syy, double Szz, double [] n0, 
				   double [] nh0, double[] ca0, Matrix rg, int depth, int N, double [] ppS, 
				   Vector ppVec, double[] nToNHVec2, double[] nToCAVec2, double[] ca2){
	int i, j, k;
	double [] n  = {n0[0],   n0[1],  n0[2]};
	double [] nh = {nh0[0], nh0[1], nh0[2]};
	double [] ca = {ca0[0], ca0[1], ca0[2]};
	double [] n2nh = {nh0[0] - n0[0], nh0[1] - n0[1], nh0[2] - n0[2]}; //n2nh vector
	double [] n2ca = {ca0[0] - n0[0], ca0[1] - n0[1], ca0[2] - n0[2]}; //n2ca vector
	if(depth == N ){ //at the end
	    n  = new double[]{n0[0],   n0[1],  n0[2]};
	    nh = new double[]{nh0[0], nh0[1], nh0[2]};
	    ca = new double[]{ca0[0], ca0[1], ca0[2]};
//  	    System.out.println(" Complete ");
// 	    if ( (interAngle(n2nh, nToNHVec2) < 15.0 * Const.cst) 
// 		 && (interAngle(n2ca, nToCAVec2) < 15.0 * Const.cst)){
		for (j=0; j<2*N; j++){
		    System.out.println(ppS[j]/Const.cst);
		    ppVec.addElement(new Double(ppS[j]));
		}
		System.out.println();
		return true;
// 	    } else return false;
	}
	ModelRdc mdc = new ModelRdc();
	Vector phiVec = new Vector();
	Vector psiVec = new Vector();
	double phi = 0.0, psi = 0.0;
	Matrix mat = new Matrix(3,3);
	int resolution = 10;
	//compute the psi_i using either NH RDCs for residue i+1, or the seqNOE, dan(i, i+1),
	//if no exp data just compute it by grid-search between [-pi, pi] 
	//compute the phi_i using either CH RDCs for residue i, or the intraNOE dna(i, i),
	//if no exp data available, just compute it by grid-search between [-pi, pi] 
	double chRdc = restraints[depth][0]; //chRDC of residue $i$
	double nhRdc = restraints[depth][1]; //nhRDC of residue $i + 1$
	double dna   = restraints[depth][2];  //dna of residue $i$
	double dan   = restraints[depth][3];  //dan of residue HA(i) and NH(i+1)
	double dnn   = restraints[depth][4];  //dnn of residue NA(i) and NH(i+1)
	double dnnCal  = 0.0;
	if (Math.abs(chRdc - Const.noDataIndicator) > Const.eps){
	    phiVec = phiCal(Syy, Szz, chRdc, rg);
 	    System.out.println(" phi by Rdc "+depth+"  "+phiVec.size());
	}else if(Math.abs(dna - Const.noDataIndicator) > Const.eps){
	    phiVec = phiCalNoe(dna);
// 	    System.out.println(dna+" ---- "+phiVec.size());
	}
	double[][] haco = new double[2][3];
	double[] ha = new double[3];
	double[] co = new double[3];
	double disToCa2 = 0.0;
	double [][] nNhCaCoords = new double[3][3];

	if (depth == 1){ 
	    phi = -65.0 * Const.cst; //this is a proline
	    if (Math.abs(nhRdc - Const.noDataIndicator) > Const.eps){
		psiVec = psiCal(Syy, Szz, nhRdc / Const.nhRatio, phi, rg);
	    }else if(Math.abs(dan - Const.noDataIndicator) > Const.eps){
		// 		    System.out.println(dan+" --*** ");
		haco = mdc.haCoCoords(phi, n, nh, ca);
		ha = haco[0];
		co = haco[1];
		psiVec = psiCalNoe(dan);
		// 		    System.out.println(dan+" --** "+psiVec.size());
	    }else psiVec = new Vector();
	    if (!psiVec.isEmpty()){
		for (k =0; k<psiVec.size(); k++){
		    psi = ((Double)psiVec.elementAt(k)).doubleValue();
		    ppS[2*depth] = phi;
		    ppS[2*depth + 1] = psi;
		    nNhCaCoords = mdc.nNhCaCal(phi, psi, n, nh, ca); //compute the n, nh and ca residue (i+1)
		    dnnCal = internuclearDistance(nNhCaCoords[2], nh);
		    disToCa2 = internuclearDistance(ca, ca2);
		    System.out.println(depth+" ****  "+dnnCal+"  "+dnn+"   "+disToCa2);
 		    if (disToCa2 < (N - depth) * Const.dCA2CA && Math.abs(dnn-dnnCal) < 1.80){
			phiPsiByRdcNoe(asgVec, restraints, Syy, Szz, nNhCaCoords[0], nNhCaCoords[2], 
				       nNhCaCoords[1], mat, depth+1, N, ppS, ppVec, nToNHVec2, nToCAVec2, ca2);
			mat = newRG(phi, psi, rg, true); //compute a new rotation matrix rg
 		    }
		}
	    }else{
		for (k=0; k < (360 / resolution); k++){
		    psi = k * resolution * Math.PI / 360.0 - Math.PI;
		    ppS[2*depth] = phi;
		    ppS[2*depth + 1] = psi;
		    nNhCaCoords = mdc.nNhCaCal(phi, psi, n, nh, ca); //compute the n, nh and ca for next residue
		    dnnCal = internuclearDistance(nNhCaCoords[2], nh);
		    disToCa2 = internuclearDistance(ca, ca2);
		    System.out.println(depth+" *****  "+dnnCal+"  "+dnn+"   "+disToCa2);
 		    if (disToCa2 < (N - depth) * Const.dCA2CA && Math.abs(dnn-dnnCal) < 1.80){
			mat = newRG(phi, psi, rg, true); //compute a new rotation matrix rg
			phiPsiByRdcNoe(asgVec, restraints, Syy, Szz, nNhCaCoords[0], nNhCaCoords[2], 
				       nNhCaCoords[1], mat, depth+1, N, ppS, ppVec, nToNHVec2, nToCAVec2, ca2);
 		    }
		}
	    }
	}else if (!phiVec.isEmpty()){
	    for (j =0; j<phiVec.size(); j++){
		phi = ((Double)phiVec.elementAt(j)).doubleValue();
		System.out.println(depth+":  "+phi/Const.cst);
		if (Math.abs(nhRdc - Const.noDataIndicator) > Const.eps){
		    // 		    System.out.println("dan = "+dan+" -*** ");
		    psiVec = psiCal(Syy, Szz, nhRdc / Const.nhRatio, phi, rg);
		}else if(Math.abs(dan - Const.noDataIndicator) > Const.eps){
		    // 		    System.out.println(dan+" --*** ");
		    haco = mdc.haCoCoords(phi, n, nh, ca);
		    ha = haco[0];
		    co = haco[1];
		    psiVec = psiCalNoe(dan);
		    // 		    System.out.println(dan+" --** "+psiVec.size());
		}
		if (!psiVec.isEmpty()){
		    for (k =0; k<psiVec.size(); k++){
			psi = ((Double)psiVec.elementAt(k)).doubleValue();
			ppS[2*depth] = phi;
			ppS[2*depth + 1] = psi;
			// 			System.out.println(depth+" *  "+dnnCal+"  "+dnn);
			nNhCaCoords = mdc.nNhCaCal(phi, psi, n, nh, ca); //compute the n, nh and ca residue (i+1)
			dnnCal = internuclearDistance(nNhCaCoords[2], nh);
// 			n  = nNhCaCoords[0];
// 			ca = nNhCaCoords[1];
// 			nh = nNhCaCoords[2];
			disToCa2 = internuclearDistance(nNhCaCoords[1], ca2);
			System.out.println(depth+" *  "+dnnCal+"  "+dnn+"   "+disToCa2);
 			if (disToCa2 < (N - depth) * Const.dCA2CA && Math.abs(dnn-dnnCal) < 1.80){
			    mat = newRG(phi, psi, rg, true); //compute a new rotation matrix rg
			    phiPsiByRdcNoe(asgVec, restraints, Syy, Szz, nNhCaCoords[0], nNhCaCoords[2], 
					   nNhCaCoords[1], mat, depth+1, N, ppS, ppVec, nToNHVec2, nToCAVec2, ca2);
 			}
		    }
		}else{
		    for (k=0; k < (360 / resolution); k++){
			psi = k * resolution * Math.PI / 360.0 - Math.PI;
			ppS[2*depth] = phi;
			ppS[2*depth + 1] = psi;
			nNhCaCoords = mdc.nNhCaCal(phi, psi, n, nh, ca); //compute the n, nh and ca for next residue
			dnnCal = internuclearDistance(nNhCaCoords[2], nh);
			disToCa2 = internuclearDistance(ca, ca2);
			System.out.println(depth+" **  "+dnnCal+"  "+dnn+"   "+disToCa2);
 			if (disToCa2 < (N - depth) * Const.dCA2CA && Math.abs(dnn-dnnCal) < 1.80){
			    mat = newRG(phi, psi, rg, true); //compute a new rotation matrix rg
			    phiPsiByRdcNoe(asgVec, restraints, Syy, Szz, nNhCaCoords[0], nNhCaCoords[2], 
					   nNhCaCoords[1], mat, depth+1, N, ppS, ppVec, nToNHVec2, nToCAVec2, ca2);
 			}
		    }
		}
	    }
	}else{	
	    for (j=0; j < (360 / resolution); j++){
		phi = j * resolution * Math.PI / 360.0 - Math.PI;
		if (Math.abs(nhRdc - Const.noDataIndicator) > Const.eps){
		    psiVec = psiCal(Syy, Szz, nhRdc, phi, rg);
		}else if(Math.abs(dan - Const.noDataIndicator) > Const.eps){
		    haco = mdc.haCoCoords(phi, n, nh, ca, false);
		    ha = haco[0];
		    co = haco[1];
		    psiVec = psiCalNoe(dan);
		}
		if (!psiVec.isEmpty()){
		    for (k =0; k<psiVec.size(); k++){
			psi = ((Double)psiVec.elementAt(k)).doubleValue();
			ppS[2*depth] = phi;
			ppS[2*depth + 1] = psi;
			nNhCaCoords = mdc.nNhCaCal(phi, psi,n, nh, ca); //compute the n, nh and ca for next residue
			dnnCal = internuclearDistance(nNhCaCoords[2], nh);
			disToCa2 = internuclearDistance(ca, ca2);
			System.out.println(depth+" ***  "+dnnCal+"  "+dnn+"   "+disToCa2);
 			if (disToCa2 < (N - depth) * Const.dCA2CA && Math.abs(dnn-dnnCal) < 1.80){
			    mat = newRG(phi, psi, rg, true); //compute a new rotation matrix rg
			    phiPsiByRdcNoe(asgVec, restraints, Syy, Szz, nNhCaCoords[0], nNhCaCoords[2], 
					   nNhCaCoords[1], mat, depth+1, N, ppS, ppVec, nToNHVec2, nToCAVec2, ca2);
 			}
		    }
		}else{
		    for (k=0; k < (360 / resolution); k++){
			psi = k * resolution * Math.PI / 360.0 - Math.PI;
			ppS[2*depth] = phi;
			ppS[2*depth + 1] = psi;
			nNhCaCoords = mdc.nNhCaCal( phi, psi,n, nh, ca); //compute the n, nh and ca for next residue
			dnnCal = internuclearDistance(nNhCaCoords[2], nh);
			disToCa2 = internuclearDistance(ca, ca2);
			System.out.println(depth+" ****  "+dnnCal+"  "+dnn+"   "+disToCa2);
 			if (disToCa2 < (N - depth) * Const.dCA2CA && Math.abs(dnn-dnnCal) < 1.80){
			    mat = newRG(phi, psi, rg, true); //compute a new rotation matrix rg
			    phiPsiByRdcNoe(asgVec, restraints, Syy, Szz, nNhCaCoords[0], nNhCaCoords[2], 
					   nNhCaCoords[1], mat, depth+1, N, ppS, ppVec, nToNHVec2, nToCAVec2, ca2);
 			}
		    }
		}
	    }
	}
	return false;
    }
    
  //  /** A recusive function to compute all the backbone (phi, psi) angles for an $m$-residue fragment 
//     * This is a version for real computation, NOT for collecting running statistics.
//     @param chNhRdc the RDC data in double array.
//     @param rg  a matrix to rotate the POF to a coodinate system defined in the first peptide plane.
//     @param Syy  Saupe elements computed from both NH and CH rdcs
//     @param Szz  Saupe elements computed from both NH and CH rdcs
//     @param i    the current depth of the depth-first search. Begin with i = 0; 
//     @param N    the total depth of the depth-first search, the same as the number of residues
//     @param ppS  a temporary array for saving intermediate (\phi, \psi) angles.
//     @param ppVec an array for returning final (\phi, \psi) angles.
//     */
//     public boolean dfsBackward (double[][] chNhRdc, double Syy, double Szz, Matrix rg, int i, int N, 
// 				double[][] ppS, Vector ppVec,  double phi59, double [] n2ca59, double [] n2nh59){
// 	int j, m;
// 	Vector phiVec = new Vector();
// 	Vector psiVec = new Vector();
// 	double phi = 0.0, psi = 0.0;
// 	Matrix mat = new Matrix(3,3);
// 	Matrix r2yInv = new Matrix(3,3);
// 	Matrix r4zInv = new Matrix(3,3);
// 	boolean hasSolution = false;
// 	double [] n2caVec = new double[3];
// 	double [] n2nhVec = new double[3];
// 	if(i == N-4 ){ //at the end
// 	    psiVec = psiCalBackward(Syy, Szz, chNhRdc[i][0], rg); //
// 	    if (!psiVec.isEmpty()){
// 		System.out.println("psi "+i+"  "+psiVec.size());
// 		for (m = 0; m <psiVec.size(); m++){
// 		    psi = ((Double)psiVec.elementAt(m)).doubleValue();
// 		    phiVec = phiCalBackward(Syy, Szz, chNhRdc[i][1], psi, rg); //THE NH RDC has been normalized
// 		    if ( !phiVec.isEmpty() ){
// 			System.out.println("phi "+i+"  "+phiVec.size());
// 			for (j=0; j < phiVec.size(); j++){
// 			    phi = ((Double)phiVec.elementAt(j)).doubleValue();
// 			    r2yInv = mat.rotationMat(phi, "-y");
// 			    r4zInv = mat.rotationMat(psi, "-z"); 
// 			    mat=Const.r1x9yInv.times(r2yInv.times(Const.r3xInv).times(r4zInv.times(Const.rot1.times(rg))));
// 			    n2caVec = mat.transpose().times(Const.unitN2CA);
// 			    n2nhVec = mat.transpose().times(Const.unitMinusZ);
// 			    solutionVec = phiPsiByPP13(n2caVec, n2nhVec, n2ca59, n2nh59, phi59);
// 			    if (!solutionVec.isEmpty()){
// 				for (int tt=0; tt< solutionVec.size(); tt++){
// 				    solutions = (double[])solutionVec.elementAt(tt);
// 				    psi59 = solutions[0];
// 				    phi60 = solutions[1];
// 				    psi60 = solutions[2];
// 				    ppS[i][0] = solutions[1];
// 				    ppS[i][1] = solutions[2];
// 				    ppS[i+1][0] = phi59;
// 				    ppS[i+1][1] = solutions[0];
// 				    ppVec.addElement(ppS);
// 				}
// 				hasSolution= true;
// 			    }
// 			}
// 		    }
// 		}
// 	    }
// 	    return hasSolution;
// 	}else{
// 	    psiVec = psiCalBackward(Syy, Szz, chNhRdc[i][0], rg);
// 	    if (!psiVec.isEmpty()){
// 		System.out.println("psi "+i+"   "+psiVec.size());
// 		for (m = 0; m <psiVec.size(); m++){
// 		    psi = ((Double)psiVec.elementAt(m)).doubleValue();
// 		    phiVec = phiCalBackward(Syy, Szz, chNhRdc[i][1], psi, rg);
// 		    if ( !phiVec.isEmpty() ){
// 			System.out.println("phi "+i+"  "+phiVec.size());
// 			for (j=0; j < phiVec.size(); j++){
// 			    phi = ((Double)phiVec.elementAt(j)).doubleValue();
// 			    ppS[i][0] = phi;
// 			    ppS[i][1] = psi;
// 			    System.out.println(phi+"  "+psi);
// 			    r2yInv = mat.rotationMat(phi, "-y");
// 			    r4zInv = mat.rotationMat(psi, "-z"); 
// 			    mat=Const.r1x9yInv.times(r2yInv.times(Const.r3xInv).times(r4zInv.times(Const.rot1.times(rg))));
//  			    dfsBackward(chNhRdc, Syy, Szz, mat, i+1, N, ppS, ppVec, phi59, n2ca59, n2nh59);
// //  			    if (!(hasSolution = dfsBackward(chNhRdc, Syy, Szz, mat, i+1, N, ppS, ppVec)))
// // 				break;
// //   				return hasSolution;
// 			}
// 		    }
// 		}
// 	    }
// // 	    return false;
// 	}
//  	return false;
//     }
    
//     public void minLoop(final Vector h1Vec, final Vector asgVec2, Vector pdbVec, final Vector chRdcVec,
// 			final Vector nhRdcVec,double Syy,double Szz, int no1, int no2,double a1,double a2){

// 	double[][] restraints =restraintsBackward(no2, no1, h1Vec, asgVec2, chRdcVec, nhRdcVec, a1, a2);
// 	int i = 0, j, k;
// 	for (i=0; i<restraints.length; i++){
// 	    System.out.print( (no2 - i)+"  ");
// 	    for (j=0; j<5; j++)
// 		System.out.print(restraints[i][j]+"  ");
// 	    System.out.println();
// 	}
// 	int N = 6; //64-59 no2 - no1; //5
// 	double [][] chNhRdc = new double[N][2];
// 	for (i=0; i<chNhRdc.length; i++){
// 	    chNhRdc[i][0] = restraints[i][0];
// 	    chNhRdc[i][1] = restraints[i][1];
// 	}
// 	double [] n  = new double[3];
// 	double [] nh = new double[3];
// 	double [] ca = new double[3];
// 	double [] n2  = new double[3];
// 	double [] nh2 = new double[3];
// 	double [] ca2 = new double[3];
// 	Cartesian cc  = new Cartesian();
// 	String atom = "";
// 	int index=Collections.binarySearch(pdbVec, new Pdb(no1), new Pdb.PdbComparator()); //No.50

// 	Pdb pp = (Pdb)pdbVec.elementAt(index); 
// 	String resid = pp.getResidue();
// 	Vector atomVec = pp.getAtomVec();
// 	for (j=0; j<atomVec.size(); j++){
// 	    cc = (Cartesian)atomVec.elementAt(j);
// 	    atom = cc.getAtom();
// 	    if (atom.equals("N"))
// 		n = cc.getXYZ();
// 	    else if (atom.equals("CA"))
// 		ca = cc.getXYZ();
// 	    else if (atom.equals("H"))
// 		nh = cc.getXYZ();
// 	}
// 	double[] nToNHVec = internuclearVec(n, nh);
// 	double[] nToCAVec = internuclearVec(n, ca);
// 	Matrix rg = RgCal(nToNHVec, nToCAVec);

// 	index =Collections.binarySearch(pdbVec, new Pdb(no2+1), new Pdb.PdbComparator()); //No.65
// 	pp = (Pdb)pdbVec.elementAt(index); 
// 	resid = pp.getResidue();
// 	atomVec = pp.getAtomVec();
// 	for (j=0; j<atomVec.size(); j++){
// 	    cc = (Cartesian)atomVec.elementAt(j);
// 	    atom = cc.getAtom();
// 	    if (atom.equals("N"))
// 		n2 = cc.getXYZ();
// 	    else if (atom.equals("CA"))
// 		ca2 = cc.getXYZ();
// 	    else if (atom.equals("H"))
// 		nh2 = cc.getXYZ();
// 	}
// 	double[] nToNHVec2 = internuclearVec(n2, nh2);
// 	double[] nToCAVec2 = internuclearVec(n2, ca2);
// 	Matrix rg2 = RgCal(nToNHVec2, nToCAVec2);
// 	double dCaCa = internuclearDistance(ca2, ca);
// // 	System.out.println("dCaCa "+ dCaCa);
// 	int depth = 0;
// 	Vector ppVec = new Vector();
// 	double [][] ppS = new double[N][2];
// 	double [][] ppAngles = new double[N][2];
// 	boolean hasSolution = false;
// 	double chRmsd = 0.0;
// 	double nhRmsd = 0.0;
// 	long seed = 78312434;
// 	Random rr = new Random(seed);
// 	int nCycle = 1;//1024;
// 	for (int mm=0; mm<nCycle; mm++){
// 	    for (i=0; i<chNhRdc.length; i++){
// 		chNhRdc[i][0] = restraints[i][0] + chRmsd * rr.nextGaussian();
// 		chNhRdc[i][1] = restraints[i][1] / Const.nhRatio + nhRmsd*rr.nextGaussian();
// 	    }
// 	    ppVec = new Vector();
// 	    hasSolution = dfsBackward(chNhRdc, Syy, Szz, rg2, depth, N, ppS, ppVec);
// 	    if (!ppVec.isEmpty()){
// 		System.out.println(ppVec.size());
// 		for(j=0; j<ppVec.size(); j++){
// 		    ppAngles = (double[][])ppVec.elementAt(j);
// 		    for (k=0; k<ppAngles.length; k++)
// 			printArray(ppAngles[k]);
// 		    System.out.println();
// 		}
// 	    }
// 	}
// // 	boolean hasSln = phiPsiByRdcNoe(asgVec2, restraints, Syy, Szz, n, nh, ca, rg, depth, N,
// // 					ppS, ppVec, nToNHVec2, nToCAVec2, ca2);
//     }

    /**
   * Loop34 to41.
   * 
   * @param restraints the restraints
   * @param Syy the syy
   * @param Szz the szz
   * @param no1 the no1
   * @param no2 the no2
   * @param n the n
   * @param nh the nh
   * @param ca the ca
   * @param n2 the n2
   * @param nh2 the nh2
   * @param ca2 the ca2
   * @param phiPsiVec the phi psi vec
   * @param rdcRmsd the rdc rmsd
   * 
   * @return true, if successful
   */
  public boolean loop34To41(double[][] restraints, double Syy, double Szz, int no1, int no2, double [] n, 
			      double[] nh, double[] ca, double[] n2, double[] nh2, double[] ca2, Vector phiPsiVec, double[] rdcRmsd){
	int i, j, k;
	ModelRdc mdc = new ModelRdc();
	int N = no2 - no1 + 1; //40-34+1 = 7
	double[] nToNHVec = internuclearVec(n, nh);
	double[] nToCAVec = internuclearVec(n, ca);
	Matrix rg34 = RgCal(nToNHVec, nToCAVec); 
	Matrix rgInv34 = rg34.transpose();
	double[] nToNHVec2 = internuclearVec(n2, nh2);
	double[] nToCAVec2 = internuclearVec(n2, ca2);
	Matrix rg41 = RgCal(nToNHVec2, nToCAVec2); 

	double disCa2Ca = length(internuclearVec(ca, ca2));
	double chRdc40 = restraints[0][0];
	double nhRdc40 = restraints[0][1] / Const.nhRatio;
	double chRdc39 = restraints[1][0];
	double nhRdc39 = restraints[1][1] / Const.nhRatio;
	double chRdc38 = restraints[2][0];
	double nhRdc38 = restraints[2][1] / Const.nhRatio;
	double chRdc37 = restraints[3][0];
	double nhRdc37 = restraints[3][1] / Const.nhRatio;
	double chRdc36 = restraints[4][0];
	double nhRdc36 = restraints[4][1] / Const.nhRatio;
	double chRdc35 = restraints[5][0];
	double nhRdc35 = restraints[5][1] / Const.nhRatio;
	double chRdc34 = restraints[6][0];
	double nhRdc34 = restraints[6][1] / Const.nhRatio;
	Vector phiVec40 = new Vector();
	Vector psiVec40 = new Vector();
	Vector phiVec39 = new Vector();
	Vector psiVec39 = new Vector();
	Vector psiVec38 = new Vector();
	Vector psiVec37 = new Vector();
	Vector phiVec36 = new Vector();
	Vector psiVec36 = new Vector();
	Vector phiVec35 = new Vector();
	Vector psiVec35 = new Vector();
	Vector phiVec34 = new Vector();
	Vector psiVec34 = new Vector();

	Matrix r2yInv = new Matrix(3, 3);
	Matrix r4zInv = new Matrix(3, 3);
	Matrix rg40  = new Matrix(3, 3);
	Matrix rg39  = new Matrix(3, 3);
	Matrix rg38  = new Matrix(3, 3);
	Matrix rg37  = new Matrix(3, 3);
	Matrix rg36  = new Matrix(3, 3);
	Matrix rg35  = new Matrix(3, 3);

	Matrix r2y = new Matrix(3, 3);
	Matrix r4z = new Matrix(3, 3);
	Matrix mat = new Matrix(3, 3);
	double phi40 = 0.0, psi40 = 0.0;
	double phi39 = 0.0, psi39 = 0.0;
	double phi38 = Const.phiPro; //Pro38
	double psi38 = 0.0;
	double phi37 = Const.phiPro; //Pro37
	double psi37 = 0.0;
	double phi36 = 0.0, psi36 = 0.0;
	double phi35 = 0.0, psi35 = 0.0;
	double phi34 = 0.0, psi34 = 0.0;
	double [] solutions = new double[3];
	double [] chVec34 = new double[3];
	double [] nhVec34 = new double[3];
	double [] chVec35 = new double[3];
	double [] nhVec35 = new double[3];
	double [] chVec36 = new double[3];
	double [] nhVec36 = new double[3];

	double [] phiS = new double [N];
	double [] psiS = new double [N];
	double [][] nNhCaHa40 = new double[4][3];
	double [][] nNhCaHa39 = new double[4][3];
	double [][] nNhCaHa38 = new double[4][3];
	double [][] nNhCaHa37 = new double[4][3];
	long seed = 74851987;
	Random rr = new Random(seed);
	int nCycle = 512;
	double sigma = 8.0;
	double [] rdcRms = new double[2*N];
	for (i=0; i<2*N; i++) //in the order of CH and NH
	    rdcRms[i] = 0.0;
	double score = 0.0;
	double rmsCal = 100.0;
	double rT = 100.0;
	int count = 1;
	double dis2LastCA = 0.0;
	int slnNo = 0;
	double[] solution1 = new double[2];
	double[] solution2 = new double[2];
	double[] solution3 = new double[2];
	Vector solutionVec = new Vector();
	boolean hasSolution = false;
	boolean hasPhiPsi   = false;
// 	Vector helixPdb = pp.extractFragment(pdbVec, 23, 34);
	for(int mm=0; mm<nCycle; mm++){
	    score = 0.0;
	    for (int iii=0; iii<2*N; iii++) //in the order of CH and NH
		rdcRms[iii] = 0.0;
	    //Q40
	    chRdc40 = restraints[0][0] + sigma * rr.nextGaussian();
	    rdcRms[0]=(chRdc40 -restraints[0][0]) * (chRdc40 -restraints[0][0]);
	    psiVec40 = psiCalBackward(Syy, Szz, chRdc40, rg41);
	    for (i=0; i < psiVec40.size(); i++){
		psi40 = ((Double)psiVec40.elementAt(i)).doubleValue();
		nhRdc40 = (restraints[0][1] + 0.50*sigma *rr.nextGaussian())/Const.nhRatio;
		rdcRms[1] =(nhRdc40*Const.nhRatio -restraints[0][1]) * (nhRdc40*Const.nhRatio -restraints[0][1]);
		phiVec40 = phiCalBackward(Syy, Szz, nhRdc40, psi40, rg41);
		for (j=0; j < phiVec40.size(); j++){
		    phi40 = ((Double)phiVec40.elementAt(j)).doubleValue();
		    nNhCaHa40 = mdc.nNhCaHaByBackward(phi40, psi40, n2, nh2, ca2);
		    r2yInv = mat.rotationMat(phi40, "-y");
		    r4zInv = mat.rotationMat(psi40, "-z"); 
		    rg40 =Const.r1x9yInv.times(r2yInv.times(Const.r3xInv).times(r4zInv.times(Const.rot1.times(rg41))));
		    //D39
		    chRdc39 = restraints[1][0] + sigma * rr.nextGaussian();
		    rdcRms[2] =(chRdc39-restraints[1][0]) * (chRdc39-restraints[1][0]);
		    psiVec39 = psiCalBackward(Syy, Szz, chRdc39, rg40);
		    for (k=0; k < psiVec39.size(); k++){
			nhRdc39 = (restraints[1][1] + 0.50 * sigma * rr.nextGaussian()) / Const.nhRatio;
			rdcRms[3]=(nhRdc39*Const.nhRatio -restraints[1][1]) *(nhRdc39*Const.nhRatio-restraints[1][1]);
			psi39 = ((Double)psiVec39.elementAt(k)).doubleValue();
			phiVec39 = phiCalBackward(Syy, Szz, nhRdc39, psi39, rg40);
			for (int jj=0; jj < phiVec39.size(); jj++){
			    phi39 = ((Double)phiVec39.elementAt(jj)).doubleValue();
			    nNhCaHa39 = mdc.nNhCaHaByBackward(phi39, psi39, nNhCaHa40[0], nNhCaHa40[1], nNhCaHa40[2]);
			    r2yInv = mat.rotationMat(phi39, "-y");  //we need their transpose
			    r4zInv = mat.rotationMat(psi39, "-z"); 
			    rg39 =Const.r1x9yInv.times(r2yInv.times(Const.r3xInv).times(r4zInv.times(Const.rot1.times(rg40))));
			    //Pro38
			    chRdc38 = restraints[2][0] + sigma * rr.nextGaussian();
			    rdcRms[4] =(chRdc38-restraints[2][0]) * (chRdc38-restraints[2][0]);
			    rdcRms[5] = 0.0;
			    psiVec38 = psiCalBackward(Syy, Szz, chRdc38, rg39);
			    for (int kk=0; kk < psiVec38.size(); kk++){
				psi38 = ((Double)psiVec38.elementAt(kk)).doubleValue();
				nNhCaHa38 =mdc.nNhCaHaByBackward(phi38, psi38, nNhCaHa39[0], nNhCaHa39[1], nNhCaHa39[2]);
				r4zInv = mat.rotationMat(psi38, "-z"); 
				rg38 =Const.r1x9yInv.times(Const.r2yInvPro.times(Const.r3xInv).times(r4zInv.times(Const.rot1.times(rg39))));
				//Pro37
				chRdc37 = restraints[3][0] + sigma * rr.nextGaussian();
				rdcRms[6] =(chRdc37-restraints[3][0]) * (chRdc37-restraints[3][0]);
				rdcRms[7] = 0.0;
				psiVec37 = psiCalBackward(Syy, Szz, chRdc37, rg38);
				for (int qq=0; qq < psiVec37.size(); qq++){
				    psi37 = ((Double)psiVec37.elementAt(qq)).doubleValue();
				    nNhCaHa37 =mdc.nNhCaHaByBackward(phi37, psi37, nNhCaHa38[0], nNhCaHa38[1], nNhCaHa38[2]);
				    r4zInv = mat.rotationMat(psi37, "-z"); 
				    rg37=Const.r1x9yInv.times(Const.r2yInvPro.times(Const.r3xInv).times(r4zInv.times(Const.rot1.times(rg38))));
// 				    System.out.println ("phi40 = "+phi40 / Const.cst+", "+psi40 / Const.cst);
// 				    System.out.println ("phi39 = "+phi39 / Const.cst+", "+psi39 / Const.cst);
// 				    System.out.println ("phi38 = "+phi38 / Const.cst+", "+psi38 / Const.cst);
// 				    System.out.println ("phi37 = "+phi37 / Const.cst+", "+psi37 / Const.cst);
				    disCa2Ca = length(internuclearVec(ca, nNhCaHa37[2]));
				    if (disCa2Ca < 11){
// 					System.out.println(" dis ="+disCa2Ca);
					hasSolution =phiPsiByPP14(ca, nNhCaHa37[0], nNhCaHa37[2], nNhCaHa37[1], rg34, rg37, solutionVec);
					if (hasSolution){
					    slnNo = solutionVec.size() / 3;
					    for (int ww=0; ww< slnNo; ww += 3){
						solution1 =(double[])solutionVec.elementAt(ww);
						phi34 = solution1[0];
						psi34 = solution1[1];
						r2y = mat.rotationMat(phi34, "+y"); 
						r4z = mat.rotationMat(psi34, "+z"); 
						chVec34 = rgInv34.times(Const.r1x9yInv.times(r2y.transpose().times(Const.dirCosCHcnt)));
						chRdc34 =(-Syy-Szz) *chVec34[0]*chVec34[0] +Syy *chVec34[1]*chVec34[1] +Szz *chVec34[2]*chVec34[2];
						rdcRms[12] = (chRdc34 - restraints[6][0])*(chRdc34 - restraints[6][0]);
						rg35 = Const.rot1Inv.times(r4z.times(Const.r3x).times(r2y.times(Const.r9y1x.times(rg34))));
						rdcRms[13] = 0.0; //the NH RDC of E34 is NOT used
						solution2 =(double[])solutionVec.elementAt(ww+1);
						phi35 = solution2[0];
						psi35 = solution2[1];
						r2y = mat.rotationMat(phi35, "+y"); 
						r4z = mat.rotationMat(psi35, "+z"); 
						rg36 = Const.rot1Inv.times(r4z.times(Const.r3x).times(r2y.times(Const.r9y1x.times(rg35))));
// 						chVec35 = rg35.transpose().times(Const.r1x9yInv.times(r2y.transpose().times(Const.dirCosCHcnt)));
// 						chRdc35 =(-Syy-Szz) *chVec35[0]*chVec35[0] +Syy *chVec35[1]*chVec35[1] +Szz *chVec35[2]*chVec35[2];
						rdcRms[10] = 0.0;//(chRdc35 - restraints[5][0])*(chRdc35 - restraints[5][0]);
						nhVec35 = rg35.transpose().times(Const.unitMinusZ);
						nhRdc35 =(-Syy-Szz) *nhVec35[0]*nhVec35[0] +Syy *nhVec35[1]*nhVec35[1] +Szz *nhVec35[2]*nhVec35[2];
						rdcRms[11] = (nhRdc35*Const.nhRatio - restraints[5][1])*(nhRdc35*Const.nhRatio - restraints[5][1]);
						solution3 =(double[])solutionVec.elementAt(ww+2);
						phi36 = solution3[0];
						psi36 = solution3[1];
						r2y = mat.rotationMat(phi36, "+y"); 
						r4z = mat.rotationMat(psi36, "+z"); 
						chVec36 = rg36.transpose().times(Const.r1x9yInv.times(r2y.transpose().times(Const.dirCosCHcnt)));
						chRdc36 =(-Syy-Szz) *chVec36[0]*chVec36[0] +Syy *chVec36[1]*chVec36[1] +Szz *chVec36[2]*chVec36[2];
						rdcRms[8] = (chRdc36 - restraints[4][0])*(chRdc36 - restraints[4][0]);
						nhVec36 = rg36.transpose().times(Const.unitMinusZ);
						nhRdc36 =(-Syy-Szz) *nhVec36[0]*nhVec36[0] +Syy *nhVec36[1]*nhVec36[1] +Szz *nhVec36[2]*nhVec36[2];
						rdcRms[9] = (nhRdc36*Const.nhRatio -restraints[4][1])*(nhRdc36*Const.nhRatio -restraints[4][1]);
						score = 0.0;
						for (int kkk=0; kkk < 2 *N; kkk++)
						    score += rdcRms[kkk];
						rmsCal = Math.sqrt(0.50 * score / N);
//  						System.out.println("RMSd= "+rmsCal);
						if (rmsCal < 8.0 && rmsCal < rT ){
						    rT = rmsCal;
						    phiS[0] = phi34;
						    psiS[0] = psi34;
						    phiS[1] = phi35;
						    psiS[1] = psi35;
						    phiS[2] = phi36;
						    psiS[2] = psi36;
						    phiS[3] = phi37;
						    psiS[3] = psi37;
						    phiS[4] = phi38;
						    psiS[4] = psi38;
						    phiS[5] = phi39;
						    psiS[5] = psi39;
						    phiS[6] = phi40;
						    psiS[6] = psi40;
						    hasPhiPsi = true;
// 						    rdcRmsd[0] = rmsCal;
// 						    System.out.println("RMS34-40= "+rmsCal);
// 						    pdbVec34To40 = mdc.modelBuild(phiS, psiS, n, nh, ca, no1, false);
// 						    printArray(n);
// 						    printArray(nh);
// 						    printArray(ca);
// 						    pp= (Pdb)pdbVec34To40.lastElement();
// 						    System.out.println(pp);
// 						    System.out.println("MODEL          "+ count);
// 						    pp.print(helixPdb);
//  						    pp.print(pdbVec34To40);
// 						    System.out.println("TER");
// 						    System.out.println("ENDMODEL");
// 						    System.out.println();
// 						    count++;
						}
						score = 0.0;
					    }
					}
					solutionVec = new Vector();
				    }
				}
			    }
			}
		    }
		}
	    }
	}
	if (hasPhiPsi){
	    phiPsiVec.add(phiS);
	    phiPsiVec.add(psiS);
	    rdcRmsd[0] = rT;
	}
	return hasPhiPsi;
    }


    /**
     * Compute the CA directions of residues 2 and 3 specified by polar
     * angles (theta, phi), given the CA coordinates for residue 1 and
     * 3 and the theta polar angle for residue 2.  See Math derivation
     * for the detail; Be careful to return all the solutions.
     * 
     * @param ca1 the coordinate of the CA nucleus of residue i
     * @param n4 the n4
     * @param ca4 the ca4
     * @param nh4 the nh4
     * @param rg1 the rg1
     * @param rg4 the rg4
     * @param solutionVec the solution vec
     * 
     * @return true, if phi psi by p p14
     */

    public boolean phiPsiByPP14(double[] ca1,double[] n4,double[] ca4,double [] nh4, Matrix rg1, Matrix rg4,
				Vector solutionVec){
	int i, j, k;
	double[] n2nhVec4 = internuclearVec(n4, nh4);
	double [] ca3 = caByNextPeptidePlane(n4, rg4);
	double [] ca3Toca4Dir = internuclearVec(ca3, ca4);
	double [] ca4Cal = new double[3]; 
	double c1 = (ca3[0] - ca1[0]) / Const.dCA2CA;
	double c2 = (ca3[1] - ca1[1]) / Const.dCA2CA;
	double c3 = (ca3[2] - ca1[2]) / Const.dCA2CA;
	double cosTheta2 = 0.0; //Math.cos(theta2);
	double sinTheta2 = 0.0; //Math.sin(theta2);
	double theta2, theta3, sinTheta3;
	double[] phi0Arr = new double[2];
	double[] phi1Arr = new double[2];

	double sinPhi1 = 0.0;
	double cosPhi1 = 0.0;
	double phiCnt  = 0.0;
	double sinPhiCnt = 0.0;
	double cosPhiCnt = 0.0;
	double c = 1.0;
	double sinPhi0 = 0.0;
	double cosPhi0 = 0.0;
	double c1Cal = 0.0;
 	double c2Cal = 0.0;
	double phi2 = 0.0, phi3 = 0.0;
	boolean hasSolution = false;
	boolean hasPhiPsi = false;
	double[][] phiPsiSolution1 = new double[2][3];
	double[][] phiPsiSolution2 = new double[2][3];
	double[][] phiPsiSolution3 = new double[2][3];
	double phi34BB, psi34BB, phi35BB, psi35BB, phi36BB, psi36BB;
	Matrix rg35 = new Matrix(3, 3);
	Matrix rg36 = new Matrix(3, 3);
	Matrix rg37 = new Matrix(3, 3);
	Matrix mat = new Matrix(3, 3);
	Matrix r2y = new Matrix(3, 3);
	Matrix r4z = new Matrix(3, 3);
	double[] ca2Dir = new double[3];
	double[] ca3Dir = new double[3];
	double[] nh4Dir = new double[3];
	double[] ca2Cal = new double[3];
	double[] ca3Cal = new double[3];

	for (i = 0; i < 360; i++){ //grid-search
	    theta2 = i * Math.PI / 360;
	    sinTheta2 = Math.sin(theta2);
	    cosTheta2 = Math.cos(theta2);
	    if (Math.abs(c3 - cosTheta2) <= 1.0){
		theta3 = Math.acos(c3 - cosTheta2); //for polar angle, theta is in [0. pi] 
		sinTheta3 = Math.sin(theta3);
		if (sinTheta2 != 0.0 && sinTheta3 != 0.0){
		    cosPhi0 = 0.50*(c1*c1 + c2*c2 - sinTheta2 * sinTheta2 - sinTheta3 * sinTheta3) / (sinTheta2 * sinTheta3);
		    if (Math.abs(cosPhi0) <= 1.0){
			phi0Arr[0] = Math.acos(cosPhi0); //2 solutions for Phi0
			phi0Arr[1] = -phi0Arr[0];
			for (j = 0; j < 2; j++){
			    sinPhi0 = Math.sin(phi0Arr[j]);
			    c = Math.sqrt( (sinTheta2 * cosPhi0 + sinTheta3) * (sinTheta2 * cosPhi0 + sinTheta3) 
					   + sinTheta2 * sinTheta2 * sinPhi0 * sinPhi0);
			    if ( c != 0.0 && Math.abs(c1 / c) <= 1.0){
				sinPhiCnt = (sinTheta2 * cosPhi0 + sinTheta3) / c; //compute phiC
				cosPhiCnt = sinTheta2 * sinPhi0 / c;
				if ( cosPhiCnt >= 0 )
				    phiCnt = Math.asin(sinPhiCnt);
				else if ( cosPhiCnt < 0)
				    phiCnt = Math.PI - Math.asin(sinPhiCnt);
				phi1Arr[0] = Math.asin(c1/c);
				phi1Arr[1] = Math.PI - phi1Arr[0];
				for (k=0; k<2; k++){
				    phi2 = phiCnt - phi1Arr[k] + phi0Arr[j]; //phi2
				    phi3 = phiCnt - phi1Arr[k];   //phi3
				    if (phi2 < 0.0)
					phi2 += 2.0*Math.PI;
				    if (phi3 < 0.0)
					phi3 += 2.0*Math.PI;
				    c2Cal = sinTheta2 * Math.sin(phi2) + sinTheta3 * Math.sin(phi3);
				    if (Math.abs(c2Cal - c2) < Const.eps){
					ca2Dir[0] = Const.dCA2CA * Math.sin(theta2) * Math.cos(phi2); //ca2ca dir from residue 8 -> 9
					ca2Dir[1] = Const.dCA2CA * Math.sin(theta2) * Math.sin(phi2);
					ca2Dir[2] = Const.dCA2CA * Math.cos(theta2);
					hasPhiPsi = phiPsiByCa(rg1, ca2Dir, phiPsiSolution1); //compute phi_34, psi_34
					if(hasPhiPsi){
					    ca2Cal = addCoords(ca1, ca2Dir);  //ca of residue 35
					    ca3Dir[0] = Const.dCA2CA * Math.sin(theta3) * Math.cos(phi3);//ca35-->ca36
					    ca3Dir[1] = Const.dCA2CA * Math.sin(theta3) * Math.sin(phi3);
					    ca3Dir[2] = Const.dCA2CA * Math.cos(theta3);
					    ca3Cal = addCoords(ca2Cal, ca3Dir);  //ca of residue 36
					    ca4Cal = addCoords(ca3Cal, ca3Toca4Dir);  //ca of residue 10
					    for (int tt = 0; tt < 2; tt++){
						phi34BB = phiPsiSolution1[tt][0];
						psi34BB = phiPsiSolution1[tt][1];
						r2y = mat.rotationMat(phi34BB, "+y"); 
						r4z = mat.rotationMat(psi34BB, "+z"); 
						rg35 =Const.rot1Inv.times(r4z.times(Const.r3x).times(r2y.times(Const.r9y1x.times(rg1))));
						hasPhiPsi = phiPsiByCa(rg35, ca3Dir, phiPsiSolution2);  //compute phi_9, psi_9
						if(hasPhiPsi){
						    for (int qq = 0; qq < 2; qq++){
							phi35BB = phiPsiSolution2[qq][0];
							psi35BB = phiPsiSolution2[qq][1];
// 							System.out.println(" Angle = "+phi35BB+" G35  "+psi35BB);
							r2y = mat.rotationMat(phi35BB, "+y");
							r4z = mat.rotationMat(psi35BB, "+z"); 
							rg36 =Const.rot1Inv.times(r4z.times(Const.r3x).times(r2y.times(Const.r9y1x.times(rg35))));
							hasPhiPsi = phiPsiByCa(rg36, ca3Toca4Dir, phiPsiSolution3);  //compute phi_10, psi_10
							if(hasPhiPsi){
							    for (int uu= 0; uu < 2; uu++){
								phi36BB = phiPsiSolution3[uu][0];
								psi36BB = phiPsiSolution3[uu][1];
// 								System.out.println(" Angle = "+phi36BB+" L36  "+psi36BB);
								r2y = mat.rotationMat(phi36BB, "+y"); //transpose == inverse == rotate in opposite direction
								r4z = mat.rotationMat(psi36BB, "+z"); 
								rg37 =Const.rot1Inv.times(r4z.times(Const.r3x).times(r2y.times(Const.r9y1x.times(rg36))));
								nh4Dir = rg37.transpose().times(Const.coordNH);
// 								System.out.println(" Angle = "+interAngle(nh4Dir, n2nhVec4) / Const.cst);
								if (interAngle(nh4Dir, n2nhVec4) < 0.501 * Const.cst){
// 								    System.out.println(phi34BB/Const.cst+"  "+psi34BB/Const.cst);
// 								    System.out.println(phi35BB/Const.cst+"  "+psi35BB/Const.cst);
// 								    System.out.println(phi36BB/Const.cst+"  "+psi36BB/Const.cst);
// 								    System.out.println(" Angle = "+interAngle(nh4Dir, n2nhVec4) / Const.cst);
								    solutionVec.add(new double[]{phi34BB, psi34BB});
								    solutionVec.add(new double[]{phi35BB, psi35BB});
								    solutionVec.add(new double[]{phi36BB, psi36BB});
								    hasSolution = true;
								}//else {System.out.println(" Still error?"+ interAngle(nh4Dir, nToNHVec4) / Const.cst);}
							    }
							}
						    }
						}
					    }
					}
				    }//
				}
			    }
			}
		    }
		}
	    }
	}
	return hasSolution;
    } 

    /**
     * There is a unfinished backward method for backward computing, see the backfile in usb drive.
     * 
     * @param phi the phi
     * @param psi the psi
     * @param rg the rg
     * 
     * @return the double[]
     */

    /**
     * Method for the computation of new N->CA vector for residue $i+1$ from the computed \Phi and Psi angles.
     @param phi the \Phi angle of residue $i$ 
     @param phi the \Phi angle of residue $i$ 
     @param rg the first rotation matrix to the coordinate frame defined in the Math derivation
     @return a new n2CA vector (n2CA vector of residue $i+1$)
    */
    public double[] newNCA(double phi, double psi, Matrix rg){
	Matrix rgInv  = rg.transpose();
	Matrix r2yInv = rg.rotationMat(phi, "-y");  //we need their transpose
	Matrix r4zInv = rg.rotationMat(psi + Math.PI, "-z"); 
	double[] n2CA = rgInv.times((Const.matL).times(r2yInv.times((Const.r3xInv).times(r4zInv.times(Const.matR)))));
// 	System.out.println(n2CA[0]+"   "+n2CA[1]+"   "+n2CA[2]);
	return n2CA;
    }


//     /**
//      * Method for the computation of the rotation matrix RG for residue i
//      * from the computed \Phi and Psi angles of residue $i$ and the RG for residue i+1.
//      @param phi the \Phi angle of residue $i$ 
//      @param phi the \Psi angle of residue $i$ 
//      @param rg2 the rotation matrix to the coordinate frame defined in peptide plane $i+1$
//      @return a new rotation matrix rg, i.e. the rotation matrix for the residue $i$.
//     */
//     public Matrix newRGbackward(double phi, double psi, Matrix rg2){
// 	Matrix r2yInv = rg.rotationMat(phi, "-y");  //we need their transpose
// 	Matrix r4zInv = rg.rotationMat(psi, "-z"); 
// 	Matrix mat = rgInv.times((Const.matL).times(r2yInv.times((Const.r3xInv).times(r4zInv))));
// 	double[] nhDir = mat.times(dirCosNH);
// 	double[] n2CA = mat.times(Const.matR);
//  	ppGlobal ppg = new  ppGlobal();
// 	return ppg.RgCal(nhDir, n2CA);
//     }

    /**
 * Method for the computation of new N->CA vector for residue
 * $i+1$ from the computed \Phi and Psi angles of residue $i$.
 * 
 * @param phi the \Psi angle of residue $i$
 * @param rg the rotation matrix to the coordinate frame defined in the peptide plane $i$ as in the Math derivation
 * @param psi the psi
 * 
 * @return a new rotation matrix rg, i.e. the rotation matrix for the residue $i+1$.
 */
    public Matrix newRG(double phi, double psi, Matrix rg){
	Matrix rgInv  = rg.transpose();
	Matrix r2yInv = rg.rotationMat(phi, "-y");  //we need their transpose
	Matrix r4zInv = rg.rotationMat(psi + Math.PI, "-z"); 
	Matrix mat = rgInv.times((Const.matL).times(r2yInv.times((Const.r3xInv).times(r4zInv))));
	double[] nhDir = mat.times(dirCosNH);
	double[] n2CA = mat.times(Const.matR);
 	ppGlobal ppg = new  ppGlobal();
	return ppg.RgCal(nhDir, n2CA);
    }

    /**
     * Method for the computation of new N->CA vector for residue
     * $i+1$ from the computed \Phi and Psi angles of residue $i$.
     * 
     * @param phi the \Psi angle of residue $i$
     * @param rg the rotation matrix to the coordinate frame defined in the peptide plane $i$ as in the Math derivation
     * @param psi the psi
     * @param rightHand the right hand
     * 
     * @return a new rotation matrix rg, i.e. the rotation matrix for the residue $i+1$.
     */
    public Matrix newRG(double phi, double psi, Matrix rg, boolean rightHand){
	Matrix rgInv  = rg.transpose();
	Matrix r2yInv = rg.rotationMat(phi, "-y");  //we need their transpose
	Matrix r4zInv = rg.rotationMat(psi + Math.PI, "-z"); 
	Matrix mat = rgInv.times((Const.matL).times(r2yInv.times((Const.r3xInv).times(r4zInv))));
	double[] nhDir = mat.times(dirCosNH);
	double[] n2CA = mat.times(Const.matR);
 	ppGlobal ppg = new  ppGlobal();
	return ppg.RgCal(nhDir, n2CA, rightHand);
    }

    /**
     * Vdw energy.
     * 
     * @param pdbVec1 the pdb vec1
     * @param pdbVec2 the pdb vec2
     * @param count the count
     * 
     * @return the double
     */
    public double vdwEnergy (Vector pdbVec1, Vector pdbVec2, int[] count){
	int i, j, k, m;
	Cartesian cc = new Cartesian();
	String atom1 = "";
	String atom2 = "";
	Pdb pp1 = new Pdb();
	Pdb pp2 = new Pdb();
	Vector atomVec1 = new Vector();
	Vector atomVec2 = new Vector();
	double[] coord1 = new double[3];
	double[] coord2 = new double[3];
	double score = 0.0;
	double vdwDis = 0.0;
	int cnt = 0;
	int no1, no2;
	double vdwRadio = 1.9;
	for(i=0; i<pdbVec2.size(); i++){
	    pp2 = (Pdb)pdbVec2.elementAt(i);
	    atomVec2 = pp2.getAtomVec();
	    no2 = pp2.getResidueNo();
	    for (j=0; j<atomVec2.size(); j++){
		cc = (Cartesian)atomVec2.elementAt(j);
		atom2 = cc.getAtom();
		coord2 = cc.getXYZ();
		for(m=0; m<pdbVec1.size(); m++){
		    pp1 = (Pdb)pdbVec1.elementAt(m);
		    no1 = pp1.getResidueNo();
		    atomVec1 = pp1.getAtomVec();
		    for (k=0; k<atomVec1.size(); k++){
			cc = (Cartesian)atomVec1.elementAt(k);
			atom1 = cc.getAtom();
			coord1 = cc.getXYZ();
			vdwDis = length(internuclearVec(coord1, coord2));
			if (vdwDis < vdwRadio && no1 != no2 ){
			    if (Math.abs(no1 - no2) != 1 ){
				score += (vdwDis - vdwRadio) * (vdwDis - vdwRadio);
				cnt++;
				System.out.println(no2+" "+atom2+ "<---> "+no1+" "+atom1+ "  "+Math.sqrt(vdwDis));
			    }// else if ( ((Math.abs(no1 - no2) == 1) && ( !atom1.equals("N") && !atom2.equals("C") ) ||  (!atom1.equals("C") && !atom2.equals("N") ) )){
// 				score += (vdwDis - vdwRadio) * (vdwDis - vdwRadio);
// 				cnt++;
// 				System.out.println(no2+" "+atom2+ "<---> "+no1+" "+atom1+ "  "+Math.sqrt(vdwDis));
// 			    }
			}
		    }
		}
	    }
	}
	count[0] = cnt;
	if (cnt > 0)
	    return Math.sqrt(score/cnt);
	else return 0.0;
    }
    
    /**
     * A very simple reader for talos angle.
     * 
     * @param talosFile the name of the file
     * 
     * @return a vector of PhiPsi object
     */
    public Vector<PhiPsi> talosAngleReader(String talosFile)
    {
		Vector<PhiPsi> inputs = new Vector<PhiPsi>();
		int no;
		String resName, dih;
		double phi_lo=0.0, phi_up=0.0;
		double psi_lo=0.0, psi_up=0.0;
		double lower=0.0,upper=0.0;
		int preNo=-1;
		try{
			StreamTokenizer in =new StreamTokenizer(new FileReader(talosFile));
			while( in.nextToken() != StreamTokenizer.TT_EOF )
			{ //ignore the comments
				if (in.ttype == StreamTokenizer.TT_WORD)
					;
				else if (in.ttype == StreamTokenizer.TT_NUMBER)
				{
					no = (int)in.nval;
		  		    in.nextToken();         
		  		    resName = in.sval;
				    in.nextToken();
				    dih = in.sval;
				    in.nextToken();
				    lower = (double)in.nval;
				    in.nextToken();
				    upper=(double)in.nval;
				    if (dih.equalsIgnoreCase("PHI"))
				    {
				    	phi_lo=lower;
				    	phi_up=upper;
				    }
				    else
				    {
				    	psi_lo=lower;
				    	psi_up=upper;
				    }
				    
				    if(no==preNo)
				    	inputs.add(new PhiPsi(no, resName, 0.0, 0.0, phi_lo, phi_up,psi_lo,psi_up));
				    preNo=no;
				}	
			}			
		}catch (FileNotFoundException e) {
		    System.out.println("File not found: " + talosFile);
		}catch (IOException e) {
		    System.out.println("IOException: the stack trace is:");
		    e.printStackTrace();
		}
		return inputs;
    }

    /**
     * Compute the loop L50-L56 given the orientation of the peptide
     * planes L50 and L56. Since there is no RDCs for G53, we compute
     * the backbone angles of L50 and E51 and the phi angle of D52 by
     * forward method.  the backbone angles for R54 and T55 are
     * computed by backward method.  FOr each computed conformation,
     * we compute the link between Y59--E64 as presented in the following.
     * 
     * @param no1 L50
     * @param no2 L56
     * @param pdbVec the coordinates for the entire protein minus the loop (E51-L63)
     * @param h56To59Pdb the RDC-oriented helix L56-Y59
     * @param h1Vec the h1 vec
     * @param asgVec2 the asg vec2
     * @param chRdcVec the ch rdc vec
     * @param nhRdcVec the nh rdc vec
     * @param Syy the syy
     * @param Szz the szz
     * @param a1 the a1
     * @param a2 the a2
     */
    public void loop50To56(final Vector h1Vec, final Vector asgVec2, Vector pdbVec, Vector h56To59Pdb, Vector chRdcVec,
			   Vector nhRdcVec, double Syy,double Szz, int no1,int no2,double a1,double a2){
	int i, j, k;
	ModelRdc mdc = new ModelRdc();
	Vector phiPsiVec = new Vector();
	double[][] restraints =restraintsBackward(no2-1, no1, h1Vec, asgVec2, chRdcVec,nhRdcVec, a1,a2);
	double[][] restraints2=restraintsBackward(64, 59, h1Vec, asgVec2, chRdcVec,nhRdcVec, a1,a2);
// 	for (i=0; i < restraints.length; i++){
// 	    System.out.print( (no2-i-1)+"  ");
// 	    for (j=0; j<5; j++)
// 		System.out.print(restraints[i][j]+"  ");
// 	    System.out.println();
// 	}
// 	for (i=0; i < restraints2.length; i++){
// 	    System.out.print( (64-i)+"  ");
// 	    for (j=0; j<5; j++)
// 		System.out.print(restraints2[i][j]+"  ");
// 	    System.out.println();
// 	}

	double [] n = new double[3];
	double [] nh = new double[3];
	double [] ca = new double[3];
	Cartesian cc = new Cartesian();
	String atom = "";
	int index = Collections.binarySearch(pdbVec, new Pdb(no1-1), new Pdb.PdbComparator()); //L50,or Q49 (no1-1)
	Pdb pp = (Pdb)pdbVec.elementAt(index); 
	String resid = pp.getResidue();
	Vector atomVec = pp.getAtomVec();
	for (j=0; j<atomVec.size(); j++){
	    cc = (Cartesian)atomVec.elementAt(j);
	    atom = cc.getAtom();
	    if (atom.equals("N"))
		n = cc.getXYZ();
	    else if (atom.equals("CA"))
		ca = cc.getXYZ();
	    else if (atom.equals("H"))
		nh = cc.getXYZ();
	}
	double[] nToNHVec = internuclearVec(n, nh);
	double[] nToCAVec = internuclearVec(n, ca);
	Matrix rg49 = RgCal(nToNHVec, nToCAVec); 
	Matrix rgInv49 = rg49.transpose();

	double[] n22 = {-5.491,    -6.452,  1.940};
	double[] ha19 = {-14.351,  -2.019,  3.524};
	double[] hn23 = {-2.334,  -7.568,   4.014};//
	double disHn23Ha55 = 0.0;	
	double hn23Ha55Noe = 4.0;	
	double disN22N55 = 0.0;
	double [] n2  = new double[3];
	double [] nh2 = new double[3];
	double [] ca2 = new double[3];
	index =Collections.binarySearch(h56To59Pdb, new Pdb(no2), new Pdb.PdbComparator()); //L56
 	pp = (Pdb)h56To59Pdb.elementAt(index); 
	resid = pp.getResidue();
	atomVec = pp.getAtomVec();
	for (j=0; j<atomVec.size(); j++){
	    cc = (Cartesian)atomVec.elementAt(j);
	    atom = cc.getAtom();
	    if (atom.equals("N"))
		n2 = cc.getXYZ();
	    else if (atom.equals("CA"))
		ca2 = cc.getXYZ();
	    else if (atom.equals("H"))
		nh2 = cc.getXYZ();
	}
	double[] nToNHVec2 = internuclearVec(n2, nh2);
	double[] nToCAVec2 = internuclearVec(n2, ca2);
	Matrix rg56 = RgCal(nToNHVec2, nToCAVec2); 
	double [] n56Rot = new double[3];
// 	double chRdc55 = restraints[0][0];
// 	double nhRdc55 = restraints[0][1] / Const.nhRatio;
// 	double chRdc54 = restraints[1][0];
// 	double nhRdc54 = restraints[1][1] / Const.nhRatio;
// 	double chRdc53 = restraints[2][0];  //NO NH and CH RDCs for G53
// 	double nhRdc53 = restraints[2][1] / Const.nhRatio;
// 	double chRdc52 = restraints[3][0];
// 	double nhRdc52 = restraints[3][1] / Const.nhRatio;
// 	double chRdc51 = restraints[4][0];
// 	double nhRdc51 = restraints[4][1] / Const.nhRatio;
// 	double chRdc50 = restraints[5][0];
// 	double nhRdc50 = restraints[5][1] / Const.nhRatio;

	Vector phiVec55 = new Vector(); //55 and 54 were computed by backward-DFS
	Vector psiVec55 = new Vector();
	Vector phiVec54 = new Vector();
	Vector psiVec54 = new Vector();
	Vector phiVec50 = new Vector(); //50--52 were computed by forward-DFS
	Vector psiVec50 = new Vector();
	Vector phiVec51 = new Vector();
	Vector psiVec51 = new Vector();
	Vector phiVec52 = new Vector(); //the psi of D52 is computed by matching the two terminal peptide planes. 

	Matrix r2yInv = new Matrix(3, 3);
	Matrix r4zInv = new Matrix(3, 3);
	Matrix rg50  = new Matrix(3, 3);
	Matrix rg51  = new Matrix(3, 3);
	Matrix rg52  = new Matrix(3, 3);
	Matrix rg53 = new Matrix(3, 3);
	Matrix rg54  = new Matrix(3, 3);
	Matrix rg55  = new Matrix(3, 3);
	Matrix rg56N  = new Matrix(3, 3);
	Matrix r2y = new Matrix(3, 3);
	Matrix r4z = new Matrix(3, 3);
	Matrix mat = new Matrix(3, 3);
	double phi55 = 0.0, psi55 = 0.0;
	double phi54 = 0.0, psi54 = 0.0;
	double phi53 = 0.0, psi53 = 0.0;
	double phi52 = 0.0, psi52 = 0.0;
	double phi51 = 0.0, psi51 = 0.0;
	double phi50 = 0.0, psi50 = 0.0;
	double phi49 = 0.0, psi49 = 0.0;
	Vector solutionVec = new Vector();
	double [] solutions = new double[3];
	//the N-terminal of the computed loop 
	double [] n55Cal  = new double[3]; 
	double [] nh55Cal = new double[3];
	double [] ha55Cal = new double[3];
	double [] ca55Cal = new double[3];
	double [] n56Cal  = new double[3]; 
	double [] nh56Cal = new double[3];
	double [] ca56Cal = new double[3];
	double [] n59  = new double[3]; //the C-terminal peptide plane of the positioned helix
	double [] nh59  = new double[3];
	double [] ca59 = new double[3];
	double [] n2nh52 = new double[3];
	double [] n2ca52 = new double[3];
	double [] n2nh54 = new double[3];
	double [] n2ca54 = new double[3];
	int lengthOf2ndLoop = 6; //64-59+1
	double [] phiArray = new double[lengthOf2ndLoop]; //7?
	double [] psiArray = new double[lengthOf2ndLoop];

	int no3 = 65;
	double [] n65  = new double[3];
	double [] nh65  = new double[3];
	double [] ca65 = new double[3];
	index =Collections.binarySearch(pdbVec, new Pdb(no3), new Pdb.PdbComparator()); //E64
	pp = (Pdb)pdbVec.elementAt(index); 
	resid = pp.getResidue();
	atomVec = pp.getAtomVec();
	for (j=0; j<atomVec.size(); j++){
	    cc = (Cartesian)atomVec.elementAt(j);
	    atom = cc.getAtom();
	    if (atom.equals("N"))
		n65= cc.getXYZ();
	    else if (atom.equals("CA"))
		ca65 = cc.getXYZ();
	    else if (atom.equals("H"))
		nh65 = cc.getXYZ();
	}
	Vector ppVec = new Vector();
	int N = no2 - no1 + 1; //56-49 = 6
	double [] phiS = new double [N]; 
	double [] psiS = new double [N];

	long seed = 74851987;
	Random rr = new Random(seed);
	int nCycle = 128*1024;
	double sigma = 8.0;
	double [] rdcRms = new double[2*N];
	double nhRdc20Cal;
	for (i=0; i<2*N; i++) //in the order of CH and NH
	    rdcRms[i] = 0.0;
	double score = 0.0;
	double rmsCal = 100.0;
	double rT = 100.0;
	Vector pdbVec2To50 = pp.extractFragment(pdbVec, 2, 49); //50 before
	Vector pdbVec65To71 = pp.extractFragment(pdbVec, 65, 71); //50 before
	Vector helixPdb    = pp.extractFragment(h56To59Pdb, 56, 59);
	Vector pdbVec2 = new Vector();
	pdbVec2.addAll(pdbVec2To50);
	Vector helixPdbN = new Vector();
	Vector pdbVec50To56 = new Vector();
	Vector pdbVec59To64 = new Vector();
	int count = 1;
	double dis2LastCA = 0.0;
	double dis19Ha56Nh = 0.0;
	double nhAngle, caAngle;
	double [] n2nh23 = new double[3];
	double[] rdcRmsd = new double[1];
	int noSln = 0;
	boolean hasLoop59To65 = false;
	double chRdc, nhRdc;
	int uuu, nn, ii, iii, nnn, jj, jjj;
	double vdwScore = 0.0;
	double alpha = 0.0, beta= 0.0,   gamma = 0.0;
// 	Matrix eulerMat = mat.eulerMat(alpha, beta, gamma);
 	Matrix eulerMat = new Matrix(3, 3);
	double[]  h56To59Phi = new double[3];
	double[]  h56To59Psi = new double[3];
        for (i=0; i<3; i++){
	    h56To59Phi[i] =  Const.phiAveHelix;
	    h56To59Psi[i] =  Const.psiAveHelix+10.0 * Const.cst;
	}
	int [] cnt = new int[1];
	for(int mm=0; mm<nCycle; mm++){
	    for (uuu=0; uuu<2*N; uuu++) //in the order of CH and NH
		rdcRms[uuu] = 0.0;
	    //Q49
	    phi49 = Math.PI -2.0*Math.PI* rr.nextDouble();
	    psi49 =  Math.PI -2.0*Math.PI * rr.nextDouble();
	    r2y = mat.rotationMat(phi49, "+y"); 
	    r4z = mat.rotationMat(psi49, "+z"); 
	    rg50 = Const.rot1Inv.times(r4z.times(Const.r3x).times(r2y.times(Const.r9y1x.times(rg49))));
	    //L50
	    chRdc  = restraints[5][0] + sigma * rr.nextGaussian();
	    phiVec50 = phiCal(Syy, Szz, chRdc, rg50);
	    rdcRms[0] = (chRdc - restraints[5][0])*(chRdc - restraints[5][0]);
	    rdcRms[1] = 0.0; //no NH RDC of L50 is used in the computation
	    for (nn=0; nn< phiVec50.size(); nn++){
		phi50 =  ((Double)phiVec50.elementAt(nn)).doubleValue();
		//Use NH RDC of residue E51, (i + 1), 
		nhRdc =  (restraints[4][1] + 0.50*sigma * rr.nextGaussian()) / Const.nhRatio;
		rdcRms[3] =(nhRdc*Const.nhRatio -restraints[4][1]) * (nhRdc*Const.nhRatio -restraints[4][1]);
		psiVec50 = psiCal(Syy, Szz, nhRdc, phi50, rg50);
		for (nnn = 0; nnn<psiVec50.size(); nnn++){
		    psi50 = ((Double)psiVec50.elementAt(nnn)).doubleValue();
		    r2y = mat.rotationMat(phi50, "+y"); 
		    r4z = mat.rotationMat(psi50, "+z"); 
		    //E51
		    rg51 = Const.rot1Inv.times(r4z.times(Const.r3x).times(r2y.times(Const.r9y1x.times(rg50))));
		    chRdc  = restraints[4][0] + sigma * rr.nextGaussian();
		    rdcRms[2] = (chRdc - restraints[4][0])*(chRdc - restraints[4][0]);
		    phiVec51 = phiCal(Syy, Szz, chRdc, rg51);
		    for (ii=0; ii< phiVec51.size(); ii++){
			phi51 =  ((Double)phiVec51.elementAt(ii)).doubleValue();
			nhRdc =  (restraints[3][1] + 0.50*sigma * rr.nextGaussian()) / Const.nhRatio;
			rdcRms[5] =(nhRdc*Const.nhRatio -restraints[3][1]) * (nhRdc*Const.nhRatio -restraints[3][1]);
			psiVec51 = psiCal(Syy, Szz, nhRdc, phi51, rg51);
			for (iii = 0; iii<psiVec51.size(); iii++){
			    psi51 =  ((Double)psiVec51.elementAt(iii)).doubleValue();
			    r2y = mat.rotationMat(phi51, "+y"); 
			    r4z = mat.rotationMat(psi51, "+z"); 
			    //D52, only the phi angle is computed, 
			    rg52 = Const.rot1Inv.times(r4z.times(Const.r3x).times(r2y.times(Const.r9y1x.times(rg51))));
			    n2nh52 = rg52.transpose().times(Const.unitMinusZ);
			    n2ca52 = rg52.transpose().times(Const.n2caUnit);
			    chRdc  = restraints[3][0] + sigma * rr.nextGaussian();
			    rdcRms[4] = (chRdc - restraints[3][0])*(chRdc - restraints[3][0]);
			    rdcRms[7] = 0.0;
			    rdcRms[6] = 0.0;  //No RDCs for G53
			    phiVec52 = phiCal(Syy, Szz, chRdc, rg52);
			    for (jj=0; jj< phiVec52.size(); jj++){
				phi52 =  ((Double)phiVec52.elementAt(jj)).doubleValue();
				//T55, beginning the backward computation
				chRdc = restraints[0][0] + sigma * rr.nextGaussian();
				rdcRms[10]=(chRdc -restraints[0][0]) * (chRdc -restraints[0][0]);
				alpha = rr.nextDouble() * Math.PI/4 ;
				beta = rr.nextDouble() * Math.PI/8 ;
				gamma = rr.nextDouble() * Math.PI/4 ;
				eulerMat = mat.eulerMat(alpha, beta, gamma);
				rg56N = rg56.times(eulerMat);
				psiVec55 = psiCalBackward(Syy, Szz, chRdc, rg56N);
				for (i=0; i < psiVec55.size(); i++){
				    psi55 = ((Double)psiVec55.elementAt(i)).doubleValue();
				    nhRdc = (restraints[0][1] + 0.50*sigma * rr.nextGaussian()) / Const.nhRatio;
				    rdcRms[11] =(nhRdc*Const.nhRatio -restraints[0][1]) * (nhRdc*Const.nhRatio -restraints[0][1]);
				    phiVec55 = phiCalBackward(Syy, Szz, nhRdc, psi55, rg56N);
				    for (j=0; j < phiVec55.size(); j++){
					phi55 = ((Double)phiVec55.elementAt(j)).doubleValue();
					r2yInv = mat.rotationMat(phi55, "-y");
					r4zInv = mat.rotationMat(psi55, "-z"); 
					rg55 =Const.r1x9yInv.times(r2yInv.times(Const.r3xInv).times(r4zInv.times(Const.rot1.times(rg56N))));
					//R54
					chRdc = restraints[1][0] + sigma * rr.nextGaussian();
					rdcRms[8] =(chRdc-restraints[1][0]) * (chRdc-restraints[1][0]);
					psiVec54 = psiCalBackward(Syy, Szz, chRdc, rg55);
					for (k=0; k < psiVec54.size(); k++){
					    psi54 = ((Double)psiVec54.elementAt(k)).doubleValue();
					    nhRdc = (restraints[1][1] + 0.50 * sigma * rr.nextGaussian()) / Const.nhRatio;
					    rdcRms[9]=(nhRdc*Const.nhRatio -restraints[1][1]) *(nhRdc*Const.nhRatio-restraints[1][1]);
					    phiVec54 = phiCalBackward(Syy, Szz, nhRdc, psi54, rg55);
					    for (jjj=0; jjj < phiVec54.size(); jjj++){
						phi54 = ((Double)phiVec54.elementAt(jjj)).doubleValue();
						r2yInv = mat.rotationMat(phi54, "-y");  //we need their transpose
						r4zInv = mat.rotationMat(psi54, "-z"); 
						rg54 =Const.r1x9yInv.times(r2yInv.times(Const.r3xInv).times(r4zInv.times(Const.rot1.times(rg55))));
						n2nh54 = rg54.transpose().times(Const.unitMinusZ);
						n2ca54 = rg54.transpose().times(Const.n2caUnit);
						solutionVec =phiPsiByPP13(n2ca52, n2nh52, n2ca54, n2nh54, phi52);
						if (!solutionVec.isEmpty()){
						    for (int tt=0; tt< solutionVec.size(); tt++){
							solutions = (double[])solutionVec.elementAt(tt);
							score = 0.0;
							for (int kk=0; kk< 2 *N; kk++)
							    score += rdcRms[kk];
							rmsCal = Math.sqrt(0.50 * score / N);
							if (rmsCal < 8.0 ){
							    phiS[0] = phi49;
							    psiS[0] = psi49;
							    phiS[1] = phi50;
							    psiS[1] = psi50;
							    phiS[2] = phi51;
							    psiS[2] = psi51;
							    phiS[3] = phi52;
							    psiS[3] = solutions[0];
							    phiS[4] = solutions[1];
							    psiS[4] = solutions[2];
							    phiS[5] = phi54;
							    psiS[5] = psi54;
							    phiS[6] = phi55;
							    psiS[6] = psi55;
							    pdbVec50To56 = mdc.modelBuild(phiS, psiS, n, nh, ca, no1-1, false);
							    pp = (Pdb)pdbVec50To56.elementAt(6);
// 							    System.out.println(pp);
							    atomVec = pp.getAtomVec();
							    for (int uu=0; uu<atomVec.size(); uu++){
								cc = (Cartesian)atomVec.elementAt(uu);
								atom = cc.getAtom();
								if (atom.equals("N")) 
								    n55Cal = cc.getXYZ();
								else if (atom.equals("H")) 
								    nh55Cal = cc.getXYZ();
								else if (atom.equals("CA")) //we use N
								    ca55Cal = cc.getXYZ();
								else if (atom.equals("HA")) 
								    ha55Cal = cc.getXYZ();
							    }
							    pp = (Pdb)pdbVec50To56.lastElement(); 
							    atomVec = pp.getAtomVec();
							    for (int uu=0; uu<atomVec.size(); uu++){
								cc = (Cartesian)atomVec.elementAt(uu);
								atom = cc.getAtom();
								if (atom.equals("N")) 
								    n56Cal = cc.getXYZ();
								else if (atom.equals("H")) 
								    nh56Cal = cc.getXYZ();
								else if (atom.equals("CA")) //we use N
								    ca56Cal = cc.getXYZ();
							    }
							    dis19Ha56Nh = length(internuclearVec(ha19, nh56Cal));
							    disN22N55 = length(internuclearVec(n22, n55Cal));
							    disHn23Ha55 = length(internuclearVec(hn23, ha55Cal));
							    helixPdbN = mdc.modelBuild(h56To59Phi, h56To59Psi, n56Cal, nh56Cal, ca56Cal, 56, false);
// 							    if (disN22N55 > 4.0 && disN22N55 < 6.0)
							    if (disN22N55 > 4.0 && disN22N55 < 6.0 && dis19Ha56Nh < 8.0 && dis19Ha56Nh > 2.0 && disHn23Ha55 < 6.0){ 
// 								System.out.println( disN22N55+"  "+dis19Ha56Nh);
								System.out.println("Dis to ha19 "+dis19Ha56Nh+"  "+disN22N55+"  "+disHn23Ha55+"   "+rmsCal);
								System.out.println("Angles:   "+alpha/Const.cst+"  "+beta/Const.cst+"  "+gamma/Const.cst);
								//connect the two fragments, and compute the CA(59)<-->CA(64) distance
// 								helixPdbN =  pp.newPdb(helixPdb, eulerMat);
// 								pp = (Pdb)helixPdbN.elementAt(0); 
// // 								System.out.println(pp);
// 								atomVec = pp.getAtomVec();
// 								for (int j4=0; j4<atomVec.size(); j4++){
// 								    cc = (Cartesian)atomVec.elementAt(j4);
// 								    atom = cc.getAtom();
// 								    if (atom.equals("N"))
// 									n56Rot = cc.getXYZ();
// 								}
// 								helixPdbN = pp.newPdbByTranslation(helixPdbN, internuclearVec(n56Rot, n56Cal));
								pp = (Pdb)helixPdbN.elementAt(3);//lastElement(); 
								atomVec = pp.getAtomVec();
//  								System.out.println(pp);
// 								atomVec = pp.getAtomVec();
								for (int vv=0; vv<atomVec.size(); vv++){
								    cc = (Cartesian)atomVec.elementAt(vv);
								    atom = cc.getAtom();
								    if (atom.equals("N")) //we use N
									n59 = cc.getXYZ();
								    else if (atom.equals("H")) 
									nh59 = cc.getXYZ();
								    else if (atom.equals("CA")) 
									ca59 = cc.getXYZ();
								}
								dis2LastCA = length(internuclearVec(ca59, ca65));
 								System.out.println("disCa2Ca = "+dis2LastCA);
								if (dis2LastCA > 8.50  && dis2LastCA < 13.0){
								    System.out.println("disCa2Ca == "+dis2LastCA);
								    phiPsiVec = new Vector();
								    pdbVec2.addAll(pdbVec50To56);
								    pdbVec2.addAll(helixPdbN);
								    pdbVec2.addAll(pdbVec65To71);
//  								    pp.print(pdbVec50To56);
//  								    pp.print(pdbVec2);
								    vdwScore = vdwEnergy(pdbVec50To56, pdbVec2, cnt);
								    pdbVec2 = new Vector();
								    pdbVec2.addAll(pdbVec2To50);
								    System.out.println("vdw "+vdwScore);
								    if ( (vdwScore < 0.5 && cnt[0] < 9) || (vdwScore < 0.75 && cnt[0] < 5)  ||  (vdwScore < 1.0 && cnt[0] < 3) ){
									//rT = rdcRmsd[0] + rmsCal;
									hasLoop59To65=loop59To65(restraints2,Syy,Szz,59,64, n59,nh59,ca59,n65,nh65,ca65,phiPsiVec,rdcRmsd);
									//System.out.println(dis2LastCA+"  "+rmsCal+"  Score= "+rT+"  "+rdcRmsd[0]);
									if(!phiPsiVec.isEmpty()){
									    rT = rdcRmsd[0] + rmsCal;
									    noSln = phiPsiVec.size() / 2;
									    System.out.println(noSln+"  "+dis2LastCA+"  "+rmsCal+"  Score= "+rT+"  "+rdcRmsd[0]+"  "+vdwScore);
									    for (int ww=0; ww<noSln; ww += 2){
										phiArray = (double[])phiPsiVec.elementAt(ww);
										psiArray = (double[])phiPsiVec.elementAt(ww+1);
										pdbVec59To64 = mdc.modelBuild(phiArray, psiArray, n59, nh59, ca59, 59, false);
										System.out.println("MODEL      "+ count);
										pp.print(pdbVec2To50);
										pp.print(pdbVec50To56);
										pp.print(helixPdbN);
										pp.print(pdbVec59To64);
										pp.print(pdbVec65To71);
										System.out.println("TER");
										System.out.println("ENDMODEL");
										System.out.println();
										count++;
									    }
									}
								    }
								}
							    }
							}
						    }
						}
					    }
					}
				    }
				}
			    }
			}
		    }
		}
	    }
	}
    }

    /**
     * Compute the loop Y59 <--> K63 given the absolute coordinates for Y59 and E64.
     * 
     * @param no1 the N-terminal of the loop. (Y59 in ubiquitin)
     * @param no2 the C-terminal of the loop. (K63 in ubiquitin)
     * @param n   the amide coordinate of the beginning residue (Y59) of the loop
     * @param nh  the amide proton coordinate of the beginning residue (Y59) of the loop
     * @param ca  the CA coordinate of the beginning residue (Y59) of the loop
     * @param n2   the amide coordinate of the ending residue (E64) of the loop
     * @param nh2  the amide proton coordinate of the ending residue (E64) of the loop
     * @param ca2  the CA coordinate of the beginning residue (E64) of the loop
     * @param phiPsiVec for returning the backbone angles.
     * @param restraints the restraints
     * @param Syy the syy
     * @param Szz the szz
     * @param rdcRmsd the rdc rmsd
     * 
     * @return true, if loop59 to65
     */
    public boolean loop59To65(double[][] restraints, double Syy, double Szz, int no1, int no2, double [] n, 
			      double[] nh, double[] ca, double[] n2, double[] nh2, double[] ca2, Vector phiPsiVec, double[] rdcRmsd){
	int i, j, k;
	ModelRdc mdc = new ModelRdc();
	int N = no2 - no1 + 1; //64-59+1 = 6
	double[] nToNHVec = internuclearVec(n, nh);
	double[] nToCAVec = internuclearVec(n, ca);
	Matrix rg59 = RgCal(nToNHVec, nToCAVec); 
	Matrix rgInv59 = rg59.transpose();
	double[] nToNHVec2 = internuclearVec(n2, nh2);
	double[] nToCAVec2 = internuclearVec(n2, ca2);
	Matrix rg65 = RgCal(nToNHVec2, nToCAVec2); 

	double disCa2Ca = length(internuclearVec(ca, ca2));
// 	double chRdc63 = restraints[0][0];
// 	double nhRdc63 = restraints[0][1] / Const.nhRatio;
// 	double chRdc62 = restraints[1][0];
// 	double nhRdc62 = restraints[1][1] / Const.nhRatio;
// 	double chRdc61 = restraints[2][0];
// 	double nhRdc61 = restraints[2][1] / Const.nhRatio;
// 	double chRdc60 = restraints[3][0];
// 	double nhRdc60 = restraints[3][1] / Const.nhRatio;
// 	double chRdc59 = restraints[4][0];
// 	double nhRdc59 = restraints[4][1] / Const.nhRatio;
	Vector phiVec64 = new Vector();
	Vector psiVec64 = new Vector();
	Vector phiVec63 = new Vector();
	Vector psiVec63 = new Vector();
	Vector phiVec62 = new Vector();
	Vector psiVec62 = new Vector();

	Matrix r2yInv = new Matrix(3, 3);
	Matrix r4zInv = new Matrix(3, 3);
	Matrix rg64  = new Matrix(3, 3);
	Matrix rg63  = new Matrix(3, 3);
	Matrix rg62  = new Matrix(3, 3);
	Matrix rg61  = new Matrix(3, 3);
	Matrix rg60  = new Matrix(3, 3);

	Matrix r2y = new Matrix(3, 3);
	Matrix r4z = new Matrix(3, 3);
	Matrix mat = new Matrix(3, 3);
	double phi64 = 0.0, psi64 = 0.0;
	double phi63 = 0.0, psi63 = 0.0;
	double phi62 = 0.0, psi62 = 0.0;
	double phi61 = 0.0, psi61 = 0.0;
	double phi60 = 0.0, psi60 = 0.0;
	double phi59 = 0.0, psi59 = 0.0;
	double [] chVec59 = new double[3];
	double [] chVec60 = new double[3];
	double [] nhVec60 = new double[3];
	double [] chVec61 = new double[3];
	double [] nhVec61 = new double[3];
	double [] solutions = new double[3];
	double chRdc59, chRdc60, nhRdc60, chRdc61, nhRdc61;
	double [] phiS = new double [N];
	double [] psiS = new double [N];
	double [][] nNhCaHa64 = new double[4][3];
	double [][] nNhCaHa63 = new double[4][3];
	double [][] nNhCaHa62 = new double[4][3];
	double [][] nNhCaHa61 = new double[4][3];
	double [][] nNhCaHa60 = new double[4][3];
	long seed = 74851987;
	Random rr = new Random(seed);
	int nCycle = 1024;
	double sigma = 8.0;
	double [] rdcRms = new double[2*N];
	for (i=0; i<2*N; i++) //in the order of CH and NH
	    rdcRms[i] = 0.0;
	double score = 0.0;
	double rmsCal = 100.0;
	double rT = 100.0;
	int count = 1;
	double dis2LastCA = 0.0;
	int slnNo = 0;
	double[] solution1 = new double[2];
	double[] solution2 = new double[2];
	double[] solution3 = new double[2];
	Vector solutionVec = new Vector();
	boolean hasSolution = false;
	boolean hasPhiPsi   = false;
// 	Vector helixPdb = pp.extractFragment(pdbVec, 23, 34);
	double chRdc, nhRdc;
	int uu, ii, vv;
	for(int mm=0; mm<nCycle; mm++){
	    score = 0.0;
	    for (int iii=0; iii<2*N; iii++) //in the order of CH and NH
		rdcRms[iii] = 0.0;
	    //E64
	    chRdc = restraints[0][0] + sigma * rr.nextGaussian();
	    rdcRms[0]=(chRdc -restraints[0][0]) * (chRdc -restraints[0][0]);
	    psiVec64 = psiCalBackward(Syy, Szz, chRdc, rg65);
	    for (i=0; i < psiVec64.size(); i++){
		psi64 = ((Double)psiVec64.elementAt(i)).doubleValue();
		nhRdc = (restraints[0][1] + 0.50*sigma *rr.nextGaussian())/Const.nhRatio;
		rdcRms[1] =(nhRdc*Const.nhRatio -restraints[0][1]) * (nhRdc*Const.nhRatio -restraints[0][1]);
		phiVec64 = phiCalBackward(Syy, Szz, nhRdc, psi64, rg65);
		for (uu=0; uu < phiVec64.size(); uu++){
		    phi64 = ((Double)phiVec64.elementAt(uu)).doubleValue();
		    nNhCaHa64 = mdc.nNhCaHaByBackward(phi64, psi64, n2, nh2, ca2);
		    r2yInv = mat.rotationMat(phi64, "-y");
		    r4zInv = mat.rotationMat(psi64, "-z"); 
		    rg64 =Const.r1x9yInv.times(r2yInv.times(Const.r3xInv).times(r4zInv.times(Const.rot1.times(rg65))));
		    //K63
		    chRdc = restraints[1][0] + sigma * rr.nextGaussian();
		    rdcRms[2]=(chRdc -restraints[1][0]) * (chRdc -restraints[1][0]);
		    psiVec63 = psiCalBackward(Syy, Szz, chRdc, rg64);
		    for (ii=0; ii < psiVec63.size(); ii++){
			psi63 = ((Double)psiVec63.elementAt(ii)).doubleValue();
			nhRdc = (restraints[1][1] + 0.50*sigma *rr.nextGaussian())/Const.nhRatio;
			rdcRms[3] =(nhRdc*Const.nhRatio -restraints[1][1]) * (nhRdc*Const.nhRatio -restraints[1][1]);
			phiVec63 = phiCalBackward(Syy, Szz, nhRdc, psi63, rg64);
			for (j=0; j < phiVec63.size(); j++){
			    phi63 = ((Double)phiVec63.elementAt(j)).doubleValue();
			    nNhCaHa63 = mdc.nNhCaHaByBackward(phi63, psi63, nNhCaHa64[0], nNhCaHa64[1], nNhCaHa64[2]);
			    r2yInv = mat.rotationMat(phi63, "-y");
			    r4zInv = mat.rotationMat(psi63, "-z"); 
			    rg63 =Const.r1x9yInv.times(r2yInv.times(Const.r3xInv).times(r4zInv.times(Const.rot1.times(rg64))));
			    //Q62
			    chRdc = restraints[2][0] + sigma * rr.nextGaussian();
			    rdcRms[4] =(chRdc-restraints[2][0]) * (chRdc-restraints[2][0]);
			    psiVec62 = psiCalBackward(Syy, Szz, chRdc, rg63);
			    for (k=0; k < psiVec62.size(); k++){
				psi62 = ((Double)psiVec62.elementAt(k)).doubleValue();
				nhRdc = (restraints[2][1] + 0.50 * sigma * rr.nextGaussian()) / Const.nhRatio;
				rdcRms[5]=(nhRdc*Const.nhRatio -restraints[2][1]) *(nhRdc*Const.nhRatio-restraints[2][1]);
				phiVec62 = phiCalBackward(Syy, Szz, nhRdc, psi62, rg63);
				for (int jj=0; jj < phiVec62.size(); jj++){
				    phi62 = ((Double)phiVec62.elementAt(jj)).doubleValue();
				    nNhCaHa62 = mdc.nNhCaHaByBackward(phi62, psi62, nNhCaHa63[0], nNhCaHa63[1], nNhCaHa63[2]);
				    r2yInv = mat.rotationMat(phi62, "-y");  //we need their transpose
				    r4zInv = mat.rotationMat(psi62, "-z"); 
				    rg62 =Const.r1x9yInv.times(r2yInv.times(Const.r3xInv).times(r4zInv.times(Const.rot1.times(rg63))));
				    //Compute the backbone angles for residue Y59, N60 and I61
				    disCa2Ca = length(internuclearVec(ca, nNhCaHa62[2]));
// 				    System.out.println(" disCa2Ca **= "+disCa2Ca);
				    if (disCa2Ca < 11 && disCa2Ca > 8){
					hasSolution =phiPsiByPP14(ca, nNhCaHa62[0], nNhCaHa62[2], nNhCaHa62[1], rg59, rg62, solutionVec);
					if (hasSolution){
					    slnNo = solutionVec.size() / 3;
					    for (int ww=0; ww< slnNo; ww += 3){
						solution1 =(double[])solutionVec.elementAt(ww); //Y59
						phi59 = solution1[0];
						psi59 = solution1[1];
						r2y = mat.rotationMat(phi59, "+y"); 
						r4z = mat.rotationMat(psi59, "+z"); 
						chVec59 = rgInv59.times(Const.r1x9yInv.times(r2y.transpose().times(Const.dirCosCHcnt)));
						chRdc59 =(-Syy-Szz) *chVec59[0]*chVec59[0] +Syy *chVec59[1]*chVec59[1] +Szz *chVec59[2]*chVec59[2];
						rdcRms[10] = (chRdc59 - restraints[5][0])*(chRdc59 - restraints[5][0]);
						rdcRms[11] = 0.0;  //NH RDC for 59 is not computed
						//N60
						rg60 = Const.rot1Inv.times(r4z.times(Const.r3x).times(r2y.times(Const.r9y1x.times(rg59))));
						nhVec60 = rg60.transpose().times(Const.unitMinusZ);
						nhRdc60 =(-Syy-Szz) *nhVec60[0]*nhVec60[0] +Syy *nhVec60[1]*nhVec60[1] +Szz *nhVec60[2]*nhVec60[2];
						rdcRms[9] = (nhRdc60*Const.nhRatio - restraints[4][1])*(nhRdc60*Const.nhRatio - restraints[4][1]);
						solution2 =(double[])solutionVec.elementAt(ww+1);
						phi60 = solution2[0];
						psi60 = solution2[1];
						r2y = mat.rotationMat(phi60, "+y"); 
						r4z = mat.rotationMat(psi60, "+z"); 
						chVec60 = rg60.transpose().times(Const.r1x9yInv.times(r2y.transpose().times(Const.dirCosCHcnt)));
						chRdc60 =(-Syy-Szz) *chVec60[0]*chVec60[0] +Syy *chVec60[1]*chVec60[1] +Szz*chVec60[2]*chVec60[2];
						rdcRms[8] = (chRdc60 - restraints[4][0])*(chRdc60 - restraints[4][0]);
						//I61
						rg61 = Const.rot1Inv.times(r4z.times(Const.r3x).times(r2y.times(Const.r9y1x.times(rg60))));				
						nhVec61 = rg61.transpose().times(Const.unitMinusZ);
						nhRdc61 =(-Syy-Szz) *nhVec61[0]*nhVec61[0] +Syy *nhVec61[1]*nhVec61[1] +Szz *nhVec61[2]*nhVec61[2];
						rdcRms[7] = (nhRdc61*Const.nhRatio - restraints[3][1])*(nhRdc61*Const.nhRatio -restraints[3][1]);
						solution3 =(double[])solutionVec.elementAt(ww+2);
						phi61 = solution3[0];
						psi61 = solution3[1];
						r2y = mat.rotationMat(phi61, "+y"); 
						r4z = mat.rotationMat(psi61, "+z"); 
						chVec61 = rg61.transpose().times(Const.r1x9yInv.times(r2y.transpose().times(Const.dirCosCHcnt)));
						chRdc61 =(-Syy-Szz) *chVec61[0]*chVec61[0] +Syy *chVec61[1]*chVec61[1] +Szz*chVec61[2]*chVec61[2];
						rdcRms[6] = (chRdc61 - restraints[3][0])*(chRdc61 - restraints[3][0]);
						//rg62 = Const.rot1Inv.times(r4z.times(Const.r3x).times(r2y.times(Const.r9y1x.times(rg61))));
						score = 0.0;
						for (int kkk=0; kkk < 2 *N; kkk++)
						    score += rdcRms[kkk];
						rmsCal = Math.sqrt(0.50 * score / N);
// 						System.out.println(" disCa2Ca59-65 "+disCa2Ca+"  "+rmsCal);
						if (rmsCal < 8.0 && rmsCal < rT ){
						    System.out.println("RMSd= "+rmsCal);
						    System.out.println(" dis ="+disCa2Ca);

						    rT = rmsCal;
						    phiS[0] = phi59;
						    psiS[0] = psi59;
						    phiS[1] = phi60;
						    psiS[1] = psi60;
						    phiS[2] = phi61;
						    psiS[2] = psi61;
						    phiS[3] = phi62;
						    psiS[3] = psi62;
						    phiS[4] = phi63;
						    psiS[4] = psi63;
						    phiS[5] = phi64;
						    psiS[5] = psi64;
						    hasPhiPsi = true;
						}
						score = 0.0;
					    }
					}
					solutionVec = new Vector();
				    }
				}
			    }
			}
		    }
		}
	    }
	}
	if (hasPhiPsi){
	    phiPsiVec.add(phiS);
	    phiPsiVec.add(psiS);
	    rdcRmsd[0] = rT;
	}
	return hasPhiPsi;
    }

    /* (non-Javadoc)
     * @see java.lang.Object#clone()
     */
    protected Object clone(){
        try {
	    Object s = super.clone();     
	    return s;                     
        } catch (CloneNotSupportedException e) {
	    throw new InternalError();
        }
    }
}
