package rdcPanda;

///////////////////////////////////////////////////////////////////////////////////////////////
//	Hncacb.java
//
//	  Version:           0.1
//
//
//	  authors:
// 	  initials            name                      organization               email
//	 ---------   -----------------------        ------------------------    ------------------
//	  LW            Lincong Wang                  Dartmouth College       wlincong@cs.dartmouth.edu
//
///////////////////////////////////////////////////////////////////////////////////////////////


/*
	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.
	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	Lesser General Public License for more details.
	
	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
	USA
	
	Contact Info:
		Bruce Donald
		Duke University
		Department of Computer Science
		Levine Science Research Center (LSRC)
		Durham
		NC 27708-0129 
		USA
		brd@cs.duke.edu
	
	If you use or publish any results derived from the use of this program please cite:
	J. Zeng, J. Boyles, C. Tripathy, L. Wang, A. Yan, P. Zhou and B.R. Donald. 
	"High-Resolution Protein Structure Determination Starting with a Global Fold 
	Calculated from Exact Solutions to the RDC Equations." Submitted For Review.

	Copyright (C) 2009 Jianyang (Michael) Zeng, Lincong Wang and Bruce R. Donald		
	<signature of Bruce Donald>, June 2008 and January 2009
	Bruce Donald, Professor of Computer Science
 */




import java.io. *;
import java.util. *;

// TODO: Auto-generated Javadoc
/** * 
 *  
*   This class defines data structures related to Hncacb. Written by Lincong Wang (2001-2005).
*/
public class Hncacb{
    
    /** The peak no. */
    private int peakNo;
    
    /** The residue no. */
    private int    residueNo; 
    
    /** The residue. */
    private String residue;
    
    /** The cs h1. */
    private double csH1; 
    
    /** The cs n15. */
    private double csN15;
    
    /** The cs c13. */
    private double csC13;
    
    /** The intensity. */
    private double intensity;

  /**
   * Instantiates a new hncacb.
   */
  public Hncacb(){
	peakNo = 0;
	residueNo = 0;
	residue = null;
	csH1 = 0.0; 
        csN15 = 0.0;
        csC13 =0.0;
	intensity = 0.0;
    }		
    
    /**
     * Instantiates a new hncacb.
     * 
     * @param no the no
     */
    public Hncacb(int no){
	peakNo = 0;
	residueNo = no;
	residue = null;
	csH1 = 0.0; 
        csN15 = 0.0;
        csC13 = 0.0;
	intensity = 0.0;
    }		
    
    /**
     * Instantiates a new hncacb.
     * 
     * @param pkNo the pk no
     * @param no the no
     * @param resid the resid
     * @param H1 the h1
     * @param N15 the n15
     * @param c13Vec the c13 vec
     * @param inten the inten
     */
    public Hncacb(int pkNo, int no,String resid, double H1, double N15, double c13Vec, double inten){
	peakNo = pkNo;
	residueNo = no;
	residue = resid;
	csH1 = H1; 
        csN15 = N15;
        csC13 = c13Vec;
	intensity = inten;
    }	

  //getting the values	
    /**
   * Gets the peak no.
   * 
   * @return the peak no
   */
  public int getPeakNo(){
	return peakNo;
    }	
    
    /**
     * Gets the residue no.
     * 
     * @return the residue no
     */
    public int getResidueNo(){
	return residueNo;
    }	
    
    /**
     * Gets the h1.
     * 
     * @return the h1
     */
    public double getH1(){
	return csH1;
    }	
    
    /**
     * Gets the n15.
     * 
     * @return the n15
     */
    public double getN15(){
	return csN15;
    }	
    
    /**
     * Gets the c13.
     * 
     * @return the c13
     */
    public double getC13(){
	return csC13;
    }	
    
    /**
     * Gets the intensity.
     * 
     * @return the intensity
     */
    public double getIntensity(){
	return intensity;
    }	

    //set the values	
    /**
     * Sets the residue no.
     * 
     * @param R the new residue no
     */
    public void setResidueNo(int R){
	residueNo = R;
    }	

    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    public String toString(){
	String desc = String.valueOf(peakNo) +"  "+String.valueOf(residueNo) +"    "+ String.valueOf(csH1)
	    +"  "+String.valueOf(csN15)+"   "+String.valueOf(csC13)+"  "+String.valueOf(intensity);
	return desc;	
    }

    /**
     * The Class resNoComparator.
     */
    public static class resNoComparator implements Comparator{
	
	/* (non-Javadoc)
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(Object o1, Object o2){
	    Hncacb n1 = (Hncacb)o1;
	    Hncacb n2 = (Hncacb)o2;
	    double d1 = n1.getResidueNo();
	    double d2 = n2.getResidueNo();
	    if (d1 < d2)
		return -1;
	    else if (d1 > d2)
		return 1;
	    else return 0;
	}
    }

    /**
     * The Class peakNoComparator.
     */
    public static class peakNoComparator implements Comparator{
	
	/* (non-Javadoc)
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(Object o1, Object o2){
	    Hncacb n1 = (Hncacb)o1;
	    Hncacb n2 = (Hncacb)o2;
	    double d1 = n1.getPeakNo();
	    double d2 = n2.getPeakNo();
	    if (d1 < d2)
		return -1;
	    else if (d1 > d2)
		return 1;
	    else return 0;
	}
    }

    /**
     * The Class hnComparator.
     */
    public static class hnComparator implements Comparator{
	
	/* (non-Javadoc)
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(Object o1, Object o2){
	    Hncacb n1 = (Hncacb)o1;
	    Hncacb n2 = (Hncacb)o2;
	    double d1 = n1.getH1();
	    double d2 = n2.getH1();
	    if (d1 < d2)
		return -1;
	    else if (d1 > d2)
		return 1;
	    else return 0;
	}
    }

    /**
     * The Class n15Comparator.
     */
    public static class n15Comparator implements Comparator{
	
	/* (non-Javadoc)
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(Object o1, Object o2){
	    Hncacb n1 = (Hncacb)o1;
	    Hncacb n2 = (Hncacb)o2;
	    double d1 = n1.getN15();
	    double d2 = n2.getN15();
	    if (d1 < d2)
		return -1;
	    else if (d1 > d2)
		return 1;
	    else return 0;
	}
    }

    /**
     * The Class c13Comparator.
     */
    public static class c13Comparator implements Comparator{
	
	/* (non-Javadoc)
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(Object o1, Object o2){
	    Hncacb n1 = (Hncacb)o1;
	    Hncacb n2 = (Hncacb)o2;
	    double d1 = n1.getC13();
	    double d2 = n2.getC13();
	    if (d1 < d2)
		return -1;
	    else if (d1 > d2)
		return 1;
	    else return 0;
	}
    }

   /**
    * A method for reading files with RDC restraints.
    * The required file format is: residue No. RDC value, experimental error.
    * Must have only three items per line. Otherwise, errors.
    * 
    * @param filename the name of the file
    * 
    * @return a vector of RDC object
    */
    public Vector hncacbRead(String filename){
	Vector inputs = new Vector();
	int pkNo;
	double csH1, csN15, intensity;
	Vector csVec = new Vector();
	try{
	    StreamTokenizer in =
		new StreamTokenizer(
				    new FileReader(filename));
	    while( in.nextToken() != StreamTokenizer.TT_EOF ){
		if (in.ttype == StreamTokenizer.TT_WORD)
		    ;
		else if (in.ttype == StreamTokenizer.TT_NUMBER){
		    pkNo = (int)in.nval;
		    in.nextToken();
 		    csN15 = in.nval; 
		    in.nextToken();
		    in.nextToken();
		    in.nextToken();
		    csC13 = in.nval;
		    in.nextToken();
		    in.nextToken();
		    in.nextToken();
		    csH1 = in.nval;
		    in.nextToken();
		    in.nextToken();
		    in.nextToken();
 		    intensity = in.nval;
		    in.nextToken();
		    in.nextToken();
		    inputs.add(new Hncacb(pkNo, 0, "ALA", csH1, csN15, csC13, intensity));
		}	
	    }			
	}catch (FileNotFoundException e) {
	    System.out.println("File not found: " + filename);
	}catch (IOException e) {
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
	}
	return inputs;
    }
    
    /**
     * Range search.
     * 
     * @param nucType the nuc type
     * @param a the a
     * @param b the b
     * @param pkVec the pk vec
     * 
     * @return the vector
     */
    public Vector rangeSearch(String nucType, double a, double b, Vector pkVec){
	Vector outVec = new Vector();
	double csValue = 0.0;
	for (int i=0; i< pkVec.size(); i++){
	    if (nucType.equals("H1"))
		csValue = ((Hncacb)pkVec.elementAt(i)).getH1();
	    else if (nucType.equals("N15"))
		csValue = ((Hncacb)pkVec.elementAt(i)).getN15();
	    if (csValue >= a && csValue <= b)
		outVec.add(pkVec.elementAt(i));
	}
	return outVec;
    }

  /**
   * Merge hncacb.
   * 
   * @param pkVec the pk vec
   * 
   * @return the vector
   */
    public Vector mergeHNCACB(Vector pkVec){
	double epsH = 0.02;
	double epsN = 0.05;
	double csH1 = 0.0;
	double csN15 = 0.0;
	double csH1Tmp = 0.0;
	double csN15Tmp = 0.0;
	Vector outVec = new Vector();
	Vector allVec = new Vector();
	double csValue = 0.0;
	Hncacb hncc = new Hncacb();
	int i = 0, j = 0;
	Collections.sort(pkVec, new Hncacb.n15Comparator());
	int count = 0, cnt =0;
	int N = pkVec.size();
	while ( count < N){
	    hncc = (Hncacb)pkVec.elementAt(count);
	    csH1  = hncc.getH1();
	    csN15 = hncc.getN15();
	    outVec.add(hncc);
	    cnt = 1;
	    for (i=count+1; i< N; i++){
		hncc = (Hncacb)pkVec.elementAt(i);
		csH1Tmp = hncc.getH1();
		csN15Tmp = hncc.getN15();
		if ( Math.abs(csH1Tmp - csH1) < epsH && Math.abs(csN15Tmp - csN15) < epsN){
		    outVec.add(hncc);
		    cnt++;
		}
	    }
	    allVec.add(outVec);
	    outVec = new Vector();
	    count += cnt;
	}
	Vector oneVec = new Vector();
	Vector oneVecNew = new Vector();
	Vector allVecNew = new Vector();
	double csCa1 = 0.0, csCa2 = 0.0;
	double intensity1 = 0.0, intensity2 = 0.0;
	Hncacb hncc1 = new Hncacb();
	Hncacb hncc2 = new Hncacb();
	boolean bigger = false;
	for(i=0; i<allVec.size(); i++){
	    oneVec = (Vector)allVec.elementAt(i);
	    for(j=0; j<oneVec.size(); j++){
		hncc1 = (Hncacb)oneVec.elementAt(j);
		csCa1 = hncc1.getC13();
		intensity1 = hncc1.getIntensity();
 		bigger = false;
		for(int k=0; k<oneVec.size(); k++){
		    hncc2 = (Hncacb)oneVec.elementAt(k);
		    csCa2 = hncc2.getC13();
		    intensity2 = hncc2.getIntensity();
		    if ( Math.abs(csCa2 - csCa1) < epsN && Math.abs(intensity2) > Math.abs(intensity1) ){
 			bigger = true;
			hncc = hncc2;
			csCa1 = hncc.getC13();
			intensity1 = hncc.getIntensity();
		    }
		}
		if (!bigger){// && !oneVecNew.contains(hncc1))
		    if (oneVecNew.indexOf(hncc1) < 0 )
			oneVecNew.add(hncc1);
		}
		else {
		    if (oneVecNew.indexOf(hncc) < 0 )
			oneVecNew.add(hncc);  //if (bigger && !oneVecNew.contains(hncc));
		}
	    }
	    allVecNew.add(oneVecNew);
	    oneVecNew = new Vector();
	}
	return allVecNew;
    }

   
}
