package rdcPanda;

///////////////////////////////////////////////////////////////////////////////////////////////
//	Dipolar.java
//
//	  Version:           0.1
//
//
//	  authors:
// 	  initials            name                      organization               email
//	 ---------   -----------------------        ------------------------    ------------------
//	  LW            Lincong Wang                  Dartmouth College       wlincong@cs.dartmouth.edu
//    JMZ		 Jianyang (Michael) Zeng	       Duke University			zengjy@cs.duke.edu
//
///////////////////////////////////////////////////////////////////////////////////////////////



/*
	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.
	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	Lesser General Public License for more details.
	
	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
	USA
	
	Contact Info:
		Bruce Donald
		Duke University
		Department of Computer Science
		Levine Science Research Center (LSRC)
		Durham
		NC 27708-0129 
		USA
		brd@cs.duke.edu
	
	If you use or publish any results derived from the use of this program please cite:
	J. Zeng, J. Boyles, C. Tripathy, L. Wang, A. Yan, P. Zhou and B.R. Donald. 
	"High-Resolution Protein Structure Determination Starting with a Global Fold 
	Calculated from Exact Solutions to the RDC Equations." Submitted For Review.

	Copyright (C) 2009 Jianyang (Michael) Zeng, Lincong Wang and Bruce R. Donald		
	<signature of Bruce Donald>, June 2008 and January 2009
	Bruce Donald, Professor of Computer Science
 */



import java.io. *;
import java.util. *;

import Jampack.JampackException;
// TODO: Auto-generated Javadoc

/** * 
 *  
*   This class provides data structures processing experimental RDC data.
*   Written by Lincong Wang (2001-2005) and Jianyang (Michael) Zeng (2005-2009).
*/
public class Dipolar implements Cloneable
{
    
    /** The residue no. */
    private int    residueNo; 
    
    /** The residue name. */
    private String residue;
    
    /** The experimental rdc. */
    private double expRdc;    //experimental data
    
    /** The experimental err for the RDC. */
    private double expErr;    //error in RDCs, not needed by the program

    /**
     * Instantiates a new dipolar.
     */
    public Dipolar(){
	residueNo = 0;
	residue = null;
	expErr = 0.0;
	expRdc = 0.0;
    }		
    
    /**
     * Instantiates a new dipolar.
     * 
     * @param no the no
     */
    public Dipolar(int no){
	residueNo = no;
	residue = null;
	expErr = 0.0;
	expRdc = 0.0;
    }		
    
    /**
     * Instantiates a new dipolar.
     * 
     * @param no the no
     * @param resid the resid
     * @param rdc the rdc
     * @param err the err
     */
    public Dipolar(int no,String resid, double rdc, double err){
	residueNo = no;
	residue = resid;
	expRdc = rdc;
	expErr = err;	   
    }	

    //getting the values	
    /**
     * Gets the residue no.
     * 
     * @return the residue no
     */
    public int getResidueNo(){
	return residueNo;
    }	
    
    /**
     * Gets the residue.
     * 
     * @return the residue
     */
    public String getResidue(){
	return residue;
    }	
    
    /**
     * Gets the err.
     * 
     * @return the err
     */
    public double getErr(){
	return expErr;
    }	
    
    /**
     * Gets the rdc.
     * 
     * @return the rdc
     */
    public double getRdc(){
	return expRdc;
    }

    //set the values	
    /**
     * Sets the residue no.
     * 
     * @param R the new residue no
     */
    public void setResidueNo(int R){
	residueNo = R;
    }	
    
    /**
     * Sets the err.
     * 
     * @param err the new err
     */
    public void setErr(double err){
	expErr = err;
    }	
    
    /**
     * Sets the rdc.
     * 
     * @param rdc the new rdc
     */
    public void setRdc(double rdc){
	expRdc = rdc;
    }
	
    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    public String toString(){
	String desc = String.valueOf(residueNo) +"    "+ String.valueOf(expRdc)
	    +"  "+String.valueOf(expErr);
	return desc;	
    }

    /**
     * The Class rdcComparator.
     */
    public static class rdcComparator implements Comparator{
	
	/* (non-Javadoc)
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(Object o1, Object o2){
	    Dipolar n1 = (Dipolar)o1;
	    Dipolar n2 = (Dipolar)o2;
	    double d1 = n1.getResidueNo();
	    double d2 = n2.getResidueNo();
	    if (d1 < d2)
		return -1;
	    else if (d1 > d2)
		return 1;
	    else return 0;
	}
    }
    
    /**
     * output the RDCs in the xplor format.
     * 
     * @param vecRDCs RDCs
     * @param atom1 first atom name of the bond vector
     * @param atom2 second atom name of the bond vector
     */
    public void outPutRDCXplorFormat (Vector vecRDCs, String atom1, String atom2)
    {
    	
    }
    		
    /**
     * read RDC data file for a fragment as defined by secondary structure identification.
     * file format is: residue No., RDC value, experimental error.
     * Must have only three items per line. Otherwise, errors.
     * 
     * @param filename the name of the file
     * @param idVec the 2ary structure identification for the protein
     * @param idType the 2ary type of the fragment
     * 
     * @return a vector of RDC objects associated with the fragment.
     */
    public Vector<Vector<Dipolar>> rdcReader(String filename, final Vector<IDof2aryStructure> idVec, String idType){
	Vector<Dipolar> rdcVec = new Vector<Dipolar>();
	int i, j, no;
	double rdc, err;
	try{
	    StreamTokenizer in =
		new StreamTokenizer(
				    new FileReader(filename));
	    while( in.nextToken() != StreamTokenizer.TT_EOF ){
		if (in.ttype == StreamTokenizer.TT_WORD)
		    ;
		else if (in.ttype == StreamTokenizer.TT_NUMBER){
		    no = (int)in.nval;
  		    in.nextToken();         
 		    rdc = in.nval;  
		    in.nextToken();
 		    err = in.nval;
  		    in.nextToken();   //If the file has four columns
		    rdcVec.add(new Dipolar(no,"", rdc,err));
		}	
	    }			
	}catch (FileNotFoundException e) {
	    System.out.println("File not found: " + filename);
	}catch (IOException e) {
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
	}
	Collections.sort(rdcVec,  new rdcComparator() ); 
	//Extract RDCs for individual strand
	IDof2aryStructure id2ary = new IDof2aryStructure();
	int index = -1;
	int beginNo = 0; 
	int endNo = 0; 
	int residueNo = 0;
	String type2ary; 
	String type;
	Vector<Vector<Dipolar>> allRdcVec = new Vector<Vector<Dipolar>>();
	Vector<Dipolar> oneElementRdc = new Vector<Dipolar>();
	for (i=0; i<idVec.size(); i++){
	    id2ary = idVec.elementAt(i);
	    beginNo = id2ary.getResidueNoA();
	    endNo   = id2ary.getResidueNoB();
	    type2ary= id2ary.getTypeAndID();
	    type = type2ary.substring(0, 1);
	    if (type.equals(idType)){
		for (j = 0; j < (endNo - beginNo+1); j++){
		    residueNo = beginNo+j;
		    index = Collections.binarySearch(rdcVec, new Dipolar(residueNo), new Dipolar.rdcComparator() );
		    if (index > -1 )
			oneElementRdc.add(rdcVec.elementAt(index));
		}
		allRdcVec.add(oneElementRdc);
		oneElementRdc = new Vector<Dipolar>();
	    }
	}	    
	return allRdcVec;
    }
    
    /**
     * read RDC data file for a fragment as defined by secondary structure identification.
     * file format is: residue No., RDC value, experimental error.
     * Must have only three items per line. Otherwise, errors.
     * 
     * @param filename the name of the file
     * @param idVec the 2ary structure identification for the protein
     * @param idType the 2ary type of the fragment
     * @param dbSc the db sc
     * 
     * @return a vector of RDC objects associated with the fragment.
     */
    public Vector<Vector<Dipolar>> rdcReader(String filename, final Vector<IDof2aryStructure> idVec, String idType,double dbSc){
	Vector<Dipolar> rdcVec = new Vector<Dipolar>();
	int i, j, no;
	double rdc, err;
	try{
	    StreamTokenizer in =
		new StreamTokenizer(
				    new FileReader(filename));
	    while( in.nextToken() != StreamTokenizer.TT_EOF ){
		if (in.ttype == StreamTokenizer.TT_WORD)
		    ;
		else if (in.ttype == StreamTokenizer.TT_NUMBER){
		    no = (int)in.nval;
  		    in.nextToken();         
 		    rdc = (in.nval)*dbSc;  
		    in.nextToken();
 		    err = (in.nval)*dbSc;
  		    in.nextToken();   //If the file has four columns
		    rdcVec.add(new Dipolar(no,"", rdc,err));
		}	
	    }			
	}catch (FileNotFoundException e) {
	    System.out.println("File not found: " + filename);
	}catch (IOException e) {
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
	}
	Collections.sort(rdcVec,  new rdcComparator() ); 
	//Extract RDCs for individual strand
	IDof2aryStructure id2ary = new IDof2aryStructure();
	int index = -1;
	int beginNo = 0; 
	int endNo = 0; 
	int residueNo = 0;
	String type2ary; 
	String type;
	Vector<Vector<Dipolar>> allRdcVec = new Vector<Vector<Dipolar>>();
	Vector<Dipolar> oneElementRdc = new Vector<Dipolar>();
	for (i=0; i<idVec.size(); i++){
	    id2ary = idVec.elementAt(i);
	    beginNo = id2ary.getResidueNoA();
	    endNo   = id2ary.getResidueNoB();
	    type2ary= id2ary.getTypeAndID();
	    type = type2ary.substring(0, 1);
	    if (type.equals(idType)){
		for (j = 0; j < (endNo - beginNo+1); j++){
		    residueNo = beginNo+j;
		    index = Collections.binarySearch(rdcVec, new Dipolar(residueNo), new Dipolar.rdcComparator() );
		    if (index > -1 )
			oneElementRdc.add(rdcVec.elementAt(index));
		}
		allRdcVec.add(oneElementRdc);
		oneElementRdc = new Vector<Dipolar>();
	    }
	}	    
	return allRdcVec;
    }



    /**
     * This is an old function.
     * A method for reading files with RDC data for a fragment as
     * defined by secondary structure identification.
     * file format is: residue No. RDC value, experimental error.
     * Must have only three items per line. Otherwise, errors.
     * 
     * @param filename the name of the file
     * @param idVec the secondary identification for the protein
     * @param idType the 2ary type of the fragment
     * 
     * @return a vector of RDC object associated with the fragment.
     */
    public Vector rdcReader_old(String filename, final Vector idVec, String idType){
	Vector rdcVec = new Vector();
	int i, j, no;
	double rdc, err;
	try{
	    StreamTokenizer in =
		new StreamTokenizer(
				    new FileReader(filename));
	    while( in.nextToken() != StreamTokenizer.TT_EOF ){
		if (in.ttype == StreamTokenizer.TT_WORD)
		    ;
		else if (in.ttype == StreamTokenizer.TT_NUMBER){
		    no = (int)in.nval;
  		    in.nextToken();         
 		    rdc = in.nval;  
		    in.nextToken();
 		    err = in.nval;

		    rdcVec.add(new Dipolar(no,"", rdc,err));
		}	
	    }			
	}catch (FileNotFoundException e) {
	    System.out.println("File not found: " + filename);
	}catch (IOException e) {
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
	}
	Collections.sort(rdcVec,  new rdcComparator() ); 
	//Extract RDCs for individual strand
	IDof2aryStructure id2ary = new IDof2aryStructure();
	int index = -1;
	int beginNo = 0; 
	int endNo = 0; 
	int residueNo = 0;
	String type2ary; 
	String type;
	Vector allRdcVec = new Vector();
	Vector oneElementRdc = new Vector();
	for (i=0; i<idVec.size(); i++){
	    id2ary = (IDof2aryStructure)idVec.elementAt(i);
	    beginNo = id2ary.getResidueNoA();
	    endNo   = id2ary.getResidueNoB();
	    type2ary= id2ary.getTypeAndID();
	    type = type2ary.substring(0, 1);
	    if (type.equals(idType)){
		for (j = 0; j < (endNo - beginNo+1); j++){
		    residueNo = beginNo+j;
		    index = Collections.binarySearch(rdcVec, new Dipolar(residueNo), new Dipolar.rdcComparator() );
		    if (index > -1 )
			oneElementRdc.add(rdcVec.elementAt(index));
		}
		allRdcVec.add(oneElementRdc);
		oneElementRdc = new Vector();
	    }
	}	    
	return allRdcVec;
    }

    /**
     * A method for extracting RDC restraints from CNS format
     * The required file format is: residue No. RDC value, experimental error.
     * Must have only three items per line. Otherwise, errors.
     * 
     * @param id the identification of RDC such as NH or CH RDCs
     * @param fileName the file name
     * @param ratio the ratio
     * 
     * @return a vector of RDC object
     */
    public Vector rdcReader4CNS(String fileName, String id, double ratio){
        Vector inputs = new Vector();
        int residueNo =0;
        double rdc = 0.0, err = 0.0;
        int ind1 = 0, ind2 = 0, ind3=0;
        String rdcValues, ss, str, idTmp;
        try{
            BufferedReader in = new BufferedReader(new FileReader(fileName));
            while ((ss = in.readLine()) != null){
                ss = ss.toUpperCase();
                if ( ss.indexOf(id) > -1 ){
                    ind1 = ss.indexOf("RESID");
                    ind2 = ss.indexOf("AND");
                    str = ss.substring(ind1+5, ind2).trim();

                    residueNo = Integer.parseInt(str);
                    ind3 = ss.indexOf(")");
		    idTmp = ss.substring(ind2+8, ind3).trim();

		    if (idTmp.equals(id)){
			rdcValues = ss.substring(ind3+1).trim();
			StringTokenizer tokens = new StringTokenizer(rdcValues);
			if(tokens.hasMoreTokens()){
			    rdc = new Double(tokens.nextToken()).doubleValue();
			    if(tokens.hasMoreTokens())
				err = new Double(tokens.nextToken()).doubleValue();
			    else err = 1.00;
			    inputs.add(new Dipolar(residueNo, "", rdc / ratio, err));
			}
		    }
		}
	    }
        }catch (FileNotFoundException e) {
            System.out.println("File not found: " + fileName);
        }catch (IOException e) {
            System.out.println("IOException: the stack trace is:");
            e.printStackTrace();
        }
        Collections.sort(inputs,  new rdcComparator() );
        return inputs;
    }

    /**
     * A method for reading files with RDC restraints.
     * The required file format is: residue No. RDC value, experimental error.
     * Must have only three items per line. Otherwise, errors.
     * 
     * @param filename the name of the file
     * 
     * @return a vector of RDC object
     */
    public Vector rdcRead(String filename){
	Vector inputs = new Vector();
	int no;
	double rdc, err;
	try{
	    StreamTokenizer in =
		new StreamTokenizer(
				    new FileReader(filename));
	    while( in.nextToken() != StreamTokenizer.TT_EOF ){
		if (in.ttype == StreamTokenizer.TT_WORD)
		    ;
		else if (in.ttype == StreamTokenizer.TT_NUMBER){
		    no = (int)in.nval;
  		    in.nextToken();         
 		    rdc = in.nval; 
		    in.nextToken();
 		    err = in.nval;

		    in.nextToken();   //If the file has four columns
		    inputs.add(new Dipolar(no,"", rdc, err));
		}	
	    }			
	}catch (FileNotFoundException e) {
	    System.out.println("File not found: " + filename);
	}catch (IOException e) {
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
	}
	Collections.sort(inputs,  new rdcComparator() ); 
	return inputs;
    }
    
    /**
     * A method for reading files with RDC restraints.
     * The required file format is: residue No. RDC value, experimental error.
     * Must have only three items per line. Otherwise, errors.
     * 
     * @param filename the name of the file
     * @param dbScale the db scale
     * 
     * @return a vector of RDC object
     */
    public Vector rdcRead(String filename,double dbScale){
	Vector inputs = new Vector();
	int no;
	double rdc, err;
	try{
	    StreamTokenizer in =
		new StreamTokenizer(
				    new FileReader(filename));
	    while( in.nextToken() != StreamTokenizer.TT_EOF ){
		if (in.ttype == StreamTokenizer.TT_WORD)
		    ;
		else if (in.ttype == StreamTokenizer.TT_NUMBER){
		    no = (int)in.nval;
  		    in.nextToken();         
 		    rdc = (in.nval)*dbScale; 
		    in.nextToken();
 		    err = (in.nval)*dbScale;

		    in.nextToken();   //If the file has four columns
		    inputs.add(new Dipolar(no,"", rdc, err));
		}	
	    }			
	}catch (FileNotFoundException e) {
	    System.out.println("File not found: " + filename);
	}catch (IOException e) {
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
	}
	Collections.sort(inputs,  new rdcComparator() ); 
	return inputs;
    }
    
    //generate the rdc table in the xplor format:    
    /**
     * The main method for testing.
     * 
     * @param argv the arguments
     * 
     * @throws JampackException the jampack exception
     */
    public static void main(String[] argv)throws JampackException
    {
   		long startTime = System.currentTimeMillis();
   		int i, j;
   		
   		String userDir = System.getProperty("user.dir");////
   		String src=userDir+"/inputFiles/";    		
   		String strOut=userDir+"/outFiles/"; 
   		ModelRdc mdc = new ModelRdc();
   		Model mod = new Model();
   				
   		//RDC files for one helix used to compute Saupe elements
   		String rdcNhFile = src+"gb1nhrdc.txt";
   	    String rdcChFile = src+"gb1chrdc.txt";
   	    String rdcCaCoFile=src+"cacoRdc.txt";
   	    String rdcConFile=src+"conRdc.txt";
   	  	Dipolar dd  =  new Dipolar();
   		final Vector<Dipolar> nhRdc = dd.rdcRead(rdcNhFile,1.0);
   		//final Vector<Dipolar> cahaRdc = dd.rdcRead(rdcChFile,1.0);
   		final Vector<Dipolar> cahaRdc = dd.rdcRead(rdcChFile,(1/2.008));
   		
   		final Vector<Dipolar> cacoRdc = dd.rdcRead(rdcCaCoFile,1.0);
   		final Vector<Dipolar> conRdc = dd.rdcRead(rdcConFile,1.0);
   		
   		String stroutName="",fileName="";
   		
   		stroutName="gb1_nh_rdc.tbl";
        fileName=strOut+stroutName;    	
    	try{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));    		
    		out.println("! rdc has been scaled to NH rdc");    		
	       	for (i=0;i<nhRdc.size();i++)
	    	{
	       		Dipolar dp=(Dipolar)nhRdc.elementAt(i);
	       		int resNo=dp.getResidueNo();
	       		double rdcValue=dp.getRdc();
	       		double rdcErr=0.2;//dp.getErr();
	       		out.println("assign ( resid 500  and name OO  )");//change 600 to 500
	       		out.println("       ( resid 500  and name Z   )");
	       		out.println("       ( resid 500  and name X   )");
	       		out.println("       ( resid 500  and name Y   )");
	       		out.println("       ( resid "+resNo+ "    and name N   )");
	       		out.println("       ( resid "+resNo+ "    and name HN  )"+ "   "+ rdcValue+"   "+rdcErr);
	       		
	       		out.println("");
	       		out.println("");	    		
	    	}//for (i=0;i<vecRefinedNoes.size();i++)
	    	out.close();
	    	//System.out.println("The NOE assignment table has been generated in "+ fileName); 
    	}catch (FileNotFoundException e)
		{
			System.out.println("File not found: " + fileName);
		}catch (IOException e)
		{
		   System.out.println("IOException: the stack trace is:");
		   e.printStackTrace();
		}
		
		//print out ch rdc
		stroutName="gb1_ch_rdc.tbl";
        fileName=strOut+stroutName;    	
    	try{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));    		
    		out.println("! rdc has been scaled to NH rdc");    		
	       	for (i=0;i<cahaRdc.size();i++)
	    	{
	       		Dipolar dp=(Dipolar)cahaRdc.elementAt(i);
	       		int resNo=dp.getResidueNo();
	       		double rdcValue=dp.getRdc();
	       		double rdcErr=0.2;//dp.getErr();
	       		out.println("assign ( resid 500  and name OO  )");
	       		out.println("       ( resid 500  and name Z   )");
	       		out.println("       ( resid 500  and name X   )");
	       		out.println("       ( resid 500  and name Y   )");
	       		out.println("       ( resid "+resNo+ "    and name CA  )");
	       		out.println("       ( resid "+resNo+ "    and name HA  )"+ "   "+ rdcValue+"   "+rdcErr);
	       		
	       		out.println("");
	       		out.println("");	    		
	    	}//for (i=0;i<vecRefinedNoes.size();i++)
	    	out.close();
	    	//System.out.println("The NOE assignment table has been generated in "+ fileName); 
    	}catch (FileNotFoundException e)
		{
			System.out.println("File not found: " + fileName);
		}catch (IOException e)
		{
		   System.out.println("IOException: the stack trace is:");
		   e.printStackTrace();
		}
		
		//		print out caco rdc
		stroutName="ff2_rdc_caco.tbl";
        fileName=strOut+stroutName;    	
    	try{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));    		
    		out.println("! rdc has not been scaled to NH rdc");    		
	       	for (i=0;i<cacoRdc.size();i++)
	    	{
	       		Dipolar dp=(Dipolar)cacoRdc.elementAt(i);
	       		int resNo=dp.getResidueNo();
	       		double rdcValue=dp.getRdc();
	       		double rdcErr=0.2;//dp.getErr();
	       		out.println("assign ( resid 500  and name OO  )");
	       		out.println("       ( resid 500  and name Z   )");
	       		out.println("       ( resid 500  and name X   )");
	       		out.println("       ( resid 500  and name Y   )");
	       		out.println("       ( resid "+resNo+ "    and name CA  )");
	       		out.println("       ( resid "+resNo+ "    and name C   )"+ "   "+ rdcValue+"   "+rdcErr);
	       		
	       		out.println("");
	       		out.println("");	    		
	    	}//for (i=0;i<vecRefinedNoes.size();i++)
	    	out.close();
	    	//System.out.println("The NOE assignment table has been generated in "+ fileName); 
    	}catch (FileNotFoundException e)
		{
			System.out.println("File not found: " + fileName);
		}catch (IOException e)
		{
		   System.out.println("IOException: the stack trace is:");
		   e.printStackTrace();
		}
		//		print out nco rdc
		stroutName="ff2_rdc_nco.tbl";
        fileName=strOut+stroutName;    	
    	try{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));    		
    		out.println("! rdc has not been scaled to NH rdc");    		
	       	for (i=0;i<conRdc.size();i++)
	    	{
	       		Dipolar dp=(Dipolar)conRdc.elementAt(i);
	       		int resNo=dp.getResidueNo();
	       		int resNextNo=resNo+1;
	       		double rdcValue=dp.getRdc();
	       		double rdcErr=0.2;//dp.getErr();
	       		out.println("assign ( resid 500  and name OO  )");
	       		out.println("       ( resid 500  and name Z   )");
	       		out.println("       ( resid 500  and name X   )");
	       		out.println("       ( resid 500  and name Y   )");
	       		out.println("       ( resid "+resNo+ "    and name C  )");
	       		out.println("       ( resid "+resNextNo+ "    and name N   )"+ "   "+ rdcValue+"   "+rdcErr);
	       		
	       		out.println("");
	       		out.println("");	    		
	    	}//for (i=0;i<vecRefinedNoes.size();i++)
	    	out.close();
	    	//System.out.println("The NOE assignment table has been generated in "+ fileName); 
    	}catch (FileNotFoundException e)
		{
			System.out.println("File not found: " + fileName);
		}catch (IOException e)
		{
		   System.out.println("IOException: the stack trace is:");
		   e.printStackTrace();
		}
		
		
		
		
    }
    
    
    /* (non-Javadoc)
     * @see java.lang.Object#clone()
     */
    protected Object clone(){
        try {
	    Object s = super.clone();     
	    return s;                     
        } catch (CloneNotSupportedException e) {
	    throw new InternalError();
        }
    }
}
