package Nasca;

///////////////////////////////////////////////////////////////////////////////////////////////
//NascaNew.java
//
//  Version:           1.0
//
//
//  authors:
//	  initials            name                      organization               email
// ---------   -----------------------        ------------------------    ------------------
//  JMZ		 Jianyang (Michael) Zeng	       Duke University			zengjy@cs.duke.edu
//
///////////////////////////////////////////////////////////////////////////////////////////////



/*
 NASCA NOE Assignment and Side-Chain Assignment  Software Version 1.0
 Copyright (C) 2009-2011 Bruce Donald Lab, Duke University

 NASCA is free software; you can redistribute it and/or modify it under
 the terms of the GNU Lesser General Public License as published by the Free
 Software Foundation, either version 3 of the License, or (at your option) any
 later version.

 NASCA is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 details.

 You should have received a copy of the GNU Lesser General Public License
 along with this library; if not, see:
     <http://www.gnu.org/licenses/>.

 There are additional restrictions imposed on the use and distribution of this
 open-source code, including: (A) this header must be included in any
 modification or extension of the code; (B) you are required to cite our
 papers in any publications that use this code. The citation for the various
 different modules of our software, together with a complete list of
 requirements and restrictions are found in the document license.pdf enclosed
 with this distribution.

 Contact Info:
     Bruce R. Donald
     Duke University
     Department of Computer Science
     Levine Science Research Center (LSRC)
     Durham, NC 27708-0129
     USA
     email: www.cs.duke.edu/brd/

 <signature of Bruce Donald>, 01 December, 2009
 Bruce R. Donald, Professor of Computer Science and Biochemistry
*/

import java.io.BufferedWriter;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Collections;
import java.util.Map;
import java.util.Vector;

import rdcPanda.Assign;
import rdcPanda.BackNoe;
import rdcPanda.H1CS;
import rdcPanda.Hsqc;
import rdcPanda.ModelRdc;
import rdcPanda.Nasca;
import rdcPanda.Noe;
import rdcPanda.Noesy;
import rdcPanda.Pdb;
import rdcPanda.Peak;
import rdcPanda.RotaPattern;
import Jampack.JampackException;


/**
 * This class provide the main function for side-chain assignment and 
 * NOE assignment without TOCSY data, i.e., using only NOESY data.
 */
public class NascaNew 
{

	public  void main (String[] args) throws JampackException
	{
		//output program information:
		//outputProgInfo();
		
		//read the NMR data:
		String userDir = System.getProperty("user.dir");
		String src=userDir+"/inputFiles/";  //input file directory		
		String strOut=userDir+"/outFiles/";
		String strParametersFileName="Nasca.input";
		
		NascaNew nasca=new NascaNew();
		nasca.doNasca(src,strOut,strParametersFileName);	
		
	}  // End main()
	
	/**
     * exact the backbone resonance assignment
     @param vecAsg all resonance assignments
     @param 
     @return backbone resonance assignment
     */
	public Vector<H1CS> ExtractBackboneAsignment(Vector<H1CS> vecAsg){
		Vector<H1CS> vecBBAsg=new Vector<H1CS>();
		for(int i=0;i<vecAsg.size();i++)
		{
			H1CS h1cs=(H1CS)vecAsg.elementAt(i);
			String atom=h1cs.getAtomName();
			if(atom.equalsIgnoreCase("CA") ||atom.equalsIgnoreCase("N")|| 
					atom.equalsIgnoreCase("HN") ||atom.equalsIgnoreCase("H")||
					atom.equalsIgnoreCase("HA")||atom.equalsIgnoreCase("HA1")|| atom.equalsIgnoreCase("HA2")||atom.equalsIgnoreCase("HA3")||
					atom.equalsIgnoreCase("C")||atom.equalsIgnoreCase("CO")||atom.equalsIgnoreCase("CB"))
				vecBBAsg.add(h1cs);
		}
		return vecBBAsg;
	}
	
	/**
     * Initialization.
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs  
     @param MapScore mapping score matrix
     @param vecBBAsg backbone resonance assignment list
     @param 
     @param
     @return 
     */
	public void Initialization(Vector<GraphNode> vecResonGraph,Vector<ProtonLabel> vecLabels, int [][] Map,double [][] MapScore,Vector<H1CS> vecBBAsg)
	{
		int i,j;
		double errCB=0.25;//0.2 for eta
		Peak pk=new Peak();
		for(i=0;i<vecResonGraph.size();i++)
    		for(j=0;j<vecLabels.size();j++){
    			Map[i][j]=1;//initial value
    			MapScore[i][j]=-9999.9;
    		}
		
		//updated the 1-to-1 mapping for backbone resonances
    	for(i=0;i<vecResonGraph.size();i++)	{
    		GraphNode node=(GraphNode)vecResonGraph.elementAt(i);
    		boolean isAsged=node.getIsAssigned();
    		if(isAsged==false)//side-chain resonances are unassigned now
    			continue;
    		int resNo=node.getResNo();
    		String atomName=node.getAtomName();
    		
    		for(j=0;j<vecLabels.size();j++){
    			ProtonLabel label=(ProtonLabel)vecLabels.elementAt(j);
    			int resNoStruct=label.getResNo();
    			String atomNameStuct=label.getAtomName();
    			String resNameStruct=label.getResName();
    			String subAtomNameStruct=atomNameStuct;    			
    			if(atomNameStuct.length()>2)
    				subAtomNameStruct=atomNameStuct.substring(0,2);
    			
    			if((resNoStruct==resNo && atomNameStuct.equalsIgnoreCase(atomName) ) ||
    					(resNoStruct==resNo && resNameStruct.equalsIgnoreCase("GLY") && subAtomNameStruct.equalsIgnoreCase(atomName) )) {
    				//if(nodeStruct.getID()==56)
    				//System.out.println("debugging..");
    				
    				if(node.getIsAssigned())
    					label.setIsAssigned(true);
    				Map[i][j]=1;
    				node.asgedId=j;
    				label.asgedId=i;
    				for(int k=0;k<vecResonGraph.size();k++)	{
    					if(k==i)
    						continue;
    					Map[k][j]=0;
    				}
    			}//if(resNoStruct==resNo && atomNameStuct.equalsIgnoreCase(atomName))
    			else
    				Map[i][j]=0;    			
    		}//for(j=0;j<vecGraph.size();j++)
    	}//for(i=0;i<vecResonGraph.size();i++)
    	
    	//initialize the CB-HB connection:
       	for(i=0;i<vecResonGraph.size();i++)	{
    		GraphNode node=(GraphNode)vecResonGraph.elementAt(i);
    		double cs_heavy=node.getCSHeavy();
    		boolean isAsged=node.getIsAssigned();
    		if(isAsged==true)
    			continue;
    		//int resNo=node.getResNo();
    		//String atomName=node.getAtomName();
    		
    		for(j=0;j<vecLabels.size();j++)	{
    			ProtonLabel nodeStruct=(ProtonLabel)vecLabels.elementAt(j);
    			double csHeavyStruct=0.0;
    			int resNoStruct=nodeStruct.getResNo();
    			String atomNameStuct=nodeStruct.getAtomName();
    			String resNameStruct=nodeStruct.getResName();
    			String subAtomNameStruct=atomNameStuct;
    			String heavyNameStruct=pk.GetHeavyAtomFromProton(resNameStruct,atomNameStuct);
    			if(atomNameStuct.length()>2)
    				subAtomNameStruct=atomNameStuct.substring(0,2);
    			if(!subAtomNameStruct.equalsIgnoreCase("HB"))
    				continue;
    			if(!heavyNameStruct.equalsIgnoreCase("CB"))
    				continue;
    			//find the CB chemical shift
    			for(int k=0;k<vecBBAsg.size();k++)
    			{
    				H1CS h1cs=(H1CS)vecBBAsg.elementAt(k);
    				String CB_atom=h1cs.getAtomName();
    				int CB_resNo=h1cs.getResidueNo();
    				if(!CB_atom.equalsIgnoreCase("CB"))
    					continue;
    				if(CB_resNo==resNoStruct && CB_atom.equalsIgnoreCase("CB"))
    					csHeavyStruct=h1cs.getH1CS();
    			}//for(int k=0;k<vecBBAsg.size();k++)    			
    				
    			if(Math.abs(csHeavyStruct-cs_heavy)<errCB)
    			{
    				Map[i][j]=1;    				
    			}//if(resNoStruct==resNo && atomNameStuct.equalsIgnoreCase(atomName))
    			else
    				Map[i][j]=0;    			
    		}//for(j=0;j<vecGraph.size();j++)
    	}//for(i=0;i<vecResonGraph.size();i++)
       	
       	// set up stereo IDs for two protons, which should have the same heavy atom name
		for(j=0;j<vecLabels.size();j++)	{
			ProtonLabel ndProton=(ProtonLabel)vecLabels.elementAt(j);
			String atom=ndProton.getAtomName();
			String res=ndProton.getResName();
			String heavy=pk.GetHeavyAtomFromProton(res,atom);
			int resNo=ndProton.getResNo();			
			for(int k=0;k<vecLabels.size();k++){
				if(j==k)
					continue;
				ProtonLabel ndProton2=(ProtonLabel)vecLabels.elementAt(k);
				String atom2=ndProton2.getAtomName();
				String res2=ndProton2.getResName();
				String heavy2=pk.GetHeavyAtomFromProton(res2,atom2);
				int resNo2=ndProton2.getResNo();
				if(heavy.equalsIgnoreCase(heavy2) && resNo==resNo2){
					ndProton.stereoID=k;
					ndProton2.stereoID=j;
				}
			}//for(int k=0;k<vecGraph.size();k++)			
		}//for(int j=0;j<vecGraph.size();j++)
	}
	

	/**
     * refine mapping based on BMRB outliers
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs  
     @param MapScore mapping score matrix
     @param vecNewNoesy noesy cross peak list
     @param vecBMRBNew BMRB information
     @param
     @return 
     */
	public void RefineMappingBasedBMRBOutliers(Vector<GraphNode> vecResonGraph,Vector<ProtonLabel> vecLabels, int [][] Map,Vector<H1CS> vecBMRBNew){
		int i,j;
		Peak pk=new Peak();
	       	
    	for(i=0;i<vecResonGraph.size();i++)
    	{
    		GraphNode node=(GraphNode)vecResonGraph.elementAt(i);
    		double csH=node.getCSProton();
    		double csHeavy=node.getCSHeavy();//be care of -999.9
    		boolean isAsged=node.getIsAssigned();
    		if(isAsged) 
    			continue;
    		for(j=0;j<vecLabels.size();j++)
    		{    				
    			ProtonLabel label=(ProtonLabel)vecLabels.elementAt(j);
    			String resName=label.getResName();
    			String atomName=label.getAtomName();
    			String subAtomName=atomName;
    		
    			String heavyName=pk.GetHeavyAtomFromProton(resName,atomName);    			
    			for(int k=0;k<vecBMRBNew.size();k++){    				
    				H1CS h1csB=(H1CS)vecBMRBNew.elementAt(k);
    				String resB=h1csB.getResidueType();
    				String atomB=h1csB.getAtomName();
    				double cs_L=h1csB.getCSLower();
    				double cs_U=h1csB.getCSUpper();
    				subAtomName=atomName;
    				if(subAtomName.length()>atomB.length())
    					subAtomName=atomName.substring(0,atomB.length());
    				if(resB.equalsIgnoreCase(resName) && atomB.equalsIgnoreCase(subAtomName) && atomB.substring(0,1).equalsIgnoreCase("H")){
    					if( (csH<cs_L) || (csH> cs_U))
    						Map[i][j]=0;
    				}
    				
    				if(resB.equalsIgnoreCase(resName) && atomB.equalsIgnoreCase(heavyName) ){
    					if( (csHeavy<cs_L) || (csHeavy> cs_U))
    						Map[i][j]=0;
    				}
    			}//for(int k=0;k<vecBMRBNew.size();k++)
    		
    		}//for(j=0;j<vecGraph.size();j++)
    	}//for(i=0;i<vecResonGraph.size();i++)
    	
	}
	
	/**
     * update the mapping set based on the mapping matrix
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs  
     @param 
     @param 
     @return 
     */
	public void UpdateMappingSet(Vector<GraphNode> vecResonGraph,Vector<ProtonLabel> vecLabels, int [][] Map){
		int i,j;
		for(i=0;i<vecResonGraph.size();i++)	{
			GraphNode nodeReson=(GraphNode)vecResonGraph.elementAt(i);
			nodeReson.EmptyMappingSet();
		}			
		for(j=0;j<vecLabels.size();j++)	{
			ProtonLabel label=(ProtonLabel)vecLabels.elementAt(j);
			label.EmptyMappingSet();
		}//for(int j=0;j<vecStructGraph.size();j++)						
		
		for(i=0;i<vecResonGraph.size();i++)	{
			GraphNode nodeReson=(GraphNode)vecResonGraph.elementAt(i);
			//int resonId=nodeReson.getID();
			
			for(j=0;j<vecLabels.size();j++)	{
				ProtonLabel label=(ProtonLabel)vecLabels.elementAt(j);
				if (Map[i][j]==1)	{
					nodeReson.addMappingNode(label);
					//if(!nodeReson.getIsAssigned())
					label.addMappingNode(nodeReson);//we don't add assiged resonance node
				}
			}//for(int j=0;j<vecStructGraph.size();j++)			
		}//for(int i=0;i<vecResonGraph.size();i++)		
	}
	/**
     * update the mapping set based on the mapping matrix
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs  
     @param 
     @param 
     @return 
     */
	public void UpdateIsRestrained(Vector<GraphNode> vecResonGraph,Vector<ProtonLabel> vecLabels, int [][] Map){
		for(int i=0;i<vecResonGraph.size();i++){
			GraphNode nodeReson=(GraphNode)vecResonGraph.elementAt(i);
			if(nodeReson.getMappingSet().size()<5)
				nodeReson.setIsRestrained(true);
		}	
	}
	/**
     * update stereo symmetry
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs  
   
     @return 
     */
	public void UpdateStereoSymSet(Vector<GraphNode> vecResonGraph,Vector<ProtonLabel> vecLabels, int [][] Map)	{		
		double errHeavy=0.1;
		Peak pk=new Peak();
		
		//initialization
		for(int i=0;i<vecResonGraph.size();i++)	{
			GraphNode nodeReson=(GraphNode)vecResonGraph.elementAt(i);
			nodeReson.vecStereoSymSet=new Vector<GraphNode>();
		}
		UpdateMappingSet(vecResonGraph,vecLabels,Map);	
		
		//using only chemical shift information
		for(int i=0;i<vecResonGraph.size();i++)	{
			GraphNode nodeReson=(GraphNode)vecResonGraph.elementAt(i);
			//double cs_proton=nodeReson.getCSProton();
			double cs_heavy=nodeReson.getCSHeavy();
			Vector<GraphNode> vec_sym=nodeReson.vecSymSet;
			Vector<ProtonLabel> vec_map=nodeReson.getMappingSet();
			
			for(int t=0;t<vec_sym.size();t++){
				GraphNode nodeReson2=(GraphNode)vec_sym.elementAt(t);				
				double cs_heavy2=nodeReson2.getCSHeavy();			
				Vector<ProtonLabel> vec_map2=nodeReson2.getMappingSet();
								
				boolean isMap=false;				
				
				//use the mapping information to refine stereo symmetry sets
				for(int k=0;k<vec_map.size();k++){
					ProtonLabel ndProton1=(ProtonLabel)vec_map.elementAt(k);
					String atom1=ndProton1.getAtomName();
					String res1=ndProton1.getResName();
					String heavyName1=pk.GetHeavyAtomFromProton(res1,atom1);
					for(int w=0;w<vec_map2.size();w++){
						ProtonLabel ndProton2=(ProtonLabel)vec_map2.elementAt(w);
						String atom2=ndProton2.getAtomName();
						String res2=ndProton2.getResName();
						String heavyName2=pk.GetHeavyAtomFromProton(res2,atom2);
						if(heavyName1.equalsIgnoreCase(heavyName2))
							isMap=true;						
					}//for(int w=0;w<vec_map2.size();w++)				
				}//for(int k=0;k<vec_map.size();k++)
				
				if(isMap && Math.abs(cs_heavy-cs_heavy2)< errHeavy  ){
					nodeReson.vecStereoSymSet.add(nodeReson2);
					nodeReson2.vecStereoSymSet.add(nodeReson);
					nodeReson.stereoID=nodeReson2.getID();//not used?
					nodeReson2.stereoID=nodeReson.getID();//not used?
				}

			}//for(int t=0;t<vec_adj.size();t++)
		}//for(i=0;i<vecResonGraph.size();i++)		
		
		//delete repeated stereo nodes;
		for(int i=0;i<vecResonGraph.size();i++)	{
			GraphNode nodeReson=(GraphNode)vecResonGraph.elementAt(i);
			Vector<GraphNode> vecStereoSet=nodeReson.vecStereoSymSet;
			for (int k=0;k<vecStereoSet.size();k++)	{
				GraphNode nodeStereo=(GraphNode)vecStereoSet.elementAt(k);
				int curID=nodeStereo.getID();
				boolean isInPre=false;
				for(int t=0;t<vecStereoSet.size();t++){
					GraphNode nodeTemp=(GraphNode)vecStereoSet.elementAt(t);
					if(t==k)
						continue;
					if(nodeTemp.getID()==curID)
						isInPre=true;
				}
				if(isInPre)	{
					vecStereoSet.remove(k);
					k--;
				}
				
			}//for (int k=0;k<vecStereoSet.size();k++)
		}//for(int i=0;i<vecResonGraph.size();i++)
	}
	/**
     * check whether one node is in another node's close neighbors
     @param vecAsg all resonance assignments
     @param 
     @return 
     */
	public boolean isInCloseAdj(ProtonLabel nodeSrc, ProtonLabel nodeObj){
		boolean isIn=false;
		for(int i=0;i<nodeSrc.vecCloseAdj.size();i++){
			ProtonLabel ndTemp=(ProtonLabel)nodeSrc.vecCloseAdj.elementAt(i);
			if(ndTemp.getID()==nodeObj.getID())
				isIn=true;
		}
		return isIn;
	}
	/**
     * check whether there exists at least one mapping between two node sets
      @param vecNodeSetA: adj set for resonance node
     @param vecNodeSetB: adj set of proton node
     @param M previous mapping matrix
     @param isAssigned whether only consider the assigned backbone resonance nodes
     @return 
     */
	public boolean isMappedBetwTwoSetsCalScore(GraphNode nodeReson, ProtonLabel label,Vector<GraphNode> vecNodeSetA,Vector<ProtonLabel> vecNodeSetB,
			int[][]M, boolean isConsiderAssigned,int threshold, double [] Score,Vector<Noesy> vecNoesy){
		boolean isMapped=false;
		int counter=0;
		Peak pk=new Peak();
		double cs_proton=nodeReson.getCSProton();
		double cs_heavy=nodeReson.getCSHeavy();
		String res=label.getResName();
		String atom=label.getAtomName();
		int resNo=label.getResNo();
		String heavName=pk.GetHeavyAtomFromProton(res,atom);
		Vector<BackNoe> vecBackNoe=new Vector<BackNoe>();
		
		for(int i=0;i<vecNodeSetA.size();i++){
			GraphNode nodeA=(GraphNode)vecNodeSetA.elementAt(i);	
			
			boolean isAsgedA=nodeA.getIsAssigned();
			int idA=nodeA.getID();
			if(isConsiderAssigned){
				if(!isAsgedA)			
					continue;
			}
			
			
			for(int j=0;j<vecNodeSetB.size();j++){
				ProtonLabel nodeB=(ProtonLabel)vecNodeSetB.elementAt(j);			
				boolean isAsgedB=nodeB.getIsAssigned();
				String atomName=nodeB.getAtomName();
				String subAtom=atomName;
				if(atomName.length()>=2)
					subAtom=atomName.substring(0,2);
							
				int idB=nodeB.getID();
				if(isConsiderAssigned)	{
					if(!isAsgedB)			
						continue;
				}
				else{
					if(!(isAsgedB ||subAtom.equalsIgnoreCase("HB") ))			
						continue;
				}
				
				if (M[idA][idB]==1)	{
					double cs_protonSec=nodeA.getCSProton();
					double cs_heavySec=nodeA.getCSHeavy();
					String resSec=nodeB.getResName();
					String atomSec=nodeB.getAtomName();
					String heavyNameSec=pk.GetHeavyAtomFromProton(resSec,atomSec);
					int resNoSec=nodeB.getResNo();
					double distUp=6.0;
					if(isInCloseAdj(nodeB,label))
						distUp=2.7;
					vecBackNoe.add(new BackNoe(cs_proton,cs_heavy,cs_protonSec,distUp,0.0,resNo,resNoSec,res,resSec,atom,heavName,atomSec ));
					vecBackNoe.add(new BackNoe(cs_protonSec,cs_heavySec,cs_proton,distUp,0.0,resNoSec,resNo,resSec,res,atomSec,heavyNameSec,atom ));
					counter++;
				}//if (M[idA][idB]==1)
					
			}//for(int j=0;j<vecNodeSetB.size();j++)			
		}//for(int i=0;i<vecNodeSetA.size();i++)
		if(counter>=threshold)
			isMapped=true;	
		
		Assign asg=new Assign();
		int [] numPeaks=new int[1];
		double csErrH=0.04, csErrN=0.4,csErrCA=0.4;
		BackNoe bkNoe=new BackNoe();
		Vector<BackNoe> vecBackNoeNew=bkNoe.DeleteRepeat(vecBackNoe);
		double dbScore=asg.NoePatternMatchScore(csErrH,csErrN, csErrCA,vecBackNoeNew,vecNoesy, numPeaks,true);
		dbScore=dbScore*vecBackNoeNew.size();//the more number of matched nodes, the better
		if(vecBackNoe.size()>1)
			Score[0]=dbScore;
		else
			Score[0]=0.0;
		return isMapped;
	}
	
	
	/**
     * refine mapping based on previous restrained mapping sets. Initially use the backbone resonance assignments.
     
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs  
     @param MapScore mapping score matrix
     @param vecNewNoesy noesy cross peak list
     @param vecBMRBNew BMRB information
     @param
     @return 
     */
	//@SuppressWarnings("unchecked")
	public void RefineMappingBasedRestrained(Vector<GraphNode> vecResonGraph,Vector<ProtonLabel> vecLabels, int [][] Map,double [][] MapScore, 
			Vector<Noesy> vecNewNoesy, Vector<H1CS> vecBMRBNew){		
		int sizeReson=2;
		int sizeStruct=2;
		int restrainThreshold=5;		
		Peak pk=new Peak();
		H1CS h1cs=new H1CS();
		int i,j;
		
		UpdateStereoSymSet(vecResonGraph,vecLabels,Map);
		for(j=0;j<vecLabels.size();j++)	{    						
			ProtonLabel label=(ProtonLabel)vecLabels.elementAt(j);
			String resName=label.getResName();
			String atomName=label.getAtomName();
			int resNo=label.getResNo();
			boolean isRestrained=false;
			if(label.getIsAssigned())
				continue;
			
			//if(resNo==3 && atomName.equalsIgnoreCase("HB2"))
				//System.out.println("debugging...");
			
			////////////////////
			String subAtom=atomName;
			if(atomName.length()>=2)
				subAtom=atomName.substring(0,2);
			if(!(subAtom.equalsIgnoreCase("HB") ) )
				continue;
			if(label.stereoID>=0)
				sizeReson=1;//2
			else
				sizeReson=1;
			/////////////////////		
			
			//for storing the mapping likelihood for all nodes
    		Vector<RotaPattern> vecNeighborSC=new Vector<RotaPattern>();    
	    	for(i=0;i<vecResonGraph.size();i++)	{   
	    		
	    		if(Map[i][j]==0)
	    			continue;
	    		GraphNode node=(GraphNode)vecResonGraph.elementAt(i);
	    		double cs_proton=node.getCSProton();
	    		double cs_heavy=node.getCSHeavy();
	    		Vector<GraphNode> vecAdj=node.getAdjVec();
	    		boolean isAssigned=node.getIsAssigned();
	    		if(isAssigned)
	    			continue;	  
	    		
	    		////////////////////////////
	    			    		
    			String heavyName=pk.GetHeavyAtomFromProton(resName,atomName);
    			if( !(heavyName.substring(0,1).equalsIgnoreCase("N") || heavyName.substring(0,1).equalsIgnoreCase("C") ))
    				continue;
    			
    		  	/////////////////////////////////
    			Vector<ProtonLabel> vecAdj2=label.getAdjVec();
    			
    			if(vecAdj.size()<1 || vecAdj2.size()<1){
    				System.out.println("debugging...we found an empty neiborhood...");
    				continue;
    			} 
    		
    			double[] neighborSC=new double[1];    	
    			boolean isRestrainTemp=false;
    			isRestrainTemp=isMappedBetwTwoSetsCalScore(node,label,vecAdj,vecAdj2,Map,true,restrainThreshold,neighborSC,vecNewNoesy);
    			Vector<H1CS> vecOneSCAsg=new Vector<H1CS>();
    			vecOneSCAsg.add(new H1CS(resNo,resName,atomName,cs_proton));
    			vecOneSCAsg.add(new H1CS(resNo,resName,heavyName,cs_heavy));
    			double asgScore=h1cs.BMRBSatisticsScore(vecOneSCAsg,vecBMRBNew);
        		  			
    			neighborSC[0]=neighborSC[0]+asgScore;
    			MapScore[i][j]=neighborSC[0];//////
    						
    			if(isRestrainTemp)
		   			isRestrained=true;
		   		
    			RotaPattern rp=new RotaPattern(i,neighborSC[0]);
    			vecNeighborSC.add(rp);    			    	
    		}//for(i=0;i<vecResonGraph.size();i++)  	
	    	  
	    	label.setIsRestrained(true);
    		////////////////////////////
    		//we select the top k mapped nodes
    		Collections.sort(vecNeighborSC, new RotaPattern.scoreComparator());
    	    	
    		for(i=0;i<vecResonGraph.size();i++)	{    			
    			boolean isIn=false;
    			
	    		for (int h=0;h<Math.min(8,vecNeighborSC.size());h++){ //6: good for HB#
	    			RotaPattern rotPatn=(RotaPattern)vecNeighborSC.elementAt(h);
	    			int graph_ID=rotPatn.getResNo();
	    			
	    			GraphNode nodeTemp=(GraphNode)vecResonGraph.elementAt(graph_ID);
	    			int stereoIDTemp=nodeTemp.stereoID;
	    			if(i==graph_ID || i==stereoIDTemp)
	    				isIn=true;	    			
	    			
	    		}//for (int h=0;h<Math.min(3,vecNeighborSC.size());h++)
	    		
	    		if(!isIn)
	    			Map[i][j]=0;
    		}//for(j=0;j<vecGraph.size();j++)  		
    			
    	}//for(j=0;j<vecGraph.size();j++)
    	
	}

	/**
     * refine mapping for HG protons
     * modified from "RefineMappingBasedAsg"
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs  
     @param MapScore mapping score matrix
     @param vecNewNoesy noesy cross peak list
     @param vecBMRBNew BMRB information
     @param
     @return 
     */
	public void RefineMappingForHG(Vector<GraphNode> vecResonGraph,Vector<ProtonLabel> vecLabels, int [][] Map,double [][] MapScore, 
			Vector<Noesy> vecNewNoesy, Vector<H1CS> vecBMRBNew)
	{	
		Nasca isoGraph=new Nasca();
		Peak pk=new Peak();
		H1CS h1cs=new H1CS();
		int i,j;
		int restrainThreshold=15;
		
		UpdateStereoSymSet(vecResonGraph,vecLabels,Map);
		for(j=0;j<vecLabels.size();j++)	{    						
			ProtonLabel node2=(ProtonLabel)vecLabels.elementAt(j);
			String resName=node2.getResName();
			String atomName=node2.getAtomName();
			int resNo=node2.getResNo();
			boolean isRestrained=false;
			if(node2.getIsAssigned())
				continue;
			
			if(resNo==3 && atomName.equalsIgnoreCase("HB2"))
				System.out.println("debugging...");
			
			////////////////////
			String subAtom=atomName;
			if(atomName.length()>=2)
				subAtom=atomName.substring(0,2);
			if(!(subAtom.equalsIgnoreCase("HG") || subAtom.equalsIgnoreCase("HD") ||subAtom.equalsIgnoreCase("HE")|| subAtom.equalsIgnoreCase("HH")||subAtom.equalsIgnoreCase("HZ") ) )
				continue;
			
			/////////////////////		
			
			//for storing the mapping likelihood for all nodes
    		Vector<RotaPattern> vecNeighborSC=new Vector<RotaPattern>();    
	    	for(i=0;i<vecResonGraph.size();i++)    	{   	    		
	    		if(Map[i][j]==0)
	    			continue;
	    		GraphNode node=(GraphNode)vecResonGraph.elementAt(i);
	    		double cs_proton=node.getCSProton();
	    		double cs_heavy=node.getCSHeavy();
	    		Vector<GraphNode> vecAdj=node.getAdjVec();
	    		boolean isAssigned=node.getIsAssigned();
	    		if(isAssigned)
	    			continue;	  
	    		
	    		////////////////////////////
	    		//
	    			    		
    			String heavyName=pk.GetHeavyAtomFromProton(resName,atomName);
    			if( !(heavyName.substring(0,1).equalsIgnoreCase("N") || heavyName.substring(0,1).equalsIgnoreCase("C") ))
    				continue;
    			
    		  	/////////////////////////////////
    			Vector<ProtonLabel> vecAdj2=node2.getAdjVec();
    			
    			if(vecAdj.size()<1 || vecAdj2.size()<1)
    			{
    				System.out.println("debugging...we found an empty neiborhood...");
    				continue;
    			} 
    		
    			double[] neighborSC=new double[1];    	
    			boolean isRestrainTemp=false;
    			isRestrainTemp=isMappedBetwTwoSetsCalScore(node,node2,vecAdj,vecAdj2,Map,false,restrainThreshold,neighborSC,vecNewNoesy);
    			Vector<H1CS> vecOneSCAsg=new Vector<H1CS>();
    			vecOneSCAsg.add(new H1CS(resNo,resName,atomName,cs_proton));
    			vecOneSCAsg.add(new H1CS(resNo,resName,heavyName,cs_heavy));
    			double asgScore=h1cs.BMRBSatisticsScore(vecOneSCAsg,vecBMRBNew);
        		  			
    			neighborSC[0]=neighborSC[0]+asgScore;
    			MapScore[i][j]=neighborSC[0];
    						
    			if(isRestrainTemp)
		   			isRestrained=true;
		   		
    			RotaPattern rp=new RotaPattern(i,neighborSC[0]);
    			vecNeighborSC.add(rp);    			    	
    		}//for(i=0;i<vecResonGraph.size();i++)  		

	  
	    	node2.setIsRestrained(true);
    		////////////////////////////
    		//we select the top k mapped nodes
    		Collections.sort(vecNeighborSC, new RotaPattern.scoreComparator());
    	
    	
    		for(i=0;i<vecResonGraph.size();i++)
    		{    			
    			boolean isIn=false;
    			
	    		for (int h=0;h<Math.min(20,vecNeighborSC.size());h++)//6: good for HB#
	    		{
	    			RotaPattern rotPatn=(RotaPattern)vecNeighborSC.elementAt(h);
	    			int graph_ID=rotPatn.getResNo();
	    			
	    			GraphNode nodeTemp=(GraphNode)vecResonGraph.elementAt(graph_ID);
	    			int stereoIDTemp=nodeTemp.stereoID;
	    			if(i==graph_ID || i==stereoIDTemp)
	    				isIn=true;	    			
	    			
	    		}//for (int h=0;h<Math.min(3,vecNeighborSC.size());h++)
	    		
	    		if(!isIn)
	    			Map[i][j]=0;
    		}//for(j=0;j<vecGraph.size();j++)
    	}//for(j=0;j<vecGraph.size();j++)    	
	}

	/**Use the DEE algorithm to prune possible side-chain resonance assignments.
    @param vecResonGraph resonance graph 
    @param vecHPositions proton labels   
    @param Map mapping matrix between two graphs  
    @param MapScore mapping score matrix
    @param vecNewNoesy noesy cross peak list
    @param vecBMRBNew BMRB information
    @param
    @return 
    */
	public int RefineMappingDeeCut(Vector<GraphNode> vecResonGraph,Vector<ProtonLabel> vecHPositions, int [][] Map,double [][] MapScore, 
			Vector<Noesy> vecNewNoesy, Vector<H1CS> vecBMRBNew)	{			
		Peak pk=new Peak();
		BackNoe bkNoe=new BackNoe();
		Assign asg=new Assign();
		H1CS h1cs=new H1CS();
		//Noesy noesy=new Noesy();
		double csErrH=0.04; double csErrN=0.3;double csErrCA=0.3;				
		int counter=0;//count how many are pruned, for the purpose of debugging.
		
		for(int j=0;j<vecHPositions.size();j++)	{    		
		    //	System.out.println("current proton node ID : "+ j);	
			ProtonLabel nodeProton=(ProtonLabel)vecHPositions.elementAt(j);
			if(nodeProton.getIsAssigned())
				continue;
			String res=nodeProton.getResName();
			String atom=nodeProton.getAtomName();
			String heavyName=pk.GetHeavyAtomFromProton(res,atom);
			
			////////////////////
			//new added:
			if(heavyName.equalsIgnoreCase("CB"))//////
				continue;
			/////////////////////
			
			int resNo=nodeProton.getResNo();				
			Vector<GraphNode> vecMapSet=nodeProton.getMappingSet();
			
			Vector<ProtonLabel> vecAdj=nodeProton.getAdjVec();
			if(vecMapSet.size()<1 || vecAdj.size()<1)//rarely happen
				continue;
			
			if(nodeProton.getIsAssigned())
				continue;
			for(int k=0;k<vecMapSet.size();k++)	{
				//System.out.println("   map ID: "+ k);	
				
				GraphNode ndMapCur=(GraphNode)vecMapSet.elementAt(k);
				boolean isPruned=false;
				double csCur_proton=ndMapCur.getCSProton();
				double csCur_heavy=ndMapCur.getCSHeavy();
				
			//	if(ndMapCur.getID()==255  && nodeProton.getID()==52)
					//System.out.println("stop here, for debugging...");
						
				
				//compute the best score for current assignment
				//compute the homo score:
				Vector<H1CS> vecOneSCAsg=new Vector<H1CS>();//for computing homo score.
				vecOneSCAsg.add(new H1CS(resNo,res,atom,csCur_proton));
				vecOneSCAsg.add(new H1CS(resNo,res,heavyName,csCur_heavy));
	   			double asgScore=h1cs.BMRBSatisticsScore(vecOneSCAsg,vecBMRBNew);
	       		double totalCurScore=asgScore;
				//compute the best pair score for current assignment
				for(int t=0;t<vecAdj.size();t++){
					ProtonLabel ndCurAdj=(ProtonLabel)vecAdj.elementAt(t);
					if(ndCurAdj.getIsAssigned())
						continue;
					Vector<GraphNode> vecNdCurAdjMap=ndCurAdj.getMappingSet();
					if(vecNdCurAdjMap.size()<1)//rarely happen
						continue;
					
					int resNoAdj=ndCurAdj.getResNo();
					String resAdj=ndCurAdj.getResName();
					String atomAdj=ndCurAdj.getAtomName();
					String heavyNameAdj=pk.GetHeavyAtomFromProton(resAdj,atomAdj);					
					int maxAdjID=-1;					
					double maxScore=-9999.9;
					
					Vector<BackNoe> vecBackNoe=new Vector<BackNoe>();//for computing pairwirse 
					for(int a=0;a<vecNdCurAdjMap.size();a++){
						GraphNode ndAdjMap=(GraphNode)vecNdCurAdjMap.elementAt(a);
						Vector<Noesy> vecNoesyTwo=new Vector<Noesy>();	
						vecNoesyTwo.addAll(ndMapCur.vecAdjNOESY);
						vecNoesyTwo.addAll(ndAdjMap.vecAdjNOESY);
						
						double csAdjMapProton=ndAdjMap.getCSProton();
						double csAdjMapHeavy=ndAdjMap.getCSHeavy();
						vecBackNoe.add(new BackNoe(csCur_proton,csCur_heavy,csAdjMapProton,6.0,0.0,resNo,resNoAdj,res,resAdj,atom,heavyName,atomAdj ));
						vecBackNoe.add(new BackNoe(csAdjMapProton,csAdjMapHeavy,csCur_proton,6.0,0.0,resNoAdj,resNo,resAdj,res,atomAdj,heavyNameAdj,atom ));
						
						Vector<BackNoe> vecBackNoeNew=bkNoe.DeleteRepeat(vecBackNoe);
						int [] numPeaks=new int[1];
						double dbScore=asg.NoePatternMatchScore(csErrH,csErrN, csErrCA,vecBackNoeNew,vecNoesyTwo, numPeaks,true);
						dbScore=dbScore*vecBackNoeNew.size();//the more number of matched nodes, the better
						if(dbScore>maxScore){
							maxScore=dbScore;
							maxAdjID=a;
						}
					}//for(int a=0;a<vecNdCurAdjMap.size();a++)
					totalCurScore=totalCurScore+maxScore;					
				}//for(int t=0;t<vecAdj.size();t++)
				
				//compute the worse score for other assignments
				for(int h=0;h<vecMapSet.size();h++)	{
					if(h==k)
						continue;				
					GraphNode ndMapCurOther=(GraphNode)vecMapSet.elementAt(h);					
					double csOther_proton=ndMapCurOther.getCSProton();
					double csOther_heavy=ndMapCurOther.getCSHeavy();
					//compute the homo score:
					Vector<H1CS> vecOneSCAsgOther=new Vector<H1CS>();//for computing homo score.
					vecOneSCAsgOther.add(new H1CS(resNo,res,atom,csOther_proton));
					vecOneSCAsgOther.add(new H1CS(resNo,res,heavyName,csOther_heavy));
	    			double asgOtherScore=h1cs.BMRBSatisticsScore(vecOneSCAsgOther,vecBMRBNew);
	        		double totalOtherScore=asgOtherScore;
	        		
	        		int minAdjID=-1;
	        		//compute the worse pairwise score for other nodes:
	        		for(int t=0;t<vecAdj.size();t++){
	        			ProtonLabel ndCurAdj=(ProtonLabel)vecAdj.elementAt(t);
						if(ndCurAdj.getIsAssigned())
							continue;
						Vector<GraphNode> vecNdCurAdjMap=ndCurAdj.getMappingSet();
						if(vecNdCurAdjMap.size()<1)//rarely happen
							continue;
						
						int resNoAdj=ndCurAdj.getResNo();
						String resAdj=ndCurAdj.getResName();
						String atomAdj=ndCurAdj.getAtomName();
						String heavyNameAdj=pk.GetHeavyAtomFromProton(resAdj,atomAdj);					
						minAdjID=-1;	
						
						double minScore=999999.9;
						Vector<BackNoe> vecBackNoe=new Vector<BackNoe>();//for computing pairwirse 
						for(int a=0;a<vecNdCurAdjMap.size();a++){
							GraphNode ndAdjMap=(GraphNode)vecNdCurAdjMap.elementAt(a);
							Vector<Noesy> vecNoesyTwo=new Vector<Noesy>();	
							vecNoesyTwo.addAll(ndMapCurOther.vecAdjNOESY);
							vecNoesyTwo.addAll(ndAdjMap.vecAdjNOESY);
							
							double csAdjMapProton=ndAdjMap.getCSProton();
							double csAdjMapHeavy=ndAdjMap.getCSHeavy();
							vecBackNoe.add(new BackNoe(csCur_proton,csCur_heavy,csAdjMapProton,6.0,0.0,resNo,resNoAdj,res,resAdj,atom,heavyName,atomAdj ));
							vecBackNoe.add(new BackNoe(csAdjMapProton,csAdjMapHeavy,csCur_proton,6.0,0.0,resNoAdj,resNo,resAdj,res,atomAdj,heavyNameAdj,atom ));
							
							Vector<BackNoe> vecBackNoeNew=bkNoe.DeleteRepeat(vecBackNoe);
							int [] numPeaks=new int[1];
							double dbScore=asg.NoePatternMatchScore(csErrH,csErrN, csErrCA,vecBackNoeNew,vecNoesyTwo, numPeaks,true);
							dbScore=dbScore*vecBackNoeNew.size();//the more number of matched nodes, the better
							if(dbScore<minScore){
								minScore=dbScore;
								minAdjID=a;
							}
						}//for(int a=0;a<vecNdCurAdjMap.size();a++)
						totalOtherScore=totalOtherScore+minScore;			
					}//for(int t=0;t<vecAdj.size();t++)
	        		
	        		//check whether the DEE criterion is satisfied:
	        		if(totalCurScore<totalOtherScore){
	        			isPruned=true;
	        			break;
	        		}
	        		
				}//for(int h=0;h<vecMapSet.size();h++)
				
				//remove the mapping if it is pruned:
				if(isPruned){
					Map[ndMapCur.getID()][nodeProton.getID()]=0;
					counter++;
				}
			
			}//for(int k=0;k<vecMapSet.size();k++)
						
		}//for(int j=0;j<vecGraph.size();j++)			
	
		return counter;
		//System.out.println("Total number of mappings pruned by DEE-CUT is: "+ counter);	
	
	}
	

	/**
     * print out all possible for proton nodes
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs      
     @return 
     */
	public void PrintAllPossibleMappings(Vector<GraphNode> vecResonGraph,Vector<ProtonLabel> vecLabels, int [][] Map)
	{
		int i,j;
		for(j=0;j<vecLabels.size();j++)	{
			ProtonLabel label=(ProtonLabel)vecLabels.elementAt(j);
    		int resNo=label.getResNo();
    		String resName=label.getResName();
    		String atomName=label.getAtomName();
    		String subAtomName=atomName;
    		if(atomName.length()>2)
    			subAtomName=atomName.substring(0,2);
    		if(atomName.equalsIgnoreCase("H")||subAtomName.equalsIgnoreCase("HA")||subAtomName.equalsIgnoreCase("HN"))
    			continue;
    			
    		System.out.println(resName+resNo+"-"+atomName+":");
    		
    		for(i=0;i<vecResonGraph.size();i++)	{
    			GraphNode node=(GraphNode)vecResonGraph.elementAt(i);
    			double cs_h=node.getCSProton();
    			double cs_heavy=node.getCSHeavy();
    			if(Map[i][j]==1)
    				System.out.format("        %.3f ,  %.3f%n", cs_h, cs_heavy);
    			
    		}//for(i=0;i<vecResonGraph.size();i++)
    		
    	}//for(j=0;j<vecGraph.size();j++)
    	
	}
	
	/**
     * compute the statistical information of the side-chain assignment
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs  
     @param vecRefAsg reference assignment table
     @param 
     @return 
     */
	public void StatisticsAllSCAssignment(Vector<GraphNode> vecResonGraph,Vector<ProtonLabel> vecLabels, int [][] Map,Vector<H1CS> vecRefAsg )	{
		Peak pk=new Peak();
		int totalScH=0;
		int missing=0;
		int averSize=0,sum=0;
		int maxSize=0;
		String maxStr="";
		int[] counter=new int[200];
		Assign asg=new Assign();
		int notInRef=0;
		boolean [] isProtonFound=new boolean[vecLabels.size()];
		for(int j=0;j<vecLabels.size();j++)
			isProtonFound[j]=false;
		
		UpdateMappingSet(vecResonGraph,vecLabels, Map);
		for(int k=0;k<vecRefAsg.size();k++)
		{
			H1CS h1cs=(H1CS)vecRefAsg.elementAt(k);
			int resRefNo=h1cs.getResidueNo();
			String resRef=h1cs.getResidueType();
			String atomRef=h1cs.getAtomName();
			double csRef=h1cs.getH1CS();
			double csHeavyRef=-999.9;
			if(atomRef.equalsIgnoreCase("CA") ||atomRef.equalsIgnoreCase("N")|| 
					atomRef.equalsIgnoreCase("HN") ||atomRef.equalsIgnoreCase("H")||
					atomRef.equalsIgnoreCase("HA")||atomRef.equalsIgnoreCase("HA1")|| 
					atomRef.equalsIgnoreCase("HA2")||atomRef.equalsIgnoreCase("HA3")||
					atomRef.equalsIgnoreCase("C")||atomRef.equalsIgnoreCase("CO")||atomRef.equalsIgnoreCase("CB"))
				continue;			
						
			if(!atomRef.substring(0,1).equalsIgnoreCase("H"))
				continue;
			String heavyRef=pk.GetHeavyAtomFromProton(resRef,atomRef);
			totalScH++;
			//get the heavy atom chemcial shift in ref table
			for(int t=0;t<vecRefAsg.size();t++)	{
				H1CS h1csTemp=(H1CS)vecRefAsg.elementAt(t);
				int resNoTemp=h1csTemp.getResidueNo();
				String atomTemp=h1csTemp.getAtomName();
				if(atomTemp.equalsIgnoreCase(heavyRef) && resNoTemp==resRefNo)
					csHeavyRef=h1csTemp.getH1CS();
			}//for(int t=0;t<vecRefAsg.size();t++)
			boolean isRefInCur=false;
			for(int j=0;j<vecLabels.size();j++)	{
				ProtonLabel nodeStr=(ProtonLabel)vecLabels.elementAt(j);
				
				String atom=nodeStr.getAtomName();
				String subAtom=atom;
				if(atom.length()>atomRef.length())
					subAtom=atom.substring(0,atomRef.length());				
			
				int resNo=nodeStr.getResNo();							
				if( !(resNo==resRefNo && subAtom.equalsIgnoreCase(atomRef)) )
					continue;
				Vector<GraphNode> vecMapSet=nodeStr.getMappingSet();
				for(int w=0;w<vecMapSet.size();w++)	{
					GraphNode resonNode=(GraphNode)vecMapSet.elementAt(w);
					double csProton=resonNode.getCSProton();
					double csHeavy=resonNode.getCSHeavy();
					if(Math.abs(csProton-csRef)<0.05 && Math.abs(csHeavy-csHeavyRef)<0.5 )
						isRefInCur=true;
				}//for(int w=0;w<vecMapSet.size();w++)				
			}//for(int j=0;j<vecStructGraph.size();j++)
			
			if(!isRefInCur)	{
				missing++;
				System.out.println("Missing proton assignment: "+ resRef+" "+resRefNo+" - "+atomRef);
			}			
						
			
		}//for(int k=0;k<vecRefAsg.size();k++)		
		
		System.out.println("Total number of sidechain protons in ref table: "+ totalScH);
		System.out.println("Total number of missing sidechain assignments: "+ missing);
		
	}


	
	/** 
     *Implement the NOE assignment (NASCA) algorithm without using the side-chain resonance information.
     @param src location of the input file
     @param strOut location of the output file
     @param strInput  input file name
     @return void
    */       
   // @SuppressWarnings("unchecked")
	public void doNasca(String src, String strOut, String strInput)throws JampackException
	{
    	boolean isDebug=false;//whether outputing the debugging information
    	Hsqc hqc = new Hsqc();
    	Peak pk = new Peak();
    	
    	ModelRdc mdc = new ModelRdc();    	
    	Assign asg = new Assign();
    	long startTime = System.currentTimeMillis();    	
    	/////////////////////////////////////////////
    	/// 1. Read the input files
    	//
    	/////////////////////////////////////////////
    	int i, j;    	
    	Pdb  pp = new Pdb();    	
    	Vector<Map<String, String>> paraVec = asg.ParamReader(src+strInput);
    	double haErr  = 0.0;
    	double h1Err  = 0.0;
    	double c13Err = 0.0;
    	double hnErr  = 0.0;
    	double nErr   = 0.0;
    	
    	int nIs4DNoesy=0;
    	
    	String strReson="", strSeq="", strNoesy2D="", strHnNoesy3D="", strResNameScheme="",strIsCheckLongAA="",
    		strHaNoesy3D="", strResFormat="",strBB="",stroutName="",strNoesyFormat="",strOutOrFormat="",
    		strIsCheck="",strRefPdb="",strRefNameSchem="",strIsWholeStr="",strPdbNameScheme="",str4DNoesy="",
    		stroutResonName="",strOutResonFormat="";
    	int nIsoriginalUp=0; 
    	double noeLimit=0.0,metCor=0.0;
    	
    	for (i=0;i<paraVec.size();i++)
    	{
    		//get the error torlence for each dimension
    		Map<String, String> paraMap = paraVec.elementAt(i);
    		if(paraMap.containsKey("HAERR"))
    			 haErr  =  Double.parseDouble((String)paraMap.get("HAERR"));
    		if(paraMap.containsKey("H1ERR"))
   			 	h1Err  =  Double.parseDouble((String)paraMap.get("H1ERR"));
    		if(paraMap.containsKey("C13ERR"))
    			c13Err  =  Double.parseDouble((String)paraMap.get("C13ERR"));
    		if(paraMap.containsKey("HNERR"))
    			hnErr  =  Double.parseDouble((String)paraMap.get("HNERR"));
    		if(paraMap.containsKey("NERR"))
    			nErr  =  Double.parseDouble((String)paraMap.get("NERR"));
    		//get the input file name   
    		if(paraMap.containsKey("SEQUENCE"))
    			strSeq  =  (String)paraMap.get("SEQUENCE");
    		if(paraMap.containsKey("RESFORMAT"))
    			strResFormat  =  (String)paraMap.get("RESFORMAT");
    		if(paraMap.containsKey("RESONANCE"))
    			strReson  =  (String)paraMap.get("RESONANCE");
    		strReson=strReson.toLowerCase();
    		if(paraMap.containsKey("2D-NOESY"))
    			strNoesy2D  =  (String)paraMap.get("2D-NOESY");
    		//strNoesy2D=strNoesy2D.toLowerCase();
    		if(paraMap.containsKey("3D-N15-NOESY"))
    			strHnNoesy3D  =  (String)paraMap.get("3D-N15-NOESY");
    		//strHnNoesy3D=strHnNoesy3D.toLowerCase();
    		if(paraMap.containsKey("3D-C13-NOESY"))
    			strHaNoesy3D  =  (String)paraMap.get("3D-C13-NOESY");
    		//strHaNoesy3D=strHaNoesy3D.toLowerCase();    		
    		if(paraMap.containsKey("PDBNAMESCHEME"))
    			strPdbNameScheme  =  (String)paraMap.get("PDBNAMESCHEME");	
    		if(paraMap.containsKey("OUTNOENAME"))
    			stroutName  =  (String)paraMap.get("OUTNOENAME");   	
    		if(paraMap.containsKey("BACKBONE"))
    			strBB  =  (String)paraMap.get("BACKBONE");
    		if(paraMap.containsKey("NOELIMIT"))
    			noeLimit  =  Double.parseDouble((String)paraMap.get("NOELIMIT"));
    		if(paraMap.containsKey("IS4DNOESY"))
    			nIs4DNoesy  =   Integer.parseInt((String)paraMap.get("IS4DNOESY"));
    		
    		if(paraMap.containsKey("4D-NOESY"))
    			str4DNoesy  =  (String)paraMap.get("4D-NOESY");
    		
    		if(paraMap.containsKey("ISORIGINALUP"))
    			nIsoriginalUp  =   Integer.parseInt((String)paraMap.get("ISORIGINALUP"));	
    		if(paraMap.containsKey("RESNAMESCHEME"))
    			strResNameScheme  =  (String)paraMap.get("RESNAMESCHEME");	
    		if(paraMap.containsKey("NOESY-FORMAT"))
    			strNoesyFormat  =  (String)paraMap.get("NOESY-FORMAT");   	
    		if(paraMap.containsKey("ISCHECKLONGAA"))
    			strIsCheckLongAA =  (String)paraMap.get("ISCHECKLONGAA"); 	
    		if(paraMap.containsKey("ISCHECK"))
    			strIsCheck  =  (String)paraMap.get("ISCHECK");   	
    		if(paraMap.containsKey("REFPDB"))
    			strRefPdb  =  (String)paraMap.get("REFPDB");  
    		if(paraMap.containsKey("METHYL-CORRECTION"))
    			metCor  =  Double.parseDouble((String)paraMap.get("METHYL-CORRECTION"));    		
    		if(paraMap.containsKey("REFNAMESCHEME"))
    			strRefNameSchem  =  (String)paraMap.get("REFNAMESCHEME");  
    		if(paraMap.containsKey("ISWHOLESTRUCTURE"))
    			strIsWholeStr  =  (String)paraMap.get("ISWHOLESTRUCTURE");  
    		if(paraMap.containsKey("ISOUTORFORMAT"))
    			strOutOrFormat  =  (String)paraMap.get("ISOUTORFORMAT");  
    		
    		if(paraMap.containsKey("OUTRESONANCENAME"))
    			stroutResonName  =  (String)paraMap.get("OUTRESONANCENAME"); 
    		if(paraMap.containsKey("OUTRESONANCEFORMAT"))
    			strOutResonFormat  =  (String)paraMap.get("OUTRESONANCEFORMAT");
    	}       	   	
    	//-------------------------------------
    	// (1.1) Read the protein sequence
    	// 
    	//-------------------------------------    	
    	String seqFile = src + strSeq;
    	//this is the vector of residues (protein sequence)
    	Vector<Assign> vecSeq=asg.ReaderSeq(seqFile);    	
    	
    	//-------------------------------------
    	// (1.2) Read the resonance list 
    	//     	
    	//following is for the general case
    	H1CS h1CS=new H1CS();    	
    	String assignFile = src+ strReson;
    	//this is the vector that stores resonance assignments;
    	//each one includes resNo, resName, atomName, csH1(chemical shift value)
    	Vector<H1CS> assignVec=new Vector<H1CS>();
    	if(strResFormat.equalsIgnoreCase("CYANA"))
    		assignVec=h1CS.h1CSReader(assignFile,vecSeq); 
    	else if(strResFormat.equalsIgnoreCase("BMRB"))
    		assignVec=h1CS.h1CSReader_BMRB(assignFile);  
    	else 
    	{
    		System.out.println("unknown format of the resonance file...");
    		System.exit(0);
    	}
    	
    	//all sorted proton resonances
    	Vector<Peak> allH1Vec = pk.allProtonSorted(assignVec);
    	Collections.sort(allH1Vec, new Peak.csComparator());     	
    	//-------------------------------------
    	// (1.3) Read the NOESY peak list 
    	// in the xeasy format
    	// including 2D NOESY peak list, 3D N15 NOESY and 3D C13 NOESY peak lists
    	//-------------------------------------
    	Noesy noesy = new Noesy();
    	
    	String strNoeFile = ""; 
    	Vector<Noesy> hnNoeVec=new Vector<Noesy>();//for storing 3d n15 NOESY peaks
    	Vector<Noesy> cnoeVec=new Vector<Noesy>();//for storing 3d c13 noesy peaks
    	Vector<Noesy> Noe4DVec=new Vector<Noesy>();//for storing 4d NOESY peaks
    	if (!strNoesy2D.equalsIgnoreCase("NULL"))
    	{
    		//to do: extension to 2d case...
    	}
    	if (!strHnNoesy3D.equalsIgnoreCase("NULL"))
    	{
    		strNoeFile=src+strHnNoesy3D;
    		if(strNoesyFormat.equalsIgnoreCase("XEASY"))
    			hnNoeVec = noesy.NoesyReader(strNoeFile);
    		else
    			hnNoeVec = noesy.NoesyReaderNMRView(strNoeFile);    		
    	}//if (!strNoesy3D.equalsIgnoreCase("NULL"))
    	
    	if (!strHaNoesy3D.equalsIgnoreCase("NULL"))
    	{
    		strNoeFile=src+strHaNoesy3D;
    		if(strNoesyFormat.equalsIgnoreCase("XEASY"))
    			cnoeVec = noesy.NoesyReader(strNoeFile);
    		else
    			cnoeVec=noesy.NoesyReaderNMRView(strNoeFile);
    	}
    	
    	//4D NOESY data
    	if (!str4DNoesy.equalsIgnoreCase("NULL"))
    	{
    		strNoeFile=src+str4DNoesy;
    		if(strNoesyFormat.equalsIgnoreCase("XEASY"))
    			Noe4DVec = noesy.NoesyReader(strNoeFile);// to be changed....
    		else
    			Noe4DVec = noesy.NoesyReaderNMRView4D(strNoeFile);    		
    	}//if (!strNoesy3D.equalsIgnoreCase("NULL"))
    	
    	Vector<Noesy> vecNoesy=new Vector<Noesy>();
    	vecNoesy.addAll(hnNoeVec);
    	vecNoesy.addAll(cnoeVec);
    	
    	Vector<Noesy> vecNoesy4D=new Vector<Noesy>();
    	vecNoesy4D.addAll(Noe4DVec);//for 4D noesy    	

    	////////////////////////////////////    	    	
    	//intensity calibration    	
    	Vector<Noesy> vecNewNoesy=new Vector<Noesy>();
    	if(vecNoesy.size()>0)
    		vecNewNoesy=noesy.SetCalibration(vecNoesy); ////  	
    	Vector<Noesy> vecNewNoesy4D=new Vector<Noesy>();
    	if(vecNoesy4D.size()>0)
    		vecNewNoesy4D=noesy.SetCalibration(vecNoesy4D);   
    	double[] constant=new double[1];
    	Vector<Pdb> vecSSE=new Vector<Pdb>();//not used in the following function:
    	noesy.SetCalibrationN15(hnNoeVec,vecSSE,assignVec,constant);//change from vecNoesy to hnNoeVec
    	
    	// -------------------------------------
    	// (1.4) Read the backbone structure     	
    	//-------------------------------------    	
    	String pdbFile = src+strBB;
    	Vector<Pdb> pdbVecBB = pp.readPdb(pdbFile);
    	Vector<Pdb> pdbVecNew=new Vector<Pdb>();
    	
    	//update backbone atom names according to protein sequence input
    	//residue name in previous backbone may be all in "ALA".
    	if(strIsWholeStr.equalsIgnoreCase("1"))    	
    		pdbVecNew= pp.residueNameUpdateNoStr(vecSeq, pdbVecBB); 
    	else
    		pdbVecNew = pp.residueNameUpdate(vecSeq, pdbVecBB);
    	
    	Vector<Pdb> pdbVecNewSec=new Vector<Pdb>();
		if(strPdbNameScheme.equalsIgnoreCase("PDB-OLD"))
			pdbVecNewSec=pp.nameConvertOrder(pdbVecNew); 
		else if(strPdbNameScheme.equalsIgnoreCase("BMRB-NEW"))
			pdbVecNewSec=pp.nameConvert2PDBNew(pdbVecNew);//need to double check
		else if(strPdbNameScheme.equalsIgnoreCase("BMRB-OLD"))
		{
			Vector<Pdb> pdbVecSSE22=pp.nameConvertOrder(pdbVecNew); 
			pdbVecNewSec=pp.nameConvert2PDBNew(pdbVecSSE22);//need to double check
		}
		else
			pdbVecNewSec.addAll(pdbVecNew);  
    	
		Vector<Pdb> vecStructureAll=new Vector<Pdb>();
		vecStructureAll.addAll(pdbVecNewSec);
		
		Vector<Pdb> vecBBTemp=pp.OutputBackbone(pdbVecNewSec);
		pdbVecNewSec=new Vector<Pdb>();
		pdbVecNewSec.addAll(vecBBTemp);    	
    	
		/////////////////////////////////////////////
    	/// 2. Implement the NASCA algorithm
    	//
    	/////////////////////////////////////////////
		ProtonLabel protonLabel=new ProtonLabel();
		GraphNode graphNode=new GraphNode();
		
		String userDir = System.getProperty("user.dir");////
    	String srcRot=userDir+"/system/rot-lib/";//path of rotamer library    	
    	
    	//------------------------------------------
    	//2.1 construct the set of proton labels (which consists of proton coordinates, rotamer index and proton name):
    	//------------------------------------------
    	//construct the label set (here we also call the structure graph), which includes
    	//the set of proton labels after placing rotamers on the RDC-defined backbone
      	Vector<ProtonLabel> vecLabels=protonLabel.ConstructProtonLabelSet(pdbVecNewSec,srcRot,5.0);
    	
      	//------------------------------------------
    	//2.2 construct the NOESY graph:
    	//------------------------------------------    	
    	Vector<H1CS> vecAsgNew=new Vector<H1CS>();		
		vecAsgNew=h1CS.CSNameConvertFromBMRBNewToPDBNew(assignVec);//assume that the default naming scheme of chemical shift is BMRB
		assignVec=new Vector<H1CS>();
		assignVec.addAll(vecAsgNew);
    	Vector<H1CS> vecBBAsg=ExtractBackboneAsignment(assignVec);		
    	
		//construct initial NOESY graph using the assigned backbone resonances
		Vector<GraphNode> vecResonNodeSet=graphNode.ConstructInitResonGraphFromAsgedList(vecBBAsg,null);
		
		//add the unassigned side-chain resonance nodes from NOESY peak list
		for(i=0;i<vecNewNoesy.size();i++){
			Noesy ne=(Noesy)vecNewNoesy.elementAt(i);
			double csH1=ne.getH1();
			double csHeavy=ne.getHeavy();
			double csH2=ne.getH2();
			Vector<GraphNode> vecResGraphTemp=new Vector<GraphNode>();
			vecResGraphTemp=graphNode.AddResonanceNode(vecResonNodeSet,csH1,csHeavy,0.02, 0.1, 0.1);//error windows: by half
			
			vecResonNodeSet=new Vector<GraphNode>();
			vecResonNodeSet.addAll(vecResGraphTemp);
		}//for(i=0;i<vecNewNoesy.size();i++){
		
		//group each resonance cluster:
		for(i=0;i<vecResonNodeSet.size();i++){
			GraphNode node=(GraphNode)vecResonNodeSet.elementAt(i);
			Vector<GraphNode> vecAdj=node.getAdjVec();
			int counter=1;
			double csProtonSum=node.getCSProton();
			double csHeavySum=node.getCSHeavy();
			for(int k=0;k<vecAdj.size();k++)///////double check this step
			{
				GraphNode ndAdj=(GraphNode)vecAdj.elementAt(k);
				csProtonSum=csProtonSum+ndAdj.getCSProton();
				csHeavySum=csHeavySum+ndAdj.getCSHeavy();
				counter++;
			}
			csProtonSum=csProtonSum/counter;
			csHeavySum=csHeavySum/counter;
			node.setProtonCS(csProtonSum);
			node.setHeavyCS(csHeavySum);
			node.EmptyAdjSet();
		}		
		//set up the edge connections for resonances:
		Vector<GraphNode> vecResonGraph=graphNode.ConstructResonGraph(vecResonNodeSet,vecNewNoesy,0.04,0.04,0.1,0.1);//0.04,0.03,0.2,0.2
		
		//set up symmetry set
		graphNode.ConstructResonGraphSymmetry(vecResonNodeSet,vecNewNoesy,0.03,0.03,0.1,0.1);//0.04,0.03,0.2,0.2
			
		//read BMRB statistical information:
		String srcBMRB=userDir+"/system/BMRB_CS.txt";    	
    	Vector<H1CS> vecBMRB=h1CS.ReadBMRBSatistics(srcBMRB);
    	Vector<H1CS> vecBMRBNew=new Vector<H1CS>();		
    	vecBMRBNew=h1CS.CSNameConvertFromBMRBNewToPDBNew(vecBMRB);//assume that the default naming scheme of cs is BMRB
    	
    	//--------------------------------------------------------
    	//2.3 pruning infeasible side-chain resonance assignments
    	//    using BMRB outliers and backbone chemical shifts.
    	//--------------------------------------------------------   
    	int [][]Map;//mapping matrix, for storing the mappings between NOESY graph and proton positions
    	double [][] MapScore;
    	Map=new int[vecResonGraph.size()][vecLabels.size()];
    	MapScore=new double[vecResonGraph.size()][vecLabels.size()];
    	
    	//initially set up all possible mappings
    	Initialization(vecResonGraph,vecLabels,Map,MapScore,vecBBAsg);
    	
    	/////////////////////////
    	//pruning side-chain resonance assignments that fall outside the BMRB intervals
    	RefineMappingBasedBMRBOutliers(vecResonGraph,vecLabels,Map,vecBMRBNew);
    	
    	//update the mapping set
    	UpdateMappingSet(vecResonGraph,vecLabels,Map);
    	UpdateIsRestrained(vecResonGraph,vecLabels, Map);    	
    	
    	//Refinement based on the bb resonance assignments:
        //Note: This function was implemented in the early stage as a pre-processing of the 
    	//side-chain resonance assignments. Now we 
    	//still keep it, since in practice it might speed up the pruning process. 
    	//In principle, the assignments pruned from this function can be also 
    	//pruned by the DEE assignment approach.
    	RefineMappingBasedRestrained(vecResonGraph,vecLabels,Map,MapScore,vecNewNoesy, vecBMRBNew);//////
    	
    	//Refinement based on previously assigned HG resonances:
        //Note: This function was implemented in the early stage as a pre-processing of the 
    	//side-chain resonance assignments. Now we 
    	//still keep it, since in practice it helps reduce the running time. 
    	//In principle, the assignments pruned from this function can be also 
    	//pruned by the DEE assignment approach.
       	RefineMappingForHG(vecResonGraph,vecLabels,Map,MapScore,vecNewNoesy, vecBMRBNew);    //////   	
     	UpdateMappingSet(vecResonGraph,vecLabels,Map);
     	
     	//-------------------------------------------
     	//2.4 Use the DEE assignment algorithm to prune side-chain 
     	//    resonance assignments that are provably not part of 
     	//    the optimal solution.
      	RefineMappingDeeCut(vecResonGraph,vecLabels,Map,MapScore,vecNewNoesy, vecBMRBNew);//new     
     	UpdateMappingSet(vecResonGraph,vecLabels,Map);     	
       	
        //print out all remaining assignments, after pruning. 
     	//This information is for debugging.
       	PrintAllPossibleMappings(vecResonGraph,vecLabels,Map);
       	
       	//output the statistical summary of the remaining assignments after pruning.
       	//This information is for debugging.
       	StatisticsAllSCAssignment(vecResonGraph,vecLabels, Map,assignVec ); 
    	
    	//update the mapping set
    	UpdateMappingSet(vecResonGraph,vecLabels,Map);
    	H1CS h1cs=new H1CS();
    	
    	//----------------------------------------------------
    	//2.5 Use the A* search algorithm to find the optimal 
    	//    side-chain resonance assignments.    	
    	//----------------------------------------------------
    	AStar astar=new AStar();
    	///Vector vecAsgNewT0=astar.AStarSearchForSCAsgsNew(vecResonGraph,vecLabels,Map,MapScore,vecNewNoesy,vecBMRBNew);
    	//Vector vecAsgNewT=h1cs.DeleteRepeatAsg(vecAsgNewT0);//delete repeated assignments.
    	
    	Vector vecAsgNewT0=astar.AStarSearchForSCAsgs(vecResonGraph,vecLabels,Map,MapScore,vecNewNoesy,vecBMRBNew);
    	Vector vecAsgNewT=h1cs.DeleteRepeatAsg(vecAsgNewT0);//delete repeated assignments.
    	
    	/*Vector vecAsgNewT0=new Vector();
    	Vector vecAsgNewT=new Vector();
    	astar.AStarSearchForSCAsgsNew(vecResonGraph,vecLabels,Map,MapScore,vecNewNoesy,vecBMRBNew);
    	
    	UpdateMappingSet(vecResonGraph,vecLabels,Map);
    	PrintAllPossibleMappings(vecResonGraph,vecLabels,Map);//debuggging...
    	StatisticsAllSCAssignment(vecResonGraph,vecLabels, Map,assignVec );
    	*/
    	
    	
    	//print out the resonance assignments in different formats:
    	//This information will be useful for debugging.
    	h1cs.PrintAllAssignment(vecAsgNewT);  //In BMRB format    	
    	h1cs.PrintAllAssignmentCyana(vecAsgNewT); //in CYANA format
    	
    	//////////////////////////////
    	//print out the resonance assignments into files:
    	String outResFileName=userDir+stroutResonName;    	
    	try{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(outResFileName)));
    		
    		if(strOutResonFormat.equalsIgnoreCase("CYANA"))        	
    			h1cs.PrintAllAssignmentToFileCyana(vecAsgNewT,out);        	
        	else        	
        		h1cs.PrintAllAssignmentToFileBMRB(vecAsgNewT,out);      	
	       
	    	out.close();
	    	System.out.println("The NOE assignment table has been generated in "+ outResFileName); 
    	}catch (FileNotFoundException e)
		{
			System.out.println("File not found: " + outResFileName);
		}catch (IOException e)
		{
		   System.out.println("IOException: the stack trace is:");
		   e.printStackTrace();
		}

    	///////////////////////////////////////   	
    	//compare with the reference resonance assignment list:
    	//this information is for debugging, and evaluating the performance.
    	h1cs.CompareSCAsgsWRefAsgs(vecAsgNewT,assignVec,vecBBAsg);    	
    	
    	//compute the completeness of the reference resonance assignment table:
    	double counter=0.0;
    	for(int t=0;t<vecLabels.size();t++)
    	{
    		ProtonLabel node=(ProtonLabel)vecLabels.elementAt(t);
    		String ndAtom=node.getAtomName();
    		int ndNo=node.getResNo();
    		
	    	for (int k=0;k<assignVec.size();k++)
	    	{	    		
	    		H1CS h1csT=(H1CS)assignVec.elementAt(k);	    		
	    		int refNo=h1csT.getResidueNo();
	    		String atom=h1csT.getAtomName();
	    		String subNdAtom=ndAtom;
	    		if(ndAtom.length()>atom.length())
	    			subNdAtom=ndAtom.substring(0,atom.length());
	    		
	    		if( !(atom.substring(0,1).equalsIgnoreCase("H")) )
	    			continue;
	    		if(ndNo==refNo && subNdAtom.equalsIgnoreCase(atom))
	    		{
	    			counter=counter+1;    	
	    			break;
	    		}
	    	}//end for (int k=0;k<assignVec.size();k++)
    	}//end for(int t=0;t<vecGraph.size();t++)
    	System.out.println("Completeness of reference assignment table is: "+(double) (counter/vecLabels.size()));
    
    	//compute the completeness of the our assignment table.
    	counter=0;
    	for(int k=0;k<vecAsgNewT.size();k++)
    	{
    		H1CS h1csT=(H1CS)vecAsgNewT.elementAt(k);
    		String atom=h1csT.getAtomName();
    		if(atom.substring(0,1).equalsIgnoreCase("H"))
    			counter=counter+1;     		
    	}
    	System.out.println("Completeness of our assignment table is: "+ (double)(counter/vecLabels.size()));
            
    	///vecBBAsg=new Vector();
    	///vecBBAsg.addAll(vecAsgNewT);
    	
    	////////////////////////////////////////////////    	
    	/*for(int k=0;k<vecBBAsg.size();k++)
    	{
    		H1CS h1_cs=(H1CS)vecBBAsg.elementAt(k);
    		
    		boolean isHydrophobic=false;
    		
    	}//for(int k=0;k<vecBBAsg.size();k++)    	
    	*/
    	//--------------------------------------------
    	//2.6 NOE assignment process
    	//
    	//--------------------------------------------
    	Vector vecAmbgAsg=new Vector();    	
    	//all sorted proton resonances
    	allH1Vec=new Vector();
    	allH1Vec = pk.allProtonSorted(vecAsgNewT);
    	Collections.sort(allH1Vec, new Peak.csComparator()); 
    	
    	//compute the initial ambiguous NOE assignments. 
    	//In principle we could use the NOESY graph to generate the set of 
    	//initial ambiugous assignments. But here we re-use the previously-written function.
    	//The result would be the same. 
    	vecAmbgAsg=asg.InitAmbiAssignment(nErr,c13Err,hnErr,haErr,h1Err,vecAsgNewT,allH1Vec,vecNewNoesy );        	
    	
    	//calibration for all initial NOE assignments
    	Vector vecAmbgAsgTemp=noesy.SetCalibrationCyana(vecAmbgAsg,vecNewNoesy,constant[0]);    	
    	vecAmbgAsg=new Vector();
    	vecAmbgAsg.addAll(vecAmbgAsgTemp);
     	 	
    	
     	//delete symmetric NOEs (which are redundent constraints):
     	Vector vecNoeRefine2=new Vector();     	
     	for(i=0;i<vecAmbgAsg.size();i++)
     	{
     		Noe noe=(Noe)vecAmbgAsg.elementAt(i);
     		int noA=noe.getResidueNoA();
     		int noB=noe.getResidueNoB();
     		String atomA=noe.getAtomA();
     		String atomB=noe.getAtomB();
     		if(noA==noB && atomA.equalsIgnoreCase(atomB))
     			continue;
     		
     		boolean isIn=true;
     		if(strOutOrFormat.equalsIgnoreCase("0"))
     		{
	     		for (j=0;j<vecNoeRefine2.size();j++)
	     		{
	     			Noe noeT=(Noe)vecNoeRefine2.elementAt(j);
	     			int noAT=noeT.getResidueNoA();
	     			int noBT=noeT.getResidueNoB();
	     			String atomAT=noeT.getAtomA();
	     			String atomBT=noeT.getAtomB();
	     			if(noA==noAT && atomA.equalsIgnoreCase(atomAT) && noB==noBT && atomB.equalsIgnoreCase(atomBT))
	     				isIn=false;
	     			if(noA==noBT && atomA.equalsIgnoreCase(atomBT) && noB==noAT && atomB.equalsIgnoreCase(atomAT))
	     				isIn=false;
	     		}//for (j=0;j<vecNoeRefine2.size();j++)
     		}//if(strOutOrFormat.equalsIgnoreCase("0"))
     		if(	isIn)
     			vecNoeRefine2.add(noe);
     	}
     	
     	vecAmbgAsg=new Vector();     	
     	vecAmbgAsg.addAll(vecNoeRefine2);
     	
     	//place the rotamers onto the RDC-defined backbone. 
     	Vector vecRotStructure=pp.RotamSelectAndStructure(hnErr, nErr, c13Err,pdbVecNewSec,vecAsgNewT,srcRot,vecNewNoesy,
    			4.5, 1,constant[0]);
     	
     	//We prune ambiguous NOE assignments based on the structural information. 
     	//Note: The NOE pruning process is a little different from what described in the paper
     	//but in the principle the results should be similar, since the structural information
     	//combined from RDC-defined backbone and rotamer library is used in a similar way.
     	Vector vecRefinedNoes=new Vector();    	
    	vecRefinedNoes=asg.PrunAmbgNoeAsg(vecRotStructure,vecAmbgAsg,noeLimit,0.0,nIsoriginalUp);
    	
     	Collections.sort(vecRefinedNoes, new Noe.NoeComparator()); 
     	
    	//sort the NOE table
    	if(!strOutOrFormat.equalsIgnoreCase("1"))
    	{
	    	for(i=0;i<vecAmbgAsg.size();i++)
	    	{
	    		Noe noe=(Noe)vecAmbgAsg.elementAt(i);
	    		int firstResNo=noe.getResidueNoA();
	    		int secondResNo=noe.getResidueNoB();
	    		String firstResName=noe.getResidueA();
	    		String secondResName=noe.getResidueB();
	    		String firstAtomName=noe.getAtomA();///
	    		String secondAtomName=noe.getAtomB();     		
	    		
	    		
	    		if(secondResNo<firstResNo)
	    		{
	    			noe.setResNoA(secondResNo);
	    			noe.setResNoB(firstResNo);
	    			noe.setAtom1Name(secondAtomName);
	    			noe.setAtom2Name(firstAtomName);
	    			noe.setResNameA(secondResName);
	    			noe.setResNameB(firstResName);
	    		}
	    	}//for(i=0;i<vecManAsgNew.size();i++)
    	
	    	Collections.sort(vecAmbgAsg, new Noe.noeComparatorC());
    	}    	      	
     	
    	if(strIsCheckLongAA.equalsIgnoreCase("1"))
    	{    		
    		Vector vecTemp=new Vector();
    		for(i=0;i<vecRefinedNoes.size();i++)
        	{
        		Noe noe=(Noe)vecRefinedNoes.elementAt(i);
        		int firstResNo=noe.getResidueNoA();
        		int secondResNo=noe.getResidueNoB();
        		String firstResName=noe.getResidueA();
        		String secondResName=noe.getResidueB();
        		String atomA=noe.getAtomA();
        		String atomB=noe.getAtomB();        		
        		String subAtomA=atomA;
        		String subAtomB=atomB;
        		if (atomA.length()>=2)
        			subAtomA=atomA.substring(0,2);
        		if(atomB.length()>=2) 
        			subAtomB=atomB.substring(0,2);
        			
        		if(Math.abs(firstResNo-secondResNo)<=4 )
        		{
        			vecTemp.add(noe);
        			continue;
        		}
        	
        		boolean isKeep=true;
        		if ( pk.isCharged(firstResName) && pk.isHydrophobic(secondResName) )
        			isKeep=false;
        		if ( pk.isCharged(secondResName) && pk.isHydrophobic(firstResName) )
        			isKeep=false;
        		if( (subAtomB.equalsIgnoreCase("HN") || subAtomB.equalsIgnoreCase("H")||subAtomB.equalsIgnoreCase("HA")||subAtomB.equalsIgnoreCase("HB")) &&
        				(subAtomA.equalsIgnoreCase("HN")||subAtomA.equalsIgnoreCase("H")||subAtomA.equalsIgnoreCase("HA")|| subAtomA.equalsIgnoreCase("HB")   ))
        			isKeep=true;
        		
        		if(isKeep)
        			vecTemp.add(noe);
        	}//for(i=0;i<vecManAsgNew.size();i++)
    		
    		vecRefinedNoes=new Vector();
    		vecRefinedNoes.addAll(vecTemp);
    	}//if(strIsCheckLongAA.equalsIgnoreCase("1"))    	
    	
    	//sort the NOE restraints, and updated the atom names for metyl groups:
    	Collections.sort(vecRefinedNoes, new Noe.NoeComparator()); 
        Noe ne=new Noe();
        Vector vecNoeNewT=ne.NameConvertFromPDBNewToPDBNew(vecRefinedNoes);
        vecRefinedNoes=new Vector();
        vecRefinedNoes.addAll(vecNoeNewT);             
       
     	//compute the numbers of unique and multiple assignments:
     	vecNoeRefine2=new Vector();
     	Vector vecNoeUniq=new Vector();
     	for(i=0;i<vecRefinedNoes.size();i++)
     	{
     		Noe noe=(Noe)vecRefinedNoes.elementAt(i);
     		int pkID=noe.getPeakID();
     		boolean isUniq=true;
     		for(j=0;j<vecRefinedNoes.size();j++)
     		{
     			if (i==j)
     				continue;
     			Noe noeT=(Noe)vecRefinedNoes.elementAt(j);
     			int pkIDT=noeT.getPeakID();
     			if(pkID==pkIDT)
     				isUniq=false;
     		}
     		if(isUniq)
     			vecNoeUniq.add(noe);
     	}
     	
    	//compare the NOE assignments with the reference structure and output the summary.
     	double[] number=new double[4];
     	if(strIsCheck.equalsIgnoreCase("1"))
     	{
     		Noe noe_temp=new Noe();
     		
     		String pdbFileXray = src+strRefPdb;
        	Vector vecTempPdbBB=pp.readPdb(pdbFileXray);
        	Vector vecTempPdbBB2=new Vector();        	
        	
        	if(strRefNameSchem.equalsIgnoreCase("PDB-OLD")) 	
        		vecTempPdbBB2 = pp.nameConvertOrder(vecTempPdbBB);
        	else if(strRefNameSchem.equalsIgnoreCase("BMRB-NEW"))
        		vecTempPdbBB2=pp.nameConvert2PDBNew(vecTempPdbBB);
    		else if(strRefNameSchem.equalsIgnoreCase("BMRB-OLD"))
    		{
    			Vector pdbVecSSE22=pp.nameConvertOrder(vecTempPdbBB); 
    			vecTempPdbBB2=pp.nameConvert2PDBNew(pdbVecSSE22);
    		}
        	else
        		vecTempPdbBB2.addAll(vecTempPdbBB);
        	System.out.println("============================================ ");
        	System.out.println("====Comparisons with reference structure===== ");
        	noe_temp.CompareAsgToRefStr(vecRefinedNoes,vecTempPdbBB2,6.0,0.0,number);
        	//System.out.println("Total number of manual NOEs = "+ vecManAsg.size());
        	System.out.println("Number of correct NOE assignments in reference structure=  "+ number[0] );
        	System.out.println("Number of wrong NOE assignments in reference structure=  "+ number[1] );    	
        	System.out.println("Number of correct Long-Rang NOE assignments in reference structure=  "+ number[2] );
        	System.out.println("Number of wrong Long-Rang NOE assignments in reference structure=  "+ number[3] );
        	
        	System.out.println("============================================ ");
     	}//if(strIsCheck.equalsIgnoreCase("1"))
     	/////////////////////////////////////////////
     	
    	//Output the final NOE assignments in standard xplor format, and statistical summary    	    	
     	String userDirTemp = System.getProperty("user.dir");
     	if(strOutOrFormat.equalsIgnoreCase("0"))
    	{
	     	String fileName=userDirTemp+stroutName;    	
	    	
	    	String xplorNoeStr="";
	    	double[] disUpper=new double[1];
	    	try{
	    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
	    		out.println("!REMARK: Total number of NOESY cross peaks is "+ (vecNoesy.size()+vecNoesy4D.size()));    		
	    		out.println("!REMARK: Total number of NOEs is "+vecRefinedNoes.size());
	    		out.println("!REMARK: the number of unique NOE assignments is "+vecNoeUniq.size());
	    		out.println("!REMARK: the number of multiple NOE assignments is "+(vecRefinedNoes.size()-vecNoeUniq.size()) );
	    		    		
	    		out.println("!REMARK: Number of correct assignments in reference structure=  "+ number[0] );
	        	out.println("!REMARK: Number of wrong assignments in reference structure=  "+ number[1] );    	
	        	out.println("!REMARK: Number of correct Long-Rang NOE assignments in reference structure=  "+ number[2] );
	        	out.println("!REMARK: Number of wrong Long-Rang NOE assignments in reference structure=  "+ number[3] );
	    		out.println("");
	    		out.println("");
	    		
		        for (i=0;i<vecRefinedNoes.size();i++)		       	
		    	{
		    		Noe noe=(Noe)vecRefinedNoes.elementAt(i);
		    		int resNo1=noe.getResidueNoA();
		    		int resNo2=noe.getResidueNoB();
		    		String res1=noe.getResidueA();
		    		String res2=noe.getResidueB();
		    		String atom1=noe.getAtomA();
		    		String atom2=noe.getAtomB();
		    		double distUpper=noe.getUpper();
		    		
		    		//xplor format:	    		
					xplorNoeStr =pk.xplorNoeStatementNew(resNo1, res1, atom1,resNo2, res2, atom2, distUpper);
					if(isDebug)
						System.out.println(xplorNoeStr); 
					out.println(xplorNoeStr);
		    	}//for (i=0;i<vecRefinedNoes.size();i++)
		    	out.close();
		    	System.out.println("The NOE assignment table has been generated in "+ fileName); 
	    	}catch (FileNotFoundException e)
			{
				System.out.println("File not found: " + fileName);
			}catch (IOException e)
			{
			   System.out.println("IOException: the stack trace is:");
			   e.printStackTrace();
			}
    	}
     	else if(strOutOrFormat.equalsIgnoreCase("1")) //we output the NOEs in OR format:
    	{
    		Noe noeTemp=new Noe();    		
    		Vector vecMultiAsg=noeTemp.ConvertSingleToMultiAsg(vecRefinedNoes);
    		String fileName=userDirTemp+stroutName;  
    		noeTemp.OutputMultiNOEAsg(vecMultiAsg,fileName);    		
    	}
     	
     	///////////////////////////////////////
     	long endTime = System.currentTimeMillis();
    	double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
    	System.out.println("Time= "+ totalTime +" minutes");
	}
}
