package rdcPanda;

 ///////////////////////////////////////////////////////////////////////////////////////////////
//	Model.java
//
//	  Version:           0.1
//
//
//	  authors:
// 	  initials            name                      organization               email
//	 ---------   -----------------------        ------------------------    ------------------
//	  LW            Lincong Wang                  Dartmouth College       wlincong@cs.dartmouth.edu
//    JMZ		 Jianyang (Michael) Zeng	       Duke University			zengjy@cs.duke.edu
//
///////////////////////////////////////////////////////////////////////////////////////////////



/*
	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.
	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	Lesser General Public License for more details.
	
	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
	USA
	
	Contact Info:
		Bruce Donald
		Duke University
		Department of Computer Science
		Levine Science Research Center (LSRC)
		Durham
		NC 27708-0129 
		USA
		brd@cs.duke.edu
	
	If you use or publish any results derived from the use of this program please cite:
	J. Zeng, J. Boyles, C. Tripathy, L. Wang, A. Yan, P. Zhou and B.R. Donald. 
	"High-Resolution Protein Structure Determination Starting with a Global Fold 
	Calculated from Exact Solutions to the RDC Equations." Submitted For Review.

	Copyright (C) 2009 Jianyang (Michael) Zeng, Lincong Wang and Bruce R. Donald		
	<signature of Bruce Donald>, June 2008 and January 2009
	Bruce Donald, Professor of Computer Science
 */

       
  
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Collections;
import java.util.Map;
import java.util.TreeMap;
import java.util.Vector;

import Jampack.Inv;
import Jampack.JampackException;
import Jampack.Zmat;

// TODO: Auto-generated Javadoc
/** * 
 *  
*   This class provides functions that compute backbone from RDC-EXACT algorithm.
*    Written by Lincong Wang (2001-2005) and Jianyang (Michael) Zeng (2005-2009)
*/
public class Model
{

    /**
     * Instantiates a new model.
     */
    public Model(){
    }
    
    /**
     * Parameter reader.
     * 
     * @param fileName the file name
     * 
     * @return the vector
     */
    public Vector parameterReader(String fileName){
	Vector parameterVec = new Vector();
	String ss ="", str ="";
	String ss2   = "";
	int    index = -1;
	int    ind2  = -1;
	String key   = "";
	String value = "";
	Map mapSaupe = new TreeMap();
	Map mapHelices = new TreeMap();
	Map map1stStrand = new TreeMap();
	Map map2ndStrand = new TreeMap();
	Map mapOtherStrand = new TreeMap();
	try{
	    BufferedReader in = new BufferedReader(new FileReader(fileName));	
	    if ((ss = in.readLine()) != null)
		ss = ss.toUpperCase();
	    while( ss != null ) {
		if ( ss.indexOf("SAUPE") > -1 && ss.indexOf("ELEMENT") > -1 ){
 		    ss = (in.readLine()).toUpperCase();
		    index = ss.indexOf("=");
		    while (index == -1){
			ss = (in.readLine()).toUpperCase();
			index = ss.indexOf("=");
		    }
		    while ( ss != null && index  > -1 ){
			key = ss.substring(0, index).trim();
			ind2 = ss.indexOf(";", index);
			if (ind2 < 0){
			    System.out.println("Error in parameter file: lines need to be ended by \";\", see Example");
			    System.exit(1);
			}
			value = ss.substring(index+1, ind2).trim();
			mapSaupe.put(key, value);
			if ((ss = in.readLine()) != null){
			    ss = ss.toUpperCase();
			    index = ss.indexOf("=");
			}
		    }
		}else if( ss.indexOf("HELICES") > -1 ){
                    ss = (in.readLine()).toUpperCase();
                    index = ss.indexOf("=");
                    while (index == -1){
                        ss = (in.readLine()).toUpperCase();
                        index = ss.indexOf("=");
                    }
                    while ( ss != null && index  > -1 ){
                        key = ss.substring(0, index).trim();
                        ind2 = ss.indexOf(";", index);
                        if (ind2 < 0){
                            System.out.println("Error in parameter file: lines must be ended by \";\", see Example");
                            System.exit(1);
                        }
                        value = ss.substring(index+1, ind2).trim();
                        mapHelices.put(key, value);

                        if ((ss = in.readLine()) != null){
                            ss = ss.toUpperCase();
                            index = ss.indexOf("=");
                        }
                    }
                }else if( ss.indexOf("FIRST") > -1 && ss.indexOf("STRAND") > -1 ){
 		    ss = (in.readLine()).toUpperCase();
		    index = ss.indexOf("=");
		    while (index == -1){
			ss = (in.readLine()).toUpperCase();
			index = ss.indexOf("=");
		    }
		    while ( ss != null && index  > -1 ){
			key = ss.substring(0, index).trim();
			ind2 = ss.indexOf(";", index);
			if (ind2 < 0){
			    System.out.println("Error in parameter file: lines need to be ended by \";\", see Example");
			    System.exit(1);
			}
			value = ss.substring(index+1, ind2).trim();
			map1stStrand.put(key, value);
			if ((ss = in.readLine()) != null){
			    ss = ss.toUpperCase();
			    index = ss.indexOf("=");
			}
		    }
		}else if( ss.indexOf("SECOND") > -1 && ss.indexOf("STRAND") > -1 ){
 		    ss = (in.readLine()).toUpperCase();
		    index = ss.indexOf("=");
		    while (index == -1){
			ss = (in.readLine()).toUpperCase();
			index = ss.indexOf("=");
		    }
		    while ( ss != null && index  > -1 ){
			key = ss.substring(0, index).trim();
			ind2 = ss.indexOf(";", index);
			if (ind2 < 0){
			    System.out.println("Error in parameter file: lines need to be ended by \";\", see Example");
			    System.exit(1);
			}
			value = ss.substring(index+1, ind2).trim();
			map2ndStrand.put(key, value);
			if ((ss = in.readLine()) != null){
			    ss = ss.toUpperCase();
			    index = ss.indexOf("=");
			}
		    }
		}else if( ss.indexOf("OTHER") > -1 && ss.indexOf("STRAND") > -1 ){
 		    ss = (in.readLine()).toUpperCase();
		    index = ss.indexOf("=");
		    while (index == -1){
			ss = (in.readLine()).toUpperCase();
			index = ss.indexOf("=");
		    }
		    while ( ss != null && index  > -1 ){
			key = ss.substring(0, index).trim();
			ind2 = ss.indexOf(";", index);
			if (ind2 < 0){
			    System.out.println("Error in parameter file: lines need to be ended by \";\", see Example");
			    System.exit(1);
			}
			value = ss.substring(index+1, ind2).trim();
			mapOtherStrand.put(key, value);
			if ((ss = in.readLine()) != null){
			    ss = ss.toUpperCase();
			    index = ss.indexOf("=");
			}
		    }
		}
		if ((ss = in.readLine()) != null)
		    ss = ss.toUpperCase();
	    }
	    in.close();
	}catch (FileNotFoundException e) {
	    System.out.println("File not found: " + fileName);
	}catch (IOException e) {
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
	}
	parameterVec.add(mapSaupe);
	parameterVec.add(mapHelices);
	parameterVec.add(map1stStrand);
	parameterVec.add(map2ndStrand);
	parameterVec.add(mapOtherStrand);
	return parameterVec;
    }


    /**
     * rotate the whole protein according to the reference backbone
     * such that the backbone coordinates are the same (both should have the same residues).
     * 
     * @param pdbAll  pdb to be rotated
     * @param pdbBB  reference backbone pdb
     * 
     * @return the pdb vector after rotated.
     * 
     * @throws JampackException the jampack exception
     */	
    public Vector BackonbeRotation(final Vector pdbAll, final Vector pdbBB) throws JampackException
    {
    	int i,j,k,h;
    	Pdb pdbA =new Pdb();
    	Pdb pdbB =new Pdb();
    	Vector atomVecA=new Vector();
    	Vector atomVecB=new Vector();
    	Cartesian ccA = new Cartesian();
    	Cartesian ccB = new Cartesian();
    	int resNoA, resNoB;
    	String resNameA="", resNameB="", nucleusA="", nucleusB="",atomA="",atomB="";
    	double [] coordA = {0.0, 0.0, 0.0};
    	double [] coordB = {0.0, 0.0, 0.0};
    	double [] ha1 = new double[3];
    	double [] ca1 = new double[3];
    	double [] amide1 = new double[3];
    	double [] ha2 = new double[3];
    	double [] ca2 = new double[3];
    	double [] amide2 = new double[3];
    	
    	//get the ca coordinates the first pdb
    	pdbA = (Pdb)pdbAll.elementAt(0);
		atomVecA=pdbA.getAtomVec();
		resNoA=pdbA.getResidueNo();
		resNameA=pdbA.getResidue();
    	
		for (j=0; j<atomVecA.size();j++)
		{
			ccA=(Cartesian)atomVecA.elementAt(j);
			atomA=ccA.getAtom();
			
			if(atomA.equalsIgnoreCase("CA"))	
			{
				coordA=ccA.getXYZ();	
				ca1=ccA.getXYZ();
			}
			if(atomA.equalsIgnoreCase("N"))
				amide1=ccA.getXYZ();
			//for gly, we consider the bmrb format:
			if(atomA.equalsIgnoreCase("HA")||atomA.equalsIgnoreCase("HA2"))
				ha1=ccA.getXYZ();			
		}
		//get the ca coordinates the second pdb
    	pdbB = (Pdb)pdbBB.elementAt(0);
		atomVecB=pdbB.getAtomVec();
		resNoB=pdbB.getResidueNo();
		resNameB=pdbB.getResidue();
		
		for (j=0; j<atomVecB.size();j++)
		{
			ccB=(Cartesian)atomVecB.elementAt(j);
			atomB=ccB.getAtom();
			
			if(atomB.equalsIgnoreCase("CA"))	
			{
				coordB=ccB.getXYZ();	
				ca2=ccB.getXYZ();
			}
			if(atomB.equalsIgnoreCase("N"))
				amide2=ccB.getXYZ();
			if(atomB.equalsIgnoreCase("HA")|| atomB.equalsIgnoreCase("HA2"))
				ha2=ccB.getXYZ();
		}
		
		Pdb pp=new Pdb();				
		PhiPsi ff = new PhiPsi();
		
		//find the rotation matrix for the first pdb (all) to rotate to stardard 
		//coordinate system
		double [] CaToHaVecRot1 =  pp.internuclearVec(ca1,  ha1);
    	double [] CaToNVecRot1 =  pp.internuclearVec(ca1, amide1);//////
		Matrix rg1 = Matrix.identity(3,3);
		rg1 = ff.RgCal(CaToHaVecRot1, CaToNVecRot1);
		
		//find the rotation matrix for the second pdb (bb) to rotate to stardard 
		//coordinate system
		double [] CaToHaVecRot2 =  pp.internuclearVec(ca2,  ha2);
    	double [] CaToNVecRot2 =  pp.internuclearVec(ca2, amide2);//////
		Matrix rg2 = Matrix.identity(3,3);
		rg2 = ff.RgCal(CaToHaVecRot2, CaToNVecRot2);
		
		//find the version of the second rotation matrix		
		double [][] A2=rg2.getArray();
		
		Zmat zmat=new Zmat(A2);
		Zmat A2i = Inv.o(zmat);		
		Matrix rg2N=new Matrix(A2i.getRe());
		Matrix rotA=rg2N.times(rg1);
		
		Vector vecPdbAllNew=pp.newPdb(pdbAll,rotA);
				
		//translate the first pdb to the same orgin as the bb pdb
		//get the ca coordinates the first pdb
    	pdbA = (Pdb)vecPdbAllNew.elementAt(0);
		atomVecA=pdbA.getAtomVec();
		    	
		for (j=0; j<atomVecA.size();j++)
		{
			ccA=(Cartesian)atomVecA.elementAt(j);
			atomA=ccA.getAtom();
			
			if(atomA.equalsIgnoreCase("CA"))	
				coordA=ccA.getXYZ();		
		}
		//get the ca coordinates the second pdb
    	pdbB = (Pdb)pdbBB.elementAt(0);
		atomVecB=pdbB.getAtomVec();
		
		for (j=0; j<atomVecB.size();j++)
		{
			ccB=(Cartesian)atomVecB.elementAt(j);
			atomB=ccB.getAtom();
			
			if(atomB.equalsIgnoreCase("CA"))	
				coordB=ccB.getXYZ();			
		}
		
		double [] trans_vec = new double[3];
		trans_vec=pp.internuclearVec(coordA,coordB);
		
		//new pdbAll after translation
		Vector pdbAllNew2=pp.newPdbByTranslation(vecPdbAllNew,trans_vec );
		
		
		return pdbAllNew2;
    	
    
    }
    
    /**
     * rotate the whole protein according to the reference backbone at specific residue,
     * such that the backbone coordinates are the same (both should have the same residues).
     * 
     * @param pdbAll  pdb to be rotated
     * @param pdbBB  reference backbone pdb
     * @param resNo the res no
     * 
     * @return the pdb vector after rotated.
     * 
     * @throws JampackException the jampack exception
     */	
    public Vector BackonbeRotationAtResidue(final Vector pdbAll, final Vector pdbBB,int resNo) throws JampackException
    {
    	int i,j,k,h;
    	Pdb pdbA =new Pdb();
    	Pdb pdbB =new Pdb();
    	Vector atomVecA=new Vector();
    	Vector atomVecB=new Vector();
    	Cartesian ccA = new Cartesian();
    	Cartesian ccB = new Cartesian();
    	int resNoA, resNoB;
    	String resNameA="", resNameB="", nucleusA="", nucleusB="",atomA="",atomB="";
    	double [] coordA = {0.0, 0.0, 0.0};
    	double [] coordB = {0.0, 0.0, 0.0};
    	double [] ha1 = new double[3];
    	double [] ca1 = new double[3];
    	double [] amide1 = new double[3];
    	double [] ha2 = new double[3];
    	double [] ca2 = new double[3];
    	double [] amide2 = new double[3];
    	
    	//get the ca coordinates the first pdb
    	int index0 = Collections.binarySearch(pdbAll, new Pdb(resNo), new Pdb.PdbComparator());
		if (index0 <0)
		{
			System.out.println("error: the aligned residue not found...exist...");
			System.exit(0);		
		}		
    	pdbA = (Pdb)pdbAll.elementAt(index0);
		atomVecA=pdbA.getAtomVec();
		resNoA=pdbA.getResidueNo();
		resNameA=pdbA.getResidue();
    	
		for (j=0; j<atomVecA.size();j++)
		{
			ccA=(Cartesian)atomVecA.elementAt(j);
			atomA=ccA.getAtom();
			
			if(atomA.equalsIgnoreCase("CA"))	
			{
				coordA=ccA.getXYZ();	
				ca1=ccA.getXYZ();
			}
			if(atomA.equalsIgnoreCase("N"))
				amide1=ccA.getXYZ();
			//for gly, we consider the bmrb format:
			if(atomA.equalsIgnoreCase("HA")||atomA.equalsIgnoreCase("HA2"))
				ha1=ccA.getXYZ();			
		}
		
		//get the ca coordinates the second pdb
		int index = Collections.binarySearch(pdbBB, new Pdb(resNo), new Pdb.PdbComparator());
		if (index <0)
		{
			System.out.println("error: the aligned residue not found...exist...");
			System.exit(0);		
		}		
		
    	pdbB = (Pdb)pdbBB.elementAt(index);
		atomVecB=pdbB.getAtomVec();
		resNoB=pdbB.getResidueNo();
		resNameB=pdbB.getResidue();
		
		for (j=0; j<atomVecB.size();j++)
		{
			ccB=(Cartesian)atomVecB.elementAt(j);
			atomB=ccB.getAtom();
			
			if(atomB.equalsIgnoreCase("CA"))	
			{
				coordB=ccB.getXYZ();	
				ca2=ccB.getXYZ();
			}
			if(atomB.equalsIgnoreCase("N"))
				amide2=ccB.getXYZ();
			if(atomB.equalsIgnoreCase("HA")|| atomB.equalsIgnoreCase("HA2"))
				ha2=ccB.getXYZ();
		}
		
		Pdb pp=new Pdb();				
		PhiPsi ff = new PhiPsi();
		
		//find the rotation matrix for the first pdb (all) to rotate to stardard 
		//coordinate system
		double [] CaToHaVecRot1 =  pp.internuclearVec(ca1,  ha1);
    	double [] CaToNVecRot1 =  pp.internuclearVec(ca1, amide1);//////
		Matrix rg1 = Matrix.identity(3,3);
		rg1 = ff.RgCal(CaToHaVecRot1, CaToNVecRot1);
		
		//find the rotation matrix for the second pdb (bb) to rotate to stardard 
		//coordinate system
		double [] CaToHaVecRot2 =  pp.internuclearVec(ca2,  ha2);
    	double [] CaToNVecRot2 =  pp.internuclearVec(ca2, amide2);//////
		Matrix rg2 = Matrix.identity(3,3);
		rg2 = ff.RgCal(CaToHaVecRot2, CaToNVecRot2);
		
		//find the version of the second rotation matrix		
		double [][] A2=rg2.getArray();
		
		Zmat zmat=new Zmat(A2);
		Zmat A2i = Inv.o(zmat);		
		Matrix rg2N=new Matrix(A2i.getRe());
		Matrix rotA=rg2N.times(rg1);
		
		Vector vecPdbAllNew=pp.newPdb(pdbAll,rotA);
				
		//translate the first pdb to the same orgin as the bb pdb
		//get the ca coordinates the first pdb
		index0 = Collections.binarySearch(vecPdbAllNew, new Pdb(resNo), new Pdb.PdbComparator());
    	pdbA = (Pdb)vecPdbAllNew.elementAt(index0);
		atomVecA=pdbA.getAtomVec();
		    	
		for (j=0; j<atomVecA.size();j++)
		{
			ccA=(Cartesian)atomVecA.elementAt(j);
			atomA=ccA.getAtom();
			
			if(atomA.equalsIgnoreCase("CA"))	
				coordA=ccA.getXYZ();		
		}
		//get the ca coordinates the second pdb
		index = Collections.binarySearch(pdbBB, new Pdb(resNo), new Pdb.PdbComparator());    	
    	pdbB = (Pdb)pdbBB.elementAt(index);
		atomVecB=pdbB.getAtomVec();
		
		for (j=0; j<atomVecB.size();j++)
		{
			ccB=(Cartesian)atomVecB.elementAt(j);
			atomB=ccB.getAtom();
			
			if(atomB.equalsIgnoreCase("CA"))	
				coordB=ccB.getXYZ();			
		}
		
		double [] trans_vec = new double[3];
		trans_vec=pp.internuclearVec(coordA,coordB);
		
		//new pdbAll after translation
		Vector pdbAllNew2=pp.newPdbByTranslation(vecPdbAllNew,trans_vec );
		
		
		return pdbAllNew2;
    	
    
    }
    
    /**
     * compute the beta sheet using RDC-EXACT.
     * 
     * @param src location of the input file
     * @param strOut location of the output file
     * @param strInput  input file name
     * 
     * @return void
     * 
     * @throws JampackException the jampack exception
     */       
    public void doRdcExactSheet(String src, String strOut, String strInput)throws JampackException
    {	 
    	int maxStructurePerEnsemble=2000;
    	boolean isDebug=Const.isDebug;
    	Assign asg = new Assign();
    	int i, j;    	
    	Pdb  pp = new Pdb();   
    	ModelRdc mdc=new ModelRdc();
    	Model mod = new Model();
    	long startTime = System.currentTimeMillis();
    	
    	/////////////////////////////////////////////
    	/// 1. Read the input files
    	//
    	/////////////////////////////////////////////    	
    	
    	Vector<Map<String, String>> paraVec = asg.ParamReader(src+strInput);
    	double Syy=0.0,Szz=0.0, wtCoCA=0.0, wtCoN=0.0;
    	String strSseBounds="",strOutPdb="",strChFile="",strNhFile="",
    		strCaCoFile="",strCoNFile="",strTalos="",strHBonds="",strSeq="";   	
    	
    	for (i=0;i<paraVec.size();i++)
    	{	    		
    		Map paraMap = paraVec.elementAt(i);    	
    		if(paraMap.containsKey("CHFILE"))
    			strChFile  =  (String)paraMap.get("CHFILE"); 
    		if(paraMap.containsKey("TALOS"))
    			strTalos  =  (String)paraMap.get("TALOS"); 
    		if(paraMap.containsKey("NHFILE"))
    			strNhFile  =  (String)paraMap.get("NHFILE"); 
    		if(paraMap.containsKey("COCAFILE"))
    			strCaCoFile  =  (String)paraMap.get("COCAFILE"); 
    		if(paraMap.containsKey("CONFILE"))
    			strCoNFile  =  (String)paraMap.get("CONFILE");     		
    		if(paraMap.containsKey("SEQUENCE"))
    			strSeq  =  (String)paraMap.get("SEQUENCE");
    		if(paraMap.containsKey("SYY"))
    			Syy  =  Double.parseDouble((String)paraMap.get("SYY"));
    		if(paraMap.containsKey("SZZ"))
    			Szz  =  Double.parseDouble((String)paraMap.get("SZZ"));   
    		if(paraMap.containsKey("WTCOCA"))
    			wtCoCA=  Double.parseDouble((String)paraMap.get("WTCOCA"));   
    		if(paraMap.containsKey("WTCON"))
    			wtCoN=  Double.parseDouble((String)paraMap.get("WTCON")); 
    		
    		if(paraMap.containsKey("SSEBOUNDARIES"))
    			strSseBounds  =  (String)paraMap.get("SSEBOUNDARIES");      		
    		  		
    		if(paraMap.containsKey("OUTPDBNAME"))
    			strOutPdb  =  (String)paraMap.get("OUTPDBNAME");  
    		
    		if(paraMap.containsKey("HBONDS"))
    			strHBonds  =  (String)paraMap.get("HBONDS");    		
    		
    	} //for (i=0;i<paraVec.size();i++)
    	
    	//read the sequence
    	String seqFile = src + strSeq;    	
    	Vector vecSeq=asg.ReaderSeq(seqFile);    	
    	
    	
    	//read the talos angle table:
    	String strTalosFile=src+strTalos;
    	PhiPsi phipsi0 =new PhiPsi();
    	Vector vecTalos=new Vector();
    	boolean existTalos = (new File(strTalosFile)).exists();
    	if(existTalos)
    		vecTalos=phipsi0.talosAngleReader(strTalosFile);
    	else
    		System.out.println("Warning: The TALOS input is not available...");
    	//read the RDC data:
    	String strChRdcFile=src+strChFile;
    	String strNhRdcFile=src+strNhFile;
    	String strCaCoRdcFile=src+strCaCoFile;
    	String strCoNRdcFile=src+strCoNFile;
    	Dipolar dd  =  new Dipolar();
		Vector<Dipolar> nhRdc = dd.rdcRead(strNhRdcFile,1.0);
		///Vector<Dipolar> cahaRdc = dd.rdcRead(strChRdcFile,2.008);
		Vector<Dipolar> cahaRdc = dd.rdcRead(strChRdcFile,1.0);		
		
		boolean existCaCo = (new File(strCaCoRdcFile)).exists();
		boolean existCoN = (new File(strCoNRdcFile)).exists();
		
		Vector<Dipolar> cacoRdc = new Vector();
		Vector<Dipolar> conRdc = new Vector();
		if(existCaCo)
			cacoRdc=dd.rdcRead(strCaCoRdcFile,0.19845);
		else
			System.out.println("Warning: The Co-Ca RDC doesn't exist...");
		if(existCoN) 			
			conRdc=dd.rdcRead(strCoNRdcFile,0.12086);	    	
		else
			System.out.println("Warning: The Co-N RDC doesn't exist...");
		
		//read the previous structure.
		Vector vecPrePdb=new Vector();
		//Currently we only deal with short beta strands, thus the following lines are not used now.
	    /*String strPdb=src+strPrePdb;
    	Vector vecPrePdb=new Vector();
    	boolean existPreFile = (new File(strPdb)).exists();
    	if(existPreFile)
    	{
    		Vector ppVec=pp.readPdb(strPdb);
    		vecPrePdb=pp.nameConvert(ppVec);
    	}   	
    	else
    		System.out.println("Warning: no previous structure was found...");
    	*/
    	
		//read H-bond file
    	Hbond hb = new Hbond();
    	String HbondFile = src+strHBonds;
    	Vector<Hbond> HbondVec = hb.HbondReader(HbondFile);
    	
    	//Read the 2ary identification file
    	IDof2aryStructure id2ary = new IDof2aryStructure();
    	String id2aryFile = src+strSseBounds;
    	Vector<IDof2aryStructure> idVec = id2ary.IDof2aryStructureReader(id2aryFile);
    	Vector[] orderVec = id2ary.getOrder(id2aryFile);
    	
    	//here we assume that the scaling factor is 1.0
    	Vector<Vector<Dipolar>> eRdcNhVec = dd.rdcReader(strNhRdcFile, idVec, "E"); //read RDCs for strands
    	Vector<Vector<Dipolar>> eRdcChVec = dd.rdcReader(strChRdcFile, idVec, "E");
    	
    	//For betaSheet
    	String userDir = System.getProperty("user.dir");////
    	String srcSheetPara=userDir+"/system/rdc-exact-sheet-parameter.txt";	
    	Vector<Map<String, String>> paraVecSheet = mod.parameterReader(srcSheetPara);
     	boolean printEulerSearch = true;
    	Vector<Pdb> sheetVec = mod.sheetModel(eRdcNhVec, eRdcChVec, Syy, Szz, HbondVec, paraVecSheet, 
    					 orderVec, printEulerSearch);
    	    		 	
    	//write the new pdb file:
    	String fileName =strOut+strOutPdb;
    	try
    	{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));   		    		   			
	    	pp.printToFile(sheetVec,fileName, out);		    	
	    	out.println("END");  
	    	out.close();
	    	
		}catch (FileNotFoundException e)
		{
			System.out.println("File not found: " + fileName);
		}catch (IOException e)
		{
			System.out.println("IOException: the stack trace is:");
			e.printStackTrace();
		}            


		long endTime = System.currentTimeMillis() ;
	 	double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
	 	System.out.println("Time= "+ totalTime +" minutes");
    }
  
 /**
  * Compute s beta-sheet with the order specified by orderVec.
  * 
  * @param eRdc1Vec  RDC of medium 1 for all the strands of the sheet
  * @param eRdc2Vec  RDC of medium 2 for all the strands of the sheet
  * @param HbondVec   the Hbonds between the strands
  * @param paraVec    the set of parameters used to compute each strands including the 1st one
  * @param printEulerSearch if true it will print out the the detail of rotation search
  * @param Syy the syy
  * @param Szz the szz
  * @param orderVec the order vec
  * 
  * @return the computed sheet Pdb vector
  */	
   public Vector sheetModel(Vector eRdc1Vec, Vector eRdc2Vec, double Syy, double Szz, 
			     Vector HbondVec, Vector paraVec, Vector[] orderVec, boolean printEulerSearch)
    {
		int i=0, j=0;
		Pdb pp = new Pdb();
		Vector strand1Rdc1 = new Vector(); 
		Vector strand1Rdc2 = new Vector(); 
		String id2ary0 = (String)orderVec[i].elementAt(0);
		int index0 = id2ary0.indexOf("E");
		if (index0 > -1 )
		{
			String strTemp=id2ary0.substring(index0+1,index0+2);
			int nn0 = Integer.parseInt(strTemp) - 1;			
			strand1Rdc1 = (Vector)eRdc1Vec.elementAt(nn0);
			strand1Rdc2 = (Vector)eRdc2Vec.elementAt(nn0);
		}		
		
		Vector sheetPdb = new Vector();
		double[] ramaFilter = new double[]{Const.phiLowBeta,Const.phiHighBeta,Const.psiLowBeta,Const.psiHighBeta};
		double phiAve = Const.phiAveBeta;
		double psiAve = Const.psiAveBeta;

		Map map1stStrand  = (Map)paraVec.elementAt(2); //The first parameters are for computing Saupe elements
		double resolution = new Double((String)map1stStrand.get("RESOLUTION")).doubleValue();
		int refineCycle = Integer.parseInt((String)map1stStrand.get("REFINECYCLE"));
		int dfsCycle    = Integer.parseInt((String)map1stStrand.get("DFSCYCLE"));
		boolean debugDFS = new Boolean((String)map1stStrand.get("DEBUGDFS")).booleanValue();
		boolean printResults = new Boolean((String)map1stStrand.get("PRINTRESULTS")).booleanValue();
		double w4Angles  = new Double((String)map1stStrand.get("WEIGHT4ANGLES")).doubleValue();
	  	PdbRmsd psd  = new PdbRmsd();
		ModelRdc mdc = new ModelRdc();
		Hbond hb = new Hbond();

		//For Strand1, the first strand refined with no H-bond terms in score funtion
		Vector pdbStrand1 = mdc.refineStrand(strand1Rdc1, strand1Rdc2, Syy, Szz, ramaFilter,
						     phiAve, psiAve, refineCycle, dfsCycle, w4Angles, resolution,
						     debugDFS, printResults);
		sheetPdb.addAll(pdbStrand1);
		//For Strands whose refinement needs H-bonds
		double w4HB = 4.0;
		Vector hBondVecOfEab = new Vector();
		int nn = 0;
		int index = -1;
		Vector strandRdc1  = new Vector();
		Vector strandRdc2  = new Vector();
		Vector pdbStrand1N = new Vector();
		Vector pdbStrand2 = new Vector();
	 	String id2ary = "";

		double resolution2 = 0.0;
		int refineCycle2 = 1;
		int dfsCycle2    = 1;
		boolean debugDFS2  =  false;
		boolean printResults2 = false;
		double w4Angles2   = 1.0;
		double w4HB2       = 1.0;

		if (paraVec.size() > 3)
		{
		    Map map2ndStrand = (Map)paraVec.elementAt(3); //The first parameters are for computing Saupe elements
		    resolution2 = new Double((String) map2ndStrand.get("RESOLUTION")).doubleValue();
		    refineCycle2 = Integer.parseInt((String) map2ndStrand.get("REFINECYCLE"));
		    dfsCycle2    = Integer.parseInt((String) map2ndStrand.get("DFSCYCLE"));
		    debugDFS2    = new Boolean((String) map2ndStrand.get("DEBUGDFS")).booleanValue();
		    printResults2 = new Boolean((String)map2ndStrand.get("PRINTRESULTS")).booleanValue();
		    w4Angles2   = new Double((String) map2ndStrand.get("WEIGHT4ANGLES")).doubleValue();
		    w4HB2       = new Double((String) map2ndStrand.get("WEIGHT4HBONDS")).doubleValue();
		}
		if (paraVec.size() > 3)
		{
		    Map mapOtherStrand = (Map)paraVec.elementAt(4); //The first parameters are for computing Saupe elements
		    resolution  = new Double((String)mapOtherStrand.get("RESOLUTION")).doubleValue();
		    refineCycle = Integer.parseInt((String)mapOtherStrand.get("REFINECYCLE"));
		    dfsCycle    = Integer.parseInt((String)mapOtherStrand.get("DFSCYCLE"));
		    debugDFS    = new Boolean((String)mapOtherStrand.get("DEBUGDFS")).booleanValue();
		    printResults = new Boolean((String)mapOtherStrand.get("PRINTRESULTS")).booleanValue();
		    w4Angles   = new Double((String)mapOtherStrand.get("WEIGHT4ANGLES")).doubleValue();
		    w4HB       = new Double((String)mapOtherStrand.get("WEIGHT4HBONDS")).doubleValue();
		    for (i=0; i<orderVec.length; i++)
		    {
		    	pdbStrand1N = pdbStrand1;
		    	for(j=0; j<orderVec[i].size(); j++)
		    	{
		    		id2ary = (String)orderVec[i].elementAt(j);
		    		index = id2ary.lastIndexOf("E");
		    		if (index > -1 )
		    		{
		    			nn = Integer.parseInt(id2ary.substring(index+1)) - 1;
		    			hBondVecOfEab = hb.hbOfTwoStrands(HbondVec, id2ary);
						strandRdc1 = (Vector)eRdc1Vec.elementAt(nn);
						strandRdc2 = (Vector)eRdc2Vec.elementAt(nn);
						System.out.println(strandRdc1.size());
						System.out.println(strandRdc2.size());
						if (i == 0)
						{
						    //specifically for the second strand
						    pdbStrand2 = mdc.refineStrand(strandRdc1, strandRdc2, Syy, Szz, phiAve, psiAve, refineCycle2,
										  dfsCycle2, resolution2, w4Angles2, w4HB2, pdbStrand1N,
										  hBondVecOfEab, printEulerSearch, debugDFS2, printResults2);
						    sheetPdb.addAll(pdbStrand2);
						    pdbStrand1N = pdbStrand2;
						}
		    		}
		    	}
		    }
		}
		Collections.sort(sheetPdb, new Pdb.PdbComparator());
		return sheetPdb;
    }

 /**
  * Compute s beta-sheet with the order specified by orderVec.
  * 
  * @param eRdc1Vec  RDC of medium 1 for all the strands of the sheet
  * @param eRdc2Vec  RDC of medium 2 for all the strands of the sheet
  * @param HbondVec   the Hbonds between the strands
  * @param paraVec    the set of parameters used to compute each strands including the 1st one
  * @param printEulerSearch if true it will print out the the detail of rotation search
  * @param pdbVec the pdb vec
  * @param Syy the syy
  * @param Szz the szz
  * @param orderVec the order vec
  * 
  * @return the computed sheet Pdb vector
  */	
     public Vector sheetModel2(Vector pdbVec, Vector eRdc1Vec, Vector eRdc2Vec, double Syy, double Szz, 
			       Vector HbondVec, Vector paraVec, Vector[] orderVec, boolean printEulerSearch){
	int i=0, j=0;
	Pdb pp = new Pdb();
	Vector strand1Rdc1 = (Vector)eRdc1Vec.elementAt(4); //65-70 was choosen first
	Vector strand1Rdc2 = (Vector)eRdc2Vec.elementAt(4);
	Dipolar dd1 = (Dipolar)strand1Rdc1.elementAt(0);
	Dipolar dd2 = (Dipolar)strand1Rdc2.elementAt(0);
	int firstResidueNo = Math.min(dd1.getResidueNo(), dd2.getResidueNo()) - 1;

	dd1 = (Dipolar)strand1Rdc1.elementAt(strand1Rdc1.size() - 1);
	dd2 = (Dipolar)strand1Rdc2.elementAt(strand1Rdc2.size() - 1);
	int lastResidueNo  = Math.max(dd1.getResidueNo(), dd2.getResidueNo()) + 1;


	Vector sheetPdb = new Vector();
	double[] ramaFilter = new double[]{Const.phiLowBeta,Const.phiHighBeta,Const.psiLowBeta,Const.psiHighBeta};
	double phiAve = Const.phiAveBeta;
	double psiAve = Const.psiAveBeta;

	Map map1stStrand  = (Map)paraVec.elementAt(2); //The first parameters are for computing Saupe elements
	double resolution = new Double((String)map1stStrand.get("RESOLUTION")).doubleValue();
	int refineCycle = Integer.parseInt((String)map1stStrand.get("REFINECYCLE"));
	int dfsCycle    = Integer.parseInt((String)map1stStrand.get("DFSCYCLE"));
	boolean debugDFS = new Boolean((String)map1stStrand.get("DEBUGDFS")).booleanValue();
	boolean printResults = new Boolean((String)map1stStrand.get("PRINTRESULTS")).booleanValue();
	double w4Angles  = new Double((String)map1stStrand.get("WEIGHT4ANGLES")).doubleValue();
  	PdbRmsd psd  = new PdbRmsd();
	ModelRdc mdc = new ModelRdc();
	Hbond hb = new Hbond();

	//Extract only the common strand (E65-70)
   	Vector pdbStrand1 = new Vector();
	int no = 0;
	for (i=0; i<pdbVec.size(); i++){
	    pp = (Pdb)pdbVec.elementAt(i);
	    no = pp.getResidueNo();
	    if ( no > firstResidueNo && no < lastResidueNo+1){
		pdbStrand1.add(pp);
	    }
	}
  	sheetPdb.addAll(pdbStrand1);
// 		For Strands whose refinement needs H-bonds
	double w4HB = 4.0;
	Vector hBondVecOfEab = new Vector();
	int nn = 0;
	int index = -1;
	Vector strandRdc1  = new Vector();
	Vector strandRdc2  = new Vector();
	Vector pdbStrand1N = new Vector();
	Vector pdbStrand2 = new Vector();
 	String id2ary = "";

	double resolution2 = 0.0;
	int refineCycle2 = 1;
	int dfsCycle2    = 1;
	boolean debugDFS2  =  false;
	boolean printResults2 = false;
	double w4Angles2   = 1.0;
	double w4HB2       = 1.0;

	if (paraVec.size() > 3){
	    Map map2ndStrand = (Map)paraVec.elementAt(3); //The first parameters are for computing Saupe elements
	    resolution2 = new Double((String) map2ndStrand.get("RESOLUTION")).doubleValue();
	    refineCycle2 = Integer.parseInt((String) map2ndStrand.get("REFINECYCLE"));
	    dfsCycle2    = Integer.parseInt((String) map2ndStrand.get("DFSCYCLE"));
	    debugDFS2    = new Boolean((String) map2ndStrand.get("DEBUGDFS")).booleanValue();
	    printResults2 = new Boolean((String)map2ndStrand.get("PRINTRESULTS")).booleanValue();
	    w4Angles2   = new Double((String) map2ndStrand.get("WEIGHT4ANGLES")).doubleValue();
	    w4HB2       = new Double((String) map2ndStrand.get("WEIGHT4HBONDS")).doubleValue();
	}
	if (paraVec.size() > 3){
	    Map mapOtherStrand = (Map)paraVec.elementAt(4); //The first parameters are for computing Saupe elements
	    resolution  = new Double((String)mapOtherStrand.get("RESOLUTION")).doubleValue();
	    refineCycle = Integer.parseInt((String)mapOtherStrand.get("REFINECYCLE"));
	    dfsCycle    = Integer.parseInt((String)mapOtherStrand.get("DFSCYCLE"));
	    debugDFS    = new Boolean((String)mapOtherStrand.get("DEBUGDFS")).booleanValue();
	    printResults = new Boolean((String)mapOtherStrand.get("PRINTRESULTS")).booleanValue();
	    w4Angles   = new Double((String)mapOtherStrand.get("WEIGHT4ANGLES")).doubleValue();
	    w4HB       = new Double((String)mapOtherStrand.get("WEIGHT4HBONDS")).doubleValue();
	    for (i=0; i<orderVec.length; i++){
 		pdbStrand1N = pdbStrand1;
		for(j=0; j<orderVec[i].size(); j++){
		    id2ary = (String)orderVec[i].elementAt(j);
		    index = id2ary.lastIndexOf("E");
		    if (index > -1 ){
			nn = Integer.parseInt(id2ary.substring(index+1)) - 1;
			hBondVecOfEab = hb.hbOfTwoStrands(HbondVec, id2ary);
			strandRdc1 = (Vector)eRdc1Vec.elementAt(nn);
			strandRdc2 = (Vector)eRdc2Vec.elementAt(nn);
			System.out.println(strandRdc1.size());
			System.out.println(strandRdc2.size());
			if (i == 1){
			    pdbStrand2 = mdc.refineStrand(strandRdc1, strandRdc2, Syy, Szz, phiAve, psiAve, refineCycle,
							  dfsCycle, resolution, w4Angles, w4HB, pdbStrand1N,
							  hBondVecOfEab, printEulerSearch, debugDFS, printResults);
			    sheetPdb.addAll(pdbStrand2);
			    pdbStrand1N = pdbStrand2;
			}
		    }
		}
	    }
	}
	Collections.sort(sheetPdb, new Pdb.PdbComparator());
	strandRdc1 = (Vector)eRdc1Vec.elementAt(2); //Specifically for E41-45
	strandRdc2 = (Vector)eRdc2Vec.elementAt(2);
	pdbStrand2 = mdc.refineStrand2(strandRdc1, strandRdc2, Syy, Szz, phiAve, psiAve, refineCycle,
				       dfsCycle, resolution, w4Angles, w4HB, sheetPdb,
				       HbondVec, printEulerSearch, debugDFS, printResults);

	return pdbStrand2;
    }
     
     //the main function used by LW.
     /**
      * The main method for testing.
      * 
      * @param argv the arguments
      * 
      * @throws JampackException the jampack exception
      */
     public static void main(String[] argv)throws JampackException
     {
    		long startTime = System.currentTimeMillis();
    		int i, j;
    		
    		String userDir = System.getProperty("user.dir");////
    		String src=userDir+"/inputFiles/";    		
    		
    		ModelRdc mdc = new ModelRdc();
    		Model mod = new Model();
    				
    		//Read Parameter file
    	//	String paraFile = src+"parameter.txt";
    		//Vector<Map<String, String>> paraVec = mod.parameterReader(paraFile);
    		
    		IDof2aryStructure id2ary = new IDof2aryStructure();
    		//String id2aryFile = src+"secondaryElements.txt";
    		//Vector<IDof2aryStructure> idVec = id2ary.IDof2aryStructureReader(id2aryFile);
    		//Vector[] orderVec = id2ary.getOrder(id2aryFile);

    	/*	for (i=0; i<idVec.size(); i++){
    		    id2ary = idVec.elementAt(i);
    		    System.out.println(id2ary);
    		}
    		*/
    		/* Temporary PDB files for pairing strands using H-bonds */
    		Pdb pp = new Pdb();

    		//RDC files for one helix used to compute Saupe elements
    		String rdcNhFile = src+"nhRdc-temp.txt";
    	    String rdcChFile = src+"chRdc-temp.txt";
    	    String rdcCaCoFile=src+"cacoRdc-temp.txt";
    	    String rdcConFile=src+"conRdc-temp.txt";
    	  	Dipolar dd  =  new Dipolar();
    		final Vector<Dipolar> nhRdc = dd.rdcRead(rdcNhFile,1.0);
    		final Vector<Dipolar> cahaRdc = dd.rdcRead(rdcChFile,2.008);
    		final Vector<Dipolar> cacoRdc = dd.rdcRead(rdcCaCoFile,0.19845);
    		final Vector<Dipolar> conRdc = dd.rdcRead(rdcConFile,0.12086);
    		
    		
    		//read the initial structures
    		Vector vecBB=new Vector();
    		String strH1File=src+"top_ff2_3.pdb";
    		
    		
    		String strH3File=src+"top_ff2_3.pdb";   		
    		
    		
    		String strH4File=src+"top_ff2_3.pdb";
    		String strH5File=src+"top_ff2_3.pdb";
    		String strH6File=src+"top_ff2_3.pdb";
    		
    		
    		Vector<Pdb> ppVecH1 = pp.readPdb(strH1File);
    	
    		Vector<Pdb> ppVecH3 = pp.readPdb(strH3File);
    	
    	//	Vector<Vector<Dipolar>> hRdc1Vec = dd.rdcReader(rdcNhFile, idVec, "H",1.0); //read RDCs for helices
    		//Vector<Vector<Dipolar>> hRdc2Vec = dd.rdcReader(rdcChFile, idVec, "H",2.008);
    		//Vector<Vector<Dipolar>> hRdcCaCoVec = dd.rdcReader(rdcCaCoFile, idVec, "H",0.19845);
    		//Vector<Vector<Dipolar>> hRdcCoNVec = dd.rdcReader(rdcConFile, idVec, "H",0.12086);
    		
    	//	Vector helixRdc1Vec = hRdc1Vec.elementAt(0); //2st helix
    	//	Vector helixRdc2Vec = hRdc2Vec.elementAt(0);
    		
    	//	Vector helixRdcCaCoVec = hRdcCaCoVec.elementAt(0);
    	//	Vector helixRdcCoNVec = hRdcCoNVec.elementAt(0);

    		double[] saupe = new double[5]; //for returning Saupe elements
    	//	Map saupeMap = paraVec.elementAt(0); //The first parameters are for computing Saupe elements
    	//	int refineCycle = Integer.parseInt((String)saupeMap.get("REFINECYCLE"));
    	//	int dfsCycle    = Integer.parseInt((String)saupeMap.get("DFSCYCLE"));
    	//	boolean debugDFS = new Boolean((String)saupeMap.get("DEBUGDFS")).booleanValue();
    	  //      boolean printResults = new Boolean((String)saupeMap.get("PRINTRESULTS")).booleanValue();
    		PdbRdc pdr = new PdbRdc();

    		//For Helix: this helix is used to compute Saupe elements for two media (7 parameters)
    		double [] ramaFilter = {Const.phiLowHelix,Const.phiHighHelix,Const.psiLowHelix,Const.psiHighHelix};
    		double phiAve = Const.phiAveHelix;
    		double psiAve = Const.psiAveHelix;

    		///////////////////changed here...
    		double Syy=    0.8193489560852427 ;//4.3973348614502905 ;//4.369119289462044;  
    		double Szz=  35.37745291254462;//8.076268764194998;//7.784097179042096;  		
    		
    		//Map helixMap = paraVec.elementAt(1); //The first parameters are for computing Saupe elements
    		
    	  //  double resolution  = new Double((String)helixMap.get("RESOLUTION")).doubleValue();
    	//	refineCycle = Integer.parseInt((String)helixMap.get("REFINECYCLE"));
    	//	dfsCycle    = Integer.parseInt((String)helixMap.get("DFSCYCLE"));
    	//	debugDFS = new Boolean((String)helixMap.get("DEBUGDFS")).booleanValue();
    	//    printResults = new Boolean((String)helixMap.get("PRINTRESULTS")).booleanValue();
    	//    double w4Angles = new Double((String) helixMap.get("WEIGHT4ANGLES")).doubleValue();
    	    
    	    //ramaFilter = {Const.phiLowHelix,Const.phiHighHelix,Const.psiLowHelix,Const.psiHighHelix};
    	    phiAve = Const.phiAveHelix;
    		psiAve = Const.psiAveHelix;
    		
    		Pdb temp2 = new Pdb();
    	 	////////////////////////////////////////////////////
    		
    		///////////
    		//parameters
    		//resolution=2.0;
    		//refineCycle=4;
    		//dfsCycle=931072; 
    		//debugDFS=true;
    		//printResults=true;
    		//w4Angles=0.1;
    	    ///////////
    		String talosStr=src+"talos.txt";
    		PhiPsi phipsi=new PhiPsi();
    		/////Vector vecTalos=phipsi.talosAngleReader(talosStr);
    		Vector vecTalos=new Vector();
    		
    		double [] n1  = {0.0, 0.0, 0.0};         //A coordinate frame in the 1st peptide plane
    		double [] nh1 = {0.0, 0.0, -Const.dN2H}; 
    	   	double [] ca1 = {0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA * Math.sin(Const.theta1)};
    	   	
    		Vector phiPsiVec=new Vector();
    		for (i=27; i<=33; i++)  
    		    phiPsiVec.add(new PhiPsi(i, "ALA", phiAve, psiAve));
    	    Vector pdbVec = mdc.modelBuild(phiPsiVec, n1, nh1, ca1);
    	    
    	    ////////////////////////////////////////////////
    	    //debug: compute the rdc rmsds for a structure given 
    	    // all four rdcs 
    	   String strTempStr=src+"top_ff2_1.pdb";//"1UBQH.pdb";
    	    Vector<Pdb> ppVecTempStr = pp.readPdb(strTempStr);
    	    
    	    i=3;
    	    Vector helixVecN = pp.newPdb(ppVecTempStr, Const.mat4ThreeDirs[i - 1]);
    	    //ppVecTempStr=new Vector();
    	   // ppVecTempStr.addAll(helixVecN);
    	    
    	    double[] rdc1Cal  = new double[nhRdc.size()+100];
    		double[] rdc2Cal  = new double[cahaRdc.size()+100];
    		Matrix mm = new Matrix(3, 3);
    		Vector pdbVecTempStr2 = new Vector();
    	    mm = pdr.bestFit(ppVecTempStr, nhRdc, cahaRdc, rdc1Cal, rdc2Cal, saupe);
    	    pdbVecTempStr2 = pp.newPdb(ppVecTempStr, mm);
    	    double[] sRmsd= new double[1];
			int[] sizeRdc=new int[1];
			double[] rdcCoNCal  = new double[nhRdc.size()+100 ];
			double[] rdcCaCoCal  = new double[cahaRdc.size()+100 ];
    	    Syy=saupe[0];
    	    Szz=saupe[1];
    	    
    	    double r_nh=pdr.BackCal(ppVecTempStr, nhRdc, "N", "H", mm, -Syy-Szz, Syy, Szz, Const.nhRatio, rdcCaCoCal,sRmsd, sizeRdc);
	    	
    	    double r_caha=pdr.BackCal(ppVecTempStr, cahaRdc, "CA", "HA", mm, -Syy-Szz, Syy, Szz, Const.cahaRatio, rdcCaCoCal,sRmsd, sizeRdc);
	    	
    	    double r_caco=pdr.BackCalCACO(ppVecTempStr, cacoRdc, "C", "CA", mm, -Syy-Szz, Syy, Szz, Const.cacoRatio, rdcCaCoCal,sRmsd, sizeRdc,true);
		    double r_con=pdr.BackCalCON(ppVecTempStr, conRdc, "C", "N",  mm, -Syy-Szz, Syy, Szz, Const.conRatio, rdcCoNCal,sRmsd, sizeRdc,true);
			
    	    ///////////////////////////////////////////////////
 	    
    	  
    	 	///////////////////////////////////////////////////////

    	 	//pdbVec=new Vector ();
    	 	//pdbVec.addAll(helixPdbVec);
    /*	    
    		System.out.println("===============helix1=====================");
    		refineCycle=3;
    	    Vector<Pdb> helixPdbVec22 = mdc.refineHelixW4RDCs(pdbVec,helixRdc1Vec, helixRdc2Vec,helixRdcCaCoVec, helixRdcCoNVec, Syy,  Szz, ramaFilter, 
    			     phiAve, psiAve, refineCycle, dfsCycle, w4Angles, resolution,debugDFS, printResults,vecTalos);
    	   // Vector<Pdb> helixPdbVec22 = mdc.refineHelix22(pdbVec,helixRdc1Vec, helixRdc2Vec, Syy,  Szz, ramaFilter, 
   			   //  phiAve, psiAve, refineCycle, dfsCycle, w4Angles, resolution,debugDFS, printResults,vecTalos);
   	    
    	    
    	    temp2 = new Pdb(); ///added by Zeng
    	    temp2.print(helixPdbVec22);///added by Zeng for debugging
    	   
    	    refineCycle=2;
    	    System.out.println("===============helix2=====================");
    	    
    	    Vector<Pdb> helixPdbVec2 = mdc.refineHelixW4RDCs(helixPdbVec22,helixRdc1Vec, helixRdc2Vec, helixRdcCaCoVec, helixRdcCoNVec, Syy,  Szz, ramaFilter, 
    			     phiAve, psiAve, refineCycle, dfsCycle, w4Angles, resolution,debugDFS, printResults,vecTalos);
    	    //Vector<Pdb> helixPdbVec2 = mdc.refineHelix22(helixPdbVec22,helixRdc1Vec, helixRdc2Vec, Syy,  Szz, ramaFilter, 
   			     //phiAve, psiAve, refineCycle, dfsCycle, w4Angles, resolution,debugDFS, printResults,vecTalos);
   	   
    	    
    	    //Pdb temp2 = new Pdb(); ///added by Zeng
    	    temp2.print(helixPdbVec2);///added by Zeng for debugging
    	   
    	    System.out.println("===============helix3=====================");
    	    Vector<Pdb> helixPdbVec3 = mdc.refineHelixW4RDCs(helixPdbVec2,helixRdc1Vec, helixRdc2Vec, helixRdcCaCoVec, helixRdcCoNVec, Syy,  Szz, ramaFilter, 
    			     phiAve, psiAve, refineCycle, dfsCycle, w4Angles, resolution,debugDFS, printResults,vecTalos);
    	    //Vector<Pdb> helixPdbVec3 = mdc.refineHelix22(helixPdbVec2,helixRdc1Vec, helixRdc2Vec, Syy,  Szz, ramaFilter, 
   			     //phiAve, psiAve, refineCycle, dfsCycle, w4Angles, resolution,debugDFS, printResults,vecTalos);
   	    
    	    
    	    temp2.print(helixPdbVec3);///added by Zeng for debugging
    		
    	    refineCycle=1;
    	    System.out.println("===============helix4=====================");
    	    Vector<Pdb> helixPdbVec4 = mdc.refineHelixW4RDCs(helixPdbVec3,helixRdc1Vec, helixRdc2Vec,helixRdcCaCoVec, helixRdcCoNVec, Syy,  Szz, ramaFilter, 
    			    phiAve, psiAve, refineCycle, dfsCycle, w4Angles, resolution,debugDFS, printResults,vecTalos);
    	    //Vector<Pdb> helixPdbVec4 = mdc.refineHelix22(helixPdbVec3,helixRdc1Vec, helixRdc2Vec, Syy,  Szz, ramaFilter, 
   			   //  phiAve, psiAve, refineCycle, dfsCycle, w4Angles, resolution,debugDFS, printResults,vecTalos);
   	   
    	    
    	    temp2.print(helixPdbVec4);///added by Zeng for debugging
    		
    	    System.out.println("===============helix5=====================");
    	    Vector<Pdb> helixPdbVec5 = mdc.refineHelixW4RDCs(helixPdbVec4,helixRdc1Vec, helixRdc2Vec, helixRdcCaCoVec, helixRdcCoNVec, Syy,  Szz, ramaFilter, 
    			     phiAve, psiAve, refineCycle, dfsCycle, w4Angles, resolution,debugDFS, printResults,vecTalos);
    	    //Vector<Pdb> helixPdbVec5 = mdc.refineHelix22(helixPdbVec4,helixRdc1Vec, helixRdc2Vec, Syy,  Szz, ramaFilter, 
   			     //phiAve, psiAve, refineCycle, dfsCycle, w4Angles, resolution,debugDFS, printResults,vecTalos);
   	   
    	    
    	    
    	    
    	    temp2.print(helixPdbVec5);///added by Zeng for debugging
    		
    	    System.out.println("===============helix6=====================");
    	    Vector<Pdb> helixPdbVec6 = mdc.refineHelixW4RDCs(helixPdbVec5,helixRdc1Vec, helixRdc2Vec,helixRdcCaCoVec, helixRdcCoNVec,  Syy,  Szz, ramaFilter, 
    			     phiAve, psiAve, refineCycle, dfsCycle, w4Angles, resolution,debugDFS, printResults,vecTalos);
    	    //Vector<Pdb> helixPdbVec6 = mdc.refineHelix22(helixPdbVec5,helixRdc1Vec, helixRdc2Vec,Syy,  Szz, ramaFilter, 
   			    // phiAve, psiAve, refineCycle, dfsCycle, w4Angles, resolution,debugDFS, printResults,vecTalos);
   	  
    	    
    	    temp2.print(helixPdbVec6);///added by Zeng for debugging
    		
    	    System.out.println("===============helix7=====================");
    	    Vector<Pdb> helixPdbVec7 = mdc.refineHelixW4RDCs(helixPdbVec6,helixRdc1Vec, helixRdc2Vec,helixRdcCaCoVec, helixRdcCoNVec,  Syy,  Szz, ramaFilter, 
    			    phiAve, psiAve, refineCycle, dfsCycle, w4Angles, resolution,debugDFS, printResults,vecTalos);
    	   // Vector<Pdb> helixPdbVec7 = mdc.refineHelix22(helixPdbVec6,helixRdc1Vec, helixRdc2Vec,Syy,  Szz, ramaFilter, 
   			     //phiAve, psiAve, refineCycle, dfsCycle, w4Angles, resolution,debugDFS, printResults,vecTalos);
   	      	    
    	    
    	    temp2.print(helixPdbVec7);///added by Zeng for debugging
    	
    	    System.out.println("===============helix8=====================");
    	    Vector<Pdb> helixPdbVec8 = mdc.refineHelixW4RDCs(helixPdbVec7,helixRdc1Vec, helixRdc2Vec,helixRdcCaCoVec, helixRdcCoNVec,  Syy,  Szz, ramaFilter, 
    			     phiAve, psiAve, refineCycle, dfsCycle, w4Angles, resolution,debugDFS, printResults,vecTalos);
    	    //Vector<Pdb> helixPdbVec8 = mdc.refineHelix22(helixPdbVec7,helixRdc1Vec, helixRdc2Vec,Syy,  Szz, ramaFilter, 
   			   //  phiAve, psiAve, refineCycle, dfsCycle, w4Angles, resolution,debugDFS, printResults,vecTalos);
   	   
    	    
    	    
    	    temp2.print(helixPdbVec8);///added by Zeng for debugging
    	
    	    
    	    System.out.println("===============helix9=====================");
    	    Vector<Pdb> helixPdbVec9 = mdc.refineHelixW4RDCs(helixPdbVec8,helixRdc1Vec, helixRdc2Vec,helixRdcCaCoVec, helixRdcCoNVec,  Syy,  Szz, ramaFilter, 
    			     phiAve, psiAve, refineCycle, dfsCycle, w4Angles, resolution,debugDFS, printResults,vecTalos);
    	    //Vector<Pdb> helixPdbVec9 = mdc.refineHelix22(helixPdbVec8,helixRdc1Vec, helixRdc2Vec,Syy,  Szz, ramaFilter, 
   			  //   phiAve, psiAve, refineCycle, dfsCycle, w4Angles, resolution,debugDFS, printResults,vecTalos);
   	  
    	    
    	    temp2.print(helixPdbVec9);///added by Zeng for debugging
    	
    	    System.out.println("===============helix10=====================");
    	    Vector<Pdb> helixPdbVec10 = mdc.refineHelixW4RDCs(helixPdbVec9,helixRdc1Vec, helixRdc2Vec,helixRdcCaCoVec, helixRdcCoNVec,  Syy,  Szz, ramaFilter, 
    			     phiAve, psiAve, refineCycle, dfsCycle, w4Angles, resolution,debugDFS, printResults,vecTalos);
    	    //Vector<Pdb> helixPdbVec10 = mdc.refineHelix22(helixPdbVec9,helixRdc1Vec, helixRdc2Vec,Syy,  Szz, ramaFilter, 
   			 //    phiAve, psiAve, refineCycle, dfsCycle, w4Angles, resolution,debugDFS, printResults,vecTalos);
   	         temp2.print(helixPdbVec10);///added by Zeng for debugging
    	*/
    	    
      	 	long endTime = System.currentTimeMillis() ;
    	 	double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
    	 	System.out.println("Time= "+ totalTime +" minutes");

    	    }

     
     //modfied main function from LW's original one
    /**
      * Main function for testing.
      * 
      * @param argv the argv
      * 
      * @throws JampackException the jampack exception
      */
     public static void main1(String[] argv) throws JampackException
    {
    	long startTime = System.currentTimeMillis();
    	Hsqc hqc = new Hsqc();
    	Peak pk = new Peak();
    	
    	ModelRdc mdc = new ModelRdc();    	
    	Assign asg = new Assign();
    	
    	/////////////////////////////////////////////
    	/// 1. Read the input files
    	//
    	/////////////////////////////////////////////
    	int i, j;
    	//String src   = "../inputFiles/"; //The directory has the input files
    	Pdb  pp = new Pdb();
    	String userDir = System.getProperty("user.dir");////
    	String src=userDir+"/inputFiles/";
    	String paramFile=src+"parameters.input";
    	Vector<Map<String, String>> paraVec = asg.ParamReader(paramFile);
    	double haErr  = 0.0;
    	double h1Err  = 0.0;
    	double c13Err = 0.0;
    	double hnErr  = 0.0;
    	double nErr   = 0.0;
    	double noeBound=0.0;
    	String strSeq="", strBB="", strReson="", strNoesy2D="", strHnNoesy3D="", 
    		strHaNoesy3D="",strManualAsg="", strSSES="";
    	for (i=0;i<paraVec.size();i++)
    	{
    		//get the error torlence for each dimension
    		Map paraMap = paraVec.elementAt(i);
    		if(paraMap.containsKey("HAERR"))
    			 haErr  =  Double.parseDouble((String)paraMap.get("HAERR"));
    		if(paraMap.containsKey("H1ERR"))
   			 	h1Err  =  Double.parseDouble((String)paraMap.get("H1ERR"));
    		if(paraMap.containsKey("C13ERR"))
    			c13Err  =  Double.parseDouble((String)paraMap.get("C13ERR"));
    		if(paraMap.containsKey("HNERR"))
    			hnErr  =  Double.parseDouble((String)paraMap.get("HNERR"));
    		if(paraMap.containsKey("NERR"))
    			nErr  =  Double.parseDouble((String)paraMap.get("NERR"));
    		if(paraMap.containsKey("NOEBOUND"))
    			noeBound  =  Double.parseDouble((String)paraMap.get("NOEBOUND"));
    		
    		//get the input file name
    		if(paraMap.containsKey("SEQUENCE"))
    			strSeq  =  (String)paraMap.get("SEQUENCE");
    		strSeq=strSeq.toLowerCase();
    		if(paraMap.containsKey("BACKBONE"))
    			strBB  =  (String)paraMap.get("BACKBONE");
    		strBB=strBB.toLowerCase();
    		if(paraMap.containsKey("RESONANCE"))
    			strReson  =  (String)paraMap.get("RESONANCE");
    		strReson=strReson.toLowerCase();
    		if(paraMap.containsKey("2D-NOESY"))
    			strNoesy2D  =  (String)paraMap.get("2D-NOESY");
    		strNoesy2D=strNoesy2D.toLowerCase();
    		if(paraMap.containsKey("3D-N15-NOESY"))
    			strHnNoesy3D  =  (String)paraMap.get("3D-N15-NOESY");
    		strHnNoesy3D=strHnNoesy3D.toLowerCase();
    		if(paraMap.containsKey("3D-C13-NOESY"))
    			strHaNoesy3D  =  (String)paraMap.get("3D-C13-NOESY");
    		strHaNoesy3D=strHaNoesy3D.toLowerCase();
    		if(paraMap.containsKey("MANUAL-ASSIGNMENT"))
    			strManualAsg  =  (String)paraMap.get("MANUAL-ASSIGNMENT");
    		strManualAsg=strManualAsg.toLowerCase();
    		
    		if(paraMap.containsKey("SSES"))
    			strSSES  =  (String)paraMap.get("SSES");
    		
    	}   	
    	   	   	
    	//-------------------------------------
    	// (1.1) Read the protein sequence
    	// 
    	//-------------------------------------
    	//String seqFile = src+"cbf.seq";
    	String seqFile = src + strSeq;
    	//this is the vector of residues (protein sequence)
    	Vector vecSeq=asg.ReaderSeq(seqFile);    	
    	
    	//-------------------------------------
    	// (1.2) Read the backbone structure 
    	// (input file name: backbone.pdb)
    	// Note: the backbone must be connected. 
    	//-------------------------------------
    	//String pdbFile = src+"backbone.pdb"; 
    	String pdbFile = src+strBB;
    	Vector pdbVecBB = pp.readPdb(pdbFile);

    	//update backbone atom names according to protein sequence input
    	//residue name in previous backbone may be all in "ALA".     		
    	Vector pdbVec3 = pp.residueNameUpdate(vecSeq, pdbVecBB);
    	
    	// this is final backbone pdb used for hosting rotamers
    	Vector pdbVecBBN = pp.nameConvert(pdbVec3);
    	Collections.sort(pdbVecBBN, new Pdb.PdbComparator());
    	    	
    	//-------------------------------------
    	// (1.3) Read the resonance list 
    	// 
    	// in the "resonance.prot" format
    	//-------------------------------------
    	   	
    	//**this is for ubq assignment table assigned by Lincong**//
    	 /*	String assignFileUbq = src+"assignByMe.txt";
    	Vector assignVecUbq  = asg.asgReader3(assignFileUbq); */
    	
    	//following is for the general case
    	H1CS h1CS=new H1CS();
    	//String assignFile = src+ "resonance.prot";    
    	String assignFile = src+ strReson; //"resonance.prot"; 
    	//this is the vector that stores resonance assignments;
    	//each one includes resNo, resName, atomName, csH1(chemical shift value)
    	Vector assignVec=h1CS.h1CSReader(assignFile,vecSeq);  //different input format
    	//Vector assignVec=h1CS.h1CSReader_BMRB(assignFile);  
    	
    	//all sorted proton resonances
    	Vector allH1Vec = pk.allProtonSorted(assignVec);
    	Collections.sort(allH1Vec, new Peak.csComparator());
    	    	
    	//-------------------------------------
    	// (1.4) Read the NOESY peak list 
    	// in the xeasy format
    	// including 2D NOESY peak list, 3D N15 NOESY and 3D C13 NOESY peak lists
    	//-------------------------------------
    	Noesy noesy = new Noesy();
    	
    	String strNoeFile = ""; 
    	Vector hnNoeVec=new Vector();//for storing 3d n15 NOESY peaks
    	Vector cnoeVec=new Vector();//for storing 3d c13 noesy peaks
    	if (!strNoesy2D.equalsIgnoreCase("NULL"))
    	{
    		//to do: extension to 2d case...
    	}
    	if (!strHnNoesy3D.equalsIgnoreCase("NULL"))
    	{
    		strNoeFile=src+strHnNoesy3D;
    		hnNoeVec = noesy.NoesyReader(strNoeFile);    		
    	}//if (!strNoesy3D.equalsIgnoreCase("NULL"))
    	
    	if (!strHaNoesy3D.equalsIgnoreCase("NULL"))
    	{
    		strNoeFile=src+strHaNoesy3D;
    		cnoeVec=noesy.NoesyReader(strNoeFile);
    	}
    	
    	Vector vecNoesy=new Vector();
    	vecNoesy.addAll(hnNoeVec);
    	vecNoesy.addAll(cnoeVec);
    	
    	//intensity calibration
    	Vector vecNewNoesy=noesy.SetCalibration(vecNoesy);    	
    	
    	/*    	    	
    	String cnoeFile2 = src+"cnoeManual.txt";   //the C13NOESY peaklist by Lincong
    	Vector cnoeVec5  =  hcch.hcchReader4(cnoeFile2);
    	String hnNoeFile = src+"ubqNoeManualPeak.txt"; //edited NOE pk list w/o repeat.
    	//Vector hnNoeVec = hnNoe.hnNoeReader2(hnNoeFile);*/
    	/*   	
    	Collections.sort(hnNoeVec, new HnNoe.n15Comparator());
     	Vector oneHnVec = new Vector();
     	Vector oneHaVec = new Vector();
     	Vector oneH1Vec = new Vector();
    	Vector assignVec2 = asg.mergeAssignAndNoe(assignVec, hnNoeVec, 
    						  cnoeVec5, hnErr, nErr, haErr, c13Err);
    	Collections.sort(assignVec2, new Assign.assignComparator());*/

    	//-------------------------------------
    	// (1.4) Read the rotamer library path 
    	// 
    	// use the small Xtal' library from Richardson's lab
    	//-------------------------------------
    	String rotSrc=  src+ "rotasamp-small/";    	
    	
    	//-------------------------------------
    	// (1.5) Read other input data, including
    	// x-ray structure, manual noe assignment, 
    	//full resonance assignment in BMRB, for checking completeness of assigned resoance list.
    	// all are not used for NOE assignment, but for measuring the assignment performance
    	//-------------------------------------
    	String pdbFileXray = src+"2A7O.pdb"; //"one-structure-pei-eta-2I5O.pdb";//"2A7O.pdb";   
    	Vector vecTempPdbBB=pp.readPdb(pdbFileXray);
    	Vector vecTempPdbBB2 = pp.nameConvert(vecTempPdbBB);
    	Vector vecBB=pp.OutputBackbone(vecTempPdbBB2);
    	//pp.print(vecBB);
    	
    	//Vector pdbVec2  = pp.readPdb(pdbFileXray);
    	// this is the X-ray backbone structure
    	//Vector pdbVecXrayBB = pp.nameConvert(pdbVec2);
    	
    	// read BMRB resoance assignment list
    	String manualAsgFile = src+ strManualAsg; //"resonance.prot";  
    	Noe noe_temp=new Noe();
    	Vector vecManAsg=noe_temp.noeReader(manualAsgFile,0.0);
    	

    	// -------------------------------------
    	// (1.6) get the order of secondary 
    	//       structure elements
    	// 
    	//-------------------------------------
    	//read the order of sse packing
    	Vector vecOrderSSES=asg.getOrder(strSSES);
    	
    	//read the pdb file of each sse
    	Vector vecPdbSSE=new Vector();
    	for(i=0; i< vecOrderSSES.size();i++)
    	{
    		String strPdbFile= (String)vecOrderSSES.elementAt(i);
    		strPdbFile=src+strPdbFile+".pdb";
    		Vector pdbVecSSE_temp = pp.readPdb(strPdbFile);

        	//update backbone atom names according to protein sequence input
        	//residue name in previous backbone may be all in "ALA".     		
        	Vector pdbVecSSE_temp2 = pp.residueNameUpdate(vecSeq, pdbVecSSE_temp);
        	
        	// this is final backbone pdb used for hosting rotamers
        	Vector pdbVecSSE = pp.nameConvert(pdbVecSSE_temp2);
        	Collections.sort(pdbVecSSE, new Pdb.PdbComparator());
        	
        	vecPdbSSE.add(pdbVecSSE);
    		
    	}//for(i=0; i< vecOrderSSES.size();i++)
    	
    	
    	/////////////////////////////////////////////////
    	// 2. Rotamer selection at each individual SSE
    	//
    	/////////////////////////////////////////////////
    	
    	//	-------------------------------------
    	// Rotamer selection at each idividual sse 
    	// select optimal rotamer at each residue 
    	// and place optimal rotamers into backbone
    	// only based on NOE from rotamer to its corresponding sse
    	//-------------------------------------
    	//this structure is in BMRB-NEW format:
    	Vector vecPdbRotSSE=new Vector();
    	Model md=new Model();
    	for(i=0;i<vecPdbSSE.size();i++)
    	{
    		Vector vecPdbSseBb=(Vector)vecPdbSSE.elementAt(i);
    		//pp.print(vecPdbSseBb);
    		
    		
    		//note: need to be fixed...
    		//be careful of rotation, we can't change the orientation of each sse
    		Vector vecPdbSseRot=pp.RotamSelectAndStructure(2*h1Err, 2*nErr, 2*c13Err,vecPdbSseBb,assignVec,rotSrc,vecNewNoesy,noeBound, 1,0.0);
    		//pp.print(vecPdbSseRot);
    		
    		//a function to rotate SSE back to the same coordinate system:
    		Vector vecPdbSseRotNew=md.BackonbeRotation(vecPdbSseRot,vecPdbSseBb); 
    		
    		vecPdbRotSSE.add(vecPdbSseRotNew);
    	}
    	
    	System.out.println("====================================");
    	
    	/////////////////////////////////////////////////
    	// 3. Grid search for the best packing ensemble
    	//
    	/////////////////////////////////////////////////
    	Vector vecPdbA=new Vector();
    	Vector vecPdbB =new Vector();
    	vecPdbA.addAll((Vector)vecPdbRotSSE.elementAt(0));
    	PdbRmsd pdr=new PdbRmsd();
    	double [] max_score=new double[4];
    	Vector[] vecEnsembPack = new Vector[4];
    	for(i=0;i<4;i++)
    		vecEnsembPack[i]=new Vector();
    	
    	for (i=1;i<vecPdbRotSSE.size();i++)
    	{
    		vecPdbB =new Vector();
    		double[] dbTemp=new double[1];
    		vecPdbB.addAll((Vector)vecPdbRotSSE.elementAt(i));
    		//vecEnsembPack[0]=new Vector();
        	vecEnsembPack[0]=pdr.positionByNOEPatterns(vecPdbA, vecPdbB,assignVec,vecNewNoesy,
        			2*h1Err, 2*nErr, 2*c13Err,  noeBound,  1,dbTemp);
        	max_score[0]=dbTemp[0];
        	
        	Vector vecTemp=(Vector)vecEnsembPack[0].elementAt(0);
        	System.out.println("i:"+i);
        	System.out.println("here we have the first packing:");
        	Vector vecBBTemp=pp.OutputBackboneNoCB(vecTemp);
        	pp.print(vecBBTemp);//debugging...
        	
        	for (j = 1; j < 4; j++)
        	{
        		Vector vecPdbANew = pp.newPdb(vecPdbA, Const.mat4ThreeDirs[j- 1]);
         	    
        		vecEnsembPack[j]=pdr.positionByNOEPatterns(vecPdbA, vecPdbB,assignVec,vecNewNoesy,
            			2*h1Err, 2*nErr, 2*c13Err,  noeBound,  1,dbTemp);        		
        		max_score[j]=dbTemp[0];
        	}//for (j = 1; j < 4; j++)
        	
        	 int maxPackIndex = 0;
        	 double max_pack_score =  max_score[0];
        		
   	         for (int t=0; t<max_score.length; t++)
        	 {
   	        	 if(max_score[t] > max_pack_score)
   	        	 {
   	        		 max_pack_score = max_score[t];
        	         maxPackIndex = t;
        	      }
        	  }//for (i=0; i<max_score.length; i++)
        	      
        	Vector vecEnsembPackNew=new Vector();
        	
        	//this is the final ensemble of packing
        	vecEnsembPackNew=(Vector)vecEnsembPack[maxPackIndex];
        	vecTemp=(Vector)vecEnsembPackNew.elementAt(0);
        	System.out.println("here we have the final packing:");
        	pp.print(vecTemp);//debugging...
        	vecPdbA=new Vector();
        	vecPdbA.addAll(vecTemp);
    	}//for (i=0;i<vecPdbSSE.size();i++)
    	System.out.println("here we have the structure after first iteration:");
    	pp.print(vecPdbA);
    	
    	/////////////////////////////////////////////////
    	// 4. refine the rotamer selection
    	//
    	/////////////////////////////////////////////////    	
    	System.out.println("====================================");
    	System.out.println("here we refine the rotamer selection:");
    	Vector vecPdbAll2=new Vector();
    	vecPdbAll2.addAll(vecPdbA);
    	Vector vecBB2=pp.OutputBackbone(vecPdbAll2);
    	
    	vecPdbRotSSE=new Vector();  
    	
//    	note: need to be fixed...
    	Vector vecPdbSSERot2=pp.RotamSelectAndStructure(2*h1Err, 2*nErr, 2*c13Err,vecBB2,assignVec,rotSrc,vecNewNoesy,noeBound, 1,0.0);
    	Vector vecPdbSseRotNew2=md.BackonbeRotation(vecPdbSSERot2,vecBB2); 
    	Collections.sort(vecPdbSseRotNew2, new Pdb.PdbComparator());
    	pp.print(vecPdbSseRotNew2);//debugging
    	
    	//break vecPdbSseRotNew2 into pieces:
    	Vector vecPdbSSERotRefine=new Vector();//store pieces of refine helices with rotamers
    	for(i=0;i<vecPdbSSE.size();i++)
    	{
    		Vector vecPdbSseBb=(Vector)vecPdbSSE.elementAt(i);
    		Pdb pdb1=(Pdb)vecPdbSseBb.elementAt(0);
    		Pdb pdb2=(Pdb)vecPdbSseBb.elementAt(vecPdbSseBb.size()-1);
    		int resNo1=pdb1.getResidueNo();
    		int resNo2=pdb2.getResidueNo();
    		
    		Vector vecPdbSSE_temp2=new Vector();
    		for (j=0;j<vecPdbSseRotNew2.size();j++)
    		{
    			Pdb pdb3=(Pdb)vecPdbSseRotNew2.elementAt(j);
    			if(pdb3.getResidueNo()>=resNo1  && pdb3.getResidueNo()<=resNo2)
    				vecPdbSSE_temp2.add(pdb3);
    		}
    		vecPdbSSERotRefine.add(vecPdbSSE_temp2);
    	}
    	
    	/////////////////////////////////////////////////
    	// 5. refine the packing using grid search
    	//
    	/////////////////////////////////////////////////   
    	System.out.println("====================================");
    	System.out.println("here we refine the packing by using grid search:");	
    	vecPdbA=new Vector();
    	vecPdbB =new Vector();
    	vecPdbA.addAll((Vector)vecPdbSSERotRefine.elementAt(0));
    	pdr=new PdbRmsd();
    	max_score=new double[4];
    	vecEnsembPack = new Vector[4];

    	for (i=1;i<vecPdbSSERotRefine.size();i++)
    	{
    		double[] dbTemp=new double[1];
    		vecPdbB =new Vector();
    		vecPdbB.addAll((Vector)vecPdbSSERotRefine.elementAt(i));
    		//vecEnsembPack[0]=new Vector();
        	vecEnsembPack[0]=pdr.positionByNOEPatterns(vecPdbA, vecPdbB,assignVec,vecNewNoesy,
        			2*h1Err, 2*nErr, 2*c13Err,  noeBound,  1,dbTemp);
        	max_score[0]=dbTemp[0];
        	
        	Vector vecTemp=(Vector)vecEnsembPack[0].elementAt(0);
        	System.out.println("i:"+i);
        	System.out.println("here we have the first packing:");
        	pp.print(vecTemp);//debugging...
        	
        	for (j = 1; j < 4; j++)
        	{
        		Vector vecPdbANew = pp.newPdb(vecPdbA, Const.mat4ThreeDirs[j- 1]);
         	    
        		vecEnsembPack[j]=pdr.positionByNOEPatterns(vecPdbA, vecPdbB,assignVec,vecNewNoesy,
            			2*h1Err, 2*nErr, 2*c13Err,  noeBound,  1,dbTemp);        		
        		max_score[j]=dbTemp[0];
        	}//for (j = 1; j < 4; j++)
        	
        	 int maxPackIndex = 0;
        	 double max_pack_score =  max_score[0];
        		
   	         for (int t=0; t<max_score.length; t++)
        	 {
   	        	 if(max_score[t] > max_pack_score)
   	        	 {
   	        		 max_pack_score = max_score[t];
        	         maxPackIndex = t;
        	      }
        	  }//for (i=0; i<max_score.length; i++)
        	      
        	Vector vecEnsembPackNew=new Vector();
        	
        	//this is the final ensemble of packing
        	vecEnsembPackNew=(Vector)vecEnsembPack[maxPackIndex];
        	vecTemp=(Vector)vecEnsembPackNew.elementAt(0);
        	System.out.println("here we have the final packing:");
        	pp.print(vecTemp);//debugging...
        	vecPdbA=new Vector();
        	vecPdbA.addAll(vecTemp);
    	}//for (i=0;i<vecPdbSSE.size();i++)
    	System.out.println("here we have the structure after second iteration:");
    	pp.print(vecPdbA);
    	
    	
    	

		long endTime = System.currentTimeMillis() ;
		double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
		System.out.println("Time= "+ totalTime +" minutes");	    
    }
}
