/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/
package libprotnmr.analysis;

import java.util.List;

import libprotnmr.geom.Vector3;
import libprotnmr.protein.AtomAddressInternal;
import libprotnmr.protein.HasAtoms;
import libprotnmr.protein.Subunit;


public class RmsdCalculator
{
	/**************************
	 *   Static Methods
	 **************************/
	
	public static double getRmsd( HasAtoms a, HasAtoms b )
	{
		return getRmsd( a, b, a.atoms(), b.atoms() );
	}
	
	public static double getBackboneRmsd( HasAtoms a, HasAtoms b )
	{
		return getRmsd( a, b, a.backboneAtoms(), b.backboneAtoms() );
	}
	
	public static double getRmsdByResidueNumbers( Subunit a, Subunit b, int startNumber, int stopNumber )
	{
		return getRmsd(
			a,
			b,
			a.getInternalAddressesByResidueNumbers( startNumber, stopNumber ),
			b.getInternalAddressesByResidueNumbers( startNumber, stopNumber )
		);
	}
	
	public static double getBackboneRmsdByResidueNumbers( Subunit a, Subunit b, int startNumber, int stopNumber )
	{
		return getRmsd(
			a,
			b,
			a.getBackboneInternalAddressesByResidueNumbers( startNumber, stopNumber ),
			b.getBackboneInternalAddressesByResidueNumbers( startNumber, stopNumber )
		);
	}
	
	public static double getRmsd( HasAtoms a, HasAtoms b, List<AtomAddressInternal> addressesA, List<AtomAddressInternal> addressesB )
	{
		// just in case...
		assert( addressesA.size() == addressesB.size() );
		
		double sum = 0.0;
		int count = 0;
		
		// for each atom...
		for( int i=0; i<addressesA.size(); i++ )
		{
			// get atoms
			Vector3 posA = new Vector3( a.getAtom( addressesA.get( i ) ).getPosition() );
			Vector3 posB = new Vector3( b.getAtom( addressesB.get( i ) ).getPosition() );
			
			// add the sd
			sum += posB.getSquaredDistance( posA );
			count++;
		}
		
		// do the rm part
		return Math.sqrt( sum / (double)count );
	}
}
