/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


#include <jni.h>

#include "cgal.h"
#include "shotCgal.h"
#include "global.h"
#include "types.h"
#include "Storage.h"


#define CLASS                       Java_edu_duke_donaldLab_jdshot_disco_cgal_Halfedge
#define Halfedge_cleanup            CLASSFN( CLASS, cleanup )
#define Halfedge_getCircle          CLASSFN( CLASS, getCircle )
#define Halfedge_getSource          CLASSFN( CLASS, getSource )
#define Halfedge_getTarget          CLASSFN( CLASS, getTarget )
#define Halfedge_getBoundingBox     CLASSFN( CLASS, getBoundingBox )
#define Halfedge_getAdjacentFace    CLASSFN( CLASS, getAdjacentFace )

typedef std::pair<int,Arrangement::Halfedge> HalfedgeMapPair;


static Storage<Arrangement::Halfedge> g_halfedges( CGALCLASS( "Halfedge" ), false );
static jclass g_halfedgeClass = NULL;


void halfedgesCleanup( JNIEnv *jvm )
{
	g_halfedges.cleanupAll( jvm );

	if( g_halfedgeClass != NULL )
	{
		jvm->DeleteGlobalRef( g_halfedgeClass );
		g_halfedgeClass = NULL;
	}
}

Arrangement::Halfedge *getHalfedge( JNIEnv *jvm, jobject self )
{
	return g_halfedges.get( jvm, self );
}

jobject newHalfedge( JNIEnv *jvm, Arrangement::Halfedge *pHalfedge )
{
	return g_halfedges.addNew( jvm, pHalfedge );
}

JNIEXPORT void JNICALL Halfedge_cleanup( JNIEnv *jvm, jclass c, jint id )
{
	START_SIGNAL_HANDLING
	{
		if( !g_halfedges.cleanup( id ) )
		{
			throwException( jvm, "Halfedge cleanup failure!" );
		}
	}
	STOP_SIGNAL_HANDLING
}

JNIEXPORT jobject JNICALL Halfedge_getCircle( JNIEnv *jvm, jobject self )
{
	START_SIGNAL_HANDLING
	{
		Arrangement::Halfedge *pHalfedge = getHalfedge( jvm, self );
		return newCircle( jvm, pHalfedge->curve().supporting_circle() );
	}
	STOP_SIGNAL_HANDLING
	return NULL;
}

JNIEXPORT jobject JNICALL Halfedge_getSource( JNIEnv *jvm, jobject self )
{
	START_SIGNAL_HANDLING
	{
		Arrangement::Halfedge *pHalfedge = getHalfedge( jvm, self );
		return newVertex( jvm, &*(pHalfedge->source()) );
	}
	STOP_SIGNAL_HANDLING
	return NULL;
}

JNIEXPORT jobject JNICALL Halfedge_getTarget( JNIEnv *jvm, jobject self )
{
	START_SIGNAL_HANDLING
	{
		Arrangement::Halfedge *pHalfedge = getHalfedge( jvm, self );
		return newVertex( jvm, &*(pHalfedge->target()) );
	}
	STOP_SIGNAL_HANDLING
	return NULL;
}

JNIEXPORT jobject JNICALL Halfedge_getBoundingBox( JNIEnv *jvm, jobject self )
{
	START_SIGNAL_HANDLING
	{
		Arrangement::Halfedge *pHalfedge = getHalfedge( jvm, self );
		return newBox( jvm, pHalfedge->curve().bbox() );
	}
	STOP_SIGNAL_HANDLING
	return NULL;
}

JNIEXPORT jobject JNICALL Halfedge_getAdjacentFace( JNIEnv *jvm, jobject self )
{
	START_SIGNAL_HANDLING
	{
		Arrangement::Halfedge *pHalfedge = getHalfedge( jvm, self );
		return newFace( jvm, &*pHalfedge->twin()->face() );
	}
	STOP_SIGNAL_HANDLING
	return NULL;
}
