/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.share.clustering.distance;

import edu.duke.donaldLab.share.math.IndexPair;

public class DistanceMatrix
{
	/**************************
	 *   Data Members
	 **************************/
	
	private double[][] m_distances;
	private long m_numDistances;
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	public DistanceMatrix( int numPoints )
	{
		// just in case...
		assert( numPoints >= 2 );
		
		m_distances = new double[numPoints-1][];
		for( int i=0; i<numPoints-1; i++ )
		{
			m_distances[i] = new double[i+1];
			
			// initialize to zero
			for( int j=0; j<i; j++ )
			{
				m_distances[i][j] = 0.0;
			}
		}
		
		m_numDistances = getSize( numPoints );
	}
	
	
	/**************************
	 *   Accessors
	 **************************/
	
	public double get( IndexPair pair )
	{
		return get( pair.left, pair.right );
	}
	
	public double get( int r, int c )
	{
		// make sure we're in the right triagnle
		if( r < c )
		{
			int swap = r;
			r = c;
			c = swap;
		}
		
		// map to the compacted matrix
		r -= 1;
		
		return m_distances[r][c];
	}
	
	public void set( IndexPair pair, double val )
	{
		set( pair.left, pair.right, val );
	}
	
	public void set( int r, int c, double val )
	{
		// make sure we're in the right triagnle
		if( r < c )
		{
			int swap = r;
			r = c;
			c = swap;
		}
		
		// map to the compacted matrix
		r -= 1;
		
		m_distances[r][c] = val;
	}
	
	public int getNumPoints( )
	{
		return m_distances.length + 1;
	}
	
	public long getNumDistances( )
	{
		return m_numDistances;
	}
	
	public void updateCompleteLinkage( int destIndex, int otherIndex )
	{
		for( int i=0; i<getNumPoints(); i++ )
		{
			// no self-pairings
			if( destIndex == i || otherIndex == i )
			{
				continue;
			}
			
			// get the max distance between i and dest/other
			double destDist = get( destIndex, i );
			double otherDist = get( otherIndex, i );
			set( destIndex, i, otherDist > destDist ? otherDist : destDist );
		}
	}
	
	
	/**************************
	 *   Static Methods
	 **************************/
	
	public static long getSize( int numPoints )
	{
		return (long)numPoints * ( (long)numPoints - 1L ) / 2L;
	}
}
