/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.test.disco;

import java.io.File;
import java.util.List;

import edu.duke.donaldLab.jdshot.context.DistanceRestraintsContext;
import edu.duke.donaldLab.jdshot.context.RdcsContext;
import edu.duke.donaldLab.jdshot.context.SymmetryContext;
import edu.duke.donaldLab.jdshot.disco.AnnulusCalculator;
import edu.duke.donaldLab.jdshot.disco.AnnulusIndex;
import edu.duke.donaldLab.jdshot.disco.FaceInfo;
import edu.duke.donaldLab.jdshot.disco.OrientationCalculator;
import edu.duke.donaldLab.jdshot.disco.PositionCalculator;
import edu.duke.donaldLab.jdshot.disco.ResultsReader;
import edu.duke.donaldLab.jdshot.disco.ViolationCalculator;
import edu.duke.donaldLab.jdshot.disco.ViolationCalculator.Violation;
import edu.duke.donaldLab.jdshot.disco.cgal.ShotCgal;
import edu.duke.donaldLab.jdshot.grid.Symmetry;
import edu.duke.donaldLab.jdshot.test.ExtendedTestCase;
import edu.duke.donaldLab.share.geom.Vector3;
import edu.duke.donaldLab.share.nmr.AlignmentTensor;
import edu.duke.donaldLab.share.nmr.DistanceRestraint;
import edu.duke.donaldLab.share.protein.AtomAddressInternal;
import edu.duke.donaldLab.share.protein.Subunit;

public class TestViolationCalculator extends ExtendedTestCase
{
	public void testDagk( )
	throws Exception
	{
		// read inputs
		SymmetryContext symmetryContext = new SymmetryContext( Symmetry.Cn, new File( "src/resources/test/2KDC.monomer.protein" ), 3 );
		RdcsContext rdcsContext = new RdcsContext( symmetryContext, "src/resources/test/2KDC.experimental.rdc" );
		DistanceRestraintsContext distanceRestraintsContext = new DistanceRestraintsContext( symmetryContext, "src/resources/test/2KDC.experimental.onlyDisulfideBonds.noe" );
		
		ShotCgal.init();
		try
		{
			// previous steps
			Subunit monomer = symmetryContext.getPseudoMonomer();
			AlignmentTensor tensor = AlignmentTensor.compute( monomer, rdcsContext.getInternalRdcs() );
			Vector3 orientation = OrientationCalculator.computeOrientations( 3, tensor ).get( 0 );
			OrientationCalculator.normalizeProtein( monomer, orientation );
			AnnulusIndex annulusIndex = AnnulusCalculator.computeAnnuli( monomer, distanceRestraintsContext.getInternalRestraints(), 3 );
			
			// load the MSRs
			List<FaceInfo> msrs = new ResultsReader().readMsrs( new File( "src/resources/test/2KDC.searchResults.dat" ) );
			
			// compute the violations
			List<DistanceRestraint<AtomAddressInternal>> inconsistentDistanceRestraints =
				PositionCalculator.getInconsistentRestraints(
					msrs,
					distanceRestraintsContext.getInternalRestraints()
				);
			List<Violation> violations = ViolationCalculator.getViolations(
				msrs,
				annulusIndex,
				inconsistentDistanceRestraints,
				symmetryContext,
				monomer
			);
			assertEquals( inconsistentDistanceRestraints.size(), violations.size() );
			
			// check the violations
			assertEquals( 3, violations.size() );
			final double Epsilon = 1e-4;
			assertViolationFound( 0.5896, 1.0212, violations.get( 0 ), Epsilon );
			assertViolationFound( 0.4727, 0.8187, violations.get( 1 ), Epsilon );
			assertViolationFound( 0.4985, 0.8635, violations.get( 2 ), Epsilon );
		}
		finally
		{
			ShotCgal.cleanup();
		}
	}
	
	public void testGb1( )
	throws Exception
	{
		// read inputs
		SymmetryContext symmetryContext = new SymmetryContext( Symmetry.Cn, new File( "src/resources/test/1Q10.monomer.protein" ), 2 );
		RdcsContext rdcsContext = new RdcsContext( symmetryContext, "src/resources/test/1Q10.experimental.rdc" );
		DistanceRestraintsContext distanceRestraintsContext = new DistanceRestraintsContext( symmetryContext, "src/resources/test/1Q10.experimental.fixed.noe" );
		
		ShotCgal.init();
		try
		{
			// previous steps
			Subunit monomer = symmetryContext.getPseudoMonomer();
			AlignmentTensor tensor = AlignmentTensor.compute( monomer, rdcsContext.getInternalRdcs() );
			Vector3 orientation = OrientationCalculator.computeOrientations( 2, tensor ).get( 0 );
			OrientationCalculator.normalizeProtein( monomer, orientation );
			AnnulusIndex annulusIndex = AnnulusCalculator.computeAnnuli( monomer, distanceRestraintsContext.getInternalRestraints(), 2 );
			
			// load the MSRs
			List<FaceInfo> msrs = new ResultsReader().readMsrs( new File( "src/resources/test/1Q10.searchResults.dat" ) );
			
			// compute the violations
			List<DistanceRestraint<AtomAddressInternal>> inconsistentDistanceRestraints =
				PositionCalculator.getInconsistentRestraints(
					msrs,
					distanceRestraintsContext.getInternalRestraints()
				);
			
			// TEMP
			for( DistanceRestraint<AtomAddressInternal> restraint : distanceRestraintsContext.getInternalRestraints() )
			{
				assertNotNull( annulusIndex.getEntries( restraint ) );
			}
			for( DistanceRestraint<AtomAddressInternal> restraint : inconsistentDistanceRestraints )
			{
				assertNotNull( annulusIndex.getEntries( restraint ) );
			}
			
			List<Violation> violations = ViolationCalculator.getViolations(
				msrs,
				annulusIndex,
				inconsistentDistanceRestraints,
				symmetryContext,
				monomer
			);
			assertEquals( inconsistentDistanceRestraints.size(), violations.size() );
			
			// check the violations
			assertEquals( 7, violations.size() );
			// HACKHACK: this epsilon is huge, why are the GB1 numbers so imprecise?
			final double Epsilon = 1e-1;
			assertViolationFound( 0.2925, 0.4612, violations.get( 0 ), Epsilon );
			assertViolationFound( 0.3178, 0.6262, violations.get( 1 ), Epsilon );
			assertViolationFound( 0.1189, 0.2347, violations.get( 2 ), Epsilon );
			assertViolationFound( 0.3509, 0.5179, violations.get( 3 ), Epsilon );
			assertViolationFound( 0.2353, 0.4102, violations.get( 4 ), Epsilon );
			assertViolationFound( 0.4275, 0.8221, violations.get( 5 ), Epsilon );
			assertViolationFound( 0.0508, 0.0000, violations.get( 6 ), Epsilon );
		}
		finally
		{
			ShotCgal.cleanup();
		}
	}
	
	private void assertViolationFound( double planeDistance, double atomDistance, Violation violation, double epsilon )
	{
		assertEquals( planeDistance, violation.distance, epsilon );
		assertEquals( atomDistance, violation.atomDistance, epsilon );
	}
}
