/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.stericFilter;

import java.io.File;
import java.io.IOException;

import edu.duke.donaldLab.jdshot.grid.PointIterator;
import edu.duke.donaldLab.jdshot.grid.PointWriter;
import edu.duke.donaldLab.jdshot.search.SearchContext;
import edu.duke.donaldLab.share.perf.Progress;

public class StericFilterer
{
	/**************************
	 *   Data Members
	 **************************/
	
	private SearchContext m_searchContext;
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	public StericFilterer( SearchContext searchContext )
	{
		m_searchContext = searchContext;
	}

	
	/**************************
	 *   Methods
	 **************************/
	
	public FilterResults filter( File outPointsFile, PointIterator iterPoint, double epsilon, int numThreads )
	throws IOException
	{
		FilterResults results = new FilterResults();
		
		// get some objects for the synch
		Progress progress = new Progress( iterPoint.getNumPoints(), 15000 );
		PointWriter writer = new PointWriter( outPointsFile );
		
		// start some processors
		StericFilterSynchronizer synch = new StericFilterSynchronizer( iterPoint, progress, writer, 20 );
		StericFilterProcessor[] processors = new StericFilterProcessor[numThreads];
		Thread[] threads = new Thread[numThreads];
		for( int i=0; i<numThreads; i++ )
		{
			processors[i] = new StericFilterProcessor( synch, m_searchContext, epsilon );
			threads[i] = new Thread( processors[i] );
			threads[i].start();
		}
		
		// wait for thread to finish
		for( int i=0; i<numThreads; i++ )
		{
			try
			{
				threads[i].join();
			}
			catch( InterruptedException ex )
			{
				// pass it along
				Thread.currentThread().interrupt();
			}
		}
		
		// cleanup the writer
		writer.close();
		
		// return the results
		results.numProcessed = synch.getNumProcessed();
		results.numAccepted = synch.getNumAccepted();
		return results;
	}
}
