/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.search;

import java.util.List;

import edu.duke.donaldLab.jdshot.context.DistanceRestraintsContext;
import edu.duke.donaldLab.jdshot.context.SymmetryContext;
import edu.duke.donaldLab.jdshot.grid.GridCell;
import edu.duke.donaldLab.jdshot.grid.SearchSpace;
import edu.duke.donaldLab.share.nmr.DistanceRestraint;
import edu.duke.donaldLab.share.nmr.Assignment;
import edu.duke.donaldLab.share.protein.AtomAddressInternal;

public class StaticSearchProcessor implements Runnable
{
	/**************************
	 *   Data Members
	 **************************/
	
	private CellChecker m_checker;
	private CellSplitter m_splitter;
	private SymmetryContext m_symmetryContext;
	private DistanceRestraintsContext m_distanceRestraintsContext;
	private List<GridCell> m_oldCells;
	private List<GridCell> m_newCells;
	private int m_startIndex;
	private int m_numCells;
	private Consistency m_consistency;
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	public StaticSearchProcessor( SymmetryContext symmetryContext, DistanceRestraintsContext distanceRestraintsContext, List<GridCell> oldCells, List<GridCell> newCells, int threadId, int numThreads )
	{
		// save parameters
		m_symmetryContext = symmetryContext;
		m_distanceRestraintsContext = distanceRestraintsContext;
		m_oldCells = oldCells;
		m_newCells = newCells;
		
		// find out what cells we're processing
		m_numCells = ( m_oldCells.size() + numThreads - 1 ) / numThreads;
		m_startIndex = threadId * m_numCells;
		if( m_startIndex + m_numCells > m_oldCells.size() )
		{
			m_numCells = m_oldCells.size() - m_startIndex;
		}
		
		// init defaults
		SearchSpace searchSpace = m_symmetryContext.newSearchSpace( m_distanceRestraintsContext.getMaxRestraint().getMaxDistance() );
		m_splitter = m_symmetryContext.getSymmetry().newCellSplitter( searchSpace );
		m_checker = m_symmetryContext.getSymmetry().newCellChecker( m_symmetryContext );
		m_consistency = new Consistency( m_numCells * 2, m_distanceRestraintsContext.getInternalRestraints() );
	}
	
	
	/**************************
	 *   Methods
	 **************************/
	
	public void run( )
	{
		// TEMP
		System.out.println( "Thread starting, reading " + m_numCells + " cells..." );
		
		for( int i=m_startIndex; i<m_numCells + m_startIndex; i++ )
		{
			for( int j=0; j<400; j++ )
			{
				for( int k=0; k<1e4; k++ );
			}
		}
		
		// TEMP
		System.out.println( "Thread finished! processed " + ( m_numCells * 2 ) + " cells" );
	}
	
	
	public void runReal( )
	{
		// TEMP
		System.out.println( "Thread starting, reading " + m_numCells + " cells..." );
		
		int leftIndex = 0;
		int rightIndex = 0;
		GridCell left = null;
		GridCell right = null;
		int bufferIndex = 0;
		
		for( int i=m_startIndex; i<m_numCells + m_startIndex; i++ )
		{
			GridCell cell = m_oldCells.get( i );
			
			// split the cell and process both children
			leftIndex = i * 2;
			rightIndex = i * 2 + 1;
			left = m_newCells.get( leftIndex );
			right = m_newCells.get( rightIndex );
			m_splitter.split( left, right, cell );
			processCell( left, leftIndex, bufferIndex++ );
			processCell( right, rightIndex, bufferIndex++ );
		}
		
		// TEMP
		System.out.println( "Thread finished! processed " + bufferIndex + " cells" );
	}
	
	
	/**************************
	 *   Functions
	 **************************/
	
	private void processCell( GridCell cell, int cellIndex, int bufferIndex )
	{
		// for each restraint...
		int numRestraints = m_distanceRestraintsContext.getInternalRestraints().size();
		for( int i=0; i<numRestraints; i++ )
		{
			DistanceRestraint<AtomAddressInternal> restraint = m_distanceRestraintsContext.getInternalRestraints().get( i );
			
			int j = 0;
			int numInconsistentAssignments = 0;
			for( Assignment<AtomAddressInternal> assignment : restraint )
			{
				if( m_checker.isConsistent( cell, restraint.getMaxDistance(), assignment ) )
				{
					m_consistency.markRestraintConsistent( i, j );
				}
				else
				{
					numInconsistentAssignments++;
				}
				
				j++;
			}
			
			if( numInconsistentAssignments == j )
			{
				m_consistency.markCellInconsistent( bufferIndex );
			}
		}
	}
}
