/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.grid.dn;

import edu.duke.donaldLab.jdshot.grid.Axes;
import edu.duke.donaldLab.jdshot.grid.GridPoint;
import edu.duke.donaldLab.jdshot.grid.StructureGenerator;
import edu.duke.donaldLab.share.geom.Vector3;
import edu.duke.donaldLab.share.math.Matrix3;
import edu.duke.donaldLab.share.protein.Protein;
import edu.duke.donaldLab.share.protein.Subunit;
import edu.duke.donaldLab.share.protein.tools.ProteinGeometry;

public class DnStructureGenerator implements StructureGenerator
{
	/**************************
	 *   Data Members
	 **************************/
	
	Subunit m_monomer;
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	public DnStructureGenerator( Subunit monomer )
	{
		m_monomer = monomer;
	}
	
	
	/**************************
	 *   Accessors
	 **************************/
	
	@Override
	public Subunit getMonomer( )
	{
		return m_monomer;
	}
	@Override
	public void setMonomer( Subunit value )
	{
		m_monomer = value;
	}
	
	
	/**************************
	 *   Methods
	 **************************/
	
	@Override
	public Protein getStructure( GridPoint point, int numSubunits )
	{
		// make sure the cast is safe
		assert( point instanceof DnGridPoint );
		
		return getStructure( (DnGridPoint)point, numSubunits );
	}
	
	public Protein getStructure( DnGridPoint point, int numSubunits )
	{
		// get the symmetry axes
		Vector3 flipAxis = new Vector3();
		Axes.getFlipVector( flipAxis, point.phi, point.theta, point.rho );
		Vector3 rotationAxis = new Vector3();
		Axes.getRotationVector( rotationAxis, point.phi, point.theta );
		
		// just in case...
		assert( numSubunits % 2 == 0 ) : "must have an even number of subunits.";
		
		// get the intersection point of the axes
		Vector3 t = new Vector3( point.x, point.y, point.z );
		
		return getStructure( t, rotationAxis, flipAxis, numSubunits );
	}
	
	public Protein getStructure( Vector3 axisPosition, Vector3 rotationAxis, Vector3 flipAxis, int numSubunits )
	{
		Matrix3 flipOverF = new Matrix3();
		Matrix3 spinAroundR = new Matrix3();
		
		// just in case...
		assert( numSubunits % 2 == 0 ) : "must have an even number of subunits.";
		
		// make a new protein with the monomer
		Protein protein = new Protein();
		Subunit firstOriginalSubunit = new Subunit( m_monomer );
		firstOriginalSubunit.setName( 'A' );
		protein.addSubunit( firstOriginalSubunit );
		Subunit firstBizzaroSubunit = new Subunit( m_monomer );
		firstBizzaroSubunit.setName( 'B' );
		protein.addSubunit( firstBizzaroSubunit );
		
		// create the dimer using the flip axis
		Matrix3.getRotationByPi( flipOverF, flipAxis );
		ProteinGeometry.transform( firstBizzaroSubunit, flipOverF, axisPosition );
		
		// create the rest of the dimers by rotating the first dimer about the rotation axis
		int numDimers = numSubunits / 2;
		for( int i=1; i<numDimers; i++ )
		{
			// get the transformation matrix
			double rotationAngle = (double)i * 2.0 * Math.PI / numDimers;
			Matrix3.getRotation( spinAroundR, rotationAxis, rotationAngle );
			
			// add the subunits
			Subunit originalSubunit = new Subunit( firstOriginalSubunit );
			originalSubunit.setName( (char)( 'A' + ( i * 2 ) ) );
			protein.addSubunit( originalSubunit );
			Subunit bizzaroSubunit = new Subunit( firstBizzaroSubunit );
			bizzaroSubunit.setName( (char)( originalSubunit.getName() + 1 ) );
			protein.addSubunit( bizzaroSubunit );
			
			ProteinGeometry.transform( originalSubunit, spinAroundR, axisPosition );
			ProteinGeometry.transform( bizzaroSubunit, spinAroundR, axisPosition );
		}
		
		// just in case...
		assert( protein.getSubunits().size() == numSubunits );
		
		return protein;
	}
}
