/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.grid.dn;

import java.io.Serializable;

import edu.duke.donaldLab.jdshot.grid.GridCell;
import edu.duke.donaldLab.jdshot.grid.GridPoint;
import edu.duke.donaldLab.jdshot.grid.Symmetry;
import edu.duke.donaldLab.share.geom.Vector3;

public class DnGridCell implements GridCell, Serializable
{
	/**************************
	 *   Definitions
	 **************************/
	
	private static final long serialVersionUID = 8851136449422768299L;
	public static final int NumACorners = 8;
	public static final int NumTCorners = 8;
	
	public enum Subspace
	{
		Translations,
		Rotations
	}
	
	
	/**************************
	 *   Fields
	 **************************/
	
	public DnGridPoint min;
	public DnGridPoint max;
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	public DnGridCell( )
	{
		min = new DnGridPoint();
		max = new DnGridPoint();
	}
	
	public DnGridCell( DnGridPoint a, DnGridPoint b )
	{
		min = new DnGridPoint( a );
		max = new DnGridPoint( b );
	}
	
	public DnGridCell( DnGridCell other )
	{
		min = new DnGridPoint( other.min );
		max = new DnGridPoint( other.max );
	}
	
	
	/**************************
	 *   Accessors
	 **************************/
	
	public void set( DnGridCell other )
	{
		min.set( other.min );
		max.set( other.max );
	}
	
	@Override
	public Symmetry getSymmetry( )
	{
		return Symmetry.Dn;
	}
	
	@Override
	public GridPoint getMin( )
	{
		return min;
	}
	
	@Override
	public void setMin( GridPoint point )
	{
		// make sure this is a valid cast
		assert( point instanceof DnGridPoint );
		
		min = (DnGridPoint)point;
	}
	
	@Override
	public GridPoint getMax( )
	{
		return max;
	}
	
	@Override
	public void setMax( GridPoint point )
	{
		// make sure this is a valid cast
		assert( point instanceof DnGridPoint );
		
		max = (DnGridPoint)point;
	}
	
	@Override
	public void set( GridCell other )
	{
		// make sure this is a valid cast
		assert( other instanceof DnGridCell );
		
		min.set( ((DnGridCell)other).min );
		max.set( ((DnGridCell)other).max );
	}
	
	
	/**************************
	 *   Methods
	 **************************/
	
	public String toString( )
	{
		return "[GridCell]\n"
			+ "\t[" + min.x + "," + max.x + "]\n" 
			+ "\t[" + min.y + "," + max.y + "]\n" 
			+ "\t[" + min.z + "," + max.z + "]\n" 
			+ "\t[" + min.phi + "," + max.phi + "]\n" 
			+ "\t[" + min.theta + "," + max.theta + "]\n" 
			+ "\t[" + min.rho + "," + max.rho + "]"; 
	}
	
	public String toCode( )
	{
		return "new GridCell(\n\t" + min.toCode() + ",\n\t" + max.toCode() + "\n)";
	}
	
	@Override
	public DnGridPoint getCenter( )
	{
		DnGridPoint temp = new DnGridPoint( min );
		temp.add( max );
		temp.scale( 0.5 );
		return temp;
	}
	
	@Override
	public DnGridPoint getDeltas( )
	{
		DnGridPoint temp = new DnGridPoint( max );
		temp.subtract( min );
		return temp;
	}
	
	@Override
	public double getVolume( )
	{
		// UNDONE: improve this method using the jacobian
		
		double volume = 0.0;
		
		// calculate the S^2 component
		volume =
			( Math.cos( min.theta ) - Math.cos( max.theta ) )
			* ( max.phi - min.phi );
		
		// calculate the S component
		volume *= max.rho - min.rho;
		
		// calculate the R^3 component
		volume *=
			( max.x - min.x )
			* ( max.y - min.y )
			* ( max.z - min.z );
		
		return volume;
	}
	
	@Override
	public void split( GridCell leftCell, GridCell rightCell, int dimension )
	{
		// make sure the cast we're about to do is safe
		assert( leftCell instanceof DnGridCell );
		assert( rightCell instanceof DnGridCell );
		
		split( (DnGridCell)leftCell, (DnGridCell)rightCell, dimension );
	}
	
	public void split( DnGridCell leftCell, DnGridCell rightCell, int dimension )
	{
		// just in case...
		assert( leftCell != null );
		assert( rightCell != null );
		
		// copy self to children
		leftCell.set( this );
		rightCell.set( this );
		
		// get the half deltas
		DnGridPoint halfDeltas = getDeltas();
		halfDeltas.scale( 0.5 );
		
		leftCell.max.set( dimension, leftCell.max.get( dimension ) - halfDeltas.get( dimension ) );
		rightCell.min.set( dimension, rightCell.min.get( dimension ) + halfDeltas.get( dimension ) );
	}
	
	public void getACorner( Vector3 out, int i )
	{
		out.set(
			( i/4 == 0 ) ? min.phi : max.phi,
			( i/2%2 == 0 ) ? min.theta : max.theta,
			( i%2 == 0 ) ? min.rho : max.rho
		);
	}
	
	public void getTCorner( Vector3 out, int i )
	{
		out.set(
			( i/4 == 0 ) ? min.x : max.x,
			( i/2%2 == 0 ) ? min.y : max.y,
			( i%2 == 0 ) ? min.z : max.z
		);
	}
}
