/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.grid.cn;

import java.io.Serializable;

import edu.duke.donaldLab.jdshot.grid.GridCell;
import edu.duke.donaldLab.jdshot.grid.GridPoint;
import edu.duke.donaldLab.jdshot.grid.Symmetry;
import edu.duke.donaldLab.share.geom.Vector3;

public class CnGridCell implements GridCell, Serializable
{
	/**************************
	 *   Definitions
	 **************************/
	
	private static final long serialVersionUID = -6265847052833470218L;
	public static final int NumACorners = 4;
	public static final int NumTCorners = 4;
	
	
	/**************************
	 *   Fields
	 **************************/
	
	public CnGridPoint min;
	public CnGridPoint max;
	public double ratio = -1; //ratio between box and hull volume
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	public CnGridCell()
	{
		min = new CnGridPoint();
		max = new CnGridPoint();
	}
	
	public CnGridCell( CnGridPoint min, CnGridPoint max )
	{
		this.min = new CnGridPoint( min );
		this.max = new CnGridPoint( max );
	}
	
	public CnGridCell( CnGridCell other )
	{
		min = new CnGridPoint( other.min );
		max = new CnGridPoint( other.max );
	}
	
	
	/**************************
	 *   Accessors
	 **************************/
	
	public void set( CnGridCell other )
	{
		min.set( other.min );
		max.set( other.max );
	}
	
	@Override
	public Symmetry getSymmetry( )
	{
		return Symmetry.Cn;
	}
	
	@Override
	public GridPoint getMin( )
	{
		return min;
	}
	
	@Override
	public void setMin( GridPoint point )
	{
		// make sure this is a valid cast
		assert( point instanceof CnGridPoint );
		
		min = (CnGridPoint)point;
	}
	
	@Override
	public GridPoint getMax( )
	{
		return max;
	}
	
	@Override
	public void setMax( GridPoint point )
	{
		// make sure this is a valid cast
		assert( point instanceof CnGridPoint );
		
		max = (CnGridPoint)point;
	}
	
	@Override
	public void set( GridCell other )
	{
		// make sure this is a valid cast
		assert( other instanceof CnGridCell );
		
		min.set( ((CnGridCell)other).min );
		max.set( ((CnGridCell)other).max );
	}
	
	
	/**************************
	 *   Methods
	 **************************/
	
	@Override
	public CnGridPoint getCenter( )
	{
		CnGridPoint temp = new CnGridPoint( min );
		temp.add( max );
		temp.scale( 0.5 );
		return temp;
	}
	
	@Override
	public double getVolume( )
	{
		// UNDONE: make sure this is the volume Himanshu intended
		
		CnGridPoint temp = new CnGridPoint( max );
		temp.subtract( min );
		return temp.x * temp.y * temp.phi * temp.theta;
	}
	
	@Override
	public void split( GridCell leftCell, GridCell rightCell, int dimension )
	{
		// make sure the cast we're about to do is safe
		assert( leftCell instanceof CnGridCell );
		assert( rightCell instanceof CnGridCell );
		
		split( (CnGridCell)leftCell, (CnGridCell)rightCell, dimension );
	}
	
	public void split( CnGridCell leftCell, CnGridCell rightCell, int dimension )
	{
		// just in case...
		assert( leftCell != null );
		assert( rightCell != null );
		
		// copy self to children
		leftCell.set( this );
		rightCell.set( this );
		
		// get the half deltas
		CnGridPoint halfDeltas = (CnGridPoint) getDeltas();
		halfDeltas.scale( 0.5 );
		
		leftCell.max.set( dimension, leftCell.max.get( dimension ) - halfDeltas.get( dimension ) );
		rightCell.min.set( dimension, rightCell.min.get( dimension ) + halfDeltas.get( dimension ) );
	}
	
	public void getACorner( Vector3 out, int i )
	{
		out.set(
			( i/2%2 == 0 ) ? min.phi : max.phi,
			( i%2 == 0 ) ? min.theta : max.theta,
			0
		);
	}
	
	public void getTCorner( Vector3 out, int i )
	{
		out.set(
			( i/2%2 == 0 ) ? min.x : max.x,
			( i%2 == 0 ) ? min.y : max.y,
			0
		);
	}
	
	@Override
	public CnGridPoint getDeltas( )
	{
		CnGridPoint temp = new CnGridPoint( max );
		temp.subtract( min );
		return temp;
	}
	
	@Override
	public String toString() {
	       return "[GridCell]\n"
	        + "\t[" + min.x + "," + max.x + "]\n"
	        + "\t[" + min.y + "," + max.y + "]\n"
	        + "\t[" + min.phi + "," + max.phi + "]\n"
	        + "\t[" + min.theta + "," + max.theta + "]\n";
	}
}
