/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.grid.base;

import java.util.ArrayList;
import java.util.List;

import edu.duke.donaldLab.jdshot.grid.DistanceRestraintReplicator;
import edu.duke.donaldLab.share.nmr.DistanceRestraint;
import edu.duke.donaldLab.share.protein.AtomAddressInternal;

public class NoeReplicatorBase implements DistanceRestraintReplicator
{
	/**************************
	 *   Data Members
	 **************************/
	
	private int m_numAdvance;
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	protected NoeReplicatorBase( int numAdvance )
	{
		m_numAdvance = numAdvance;
	}
	
	
	/**************************
	 *   Methods
	 **************************/

	@Override
	public List<DistanceRestraint<AtomAddressInternal>> replicate( List<DistanceRestraint<AtomAddressInternal>> restraints, int numSubunits )
	{
		// short circuit: if only 2 subunits, don't replicate
		if( numSubunits <= 2 )
		{
			return restraints;
		}
		
		int numRestraintsToAdd = numSubunits / m_numAdvance - 1;
		int numExpandedRestraints = restraints.size() * ( numRestraintsToAdd + 1 );
		
		// replicate the noes so each subunit around the ring is involved
		ArrayList<DistanceRestraint<AtomAddressInternal>> expandedRestraints = new ArrayList<DistanceRestraint<AtomAddressInternal>>( numExpandedRestraints );
		for( DistanceRestraint<AtomAddressInternal> noe : restraints )
		{
			// add the original restraint
			expandedRestraints.add( new DistanceRestraint<AtomAddressInternal>( noe ) );
			
			// duplicate this restraint and "rotate" it
			DistanceRestraint<AtomAddressInternal> newRestraint = null;
			for( int i=1; i<=numRestraintsToAdd; i++ )
			{
				newRestraint = new DistanceRestraint<AtomAddressInternal>( noe );				
				AtomAddressInternal.rotateSubunits( newRestraint.getLefts(), numSubunits, i * m_numAdvance );
				AtomAddressInternal.rotateSubunits( newRestraint.getRights(), numSubunits, i * m_numAdvance );
				expandedRestraints.add( newRestraint );
			}
		}
		
		// just in case
		assert( expandedRestraints.size() == numExpandedRestraints );
		
		return expandedRestraints;
	}
}
