/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.context;

import java.io.File;
import java.io.IOException;
import java.util.List;

import org.apache.log4j.Logger;

import edu.duke.donaldLab.jdshot.grid.GridPoint;
import edu.duke.donaldLab.jdshot.grid.SearchSpace;
import edu.duke.donaldLab.jdshot.grid.StructureGenerator;
import edu.duke.donaldLab.jdshot.grid.Symmetry;
import edu.duke.donaldLab.jdshot.search.CellChecker;
import edu.duke.donaldLab.share.geom.Sphere;
import edu.duke.donaldLab.share.geom.Vector3;
import edu.duke.donaldLab.share.io.Logging;
import edu.duke.donaldLab.share.mapping.NameMapper;
import edu.duke.donaldLab.share.mapping.NameScheme;
import edu.duke.donaldLab.share.nmr.DistanceRestraint;
import edu.duke.donaldLab.share.pdb.ProteinReader;
import edu.duke.donaldLab.share.protein.AtomAddressInternal;
import edu.duke.donaldLab.share.protein.Protein;
import edu.duke.donaldLab.share.protein.Subunit;
import edu.duke.donaldLab.share.protein.tools.MonomerCloner;
import edu.duke.donaldLab.share.protein.tools.ProteinGeometry;
import edu.duke.donaldLab.share.pseudoatoms.PseudoatomBuilder;

public class SymmetryContext
{
	/**************************
	 *   Data Members
	 **************************/
	
	private static final Logger m_log = Logging.getLog( SymmetryContext.class );
	
	private Symmetry m_symmetry;
	private Subunit m_monomer;
	private Subunit m_pseudoMonomer;
	private int m_numSubunits;
	private Sphere m_boundingSphere;
	private StructureGenerator m_structureGenerator;
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	public SymmetryContext( Symmetry symmetry, String monomerPath, int numSubunits )
	throws IOException
	{
		this( symmetry, new File( monomerPath ), numSubunits );
	}
	
	public SymmetryContext( Symmetry symmetry, File monomerFile, int numSubunits )
	throws IOException
	{
		// read the monomer and map names if needed
		Protein monomerProtein = new ProteinReader().read( monomerFile );
		NameMapper.ensureProtein( monomerProtein, NameScheme.New );
		
		// dump some stats
		m_log.info( String.format( "Loaded monomer %s with %d residues from:\n\t%s",
			monomerProtein.getName(),
			m_monomer.getResidues().size(),
			monomerFile.getAbsolutePath()
		) );
		
		init( symmetry, monomerProtein.getSubunit( 0 ), numSubunits );		
	}
	
	public SymmetryContext( Symmetry symmetry, Subunit subunit, int numSubunits )
	throws IOException
	{
		init( symmetry, subunit, numSubunits );
	}
	
	private void init( Symmetry symmetry, Subunit subunit, int numSubunits )
	throws IOException
	{
		m_monomer = subunit;
		m_boundingSphere = ProteinGeometry.getBoundingSphere( m_monomer );
		
		// create a monomer with pseudoatoms
		m_pseudoMonomer = new Subunit( m_monomer );
		PseudoatomBuilder.getInstance().build( m_pseudoMonomer );
		
		// save parameters
		m_symmetry = symmetry;
		m_numSubunits = numSubunits;
		
		// cache class instances
		m_structureGenerator = symmetry.newStructureGenerator( m_monomer );
	}
	
	
	/**************************
	 *   Accessors
	 **************************/
	
	public Symmetry getSymmetry( )
	{
		return m_symmetry;
	}
	
	public Subunit getMonomer( )
	{
		return m_monomer;
	}
	
	public Subunit getPseudoMonomer( )
	{
		return m_pseudoMonomer;
	}
	
	public int getNumSubunits( )
	{
		return m_numSubunits;
	}
	
	public Sphere getBoundingSphere( )
	{
		return m_boundingSphere;
	}
	
	
	/**************************
	 *   Methods
	 **************************/
	
	// UNDONE: implement print-stats methods
	
	public void normalizePosition( )
	{
		// translate the subunit such that the origin coincides with its bounding sphere
		Vector3 translation = new Vector3( m_boundingSphere.center );
		translation.negate();
		m_boundingSphere.center.add( translation );
		ProteinGeometry.translate( m_monomer, translation );
	}
	
	public Protein getOligomer( GridPoint point )
	{
		Protein oligomer = m_structureGenerator.getStructure( point, getNumSubunits() );
		return oligomer;
	}
	
	public Protein getOligomer( GridPoint point, Subunit monomer )
	{
		StructureGenerator structureGenerator = m_symmetry.newStructureGenerator( monomer );
		Protein oligomer = structureGenerator.getStructure( point, getNumSubunits() );
		return oligomer;
	}
	
	public Protein getClonedMonomer( )
	{
		return MonomerCloner.clone(
			m_pseudoMonomer,
			m_numSubunits
		);
	}
	
	public SearchSpace newSearchSpace( double maxDistance )
	{
		return m_symmetry.newSearchSpace(
			m_boundingSphere.radius,
			maxDistance,
			m_numSubunits
		);
	}
	
	public CellChecker newCellChecker( )
	{
		return m_symmetry.newCellChecker( this );
	}
	
	public List<DistanceRestraint<AtomAddressInternal>> replicateDistanceRestraints( List<DistanceRestraint<AtomAddressInternal>> restraints )
	{
		List<DistanceRestraint<AtomAddressInternal>> replicatedRestraints = m_symmetry.newNoeReplicator().replicate( restraints, getNumSubunits() ); 
		m_log.info( String.format( "Replicated %d restraints across %d subunits: %d restraints total",
			restraints.size(),
			getNumSubunits(),
			replicatedRestraints.size()
		) );
		return replicatedRestraints;
	}
}
