/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot;

import java.io.File;

import junit.framework.TestSuite;
import junit.textui.TestRunner;

import edu.duke.donaldLab.jdshot.analyze.AnalysisContext;
import edu.duke.donaldLab.jdshot.context.SymmetryContext;
import edu.duke.donaldLab.jdshot.grid.Symmetry;
import edu.duke.donaldLab.jdshot.search.SearchContext;
import edu.duke.donaldLab.jdshot.test.disco.*;
import edu.duke.donaldLab.jdshot.test.grid.*;
import edu.duke.donaldLab.jdshot.test.search.*;
import edu.duke.donaldLab.jdshot.test.analyze.*;
import edu.duke.donaldLab.share.io.ArgumentType;
import edu.duke.donaldLab.share.io.ArgumentsProcessor;
import edu.duke.donaldLab.share.protein.SubunitOrder;

public class TestMain extends TestSuite
{
	/**************************
	 *   Data Members
	 **************************/
	
	private static SearchContext m_searchContext = null;
	private static AnalysisContext m_analysisContext = null;
	private static SymmetryContext m_symmetryContext = null;
	private static File m_noesFile = null;
	
	
	/**************************
	 *   Static Methods
	 **************************/
	
	public static void main( String[] args )
	throws Exception
	{
		// process the arguments
		ArgumentsProcessor argproc = new ArgumentsProcessor();
		argproc.add( "symmetry", ArgumentType.String, "symmetry type (e.g., Cn, Dn)" );
		argproc.add( "monomerPath", ArgumentType.InFile, "path to the monomer protein" );
		argproc.add( "oligomerPath", ArgumentType.InFile, "path to the oligomer protein" );
		argproc.add( "noesPath", ArgumentType.InFile, "path to the noes" );
		argproc.add( "subunitOrder", ArgumentType.String, "ring order of the subunits e.g., ABCD" );
		argproc.process( args );
		
		// read the arguments for the monomer path and the noes path
		Symmetry symmetry = Symmetry.valueOf( argproc.getString( "symmetry" ) );
		File monomerFile = argproc.getFile( "monomerPath" );
		File oligomerFile = argproc.getFile( "oligomerPath" );
		File noesFile = argproc.getFile( "noesPath" );
		SubunitOrder subunitOrder = new SubunitOrder( argproc.getString( "subunitOrder" ) );
		
		// load a search context
		m_searchContext = new SearchContext();
		m_searchContext.load( symmetry, monomerFile, noesFile, subunitOrder );
		
		// load a symmetry context
		m_symmetryContext = new SymmetryContext( symmetry, monomerFile, subunitOrder.getNumSubunits() );
		
		// load an analysis context
		m_analysisContext = new AnalysisContext();
		m_analysisContext.load( oligomerFile, m_searchContext );
		
		// save stuff the test might want later
		m_noesFile = noesFile;
		
		// start the tests
		System.out.println( "Testing starting..." );
		TestRunner.run( getSuite() );
		System.out.println( "Testing complete!" );
	}
	
	public static SearchContext getSearchContext( )
	{
		return m_searchContext;
	}
	
	public static AnalysisContext getAnalysisContext( )
	{
		return m_analysisContext;
	}
	
	public static SymmetryContext getSymmetryContext( )
	{
		return m_symmetryContext;
	}
	
	public static File getNoesFile( )
	{
		return m_noesFile;
	}
	
	
	/**************************
	 *   Functions
	 **************************/
	
	private static TestSuite getSuite( )
	{
		TestSuite suite = new TestSuite();
		
		// grid
		suite.addTestSuite( TestDnGridPoint.class );
		suite.addTestSuite( TestDnGridCell.class );
		suite.addTestSuite( TestCellIO.class );
		suite.addTestSuite( TestAxes.class );
		
		switch( m_searchContext.getSymmetry() )
		{
			case Cn:
				// UNDONE: add tests here
			break;
			
			case Dn:
				suite.addTestSuite( TestDnCellSplitter.class );
				suite.addTestSuite( TestDnCellChecker.class );
				suite.addTestSuite( TestDnSearchSpace.class );
				suite.addTestSuite( TestDnDistanceRestraintReplicator.class );
			break;
		}
		
		// search
		suite.addTestSuite( TestSearchContext.class );
		
		// analyze
		suite.addTestSuite( TestCellAnalyzer.class );
		suite.addTestSuite( TestRestraintCalculator.class );
		suite.addTestSuite( TestRmsdCalculator.class );
		suite.addTestSuite( TestStructureGenerator.class );
		suite.addTestSuite( TestStructureAligner.class );
		
		// disco
		suite.addTestSuite( TestOrientationCalculator.class );
		suite.addTestSuite( TestAnnulusCalculator.class );
		suite.addTestSuite( TestPositionCalculator.class );
		suite.addTestSuite( TestArrangement.class );
		suite.addTestSuite( TestReferenceAnalyzer.class );
		suite.addTestSuite( TestViolationCalculator.class );
		
		return suite;
	}
}
