/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.share.test.nmr;

import java.util.List;

import edu.duke.donaldLab.share.mapping.NameMapper;
import edu.duke.donaldLab.share.mapping.NameScheme;
import edu.duke.donaldLab.share.nmr.ChemicalShift;
import edu.duke.donaldLab.share.nmr.ChemicalShiftMapper;
import edu.duke.donaldLab.share.nmr.ChemicalShiftReader;
import edu.duke.donaldLab.share.nmr.MappedChemicalShift;
import edu.duke.donaldLab.share.pdb.ProteinReader;
import edu.duke.donaldLab.share.protein.AminoAcid;
import edu.duke.donaldLab.share.protein.AtomAddressInternal;
import edu.duke.donaldLab.share.protein.Element;
import edu.duke.donaldLab.share.protein.Protein;
import edu.duke.donaldLab.share.pseudoatoms.PseudoatomBuilder;
import edu.duke.donaldLab.share.test.ExtendedTestCase;

public class TestChemicalShiftMapper extends ExtendedTestCase
{
	public void testMapping( )
	throws Exception
	{
		// read the protein
		Protein protein = new ProteinReader().read( "src/resources/test/1Q10.monomer.protein" );
		NameMapper.ensureProtein( protein, NameScheme.New );
		
		// read the shifts
		List<ChemicalShift> shifts = new ChemicalShiftReader().read( "src/resources/test/chemical.shifts" );
		NameMapper.ensureShifts( protein, shifts, NameScheme.New );
		
		List<MappedChemicalShift> mappedShifts = ChemicalShiftMapper.map( shifts, protein, true );
		assertEquals( 4, mappedShifts.size() );
		
		// make sure all shifts are mapped
		assertEquals( shifts.size(), mappedShifts.size() );
		for( int i=0; i<shifts.size(); i++ )
		{
			assertNotNull( "Shift not mapped: " + shifts.get( i ).toString(), mappedShifts.get( i ) );
		}
		
		ChemicalShift shift = null;
		MappedChemicalShift mappedShift = null;
		AtomAddressInternal address = null;
		
		// 43  4 LYS CE   C  41.8662 0.2  1 
		shift = shifts.get( 0 );
		mappedShift = mappedShifts.get( 0 );
		assertEquals( shift.getValue(), mappedShift.getValue() );
		assertEquals( shift.getError(), mappedShift.getError() );
		assertEquals( 1, mappedShift.getAddresses().size() );
		address = mappedShift.getAddresses().iterator().next();
		assertEquals( 4, protein.getResidue( address ).getNumber() );
		assertEquals( AminoAcid.Lysine, protein.getResidue( address ).getAminoAcid() );
		assertEquals( "CE", protein.getAtom( address ).getName() );
		
		// 124 12 LEU HA   H   5.0068 0.02 1 
		shift = shifts.get( 1 );
		mappedShift = mappedShifts.get( 1 );
		assertEquals( shift.getValue(), mappedShift.getValue() );
		assertEquals( shift.getError(), mappedShift.getError() );
		assertEquals( 1, mappedShift.getAddresses().size() );
		address = mappedShift.getAddresses().iterator().next();
		assertEquals( 12, protein.getResidue( address ).getNumber() );
		assertEquals( AminoAcid.Leucine, protein.getResidue( address ).getAminoAcid() );
		assertEquals( "HA", protein.getAtom( address ).getName() );
		
		// 422 45 TYR H    H   8.8528 0.02 1
		shift = shifts.get( 2 );
		mappedShift = mappedShifts.get( 2 );
		assertEquals( shift.getValue(), mappedShift.getValue() );
		assertEquals( shift.getError(), mappedShift.getError() );
		assertEquals( 1, mappedShift.getAddresses().size() );
		address = mappedShift.getAddresses().iterator().next();
		assertEquals( 45, protein.getResidue( address ).getNumber() );
		assertEquals( AminoAcid.Tyrosine, protein.getResidue( address ).getAminoAcid() );
		assertEquals( "H", protein.getAtom( address ).getName() );
		
		// 466 50 LYS CG   C  24.5176 0.2  1 
		shift = shifts.get( 3 );
		mappedShift = mappedShifts.get( 3 );
		assertEquals( shift.getValue(), mappedShift.getValue() );
		assertEquals( shift.getError(), mappedShift.getError() );
		assertEquals( 1, mappedShift.getAddresses().size() );
		address = mappedShift.getAddresses().iterator().next();
		assertEquals( 50, protein.getResidue( address ).getNumber() );
		assertEquals( AminoAcid.Lysine, protein.getResidue( address ).getAminoAcid() );
		assertEquals( "CG", protein.getAtom( address ).getName() );
	}
	
	public void testMappingPseudoatoms( )
	throws Exception
	{
		// read the protein
		Protein protein = new ProteinReader().read( "src/resources/test/1Q10.monomer.protein" );
		NameMapper.ensureProtein( protein, NameScheme.New );
		PseudoatomBuilder.getInstance().build( protein );
		
		// read the shifts
		List<ChemicalShift> shifts = new ChemicalShiftReader().read( "src/resources/test/chemical.pseudo.shifts" );
		NameMapper.ensureShifts( protein, shifts, NameScheme.New );
		PseudoatomBuilder.getInstance().buildShifts( protein, shifts );
		
		List<MappedChemicalShift> mappedShifts = ChemicalShiftMapper.map( shifts, protein, true );
		assertEquals( 2, mappedShifts.size() );
		
		ChemicalShift shift = null;
		MappedChemicalShift mappedShift = null;
		AtomAddressInternal address = null;
		
		// spot check a couple addresses
		
		// 8  1 MET HE   H   2.1100 0.02 1
		// should map to HE# / me
		shift = shifts.get( 0 );
		mappedShift = mappedShifts.get( 0 );
		assertEquals( shift.getValue(), mappedShift.getValue() );
		assertEquals( shift.getError(), mappedShift.getError() );
		assertEquals( 1, mappedShift.getAddresses().size() );
		address = mappedShift.getAddresses().iterator().next();
		assertEquals( 1, protein.getResidue( address ).getNumber() );
		assertEquals( AminoAcid.Methionine, protein.getResidue( address ).getAminoAcid() );
		assertEquals( "ME", protein.getAtom( address ).getName() );
		
		// 130 12 LEU HD2  H   0.1190 0.02 2  
		// should map to HD2# / MD2
		shift = shifts.get( 1 );
		mappedShift = mappedShifts.get( 1 );
		assertEquals( shift.getValue(), mappedShift.getValue() );
		assertEquals( shift.getError(), mappedShift.getError() );
		address = mappedShift.getAddresses().iterator().next();
		assertEquals( 1, mappedShift.getAddresses().size() );
		assertEquals( 12, protein.getResidue( address ).getNumber() );
		assertEquals( AminoAcid.Leucine, protein.getResidue( address ).getAminoAcid() );
		assertEquals( "MD2", protein.getAtom( address ).getName() );
	}
	
	public void testAssociatePairsDoesntCrash( )
	throws Exception
	{
		// read the protein
		Protein protein = new ProteinReader().read( "src/resources/test/1Q10.monomer.protein" );
		NameMapper.ensureProtein( protein, NameScheme.New );
		PseudoatomBuilder.getInstance().build( protein );
		
		// read the shifts
		List<ChemicalShift> shifts = new ChemicalShiftReader().read( "src/resources/test/1Q10.experimental.shift" );
		NameMapper.ensureShifts( protein, shifts, NameScheme.New );
		PseudoatomBuilder.getInstance().buildShifts( protein, shifts );
		
		List<ChemicalShift> hydrogenShifts = ChemicalShiftMapper.filter( shifts, Element.Hydrogen );
		List<ChemicalShift> carbonShifts = ChemicalShiftMapper.filter( shifts, Element.Carbon );
		List<MappedChemicalShift> mappedHydrogenShifts = ChemicalShiftMapper.map( hydrogenShifts, protein );
		List<MappedChemicalShift> mappedCarbonShifts = ChemicalShiftMapper.map( carbonShifts, protein );
		
		ChemicalShiftMapper.associatePairs( protein, mappedHydrogenShifts, mappedCarbonShifts, Element.Carbon );
	}
}
