/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.share.protein;

import edu.duke.donaldLab.share.io.HashCalculator;


public class AtomAddressReadable implements AtomAddress<AtomAddressReadable>, Comparable<AtomAddressReadable>
{
	/**************************
	 *   Data Members
	 **************************/
	
	private char m_subunitName;
	private int m_residueNumber;
	private String m_atomName;
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	public AtomAddressReadable( )
	{
		m_subunitName = '?';
		m_residueNumber = -1;
		m_atomName = null;
	}
	
	public AtomAddressReadable( char subunitName, int residueNumber, String atomName )
	{
		m_subunitName = Character.toUpperCase( subunitName );
		m_residueNumber = residueNumber;
		m_atomName = atomName.toUpperCase();
	}
	
	public AtomAddressReadable( AtomAddressReadable other )
	{
		m_subunitName = other.m_subunitName;
		m_residueNumber = other.m_residueNumber;
		m_atomName = other.m_atomName;
	}
	
	
	/**************************
	 *   Accessors
	 **************************/
	
	public char getSubunitName( )
	{
		return m_subunitName;
	}
	public void setSubunitName( char value )
	{
		m_subunitName = Character.toUpperCase( value );
	}
	public boolean hasSubunitName( )
	{
		return m_subunitName != '?';
	}
	
	public int getResidueNumber( )
	{
		return m_residueNumber;
	}
	public void setResidueNumber( int value )
	{
		m_residueNumber = value;
	}
	
	public String getAtomName( )
	{
		return m_atomName;
	}
	public void setAtomName( String value )
	{
		m_atomName = value.toUpperCase();
	}
	
	
	/**************************
	 *   Methods
	 **************************/
	
	public String toString( )
	{
		return m_subunitName + ":" + m_residueNumber + ":" + m_atomName;
	}
	
	public String toResidueAtomString( )
	{
		return m_residueNumber + ":" + m_atomName;
	}
	
	@Override
	public boolean isAmbiguous( )
	{
		return m_atomName.indexOf( '#' ) != -1 || m_atomName.indexOf( '*' ) != -1;
	}
	
	@Override
	public int compareTo( AtomAddressReadable other )
	{
		int diff = 0;
		
		diff = Character.valueOf( m_subunitName ).compareTo( other.m_subunitName );
		if( diff != 0 )
		{
			return diff;
		}
		
		diff = m_residueNumber - other.m_residueNumber;
		if( diff != 0 )
		{
			return diff;
		}
		
		diff = m_atomName.compareTo( other.m_atomName );
		if( diff != 0 )
		{
			return diff;
		}
		
		return 0;
	}
	
	@Override
	public boolean equals( Object other )
	{
		if( other == null )
		{
            return false;
		}
        if( other == this )
        {
            return true;
        }
        if( other.getClass() != getClass() )
        {
            return false;
        }
        
        return equals( (AtomAddressReadable)other );
	}
	
	public boolean equals( AtomAddressReadable other )
	{
		// NOTE: subunitName and atomName are always lower case!
		return
			m_subunitName == other.m_subunitName
			&& m_residueNumber == other.m_residueNumber
			&& m_atomName.equals( other.m_atomName );
	}
	
	@Override
	public int hashCode( )
	{
		return HashCalculator.combineHashes(
			Character.valueOf( m_subunitName ).hashCode(),
			m_residueNumber,
			m_atomName.hashCode()
		);
	}
	
	@Override
	public AtomAddressReadable newCopy( )
	{
		return new AtomAddressReadable( this );
	}
	
	@Override
	public boolean isSameSubunit( AtomAddressReadable other )
	{
		return m_subunitName == other.m_subunitName;
	}
}
