/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.share.nmr;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

import edu.duke.donaldLab.share.bond.Bond;
import edu.duke.donaldLab.share.bond.BondGraph;
import edu.duke.donaldLab.share.bond.BondGraphBuilder;
import edu.duke.donaldLab.share.mapping.AddressMapper;
import edu.duke.donaldLab.share.protein.AtomAddressInternal;
import edu.duke.donaldLab.share.protein.Element;
import edu.duke.donaldLab.share.protein.Protein;
import edu.duke.donaldLab.share.protein.AtomAddressReadable;

public class ChemicalShiftMapper
{
	/**************************
	 *   Static Methods
	 **************************/
	
	public static ArrayList<ChemicalShift> filter( List<ChemicalShift> shifts, Element element )
	{
		ArrayList<ChemicalShift> index = new ArrayList<ChemicalShift>();
		for( ChemicalShift shift : shifts )
		{
			if( shift.getElement() == element )
			{
				index.add( shift );
			}
		}
		return index;
	}
	
	public static ArrayList<MappedChemicalShiftPair> associatePairs( Protein protein, List<MappedChemicalShift> hydrogenShifts, List<MappedChemicalShift> heavyShifts, Element element )
	throws IOException
	{
		// build the bond graphs for the protein
		ArrayList<BondGraph> bondGraphs = BondGraphBuilder.getInstance().build( protein );
		
		// associate the pairs
		ArrayList<MappedChemicalShiftPair> pairs = new ArrayList<MappedChemicalShiftPair>();
		for( MappedChemicalShift hydrogenShift : hydrogenShifts )
		{
			/* NOTE:
				With the exception of combined methyls for VAL and LEU, all hydrogens should share
				the same bonded heavy atom. So we can just use the first address in a shift
			*/
			AtomAddressInternal hydrogenAddress = hydrogenShift.getAddresses().iterator().next();
			AtomAddressInternal heavyAddress = getHeavyAddress( protein, bondGraphs, hydrogenAddress );
			
			// if the heavy atom is not the desired element, skip
			if( protein.getAtom( heavyAddress ).getElement() != element )
			{
				continue;
			}
			
			// get the shift for the heavy address
			MappedChemicalShift heavyShift = getHeavyShift( heavyShifts, heavyAddress );
			if( heavyShift == null )
			{
				System.err.println( "Warning: Chemical Shift not found for heavy atom: " + AddressMapper.mapAddress( protein, heavyAddress ) );
				continue;
			}
			
			// build the pair
			pairs.add( new MappedChemicalShiftPair( hydrogenShift, heavyShift ) );
		}
		return pairs;
	}
	
	public static ArrayList<MappedChemicalShift> getOrphanedHydrogenShifts( Protein protein, List<MappedChemicalShift> hydrogenShifts, List<MappedChemicalShift> heavyShifts )
	throws IOException
	{
		// build the bond graphs for the protein
		ArrayList<BondGraph> bondGraphs = BondGraphBuilder.getInstance().build( protein );
		
		// associate the pairs
		ArrayList<MappedChemicalShift> orphanShifts = new ArrayList<MappedChemicalShift>();
		for( MappedChemicalShift hydrogenShift : hydrogenShifts )
		{
			AtomAddressInternal hydrogenAddress = hydrogenShift.getAddresses().iterator().next();
			AtomAddressInternal heavyAddress = getHeavyAddress( protein, bondGraphs, hydrogenAddress );
			if( getHeavyShift( heavyShifts, heavyAddress ) == null )
			{
				orphanShifts.add( hydrogenShift );
			}
		}
		return orphanShifts;
	}
	
	public static ArrayList<MappedChemicalShift> map( List<ChemicalShift> shifts, Protein protein )
	{
		return map( shifts, protein, false );
	}
	
	public static ArrayList<MappedChemicalShift> map( List<ChemicalShift> shifts, Protein protein, boolean addNulls )
	{
		ArrayList<MappedChemicalShift> mappedShifts = new ArrayList<MappedChemicalShift>();
		for( ChemicalShift shift : shifts )
		{
			MappedChemicalShift mappedShift = map( shift, protein );
			if( mappedShift != null || addNulls )
			{
				mappedShifts.add( mappedShift );
			}
		}
		return mappedShifts;
	}
	
	public static MappedChemicalShift map( ChemicalShift shift, Protein protein )
	{
		MappedChemicalShift mappedShift = new MappedChemicalShift();
		HashSet<AtomAddressInternal> mappedAddresses = new HashSet<AtomAddressInternal>( shift.getAddresses().size() );
		for( AtomAddressReadable shiftAddress : shift.getAddresses() )
		{
			mappedAddresses.addAll( AddressMapper.mapAddress( protein, shiftAddress ) );
		}
		mappedShift.setAddresses( mappedAddresses );
		mappedShift.setValue( shift.getValue() );
		mappedShift.setError( shift.getError() );
		
		if( mappedShift.getAddresses().size() <= 0 )
		{
			return null;
		}
		
		return mappedShift;
	}
	
	
	/**************************
	 *   Static Functions
	 **************************/
	
	private static AtomAddressInternal getHeavyAddress( Protein protein, List<BondGraph> bondGraphs, AtomAddressInternal hydrogenAddress )
	{
		// the hydrogen should be bonded to a single heavy-atom
		ArrayList<Bond> bonds = bondGraphs.get( hydrogenAddress.getSubunitId() ).getBonds( hydrogenAddress );
		assert( bonds.size() == 1 );
		return bonds.get( 0 ).getOtherAddress( hydrogenAddress );
	}
	
	private static MappedChemicalShift getHeavyShift( List<MappedChemicalShift> heavyShifts, AtomAddressInternal heavyAddress )
	{
		for( MappedChemicalShift heavyShift : heavyShifts )
		{
			if( heavyShift.getAddresses().contains( heavyAddress ) )
			{
				return heavyShift;
			}
		}
		return null;
	}
}
