/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.share.mapping;

import java.util.List;

import edu.duke.donaldLab.share.nmr.ChemicalShift;
import edu.duke.donaldLab.share.nmr.DistanceRestraint;
import edu.duke.donaldLab.share.nmr.Rdc;
import edu.duke.donaldLab.share.protein.AminoAcid;
import edu.duke.donaldLab.share.protein.Atom;
import edu.duke.donaldLab.share.protein.Protein;
import edu.duke.donaldLab.share.protein.AtomAddressReadable;
import edu.duke.donaldLab.share.protein.Residue;

public class NameMapper
{
	/**************************
	 *   Data Members
	 **************************/
	
	NameMapperProvider m_mapper;
	NameScheme m_source;
	NameScheme m_destination;
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	public NameMapper( NameScheme source, NameScheme destination )
	{
		m_mapper = NameMapperProvider.getInstance();
		m_source = source;
		m_destination = destination;
	}


	/**************************
	 *   Static Methods
	 **************************/
	
	public static void ensureProtein( Protein protein, NameScheme scheme )
	{
		NameMapper mapper = new NameMapper( NameScheme.getSchemeForProtein( protein ), scheme );
		if( mapper.hasMap() )
		{
			mapper.mapProtein( protein );
		}
	}
	
	public static void ensureDistanceRestraints( Protein protein, List<DistanceRestraint<AtomAddressReadable>> restraints, NameScheme scheme )
	{
		NameMapper mapper = new NameMapper( NameScheme.getSchemeForDistanceRestraints( restraints ), scheme );
		if( mapper.hasMap() )
		{
			mapper.mapDistanceRestraints( protein, restraints );
		}
	}
	
	public static void ensureShifts( Protein protein, List<ChemicalShift> shifts, NameScheme scheme )
	{
		NameMapper mapper = new NameMapper( NameScheme.getSchemeForShifts( shifts ), scheme );
		if( mapper.hasMap() )
		{
			mapper.mapShifts( protein, shifts );
		}
	}
	
	public static void ensureRdcs( Protein protein, List<Rdc<AtomAddressReadable>> rdcs, NameScheme scheme )
	{
		NameMapper mapper = new NameMapper( NameScheme.getSchemeForRdcs( rdcs ), scheme );
		if( mapper.hasMap() )
		{
			mapper.mapRdcs( protein, rdcs );
		}
	}
	

	/**************************
	 *   Methods
	 **************************/
	
	public boolean hasMap( )
	{
		return m_mapper.hasMap( m_source, m_destination );
	}

	public String mapName( AminoAcid aminoAcid, String atomName )
	{
		return m_mapper.mapName( m_source, m_destination, aminoAcid, atomName );
	}
	
	public void mapAtom( Residue residue, Atom atom )
	{
		m_mapper.mapAtom( m_source, m_destination, residue, atom );
	}
	
	public void mapProtein( Protein protein )
	{
		m_mapper.mapProtein( m_source, m_destination, protein );
	}
	
	public void mapDistanceRestraints( Protein protein, List<DistanceRestraint<AtomAddressReadable>> restraints )
	{
		m_mapper.mapDistanceRestraints( m_source, m_destination, protein, restraints );
	}
	
	public void mapAddress( Protein protein, AtomAddressReadable address )
	{
		m_mapper.mapAddress( m_source, m_destination, protein, address );
	}
	
	public void mapShifts( Protein protein, List<ChemicalShift> shifts )
	{
		m_mapper.mapShifts( m_source, m_destination, protein, shifts );
	}
	
	public void mapRdcs( Protein protein, List<Rdc<AtomAddressReadable>> rdcs )
	{
		m_mapper.mapRdcs( m_source, m_destination, protein, rdcs );
	}
}
