/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.search;

import java.util.ArrayList;

import edu.duke.donaldLab.jdshot.grid.GridCell;
import edu.duke.donaldLab.share.perf.Progress;

public class SearchProcessorSynchronizer
{
	/**************************
	 *   Data Members
	 **************************/
	
	private Progress m_progress;
	private int m_numOldCells;
	private ArrayList<GridCell> m_newCells;
	private Consistency m_consistency;
	private int m_nextIndex;
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	public SearchProcessorSynchronizer( Progress progress, int numOldCells, ArrayList<GridCell> newCells, Consistency consistency )
	{
		// save parameters
		m_progress = progress;
		m_numOldCells = numOldCells;
		m_newCells = newCells;
		m_consistency = consistency;
		
		// init data members
		m_nextIndex = 0;
	}
	
	
	/**************************
	 *   Methods
	 **************************/

	public synchronized int[] getNextIndices( int numIndices )
	{
		// are there any indices left?
		if( m_nextIndex >= m_numOldCells )
		{
			return null;
		}
		
		// update the number of indices if needed
		if( numIndices + m_nextIndex > m_numOldCells )
		{
			numIndices = m_numOldCells - m_nextIndex;
		}
		
		// return a list of the indices
		int[] indices = new int[numIndices];
		for( int i=0; i<numIndices; i++ )
		{
			indices[i] = m_nextIndex++;
		}
		
		return indices;
	}
	
	public synchronized void saveCells( ArrayList<GridCell> cells, int[] cellIndices )
	{
		// just in case...
		assert( cells.size() == cellIndices.length );
		
		for( int i=0; i<cells.size(); i++ )
		{
			m_newCells.get( cellIndices[i] ).set( cells.get( i ) );
		}
		
		m_progress.incrementProgress( cells.size() );
	}
	
	public synchronized void mergeConsistency( Consistency consistency, int startIndex )
	{
		m_consistency.update( consistency, startIndex );
	}
	
	public synchronized void markCellsInconsistent( ArrayList<Integer> cellIndices )
	{
		for( int i=0; i<cellIndices.size(); i++ )
		{
			m_consistency.markCellInconsistent( cellIndices.get( i ) );
		}
	}
	
	public synchronized void markRestraintsConsistent( ArrayList<Integer> restraintIndices, ArrayList<Integer> assignmentIndices )
	{
		// just in case...
		assert( restraintIndices.size() == assignmentIndices.size() );
		
		for( int i=0; i<restraintIndices.size(); i++ )
		{
			m_consistency.markRestraintConsistent( restraintIndices.get( i ), assignmentIndices.get( i ) );
		}
	}
}
