/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.grid.dn;

import edu.duke.donaldLab.jdshot.context.SymmetryContext;
import edu.duke.donaldLab.jdshot.grid.Axes;
import edu.duke.donaldLab.jdshot.grid.GridCell;
import edu.duke.donaldLab.jdshot.grid.base.CellCheckerBase;
import edu.duke.donaldLab.jdshot.search.BoundContext;
import edu.duke.donaldLab.jdshot.search.CellChecker;
import edu.duke.donaldLab.share.geom.AxisAlignedBox;
import edu.duke.donaldLab.share.geom.Intersect;
import edu.duke.donaldLab.share.geom.Sphere;
import edu.duke.donaldLab.share.geom.Vector3;
import edu.duke.donaldLab.share.math.Matrix3;
import edu.duke.donaldLab.share.nmr.Assignment;
import edu.duke.donaldLab.share.protein.Atom;
import edu.duke.donaldLab.share.protein.AtomAddressInternal;

public class DnCellChecker extends CellCheckerBase implements CellChecker
{
	/**************************
	 *   Definitions
	 **************************/
	
	public class CheckInfo
	{
		public double diagonalSq;
		public double epsilon;
	}
	
	
	/**************************
	 *   Data Members
	 **************************/
	
	private SymmetryContext m_symmetryContext;
	private CheckInfo m_checkInfo;
	
	
	/**************************
	 *   Constructors
	 **************************/
	
	public DnCellChecker( SymmetryContext symmetryContext )
	{
		m_symmetryContext = symmetryContext;
		m_checkInfo = new CheckInfo();
	}
	
	
	/**************************
	 *   Methods
	 **************************/
	
	public CheckInfo getCheckInfo( )
	{
		return m_checkInfo;
	}
	
	@Override
	public boolean isConsistent( GridCell cell, double maxDistance, Assignment<AtomAddressInternal> assignment )
	{
		// check the cast...
		assert( cell instanceof DnGridCell );
		
		return isConsistent( (DnGridCell)cell, maxDistance, assignment );
	}
	
	public boolean isConsistent( DnGridCell cell, double maxDistance, Assignment<AtomAddressInternal> assignment )
	{
		// allocate memory
		BoundContext boundContext = new BoundContext( DnGridCell.NumTCorners );
		AxisAlignedBox bound = new AxisAlignedBox();
		Sphere sphere = new Sphere();
		
		// get the atom positions
		Atom leftAtom = m_symmetryContext.getPseudoMonomer().getAtom( assignment.getLeft() );
		Atom rightAtom = m_symmetryContext.getPseudoMonomer().getAtom( assignment.getRight() );
		
		// UNDONE: find a way to get the restraint type without using subunit order
		/*
		// get the appropriate bound context
		boundContext.type = getSubunitType( assignment.getRight().getSubunitId(), m_symmetryContext.getSubunitOrder() );
		getBoundContext( boundContext, cell, rightAtom );
		
		// just in case...
		assert( assignment.getLeft().getSubunitId() == getSubunitId( SubunitType.Main, m_symmetryContext.getSubunitOrder() ) );
		assert( boundContext.type != SubunitType.Unknown ) : "Restraint is not intersubunit!";
		*/
		// get the bound
		getBoundingBox( bound, boundContext );
		
		// calculate the check info
		m_checkInfo.diagonalSq = bound.getDiagonalSquared();
		m_checkInfo.epsilon = boundContext.epsilon;
		
		// perform intersection test
		sphere.set( leftAtom.getPosition(), maxDistance );
		return Intersect.isOverlapping( sphere, bound );
	}
	
	
	/**************************
	 *   Functions
	 **************************/
	
	protected void getBoundContext( BoundContext boundContext, DnGridCell cell, Atom atom )
	{
		// allocate memory
		Vector3 corner = new Vector3();
		Vector3 temp = new Vector3();
		Matrix3 tempMatrixA = new Matrix3();
		Matrix3 tempMatrixB = new Matrix3();
		
		// calculate q
		boundContext.q.set( atom.getPosition() );
		
		double avgPhi = ( cell.max.phi + cell.min.phi ) / 2.0;
		double avgTheta = ( cell.max.theta + cell.min.theta ) / 2.0;
		double avgRho = ( cell.max.rho + cell.min.rho ) / 2.0;
		double halfBeta = 2.0 * Math.PI / (double)m_symmetryContext.getNumSubunits();
		
		switch( boundContext.type )
		{
			case Intradimer:
				// calculate a from f
				Axes.getFlipVector( boundContext.a, avgPhi, avgTheta, avgRho, tempMatrixA );
			break;
			
			case InterdimerStandard:
				// calculate a from c
				Axes.getCVector( boundContext.a, avgPhi, avgTheta, avgRho, halfBeta, tempMatrixA, tempMatrixB );
			break;
			
			case InterdimerGoofy:
				// calculate a from r
				Axes.getRotationVector( boundContext.a, avgPhi, avgTheta, tempMatrixA );
			break;
		}
		
		// calculate alpha... hah, yeah... calculate
		boundContext.alpha = Math.PI;
		
		// calculate epsilon (need to get the max dist of all corners of A)
		double distsq = 0.0;
		double esq = 0.0;
		for( int i=0; i<DnGridCell.NumACorners; i++ )
		{
			cell.getACorner( corner, i );
			
			switch( boundContext.type )
			{
				case Intradimer:
					Axes.getFlipVector( temp, corner.x, corner.y, corner.z, tempMatrixA );
					distsq = boundContext.a.getSquaredDistance( temp );
				break;
				
				case InterdimerStandard:
					Axes.getCVector( temp, corner.x, corner.y, corner.z, halfBeta, tempMatrixA, tempMatrixB );
					distsq = boundContext.a.getSquaredDistance( temp );
				break;
				
				case InterdimerGoofy:
					Axes.getRotationVector( temp, corner.x, corner.y, tempMatrixA );
					distsq = boundContext.a.getSquaredDistance( temp );
				break;
			}
			
			if( distsq > esq )
			{
				esq = distsq;
			}
		}
		boundContext.epsilon = Math.sqrt( esq );
		
		// calculate {t_i}
		for( int i=0; i<DnGridCell.NumTCorners; i++ )
		{
			cell.getTCorner( boundContext.corners[i], i );
		}
	}
}
