/*******************************************************************************
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * Contact Info:
 * 	Bruce Donald
 * 	Duke University
 * 	Department of Computer Science
 * 	Levine Science Research Center (LSRC)
 * 	Durham
 * 	NC 27708-0129 
 * 	USA
 * 	brd@cs.duke.edu
 * 
 * Copyright (C) 2011 Jeffrey W. Martin and Bruce R. Donald
 * 
 * <signature of Bruce Donald>, April 2011
 * Bruce Donald, Professor of Computer Science
 ******************************************************************************/


package edu.duke.donaldLab.jdshot.chart;

import java.awt.Graphics2D;
import java.awt.geom.Area;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Rectangle2D;
import java.util.List;

import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.plot.XYPlot;
import org.jfree.data.Range;

import edu.duke.donaldLab.jdshot.disco.cgal.Circle;

public class CircleSubrenderer implements Subrenderer
{
	@Override
	@SuppressWarnings( "unchecked" )
	public Range findDomainBounds( GeometryDataset dataset, int series )
	{
		List<Circle> circles = (List<Circle>)dataset.getData( series );
		
		// for each circle...
		double lower = Double.POSITIVE_INFINITY;
		double upper = Double.NEGATIVE_INFINITY;
		for( Circle circle : circles )
		{
			double x = circle.getCenter().x;
			double radius = Math.sqrt( circle.getSquaredRadius() );
			
			lower = Math.min( lower, x - radius );
			upper = Math.max( upper, x + radius );
		}
        
        return new Range( lower, upper );
	}
	
	@Override
	@SuppressWarnings( "unchecked" )
	public Range findRangeBounds( GeometryDataset dataset, int series )
	{
		List<Circle> circles = (List<Circle>)dataset.getData( series );
		
		// for each circle...
		double lower = Double.POSITIVE_INFINITY;
		double upper = Double.NEGATIVE_INFINITY;
		for( Circle circle : circles )
		{
			double y = circle.getCenter().y;
			double radius = Math.sqrt( circle.getSquaredRadius() );
			
			lower = Math.min( lower, y - radius );
			upper = Math.max( upper, y + radius );
		}
        
        return new Range( lower, upper );
	}
	
	@Override
	public void drawItem( Graphics2D g2, Rectangle2D dataArea, XYPlot plot,
		ValueAxis domainAxis, ValueAxis rangeAxis, GeometryDataset dataset,
		int series, int item, GeometryRenderer renderer )
	{
		Circle circle = (Circle)dataset.getData( series ).get( item );
		
		// get the circle parameters
		double x = circle.getCenter().x;
		double y = circle.getCenter().y;
		double radius = Math.sqrt( circle.getSquaredRadius() );
		
		// calculate the coordinate transform
		double tranX = domainAxis.valueToJava2D( x, dataArea, plot.getDomainAxisEdge() );
		double tranXZero = domainAxis.valueToJava2D( 0.0, dataArea, plot.getDomainAxisEdge() );
		double tranXRadius = Math.abs( domainAxis.valueToJava2D( radius, dataArea, plot.getDomainAxisEdge() ) - tranXZero );
		
		double tranY = rangeAxis.valueToJava2D( y, dataArea, plot.getRangeAxisEdge() );
		double tranYZero = rangeAxis.valueToJava2D( 0.0, dataArea, plot.getRangeAxisEdge() );
		double tranYRadius = Math.abs( rangeAxis.valueToJava2D( radius, dataArea, plot.getRangeAxisEdge() ) - tranYZero );
		
		Ellipse2D ellipse = new Ellipse2D.Double(
			tranX - tranXRadius,
			tranY - tranYRadius,
			tranXRadius * 2.0,
			tranYRadius * 2.0
		);
		
		// do we need to draw the ellipse?
		Area area = new Area( ellipse );
		if( !area.intersects( dataArea ) || area.contains( dataArea ) )
        {
        	return;
        }
		
		// draw the ellipse
		if( renderer.getItemFillPaint( series, item ) != null )
		{
			g2.setPaint( renderer.getItemFillPaint( series, item ) );
			g2.fill( ellipse );
		}
		if( renderer.getItemOutlinePaint( series, item ) != null )
		{
			g2.setPaint( renderer.getItemOutlinePaint( series, item ) );
			g2.setStroke( renderer.getItemOutlineStroke( series, item ) );
			g2.draw( ellipse );
		}
	}
}
